# -*- coding: utf-8 -*-
"""
    pyvisa.ctwrapper.highlevel
    ~~~~~~~~~~~~~~~~~~~~~~~~~~

    Highlevel wrapper of the VISA Library.

    This file is part of PyVISA.

    :copyright: 2014 by PyVISA Authors, see AUTHORS for more details.
    :license: MIT, see LICENSE for more details.
"""

from __future__ import division, unicode_literals, print_function, absolute_import

import warnings

from pyvisa import constants, errors, highlevel, logger
from pyvisa.compat import integer_types, OrderedDict

from .cthelper import Library, find_library
from . import functions


def add_visa_methods(aclass):
    for method in functions.visa_functions:
        setattr(aclass, method, getattr(functions, method))
    return aclass


def _args_to_str(args):
    out = []
    for arg in args:
        try:
            # noinspection PyProtectedMember
            out.append(str(arg._obj))
        except AttributeError:
            out.append(arg)
    return tuple(out)


@add_visa_methods
class NIVisaLibrary(highlevel.VisaLibraryBase):
    """High level NI-VISA Library wrapper using ctypes.

    The easiest way to instantiate the library is to let `pyvisa` find the
    right one for you. This looks first in your configuration file (~/.pyvisarc).
    If it fails, it uses `ctypes.util.find_library` to try to locate a library
    in a way similar to what the compiler does:

       >>> visa_library = NIVisaLibrary()

    But you can also specify the path:

        >>> visa_library = NIVisaLibrary('/my/path/visa.so')

    :param library_path: path of the VISA library.
    """

    @staticmethod
    def get_library_paths():
        """Return a tuple of possible library paths.

        :rtype: tuple
        """

        from ..util import LibraryPath, read_user_library_path

        user_lib = read_user_library_path()
        tmp = [find_library(library_path)
               for library_path in ('visa', 'visa32', 'visa32.dll')]

        tmp = [LibraryPath(library_path)
               for library_path in tmp
               if library_path is not None]

        logger.debug('Automatically found library files: %s' % tmp)

        if user_lib:
            user_lib = LibraryPath(user_lib, 'user')
            try:
                tmp.remove(user_lib)
            except ValueError:
                pass
            tmp.insert(0, user_lib)

        return tuple(tmp)

    @staticmethod
    def get_debug_info():
        """Return a list of lines with backend info.
        """
        from pyvisa import __version__
        d = OrderedDict()
        d['Version'] = '%s (bundled with PyVISA)' % __version__

        paths = NIVisaLibrary.get_library_paths()

        for ndx, visalib in enumerate(paths, 1):
            nfo = OrderedDict()
            nfo['found by'] = visalib.found_by
            nfo['bitness'] = visalib.bitness
            try:
                lib = NIVisaLibrary(visalib)
                sess, _ = lib.open_default_resource_manager()
                print(sess)
                nfo['Vendor'] = str(lib.get_attribute(sess, constants.VI_ATTR_RSRC_MANF_NAME)[0])
                nfo['Impl. Version'] = str(lib.get_attribute(sess, constants.VI_ATTR_RSRC_IMPL_VERSION)[0])
                nfo['Spec. Version'] = str(lib.get_attribute(sess, constants.VI_ATTR_RSRC_SPEC_VERSION)[0])
                lib.close(sess)
            except Exception as e:
                e = str(e)
                if 'No matching architecture' in e:
                    nfo['Could not get more info'] = 'Interpreter and library have different bitness.'
                else:
                    nfo['Could not get more info'] = str(e).split('\n')

            d['#%d: %s' % (ndx, visalib)] = nfo

        if not paths:
            d['Binary library'] = 'Not found'

        return d

    def _init(self):
        try:
            lib = Library(self.library_path)
        except OSError as exc:
            raise errors.LibraryError.from_exception(exc, self.library_path)

        self.lib = lib

        # Set the argtypes, restype and errcheck for each function
        # of the visa library. Additionally store in `_functions` the
        # name of the functions.
        functions.set_signatures(self.lib, errcheck=self._return_handler)

        # Set the library functions as attributes of the object.
        for method_name in getattr(self.lib, '_functions', []):
            setattr(self, method_name, getattr(self.lib, method_name))

    def _return_handler(self, ret_value, func, arguments):
        """Check return values for errors and warnings.
        """

        logger.debug('%s%s -> %r',
                     func.__name__, _args_to_str(arguments), ret_value,
                     extra=self._logging_extra)

        try:
            ret_value = constants.StatusCode(ret_value)
        except ValueError:
            pass

        self._last_status = ret_value

        # The first argument of almost all registered visa functions is a session.
        # We store the error code per session
        session = None
        if func.__name__ not in ('viFindNext', ):
            try:
                session = arguments[0]
            except KeyError:
                raise Exception('Function %r does not seem to be a valid '
                                'visa function (len args %d)' % (func, len(arguments)))

            # Functions that use the first parameter to get a session value.
            if func.__name__ in ('viOpenDefaultRM', ):
                # noinspection PyProtectedMember
                session = session._obj.value

            if isinstance(session, integer_types):
                self._last_status_in_session[session] = ret_value
            else:
                # Functions that might or might have a session in the first argument.
                if func.__name__ not in ('viClose', 'viGetAttribute', 'viSetAttribute', 'viStatusDesc'):
                    raise Exception('Function %r does not seem to be a valid '
                                    'visa function (type args[0] %r)' % (func, type(session)))

        if ret_value < 0:
            raise errors.VisaIOError(ret_value)

        if ret_value in self.issue_warning_on:
            if session and ret_value not in self._ignore_warning_in_session[session]:
                warnings.warn(errors.VisaIOWarning(ret_value), stacklevel=2)

        return ret_value


