import json
import logging
import threading
from datetime import datetime, timedelta
import time

try:
    from Queue import Queue
except:
    from queue import Queue
from jsonpickle.pickler import Pickler
from concurrent.futures import Future

from wshubsapi.connected_clients_holder import ConnectedClientsHolder
from wshubsapi.function_message import FunctionMessage
from wshubsapi.utils import set_serializer_date_handler, serialize_message
from wshubsapi.messages_received_queue import MessagesReceivedQueue
# do not remove this line (hubs inspector needs to find it)
from wshubsapi import utils_api_hub, asynchronous

log = logging.getLogger(__name__)
__author__ = 'Jorge Garcia Irazabal'

_DEFAULT_PICKER = Pickler(max_depth=5, max_iter=80, make_refs=False)

set_serializer_date_handler()  # todo move this


class HubsApiException(Exception):
    pass


class CommEnvironment(object):
    def __init__(self, max_workers=MessagesReceivedQueue.DEFAULT_MAX_WORKERS,
                 unprovided_id_template="UNPROVIDED__{}",
                 serialization_max_depth=5, serialization_max_iter=80,
                 client_function_timeout=5):
        self.lock = threading.Lock()
        self.available_unprovided_ids = list()
        self.unprovided_id_template = unprovided_id_template
        self.last_provided_id = 0
        self.message_received_queue = MessagesReceivedQueue(self, max_workers)
        self.message_received_queue.start_threads()
        self.all_connected_clients = ConnectedClientsHolder.all_connected_clients
        self.serialization_args = dict(max_depth=serialization_max_depth, max_iter=serialization_max_iter)
        self.client_function_timeout = client_function_timeout
        self.__last_client_message_id = 0
        self.__new_client_message_id_lock = threading.Lock()
        self.__futures_buffer = {}
        """:type : dict[int, Future, datetime]"""

    def get_unprovided_id(self):
        if len(self.available_unprovided_ids) > 0:
            return self.available_unprovided_ids.pop(0)
        while self.unprovided_id_template.format(self.last_provided_id) in self.all_connected_clients:
            self.last_provided_id += 1
        return self.unprovided_id_template.format(self.last_provided_id)

    def on_opened(self, client, id_=None):
        with self.lock:
            if id_ is None or id_ in self.all_connected_clients:
                client.ID = self.get_unprovided_id()
            else:
                client.ID = id_
            ConnectedClientsHolder.append_client(client)
            return client.ID

    def on_message(self, client, msg_str):
        try:
            msg_str = msg_str if isinstance(msg_str, str) else msg_str.encode("utf-8")
            msg_obj = json.loads(msg_str)
            if "replay" not in msg_obj:
                self.__on_replay(client, msg_str, msg_obj)
            else:
                self.__on_replayed(msg_obj)

        except Exception as e:
            self.on_error(client, e)

    def on_async_message(self, client, message):
        self.message_received_queue.put((message, client))

    def on_closed(self, client):
        """:type client: wshubsapi.connected_client.ConnectedClient"""
        ConnectedClientsHolder.pop_client(client.ID)
        client.api_is_closed = True

    def on_error(self, client, exception):
        log.exception("Error parsing message")

    def replay(self, client, replay, origin_message):
        """
        :type client: wshubsapi.connected_client.ConnectedClient
        :param replay: serialized object to be sent as a replay of a message received
        :param origin_message: Message received (provided for overridden functions)
        """
        client.api_write_message(serialize_message(self.serialization_args, replay))

    def get_new_clients_future(self):
        with self.__new_client_message_id_lock:
            self.__last_client_message_id += 1
            id_ = self.__last_client_message_id
            self.__futures_buffer[id_] = Future()
        return self.__futures_buffer[id_], id_

    def close(self, **kwargs):
        self.message_received_queue.executor.shutdown(**kwargs)

    def __on_time_out(self, id_):
        with self.__new_client_message_id_lock:
            if id_ in self.__futures_buffer:
                future = self.__futures_buffer.pop(id_)
                future.set_exception(HubsApiException("Timeout exception"))

    def __on_replay(self, client, msg_str, msg_obj):
        hub_function = FunctionMessage(msg_obj, client)
        replay = hub_function.call_function()
        if replay is not None:
            self.replay(client, replay, msg_str)

    def __on_replayed(self, msg_obj):
        future = self.__futures_buffer.pop(msg_obj["ID"], None)
        if future is not None:
            if msg_obj["success"]:
                future.set_result(msg_obj["replay"])
            else:
                future.set_exception(Exception(msg_obj["replay"]))


