# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

# internal
import stix
import stix.utils as utils
from stix.common import vocabs
from stix.common import StructuredTextList, VocabString

# bindings
import stix.bindings.ttp as ttp_binding

from mixbox import fields, entities


class MalwareInstance(stix.Entity):
    _binding = ttp_binding
    _binding_class = _binding.MalwareInstanceType
    _namespace = "http://stix.mitre.org/TTP-1"
    _XSI_TYPE = None # defined by subclasses

    id_ = fields.IdField("id")
    idref = fields.IdrefField("idref")
    title = fields.TypedField("Title")
    descriptions = fields.TypedField("Description", type_="stix.common.StructuredTextList")
    short_descriptions = fields.TypedField("Short_Description", type_="stix.common.StructuredTextList")
    names = vocabs.VocabField("Name", type_=VocabString, multiple=True, key_name="names")
    types = vocabs.VocabField("Type", type_=vocabs.MalwareType, multiple=True, key_name="types")

    def __init__(self, id_=None, idref=None, title=None, description=None, short_description=None):
        super(MalwareInstance, self).__init__()
        self.id_ = id_
        self.idref = idref
        self.title = title
        self.description = StructuredTextList(description)
        self.short_description = StructuredTextList(short_description)

    @property
    def description(self):
        """A single description about the contents or purpose of this object.

        Default Value: ``None``

        Note:
            If this object has more than one description set, this will return
            the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`
        """
        if self.descriptions is None:
            self.descriptions = StructuredTextList()
        return next(iter(self.descriptions), None)

    @description.setter
    def description(self, value):
        self.descriptions = value

    def add_description(self, description):
        """Adds a description to the ``descriptions`` collection.

        This is the same as calling "foo.descriptions.add(bar)".
        """
        self.descriptions.add(description)

    @property
    def short_description(self):
        """A single short description about the contents or purpose of this
        object.

        Default Value: ``None``

        Note:
            If this object has more than one short description set, this will
            return the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`
        """
        if self.short_descriptions is None:
            self.short_descriptions = []
        return next(iter(self.short_descriptions), None)

    @short_description.setter
    def short_description(self, value):
        self.short_descriptions = value

    def add_short_description(self, description):
        """Adds a description to the ``short_descriptions`` collection.

        This is the same as calling "foo.short_descriptions.add(bar)".
        """
        self.short_descriptions.add(description)
        
    def add_name(self, name):
        self.names.append(name)

    def add_type(self, type_):
        self.types.append(type_)

    @staticmethod
    def lookup_class(xsi_type):
        if not xsi_type:
            raise ValueError("xsi:type is required")

        return stix.lookup_extension(xsi_type)


class MalwareInstanceFactory(entities.EntityFactory):
    @classmethod
    def entity_class(cls, key):
        return stix.lookup_extension(key, default=MalwareInstance)

# Backwards compatibility
add_extension = stix.add_extension
