# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

# external
from mixbox.cache import Cached

# internal
import stix
import stix.utils as utils
from stix.common import vocabs
from stix.common import StructuredTextList, VocabString

# bindings
import stix.bindings.ttp as ttp_binding


class MalwareInstance(Cached, stix.Entity):
    _binding = ttp_binding
    _binding_class = _binding.MalwareInstanceType
    _namespace = "http://stix.mitre.org/TTP-1"

    def __init__(self, id_=None, idref=None, title=None, description=None, short_description=None):
        self.id_ = id_
        self.idref = idref
        self.title = title
        self.description = description
        self.short_description = short_description
        self.names = None
        self.types = None

    @property
    def id_(self):
        """The ``id_`` property serves as an identifier.

        Default Value: ``None``

        Note:
            Both the ``id_`` and ``idref`` properties cannot be set at the
            same time. **Setting one will unset the other!**

        Returns:
            A string id.

        """
        return self._id

    @id_.setter
    def id_(self, value):
        if not value:
            self._id = None
        else:
            self._id = value
            self.idref = None

    @property
    def idref(self):
        """The ``idref`` property must be set to the ``id_`` value of another
        object instance of the same type. An idref does not need to resolve to
        a local object instance.

        Default Value: ``None``.

        Note:
            Both the ``id_`` and ``idref`` properties cannot be set at the
            same time. **Setting one will unset the other!**

        Returns:
            The value of the ``idref`` property

        """
        return self._idref

    @idref.setter
    def idref(self, value):
        if not value:
            self._idref = None
        else:
            self._idref = value
            self.id_ = None  # unset id_ if idref is present

    @property
    def title(self):
        return self._title

    @title.setter
    def title(self, value):
        self._title = value

    @property
    def description(self):
        """A single description about the contents or purpose of this object.

        Default Value: ``None``

        Note:
            If this object has more than one description set, this will return
            the description with the lowest ordinality value.

        Returns:
            An instance of
            :class:`.StructuredText`

        """
        return next(iter(self.descriptions), None)

    @description.setter
    def description(self, value):
        self.descriptions = value

    @property
    def descriptions(self):
        """A :class:`.StructuredTextList` object, containing descriptions about
        the purpose or intent of this object.

        This is typically used for the purpose of providing multiple
        descriptions with different classificaton markings.

        Iterating over this object will yield its contents sorted by their
        ``ordinality`` value.

        Default Value: Empty :class:`.StructuredTextList` object.

        Note:
            IF this is set to a value that is not an instance of
            :class:`.StructuredText`, an effort will ne made to convert it.
            If this is set to an iterable, any values contained that are not
            an instance of :class:`.StructuredText` will be be converted.

        Returns:
            An instance of
            :class:`.StructuredTextList`

        """
        return self._description

    @descriptions.setter
    def descriptions(self, value):
        self._description = StructuredTextList(value)

    def add_description(self, description):
        """Adds a description to the ``descriptions`` collection.

        This is the same as calling "foo.descriptions.add(bar)".

        """
        self.descriptions.add(description)

    @property
    def short_description(self):
        """"A single short description about the contents or purpose of this
        object.

        Default Value: ``None``

        Note:
            If this object has more than one short description set, this will
            return the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`

        """
        return next(iter(self.short_descriptions), None)

    @short_description.setter
    def short_description(self, value):
        self.short_descriptions = value

    @property
    def short_descriptions(self):
        """A :class:`.StructuredTextList` object, containing short descriptions
        about the purpose or intent of this object.

        This is typically used for the purpose of providing multiple
        short descriptions with different classificaton markings.

        Iterating over this object will yield its contents sorted by their
        ``ordinality`` value.

        Default Value: Empty :class:`.StructuredTextList` object.

        Note:
            IF this is set to a value that is not an instance of
            :class:`.StructuredText`, an effort will ne made to convert it.
            If this is set to an iterable, any values contained that are not
            an instance of :class:`.StructuredText` will be be converted.

        Returns:
            An instance of :class:`.StructuredTextList`

        """
        return self._short_description

    @short_descriptions.setter
    def short_descriptions(self, value):
        self._short_description = StructuredTextList(value)

    def add_short_description(self, description):
        """Adds a description to the ``short_descriptions`` collection.

        This is the same as calling "foo.short_descriptions.add(bar)".

        """
        self.short_descriptions.add(description)

    @property
    def names(self):
        return self._names

    @names.setter
    def names(self, value):
        self._names = MalwareNames(value)

    def add_name(self, name):
        self._names.append(name)

    @property
    def types(self):
        return self._types

    @types.setter
    def types(self, value):
        self._types = MalwareTypes(value)

    def add_type(self, type_):
        self._types.append(type_)

    @staticmethod
    def lookup_class(xsi_type):
        if not xsi_type:
            raise ValueError("xsi:type is required")

        return stix.lookup_extension(xsi_type)

    def to_obj(self, return_obj=None, ns_info=None):
        super(MalwareInstance, self).to_obj(return_obj=return_obj, ns_info=ns_info)

        if not return_obj:
            return_obj = self._binding_class()

        return_obj.id = self.id_
        return_obj.Title = self.title

        if self.descriptions:
            return_obj.Description = self.descriptions.to_obj(ns_info=ns_info)
        if self.short_descriptions:
            return_obj.Short_Description = self.short_descriptions.to_obj(ns_info=ns_info)
        if self.names:
            return_obj.Name = self.names.to_obj(ns_info=ns_info)
        if self.types:
            return_obj.Type = self.types.to_obj(ns_info=ns_info)

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        import stix.extensions.malware.maec_4_1_malware  # noqa
        
        if not obj:
            return None
        
        if not return_obj:
            klass = stix.lookup_extension(obj, default=cls)
            return_obj = klass.from_obj(obj, klass())
        else:
            return_obj.id_ = obj.id
            return_obj.title = obj.Title
            return_obj.descriptions = StructuredTextList.from_obj(obj.Description)
            return_obj.short_descriptions = StructuredTextList.from_obj(obj.Short_Description)
            return_obj.names = MalwareNames.from_obj(obj.Name)
            return_obj.types = MalwareTypes.from_obj(obj.Type)

        return return_obj

    def to_dict(self):
        d = utils.to_dict(self)

        if getattr(self, '_XSI_TYPE', None):
            d['xsi:type'] = self._XSI_TYPE

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        from stix.extensions.malware import maec_4_1_malware  # noqa

        if not dict_repr:
            return None

        get = dict_repr.get
    
        if not return_obj:
            klass = stix.lookup_extension(get('xsi:type'), cls)
            return_obj = klass.from_dict(dict_repr, klass())
        else:
            return_obj.id_ = get('id')
            return_obj.title = get('title')
            return_obj.descriptions = StructuredTextList.from_dict(get('description'))
            return_obj.short_descriptions = StructuredTextList.from_dict(get('short_description'))
            return_obj.names = MalwareNames.from_dict(get('names'))
            return_obj.types = MalwareTypes.from_dict(get('types'))

        return return_obj


class MalwareNames(stix.TypedList):
    _contained_type = VocabString


class MalwareTypes(stix.TypedList):
    _contained_type = VocabString

    def _fix_value(self, value):
        return vocabs.MalwareType(value)


# Backwards compatibility
add_extension = stix.add_extension
