# Copyright (c) 2014 Adam Karpierz
# SPDX-License-Identifier: BSD-3-Clause

import traceback
import builtins

import jni

from jvm.jframe  import JFrame
from jvm.jstring import JString
from jvm.jhost   import JHost
from jvm.java    import throwJavaException

from ....._jvm  import get_jvm
from ....._jvm  import jni_enter, jni_exit
from ....._jenv import JB_Env
from .....jutil import make_run_dictionary


# Class: org.cellprofiler.javabridge.CPython

# Method: native void exec(String script,
#                          java.util.Map<String, Object> locals,
#                          java.util.Map<String, Object> globals) throws WrappedException;

@jni.method("(Ljava/lang/String;Ljava/util/Map;Ljava/util/Map;)V")
def exec(env, this,
         script, locals, globals):

    jenv = env[0]

    if not script:
        throwJavaException(jenv, "java.lang.Error", "Script was null.")
        return

    #check_init(); #!!!

    with JHost.CallbackState():
        try:
            jni_enter(jenv) # Prerequisites: The GIL must be taken.
            try:
                py_script  = JString(jenv, script, own=False).str if script else None
                py_locals  = _jmap_to_dict(jenv, locals) if locals else {}
                py_globals = (py_locals if locals and jenv.IsSameObject(locals, globals)
                              else (_jmap_to_dict(jenv, globals) if globals else {}))
                _add_globals(py_globals)

                builtins.exec(builtins.compile(py_script, "<string>", "exec"),
                              py_globals, py_locals)
            finally:
                jni_exit()
        except Exception as exc:
            traceback.print_exc()
            throwJavaException(jenv, "java.lang.Error",
                                     f"Python exception at {__file__}:")

_initialized: bool = False

def check_init():

    global _initialized

    if _initialized or Py_IsInitialized():
        return

    Py_Initialize()
    if defined("_WIN32"):
        # If MSVC90.dll is on the path, we will perish horribly in Windows
        # with an R6034 exception on loading it from some of the numerous
        # pyd files that include it.
        PyRun_SimpleString("import os\n"
                           "os.environ['path'] = ';'.join("
                               "[p for p in os.environ['path'].split(';') "
                               " if 'msvcr90.dll' not in map(lambda x: x.lower(), "
                                                            "os.listdir(p))])")
    # endif
    _set_vm()
    _initialized = True

def _set_vm() -> int:
    # Run some code here to encapsulate a pointer to the VM and call
    # from jt import javabridge
    # javabridge.jvm_enter(capsule)

    global pVM

    try:
        pPyVM = PyCapsule_New(pVM, NULL, NULL)
    except:
        print("Unable to encapsulate VM for Python.", file=sys.stderr)
        return -1
    try:
        from jt import javabridge
    except:
        print("Failed to import javabridge.", file=sys.stderr)
        return -1
    try:
        jvm_enter = javabridge.jvm_enter
    except:
        print("Failed to find function, javabridge.jvm_enter", file=sys.stderr)
        return -1
    try:
        jvm_enter(pPyVM)
    except:
        print("Caught exception in jvm_enter.", file=sys.stderr)
        return -1

    return 0

def _jmap_to_dict(jenv, jmap):
    jvm = get_jvm()
    env = JB_Env()
    env.env = jenv
    jobject = env._make_jb_object(jvm.JObject(None, jmap, own=False))
    return make_run_dictionary(jobject)

def _add_globals(py_globals):
    # Add globals from __main__ to a dictionary of them.
    pass
    #py_main = PyImport_AddModule("__main__")
    #PyDict_Merge(py_globals, py_main.__dict__, 0)


__jnimethods__ = (
    exec,
)

__javacode__ = bytearray(  # Auto-generated; DO NOT EDIT!
    b"\xca\xfe\xba\xbe\x00\x00\x00\x34\x00\x7b\x0a\x00\x1f\x00\x35\x0a\x00\x03\x00\x36"
    b"\x07\x00\x37\x0a\x00\x38\x00\x39\x0a\x00\x3a\x00\x3b\x09\x00\x03\x00\x3c\x08\x00"
    b"\x3d\x0a\x00\x3e\x00\x3f\x08\x00\x40\x0a\x00\x33\x00\x41\x0a\x00\x33\x00\x42\x08"
    b"\x00\x43\x0a\x00\x33\x00\x44\x07\x00\x45\x0a\x00\x0e\x00\x46\x0a\x00\x0e\x00\x47"
    b"\x07\x00\x48\x08\x00\x49\x0a\x00\x38\x00\x4a\x0a\x00\x34\x00\x4b\x0a\x00\x34\x00"
    b"\x4c\x07\x00\x4d\x0a\x00\x33\x00\x4e\x0a\x00\x4f\x00\x50\x0a\x00\x34\x00\x51\x07"
    b"\x00\x52\x0a\x00\x1a\x00\x53\x0a\x00\x3a\x00\x54\x08\x00\x55\x0a\x00\x3e\x00\x56"
    b"\x07\x00\x57\x07\x00\x58\x01\x00\x10\x57\x72\x61\x70\x70\x65\x64\x45\x78\x63\x65"
    b"\x70\x74\x69\x6f\x6e\x01\x00\x0c\x49\x6e\x6e\x65\x72\x43\x6c\x61\x73\x73\x65\x73"
    b"\x07\x00\x59\x01\x00\x0a\x53\x74\x61\x63\x6b\x46\x72\x61\x6d\x65\x01\x00\x03\x6c"
    b"\x6f\x67\x01\x00\x1a\x4c\x6a\x61\x76\x61\x2f\x75\x74\x69\x6c\x2f\x6c\x6f\x67\x67"
    b"\x69\x6e\x67\x2f\x4c\x6f\x67\x67\x65\x72\x3b\x01\x00\x06\x3c\x69\x6e\x69\x74\x3e"
    b"\x01\x00\x03\x28\x29\x56\x01\x00\x04\x43\x6f\x64\x65\x01\x00\x07\x65\x78\x65\x63"
    b"\x75\x74\x65\x01\x00\x15\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74"
    b"\x72\x69\x6e\x67\x3b\x29\x56\x01\x00\x0a\x45\x78\x63\x65\x70\x74\x69\x6f\x6e\x73"
    b"\x01\x00\x33\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e"
    b"\x67\x3b\x4c\x6a\x61\x76\x61\x2f\x75\x74\x69\x6c\x2f\x4d\x61\x70\x3b\x4c\x6a\x61"
    b"\x76\x61\x2f\x75\x74\x69\x6c\x2f\x4d\x61\x70\x3b\x29\x56\x01\x00\x09\x53\x69\x67"
    b"\x6e\x61\x74\x75\x72\x65\x01\x00\x7f\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67"
    b"\x2f\x53\x74\x72\x69\x6e\x67\x3b\x4c\x6a\x61\x76\x61\x2f\x75\x74\x69\x6c\x2f\x4d"
    b"\x61\x70\x3c\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67"
    b"\x3b\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x3e"
    b"\x3b\x4c\x6a\x61\x76\x61\x2f\x75\x74\x69\x6c\x2f\x4d\x61\x70\x3c\x4c\x6a\x61\x76"
    b"\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x4c\x6a\x61\x76\x61\x2f"
    b"\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x3e\x3b\x29\x56\x01\x00\x04\x65"
    b"\x78\x65\x63\x01\x00\x08\x3c\x63\x6c\x69\x6e\x69\x74\x3e\x01\x00\x0d\x53\x74\x61"
    b"\x63\x6b\x4d\x61\x70\x54\x61\x62\x6c\x65\x07\x00\x5a\x07\x00\x5b\x0c\x00\x27\x00"
    b"\x28\x0c\x00\x30\x00\x2d\x01\x00\x23\x6f\x72\x67\x2f\x63\x65\x6c\x6c\x70\x72\x6f"
    b"\x66\x69\x6c\x65\x72\x2f\x6a\x61\x76\x61\x62\x72\x69\x64\x67\x65\x2f\x43\x50\x79"
    b"\x74\x68\x6f\x6e\x07\x00\x5c\x0c\x00\x5d\x00\x5e\x07\x00\x5f\x0c\x00\x60\x00\x61"
    b"\x0c\x00\x25\x00\x26\x01\x00\x0f\x6a\x61\x76\x61\x2e\x63\x6c\x61\x73\x73\x2e\x70"
    b"\x61\x74\x68\x07\x00\x62\x0c\x00\x63\x00\x64\x01\x00\x0e\x70\x61\x74\x68\x2e\x73"
    b"\x65\x70\x61\x72\x61\x74\x6f\x72\x0c\x00\x65\x00\x66\x0c\x00\x67\x00\x5e\x01\x00"
    b"\x0b\x63\x70\x79\x74\x68\x6f\x6e\x2e\x6a\x61\x72\x0c\x00\x68\x00\x69\x01\x00\x0c"
    b"\x6a\x61\x76\x61\x2f\x69\x6f\x2f\x46\x69\x6c\x65\x0c\x00\x27\x00\x2b\x0c\x00\x6a"
    b"\x00\x5e\x01\x00\x15\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x43\x6c\x61\x73\x73"
    b"\x4c\x6f\x61\x64\x65\x72\x01\x00\x09\x75\x73\x72\x5f\x70\x61\x74\x68\x73\x0c\x00"
    b"\x6b\x00\x6c\x0c\x00\x6d\x00\x6e\x0c\x00\x6f\x00\x70\x01\x00\x13\x5b\x4c\x6a\x61"
    b"\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x0c\x00\x71\x00\x69"
    b"\x07\x00\x72\x0c\x00\x73\x00\x74\x0c\x00\x75\x00\x76\x01\x00\x13\x6a\x61\x76\x61"
    b"\x2f\x6c\x61\x6e\x67\x2f\x45\x78\x63\x65\x70\x74\x69\x6f\x6e\x0c\x00\x77\x00\x5e"
    b"\x0c\x00\x78\x00\x2b\x01\x00\x11\x6a\x61\x76\x61\x2e\x6c\x69\x62\x72\x61\x72\x79"
    b"\x2e\x70\x61\x74\x68\x0c\x00\x79\x00\x7a\x01\x00\x10\x6a\x61\x76\x61\x2f\x6c\x61"
    b"\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x01\x00\x34\x6f\x72\x67\x2f\x63\x65\x6c\x6c"
    b"\x70\x72\x6f\x66\x69\x6c\x65\x72\x2f\x6a\x61\x76\x61\x62\x72\x69\x64\x67\x65\x2f"
    b"\x43\x50\x79\x74\x68\x6f\x6e\x24\x57\x72\x61\x70\x70\x65\x64\x45\x78\x63\x65\x70"
    b"\x74\x69\x6f\x6e\x01\x00\x2e\x6f\x72\x67\x2f\x63\x65\x6c\x6c\x70\x72\x6f\x66\x69"
    b"\x6c\x65\x72\x2f\x6a\x61\x76\x61\x62\x72\x69\x64\x67\x65\x2f\x43\x50\x79\x74\x68"
    b"\x6f\x6e\x24\x53\x74\x61\x63\x6b\x46\x72\x61\x6d\x65\x01\x00\x10\x6a\x61\x76\x61"
    b"\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x01\x00\x17\x6a\x61\x76\x61\x2f"
    b"\x6c\x61\x6e\x67\x2f\x72\x65\x66\x6c\x65\x63\x74\x2f\x46\x69\x65\x6c\x64\x01\x00"
    b"\x0f\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x43\x6c\x61\x73\x73\x01\x00\x07\x67"
    b"\x65\x74\x4e\x61\x6d\x65\x01\x00\x14\x28\x29\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e"
    b"\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x01\x00\x18\x6a\x61\x76\x61\x2f\x75\x74\x69"
    b"\x6c\x2f\x6c\x6f\x67\x67\x69\x6e\x67\x2f\x4c\x6f\x67\x67\x65\x72\x01\x00\x09\x67"
    b"\x65\x74\x4c\x6f\x67\x67\x65\x72\x01\x00\x2e\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61"
    b"\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x29\x4c\x6a\x61\x76\x61\x2f\x75\x74\x69"
    b"\x6c\x2f\x6c\x6f\x67\x67\x69\x6e\x67\x2f\x4c\x6f\x67\x67\x65\x72\x3b\x01\x00\x10"
    b"\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x79\x73\x74\x65\x6d\x01\x00\x0b\x67"
    b"\x65\x74\x50\x72\x6f\x70\x65\x72\x74\x79\x01\x00\x26\x28\x4c\x6a\x61\x76\x61\x2f"
    b"\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x29\x4c\x6a\x61\x76\x61\x2f\x6c"
    b"\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x01\x00\x05\x73\x70\x6c\x69\x74\x01"
    b"\x00\x27\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67"
    b"\x3b\x29\x5b\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67"
    b"\x3b\x01\x00\x0b\x74\x6f\x4c\x6f\x77\x65\x72\x43\x61\x73\x65\x01\x00\x08\x65\x6e"
    b"\x64\x73\x57\x69\x74\x68\x01\x00\x15\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67"
    b"\x2f\x53\x74\x72\x69\x6e\x67\x3b\x29\x5a\x01\x00\x09\x67\x65\x74\x50\x61\x72\x65"
    b"\x6e\x74\x01\x00\x10\x67\x65\x74\x44\x65\x63\x6c\x61\x72\x65\x64\x46\x69\x65\x6c"
    b"\x64\x01\x00\x2d\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69"
    b"\x6e\x67\x3b\x29\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x72\x65\x66\x6c\x65"
    b"\x63\x74\x2f\x46\x69\x65\x6c\x64\x3b\x01\x00\x0d\x73\x65\x74\x41\x63\x63\x65\x73"
    b"\x73\x69\x62\x6c\x65\x01\x00\x04\x28\x5a\x29\x56\x01\x00\x03\x67\x65\x74\x01\x00"
    b"\x26\x28\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b"
    b"\x29\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x01"
    b"\x00\x10\x65\x71\x75\x61\x6c\x73\x49\x67\x6e\x6f\x72\x65\x43\x61\x73\x65\x01\x00"
    b"\x10\x6a\x61\x76\x61\x2f\x75\x74\x69\x6c\x2f\x41\x72\x72\x61\x79\x73\x01\x00\x06"
    b"\x63\x6f\x70\x79\x4f\x66\x01\x00\x29\x28\x5b\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e"
    b"\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x49\x29\x5b\x4c\x6a\x61\x76\x61\x2f\x6c\x61"
    b"\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x01\x00\x03\x73\x65\x74\x01\x00\x27\x28"
    b"\x4c\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x4c\x6a"
    b"\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x4f\x62\x6a\x65\x63\x74\x3b\x29\x56\x01\x00"
    b"\x0a\x67\x65\x74\x4d\x65\x73\x73\x61\x67\x65\x01\x00\x07\x77\x61\x72\x6e\x69\x6e"
    b"\x67\x01\x00\x0b\x73\x65\x74\x50\x72\x6f\x70\x65\x72\x74\x79\x01\x00\x38\x28\x4c"
    b"\x6a\x61\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x4c\x6a\x61"
    b"\x76\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x29\x4c\x6a\x61\x76"
    b"\x61\x2f\x6c\x61\x6e\x67\x2f\x53\x74\x72\x69\x6e\x67\x3b\x00\x21\x00\x03\x00\x1f"
    b"\x00\x00\x00\x01\x00\x1a\x00\x25\x00\x26\x00\x00\x00\x06\x00\x01\x00\x27\x00\x28"
    b"\x00\x01\x00\x29\x00\x00\x00\x11\x00\x01\x00\x01\x00\x00\x00\x05\x2a\xb7\x00\x01"
    b"\xb1\x00\x00\x00\x00\x00\x01\x00\x2a\x00\x2b\x00\x02\x00\x29\x00\x00\x00\x14\x00"
    b"\x04\x00\x02\x00\x00\x00\x08\x2a\x2b\x01\x01\xb6\x00\x02\xb1\x00\x00\x00\x00\x00"
    b"\x2c\x00\x00\x00\x04\x00\x01\x00\x20\x00\x01\x00\x2a\x00\x2d\x00\x03\x00\x29\x00"
    b"\x00\x00\x14\x00\x04\x00\x04\x00\x00\x00\x08\x2a\x2b\x2c\x2d\xb6\x00\x02\xb1\x00"
    b"\x00\x00\x00\x00\x2c\x00\x00\x00\x04\x00\x01\x00\x20\x00\x2e\x00\x00\x00\x02\x00"
    b"\x2f\x00\x01\x00\x30\x00\x2b\x00\x02\x00\x29\x00\x00\x00\x14\x00\x04\x00\x02\x00"
    b"\x00\x00\x08\x2a\x2b\x01\x01\xb6\x00\x02\xb1\x00\x00\x00\x00\x00\x2c\x00\x00\x00"
    b"\x04\x00\x01\x00\x20\x01\x01\x00\x30\x00\x2d\x00\x02\x00\x2c\x00\x00\x00\x04\x00"
    b"\x01\x00\x20\x00\x2e\x00\x00\x00\x02\x00\x2f\x00\x08\x00\x31\x00\x28\x00\x01\x00"
    b"\x29\x00\x00\x01\x62\x00\x03\x00\x10\x00\x00\x00\xf2\x12\x03\xb6\x00\x04\xb8\x00"
    b"\x05\xb3\x00\x06\x12\x07\xb8\x00\x08\x4b\x12\x09\xb8\x00\x08\x4c\x01\x4d\x2a\x2b"
    b"\xb6\x00\x0a\x4e\x2d\xbe\x36\x04\x03\x36\x05\x15\x05\x15\x04\xa2\x00\xbc\x2d\x15"
    b"\x05\x32\x3a\x06\x19\x06\xb6\x00\x0b\x12\x0c\xb6\x00\x0d\x99\x00\xa3\xbb\x00\x0e"
    b"\x59\x19\x06\xb7\x00\x0f\x3a\x07\x19\x07\xb6\x00\x10\x3a\x08\x12\x11\x12\x12\xb6"
    b"\x00\x13\x3a\x09\x19\x09\x04\xb6\x00\x14\x19\x09\x01\xb6\x00\x15\xc0\x00\x16\x3a"
    b"\x0a\x03\x36\x0b\x19\x0a\x3a\x0c\x19\x0c\xbe\x36\x0d\x03\x36\x0e\x15\x0e\x15\x0d"
    b"\xa2\x00\x20\x19\x0c\x15\x0e\x32\x3a\x0f\x19\x0f\x19\x08\xb6\x00\x17\x99\x00\x09"
    b"\x04\x36\x0b\xa7\x00\x09\x84\x0e\x01\xa7\xff\xdf\x15\x0b\x9a\x00\x24\x19\x0a\x19"
    b"\x0a\xbe\x04\x60\xb8\x00\x18\xc0\x00\x16\x3a\x0c\x19\x0c\x19\x0c\xbe\x04\x64\x19"
    b"\x08\x53\x19\x09\x01\x19\x0c\xb6\x00\x19\xa7\x00\x10\x3a\x09\xb2\x00\x06\x19\x09"
    b"\xb6\x00\x1b\xb6\x00\x1c\x12\x1d\x19\x08\xb8\x00\x1e\x57\xa7\x00\x09\x84\x05\x01"
    b"\xa7\xff\x43\x2c\xc6\x00\x0a\x12\x1d\x2c\xb8\x00\x1e\x57\xb1\x00\x01\x00\x52\x00"
    b"\xc5\x00\xc8\x00\x1a\x00\x01\x00\x32\x00\x00\x00\x56\x00\x0a\xff\x00\x26\x00\x06"
    b"\x07\x00\x33\x07\x00\x33\x07\x00\x33\x07\x00\x16\x01\x01\x00\x00\xff\x00\x54\x00"
    b"\x0f\x07\x00\x33\x07\x00\x33\x07\x00\x33\x07\x00\x16\x01\x01\x07\x00\x33\x07\x00"
    b"\x0e\x07\x00\x33\x07\x00\x34\x07\x00\x16\x01\x07\x00\x16\x01\x01\x00\x00\x1d\xf8"
    b"\x00\x05\xf8\x00\x25\x42\x07\x00\x1a\x0c\xf8\x00\x0a\xf8\x00\x05\xf8\x00\x0a\x00"
    b"\x01\x00\x22\x00\x00\x00\x12\x00\x02\x00\x20\x00\x03\x00\x21\x00\x09\x00\x23\x00"
    b"\x03\x00\x24\x00\x09"
)
