# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A file-like object that writes line atomically to another file.

>>> from threading import Thread
>>> from cStringIO import StringIO
>>> TEXT = 'Hello\\nHello\\n'
>>> def writer(out):
...     for i in xrange(len(TEXT)):
...         out.write(TEXT[:i])
...         out.write(TEXT[i:])
>>> s = StringIO()
>>> o = OutputFile(s)
>>> threads = [Thread(target = writer, args=(o,)) for i in xrange(2)]
>>> for t in threads:
...     t.start()
>>> o.flush()
>>> for t in threads:
...     t.join()
>>> s.getvalue() == TEXT * (2 * len(TEXT))
True
>>> o.close()

"""

import threading

def newline_indecies(data):
    index = data.find('\n')
    while -1 != index:
        yield index
        index = data.find('\n', index + 1)


class OutputFile(object):
    __slots__ = ['_file',
                 '_lock',
                 '_local',
                 '_local_que_id',
        ]

    def __init__(self, fileobj):
        self._file = fileobj
        self._local = threading.local()
        self._lock = threading.Lock()
        self._local_que_id = '__OutputFile_que_%x_' % (id(self),)

    def _internal_write(self, buf):
        with self._lock:
            self._file.write(buf)

    def write(self, data):
        # get my que from thread local storage
        que = getattr(self._local, self._local_que_id, None)
        if que is None:
            que = ['']
            setattr(self._local, self._local_que_id, que)

        startlines_at = list(x + 1 for x in newline_indecies(data))

        if not startlines_at: # no new-lines at all
            que[0] = que[0] + data
            return

        # handle the first new-line
        self._internal_write(que[0] + data[:startlines_at[0]])

        for index in xrange(len(startlines_at) - 1):
            self._internal_write(data[startlines_at[index]:startlines_at[index+1]])

        # handle left overs
        que[0] = data[startlines_at[-1]:]

    def flush(self):
        self._file.flush()

    def close(self):
        self._file.close()
