# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
import sys

def safe_apply(func, *args, **kws):
    """
    Runs a function with given args and keywords, and returns:
    - (True, <return-value>) if function returned something;
    - (False, <sys.exc_info()> if function raised an exception.

    >>> def blah(x):
    ...     assert x > 5
    ...     return x - 5
    ...
    >>> safe_apply(blah, 10)
    (True, 5)
    >>> success, result = safe_apply(blah, 5)
    >>> success
    False
    >>> result[:2]
    (<type 'exceptions.AssertionError'>, AssertionError())
    
    """
    try:
        return True, func(*args, **kws)
    except:
        return False, sys.exc_info()
    
def unsafe(success, result):
    """
    Takes a tuple from safe_apply and returns as if function was called.

    >>> def blah(x):
    ...     assert x > 5
    ...     return x - 5
    ...
    >>> unsafe(*safe_apply(blah, 10))
    5
    >>> unsafe(*safe_apply(blah, 5))
    Traceback (most recent call last):
        ...
    AssertionError

    """
    if success:
        return result
    else:
        raise result[1]
