# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A set that remembers the insertion order.

>>> s = FrozenOrderedSet([1,2])
>>> s
FrozenOrderedSet([1, 2])
>>> len(s)
2
>>> list(s)
[1, 2]
>>> 1 in s
True
>>> 3 in s
False
>>> cmp(s, FrozenOrderedSet([1,2]))
0
>>> cmp(s, FrozenOrderedSet([1,2,3]))
-1
>>> cmp(s, FrozenOrderedSet([1]))
1
>>> cmp(s, FrozenOrderedSet([1,3]))
-1
>>> cmp(s, FrozenOrderedSet([2,1]))
-1
>>> cmp(FrozenOrderedSet([2,1]), s)
1
>>> s & [1]
FrozenOrderedSet([1])
>>> s - [1]
FrozenOrderedSet([2])
>>> s ^ [1,3]
FrozenOrderedSet([2, 3])
>>> s | [2,3]
FrozenOrderedSet([1, 2, 3])
>>> s.isdisjoint([1,3])
False
>>> s.isdisjoint([5,3])
True
>>> reversed(s)
FrozenOrderedSet([2, 1])
>>> s = OrderedSet([1,2])
>>> s
OrderedSet([1, 2])
>>> len(s)
2
>>> list(s)
[1, 2]
>>> 1 in s
True
>>> 3 in s
False
>>> cmp(s, OrderedSet([1,2]))
0
>>> cmp(s, OrderedSet([1,2,3]))
-1
>>> cmp(s, OrderedSet([1]))
1
>>> cmp(s, OrderedSet([1,3]))
-1
>>> cmp(s, OrderedSet([2,1]))
-1
>>> cmp(OrderedSet([2,1]), s)
1
>>> s & [1]
OrderedSet([1])
>>> s - [1]
OrderedSet([2])
>>> s ^ [1,3]
OrderedSet([2, 3])
>>> s | [2,3]
OrderedSet([1, 2, 3])
>>> s.isdisjoint([1,3])
False
>>> s.isdisjoint([5,3])
True
>>> reversed(s)
OrderedSet([2, 1])
>>> s = OrderedSet(xrange(10))
>>> s.add(11)
>>> s
OrderedSet([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 11])
>>> s.discard(8)
>>> s
OrderedSet([0, 1, 2, 3, 4, 5, 6, 7, 9, 11])
>>> s.remove(12)
Traceback (most recent call last):
    ...
KeyError: 12
>>> s
OrderedSet([0, 1, 2, 3, 4, 5, 6, 7, 9, 11])
>>> s.remove(9)
>>> s
OrderedSet([0, 1, 2, 3, 4, 5, 6, 7, 11])
>>> s.update([8,9,10])
>>> s
OrderedSet([0, 1, 2, 3, 4, 5, 6, 7, 11, 8, 9, 10])
>>> s -= [1,2]
>>> s
OrderedSet([0, 3, 4, 5, 6, 7, 11, 8, 9, 10])
>>> s &= xrange(8)
>>> s
OrderedSet([0, 3, 4, 5, 6, 7])
>>> s |= xrange(10)
>>> s
OrderedSet([0, 3, 4, 5, 6, 7, 1, 2, 8, 9])
>>> s ^= [3,12,13]
>>> s
OrderedSet([0, 4, 5, 6, 7, 1, 2, 8, 9, 12, 13])
>>> s.pop()
13
>>> OrderedSet().pop()
Traceback (most recent call last):
    ...
IndexError: pop from empty OrderedSet
>>> s.popfirst()
0
>>> OrderedSet().popfirst()
Traceback (most recent call last):
    ...
IndexError: popfirst from empty OrderedSet
>>> import cPickle as pickle
>>> o = OrderedSet((5, 'foo'))
>>> s = pickle.dumps(o, -1)
>>> O = pickle.loads(s)
>>> O == o
True
>>> o = o.freeze()
>>> o.freeze() is o
True
>>> s = pickle.dumps(o, -1)
>>> O = pickle.loads(s)
>>> O == o
True
>>> o.copy() == o
True
>>> OrderedSet([1,2]).issubset(OrderedSet([2,3,1]))
True
>>> OrderedSet([1,2]).issubset(OrderedSet([1]))
False
>>> OrderedSet([2,3,1]).issuperset(OrderedSet([1,2]))
True
>>> OrderedSet([1]).issuperset(OrderedSet([1, 2]))
False

"""

__all__ = ['FrozenOrderedSet', 'OrderedSet']

from bfc.linkedlist import LinkedList, DataNode

from itertools import chain

class BaseOrderedSet(object):
    __slots__ = [
        '_list',
        '_dict',
        ]

    def __init__(self, sequence = None):
        self._init()
        if sequence is not None:
            self._update(sequence)

    def _init(self):
        self._list = LinkedList()
        self._dict = {}

    def _add(self, item):
        if item in self._dict:
            return
        dn = DataNode(item)
        self._dict[item] = dn
        dn.insert_before(self._list)

    def _update(self, sequence):
        for item in sequence:
            self._add(item)

    __hash__ = None

    def __len__(self):
        return len(self._dict)

    def __iter__(self):
        for data_node in self._list:
            yield data_node.data

    def __contains__(self, item):
        return item in self._dict

    # Py3k TODO: re-write as new rich comparison functions
    def __cmp__(self, other):
        if not isinstance(other, BaseOrderedSet): # pragma: no cover
            return NotImplemented

        iter_left = iter(self)
        iter_right = iter(other)

        while True:
            try:
                left = iter_left.next()
            except StopIteration:
                left_exists = False
            else:
                left_exists = True
            try:
                right = iter_right.next()
            except StopIteration:
                right_exists = False
            else:
                right_exists = True

            if not left_exists:
                return -int(right_exists)
            if not right_exists:
                return 1
            r = cmp(left, right)
            if r:
                return r

    def __and__(self, sequence):
        return self.__class__(item for item in self if item in sequence)

    def __or__(self, sequence):
        return self.__class__(chain(self, sequence))

    def __sub__(self, sequence):
        return self.__class__(item for item in self if item not in sequence)

    def __xor__(self, sequence):
        return self.__class__(chain((item for item in self if item not in sequence),
                                    (item for item in sequence if item not in self)))

    def isdisjoint(self, sequence):
        return all(item not in sequence for item in self) and all(item not in self for item in sequence)

    def __repr__(self):
        return '%s([%s])' % (self.__class__.__name__,
                             ', '.join(repr(item) for item in self))

    def __reversed__(self):
        return self.__class__(dn.data for dn in self._list.reveresed_datanodes())

    def copy(self):
        return self.__class__(self)

    difference = __sub__
    symmetric_difference = __xor__
    intersection = __and__
    union = __or__

    def issubset(self, other):
        return all(item in other for item in self)

    def issuperset(self, other):
        return all(item in self for item in other)

    def __getstate__(self):
        return {'_':tuple(self)}

    def __setstate__(self, state):
        self._init()
        self._update(state['_'])
        

class FrozenOrderedSet(BaseOrderedSet):
    __slots__ = ['_hash_value']

    def __init__(self, sequence = None):
        BaseOrderedSet.__init__(self, sequence)
        self._hash_value = None

    def __hash__(self):
        if self._hash_value is None:
            self._hash_value = hash(tuple(self))
        return self._hash_value

    def freeze(self):
        return self

class OrderedSet(BaseOrderedSet):
    __slots__ = []

    add = BaseOrderedSet._add

    def freeze(self):
        return FrozenOrderedSet(self)

    def discard(self, item):
        dn = self._dict.pop(item, None)
        if dn is not None:
            dn.detach()

    def _pop(self, dn):
        item = dn.data
        del self._dict[item]
        dn.detach()
        return item

    def pop(self):
        if not self:
            raise IndexError('pop from empty %s' % (self.__class__.__name__,))
        return self._pop(self._list.prev)

    def popfirst(self):
        if not self:
            raise IndexError('popfirst from empty %s' % (self.__class__.__name__,))
        return self._pop(self._list.next)

    def clear(self):
        while self:
            item, dn = self._dict.popitem()
            dn.detach()

    update = BaseOrderedSet._update

    def remove(self, item):
        self._dict.pop(item).detach()

    def __isub__(self, other):
        for item in other:
            self.discard(item)
        return self

    def __iand__(self, other):
        for item in [item for item in self if item not in other]:
            self.remove(item)
        return self

    def __ior__(self, other):
        self.update(other)
        return self

    def __ixor__(self, other):
        to_add = [item for item in other if item not in self]

        for item in other:
            self.discard(item)
        self.update(to_add)
        return self

    difference_update = __isub__
    symmetric_difference_update = __ixor__
    intersection_update = __iand__
