# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Imports Python files as modules.
The returned module will not be present in sys.modules!
The importer is cached, so if you need to reload, call import_file.reset().

Note: if you import a file whose base name is an existing one in sys.modules,
the cached one will be removed.

>>> import tempfile, sys, os
>>> a = tempfile.NamedTemporaryFile(suffix = '.py')
>>> mod_name = os.path.basename(a.name)[:-3]
>>> a.write('''print "Hello"
... def foo(x): print "%s" % (x,)
... import sys
... ''')
>>> a.flush()
>>> module = import_file(a.name)
Hello
>>> module.foo(5)
5
>>> mod_name in sys.modules
False
>>> b = tempfile.NamedTemporaryFile(suffix = '.py')
>>> b.write('''raise Exception('Blah')
... ''')
>>> b.flush()
>>> import_file.reset()
>>> module = import_file(b.name)
Traceback (most recent call last):
    ...
Exception: Blah

"""
import imp, sys, os

from .cache.func import hashable_non_null_cached

__all__ = ['import_file']

@hashable_non_null_cached
def import_file(filename):
    dirname, basename = os.path.split(filename)
    assert basename.endswith('.py')
    name = basename[:-3]
    try:
        del sys.modules[name]
    except KeyError:
        pass
    fp, pathname, description = imp.find_module(name, [dirname])
    try:
        return imp.load_module(name, fp, pathname, description)
    finally:
        if fp:
            fp.close()
        # if the module tries to remove itself during its own load, imp.load_module fails
        # but if the module itself raises exception during its own run, it won't be in sys.modules anyway
        try:
            del sys.modules[name]
        except KeyError:
            pass
