# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A NamedContext is a name-based derivable configuration container, much like context.Context,
with the following difference:

Instead of derivable and non-derivable fathers, the bubble concept is introduced.
A bubble affects both the derivable and local dicts of a sibling.

>>> a = NamedContext('a')
>>> a.name
'a'
>>> a.derivable['a'] = 5
>>> a.bubble('b').derivable['B'] = 10
>>> a.bubble('b').local['b'] = 20
>>> a.bubble('b').derivable.items()
[('B', 10), ('a', 5)]
>>> a.bubble('b').local.items()
[('b', 20), ('B', 10), ('a', 5)]
>>> b = NamedContext('b',(a,))
>>> b.fathers[0] is a
True
>>> b.derivable.items()
[('B', 10), ('a', 5)]
>>> b.local.items()
[('b', 20), ('B', 10), ('a', 5)]
>>> a.bubble('c').local['c'] = 50
>>> c = NamedContext('c',(b,))
>>> c.local.items()
[('c', 50), ('B', 10), ('a', 5)]

The bubble overrides usual derivation:

>>> a = NamedContext('a')
>>> b = NamedContext('b',(a,))
>>> c = NamedContext('c',(b,))
>>> b.derivable['C'] = 1
>>> a.bubble('c').derivable['C'] = 2
>>> c.derivable.items()
[('C', 2)]

Bubbles can be nested:

>>> a = NamedContext('a')
>>> b = NamedContext('b',(a,))
>>> c = NamedContext('c',(b,))
>>> b.derivable['C'] = 1
>>> a.bubble('c').derivable['C'] = 2
>>> c.derivable.items()
[('C', 2)]

DerivedSetInsideDict and DerivedDictInsideDict work natively with derivable and local dicts:

>>> a = NamedContext('a')
>>> b = NamedContext('b',(a,))
>>> ap = DerivedSetInsideDict(a.derivable, 'path')
>>> ap.add('/')
>>> a.bubble('b').derivable['path'].add('/ab_bubble_derivable')
>>> tuple(a.bubble('b').derivable['path'])
('/ab_bubble_derivable', '/')
>>> a.bubble('b').local['path'].add('/ab_bubble_local')
>>> tuple(a.bubble('b').local['path'])
('/ab_bubble_local', '/ab_bubble_derivable', '/')
>>> tuple(b.derivable['path'])
('/ab_bubble_derivable', '/')
>>> b.local['path'].add('/tmp')
>>> tuple(b.local['path'])
('/tmp', '/ab_bubble_local', '/ab_bubble_derivable', '/')

This class can be used to read configuration files written in Python:

>>> a = NamedContext('a')
>>> b = NamedContext('b',(a,))
>>> c = NamedContext('c',(b,))
>>> a_code = '''
... DERIVABLE.A = 1
... a = 2
... DERIVABLE.DERIVEDSET('s').add(3)
... DERIVEDSET('s').add(4)
... BUBBLE('b').DERIVABLE.B = 3
... BUBBLE('b').b = 4
... BUBBLE('b').DERIVEDDICT('d')[1] = 2
... BUBBLE('b').BUBBLE('c').c = 5
... BUBBLE('b').BUBBLE('c').DERIVABLE.C = 6
... '''
>>> a.update_by_exec(a_code)
>>> a.local.items()[0::2]
[('a', 2), ('A', 1)]
>>> list(a.local['s'])
[4, 3]
>>> a._bubbles.items()
[('b', BubbledNamedContext('b', NamedContext('a', ())))]
>>> b.local.items()[:1] + b.local.items()[2:4]
[('b', 4), ('B', 3), ('A', 1)]
>>> list(b.local['s'])
[3]
>>> b.local['d'].items()
[(1, 2)]
>>> b._bubbles.items()
[]
>>> c.local.items()[:2] + c.local.items()[3:]
[('c', 5), ('C', 6), ('B', 3), ('A', 1)]
>>> list(c.local['s'])
[3]

A NamedContext can be flattened, making it independent of its fathers:

>>> a = NamedContext('a')
>>> b = NamedContext('b', (a,))
>>> c = NamedContext('c', (b,))
>>> a.derivable['A'] = 5
>>> a.bubble('c').local['c'] = 55
>>> a.bubble('d').local['d'] = 11
>>> b.derivable['A']
5
>>> c.local['c']
55
>>> b.flatten()
>>> a.derivable['A'] = 6
>>> a.bubble('c').local['c'] = 44
>>> b.derivable['A']
5
>>> c.local['c']
55
>>> d = NamedContext('d', (a,), flatten = True)
>>> a.bubble('d').local['d'] = 22
>>> d.local['d']
11

A NamedContext can also be frozen. It supports bfc.freeze API by self-freezing.
A freezed instance is hashable and comparable.
When freezing an instance, its fathers are also freezed, and all bubbles are freezed,
including new ones.
Re-freezing does nothing.

>>> a = NamedContext('a')
>>> a.derivable['a'] = 1
>>> a.bubble('b').local['b'] = 5
>>> a.bubble('d').local['d'] = 5
>>> del a.bubble('d').local['d']
>>> a.freeze()
>>> a.bubble('c').local['a']
1
>>> s = set([a])
>>> at = NamedContext('a')
>>> at.derivable['a'] = 1
>>> at.bubble('b').local['b'] = 5
>>> at.freeze()
>>> a == at
True
>>> a.freeze()

And everything is picklable:

>>> a = NamedContext('a')
>>> import cPickle as pickle
>>> a.derivable['A'] = 'A'
>>> s = pickle.dumps(a, -1)
>>> a.bubble('b').local['b'] = 'b'
>>> s = pickle.dumps(a, -1)
>>> del a
>>> a = pickle.loads(s)
>>> a.local.items()
[('A', 'A')]
>>> a.bubble('b').local.items()
[('b', 'b'), ('A', 'A')]

"""

from .dict import DerivedDict, DerivedDictInsideDict, DerivedSetInsideDict

from ..views.generator import GeneratorView
from ..frozendict import FrozenDict
from ..freeze import freeze
from ..attrdict import AttrDict
from ..executils import ExecutionContext

import itertools

__all__ = ['NamedContext',
           'EMPTY',
           'DerivedDictInsideDict',
           'DerivedSetInsideDict',
           ]

class AttrBubble(AttrDict):
    __slots__ = ['__bubble',
                 '__name_fixer']

    def __init__(self, bubble, name_fixer):
        object.__setattr__(self, '_AttrBubble__bubble', bubble)
        object.__setattr__(self, '_AttrBubble__name_fixer', name_fixer)
        AttrDict.__init__(self, ExecutionContext(bubble.local,
                                                 bubble.local.execution_reaodonly_parts,
                                                 ))

    @property
    def DERIVABLE(self):
        return AttrDict(ExecutionContext(self.__bubble.derivable,
                                         self.__bubble.derivable.execution_reaodonly_parts,
                                         ))

    def BUBBLE(self, name):
        b = self.__bubble.bubble(self.__name_fixer(name))
        return self if b is self.__bubble else AttrBubble(b, self.__name_fixer)

def _derivable_fathers_generator(nc):
    return nc._derivable_fathers_generator()

def _derivable_bubble_inside_namedcontext_generator(nc):
    return nc._derivable_bubble_inside_namedcontext_generator()

def _local_bubble_inside_namedcontext_generator(nc):
    return nc._local_bubble_inside_namedcontext_generator()

class NamedContextBase(object):
    __slots__ = ['_name',
                 '_derivable',
                 '_local',
                 '_bubbles',
                 '_essence',
                 '_extra_bubbles',
                 ]
    def __init__(self, name):
        self._name = name
        derivable_fathers = GeneratorView(_derivable_fathers_generator, self)
        self._derivable = DerivedDict(derivable_fathers = GeneratorView(_derivable_bubble_inside_namedcontext_generator, self),
                                      non_derivable_fathers = derivable_fathers)
        self._local = DerivedDict(derivable_fathers = GeneratorView(_local_bubble_inside_namedcontext_generator, self),
                                  non_derivable_fathers = derivable_fathers)
        self._bubbles = {}
        self._essence = None
        self._extra_bubbles = {}

    @property
    def name(self):
        return self._name

    @property
    def derivable(self):
        return self._derivable

    @property
    def local(self):
        return self._local

    @property
    def is_frozen(self):
        return self._essence is not None

    def bubble(self, name):
        if name == self._name:
            return self

        b = self._bubbles.get(name, None)
        if b is None:
            if self.is_frozen:
                b = self._extra_bubbles.get(name, None)
                if b is None:
                    b = BubbledNamedContext(name, self)
                    b.freeze()
                    self._extra_bubbles[name] = b
            else:
                b = BubbledNamedContext(name, self)
                self._bubbles[name] = b
        return b

    def _derivable_bubble_inside_namedcontext_generator(self):
        for father in self._bubble_fathers():
            yield father._derivable

    def _local_bubble_inside_namedcontext_generator(self):
        for father in self._bubble_fathers():
            yield father._local
        yield self._derivable

    def flatten(self):
        assert not self.is_frozen
        for b in self._bubbles.itervalues():
            b.flatten()
        self._derivable.flatten()
        self._local.flatten()

    def freeze(self):
        if self.is_frozen:
            return

        self._derivable.freeze()
        self._local.freeze()
        empty_bubbles = []
        for name, bubble in self._bubbles.iteritems():
            bubble.freeze()
            if not (bubble._bubbles or bubble._derivable._internal_data or bubble._local._internal_data):
                empty_bubbles.append(name)
        for name in empty_bubbles:
            self._extra_bubbles[name] = self._bubbles.pop(name)
        self._bubbles = FrozenDict(self._bubbles)
        self._essence = (self._local, self._derivable, self._bubbles)

    def __hash__(self):
        assert self.is_frozen
        return hash(self._essence)

    def __cmp__(self, other):
        assert self.is_frozen and other.is_frozen
        return cmp(self._essence, other._essence)

class NamedContext(NamedContextBase):
    __slots__ = ['_fathers']

    def __init__(self, name, fathers = (), flatten = False, freeze = False):
        NamedContextBase.__init__(self, name)
        self._fathers = fathers
        if flatten:
            self.flatten()
        if freeze:
            self.freeze()

    def __repr__(self):
        return '%s(%r, %r)' % (self.__class__.__name__, self.name, self._fathers)

    @property
    def fathers(self):
        return self._fathers

    def _bubble_fathers(self):
        for father in self._fathers:
            yield father.bubble(self._name)

    def _bubble_containers(self):
        return self._bubble_fathers()

    def _derivable_fathers_generator(self):
        for father in self._fathers:
            yield father._derivable
            for d in father._derivable_fathers_generator():
                yield d

    def update_by_exec(self, executed, readonly_parts = (), name_fixer = lambda x : x):
        def BUBBLE(name):
            return AttrBubble(self.bubble(name_fixer(name)), name_fixer)
        nc_part = {'BUBBLE': BUBBLE,
                   'DERIVABLE' : AttrDict(ExecutionContext(self._derivable, self._derivable.execution_reaodonly_parts))}
        self._local.update_by_exec(executed, (nc_part,) + readonly_parts)

    def _bubble_father_names(self, seen):
        for father in self._fathers:
            seen.update(father._bubbles)
            father._bubble_father_names(seen)

    def flatten(self):
        assert not self.is_frozen
        seen = set()
        self._bubble_father_names(seen)
        for b in seen:
            self.bubble(b)
        NamedContextBase.flatten(self)
        self._fathers = ()

class BubbledNamedContext(NamedContextBase):
    __slots__ = ['_container']

    def __init__(self, name, container):
        NamedContextBase.__init__(self, name)
        self._container = container

    def __repr__(self):
        return '%s(%r, %r)' % (self.__class__.__name__, self.name, self._container)

    def _bubble_fathers(self):
        for bcontainer in self._container._bubble_containers():
            yield bcontainer.bubble(self._name)

    def _bubble_containers(self):
        yield self._container

    def _derivable_fathers_generator(self):
        yield self._container._derivable
        for d in self._container._derivable_fathers_generator():
            yield d

EMPTY = NamedContext('', freeze = True)
