# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.

from ..freeze import freeze

from itertools import chain

class DerivedBase(object):

    __slots__ = ['_internal_data',
                 '_derivable_fathers',
                 '_non_derivable_fathers',
                 '_essence',
                 '__weakref__',
                 ]

    def __init__(self, internal_data, derivable_fathers = (), non_derivable_fathers = (), freeze = False):
        self._internal_data = internal_data
        self._derivable_fathers = derivable_fathers
        self._non_derivable_fathers = non_derivable_fathers
        self._essence = None
        if freeze:
            self.freeze()

    def __repr__(self):
        return '%s(%r, %r, %r)' % (self.__class__.__name__, self._internal_data, self._derivable_fathers, self._non_derivable_fathers)

    @property
    def derivable_fathers(self):
        return self._derivable_fathers

    @property
    def non_derivable_fathers(self):
        return self._non_derivable_fathers

    @property
    def fathers(self):
        return chain(self._derivable_fathers, self._non_derivable_fathers)

    def _iter_derivable_internal_data(self):
        yield self._internal_data
        for f in self._derivable_fathers:
            if isinstance(f, DerivedBase):
                for d in f._iter_derivable_internal_data():
                    yield d
            else:
                yield f

    def _iter_father_internal_data(self):
        for f in self._derivable_fathers:
            if isinstance(f, DerivedBase):
                for d in f._iter_derivable_internal_data():
                    yield d
            else:
                yield f
        for f in self._non_derivable_fathers:
            if isinstance(f, DerivedBase):
                for d in f._iter_derivable_internal_data():
                    yield d
            else:
                yield f
    
    def _iter_internal_data(self):
        yield self._internal_data
        for d in self._iter_father_internal_data():
            yield d

    ### collections.Container requirements
    def __contains__(self, obj):
        return any(obj in d for d in self._iter_internal_data())

    ### collections.Sized requirements
    def __len__(self):
        c = 0
        for x in self:
            c = c + 1
        return c

    def update(self, seq):
        assert not self.is_frozen
        self._internal_data.update(seq)

    def flatten(self):
        for d in self._iter_father_internal_data():
            self.update(d)
        self._derivable_fathers = ()
        self._non_derivable_fathers = ()

    def freeze(self):
        if self.is_frozen:
            return

        self._derivable_fathers = freeze(self._derivable_fathers)
        self._non_derivable_fathers = freeze(self._non_derivable_fathers)
        self._internal_data = freeze(self._internal_data)
        self._essence = (self._internal_data, self._derivable_fathers, self._non_derivable_fathers)

    def __hash__(self):
        assert self.is_frozen
        return hash(self._essence)

    def __cmp__(self, other):
        assert self.is_frozen and other.is_frozen
        return cmp(self._essence, other._essence)

    @property
    def is_frozen(self):
        return self._essence is not None

    def clear(self):
        assert not self.is_frozen
        self._internal_data.clear()
