# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Runs a target and its dependencies in a recursive manner,
using the DFS algorithem.
"""

from ..safeapply import safe_apply

__all__ = ['dfs_run','dfs_run_all']

def dfs_run(target_dep, environment):

    # already run?
    ready_success = getattr(target_dep, '_success', None)
    if ready_success is not None:
        return ready_success

    target_dep._schedule_to_run()
    try:
        environment.on_dfs_start(target_dep)

        # run dependencies
        try:
            dep_failed = False
            dep_view = target_dep.not_yet_run_deps
            while dep_view:
                for d in list(dep_view):
                    s = dfs_run(d, environment)
                    if not s:
                        dep_failed = True
                        if not environment.keep_on_error:
                            break

                if dep_failed:
                    return False

            # sanity
            if not target_dep.may_run: # pragma: no cover
                raise RuntimeError('Cannot run dep %r' % target_dep)
            
            success, result = safe_apply(target_dep.run, environment)
            target_dep._mark_as_run(success, result)
        finally:
            environment.on_dfs_end(target_dep)
    finally:
        target_dep._unschedule_to_run()
    
    return success

def dfs_run_all(depgraph, environment):
    success = True
    for target_dep in list(depgraph.final_dependencies):
        if not dfs_run(target_dep, environment):
            success = False
            if not environment.keep_on_error:
                break
    return success
