# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.

from ..digraph import Graph, Vertex
from ..views.generator import GeneratorView
from ..views.wr import WrView
from ..views.immutable import Immutable
from ..orderedset import OrderedSet

from weakref import ref as wr

# A -> B means that A is a dependency of B

class DependencyView(object):
    __slots__ = ['_seq',
                 ]

    def __init__(self, seq):
        self._seq = seq

    def __iter__(self):
        return iter(dv._wr_dep() for dv in self._seq)

    def __contains__(self, dep):
        return dep._full_vertex in self._seq

    def __len__(self):
        return len(self._seq)

class DependencyGraph(object):
    """
    A dependency graph.
    """
    __slots__ = ['_full_graph',
                 '_run_graph',
                 '_not_run_source_wrs',
                 '_all_dependencies',
                 '__weakref__',
                 ]

    def __init__(self):
        self._full_graph = Graph()
        self._run_graph = Graph()
        self._not_run_source_wrs = OrderedSet()
        self._all_dependencies = OrderedSet()

    def add_dependency(self, dependency):
        """Adds dependecies to the graph.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> dg.add_dependency(a)
        >>> list(dg.all_dependencies) == [a]
        True
        >>> b = Dependency()
        >>> list(dg.all_dependencies) == [a]
        True
        >>> dg.add_dependency(b)
        >>> set(dg.all_dependencies) == set([a,b])
        True
        >>> a in dg.all_dependencies
        True
        >>> a.dependency_graph is dg
        True

        """
        assert dependency._dependency_graph_weakref is None, 'The Dependency already belongs to some DependencyGraph'

        self._all_dependencies.add(dependency)
        self._full_graph.add(dependency._full_vertex)
        self._run_graph.add(dependency._run_vertex)

        dependency._dependency_graph_weakref = wr(self)

        self._not_run_source_wrs.add(wr(dependency))

    @property
    def ready_to_run_dependencies(self):
        """
        Returns a view of dependencies that can be run.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> gv = dg.ready_to_run_dependencies
        >>> a.depend_on(b)
        >>> set(gv) == set([b])
        True
        
        """
        return WrView(self._not_run_source_wrs)

    @property
    def final_dependencies(self):
        """
        Returns a view of dependencies that are not dependencies of others.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> gv = dg.final_dependencies
        >>> a.depend_on(b)
        >>> set(gv) == set([a])
        True
        >>> a in gv
        True
        >>> b in gv
        False

        """
        return DependencyView(self._full_graph.pits)

    @property
    def all_dependencies(self):
        """
        Returns a view of all dependencies in the graph.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> gv = dg.all_dependencies
        >>> a.depend_on(b)
        >>> set(gv) == set([a,b])
        True
        
        """
        return Immutable(self._all_dependencies)

class DependencyVertex(Vertex):
    __slots__ = ['_wr_dep']
    def __init__(self, dep):
        Vertex.__init__(self)
        self._wr_dep = wr(dep)

class Dependency(object):
    __slots__ = ['_dependency_graph_weakref',
                 '_full_vertex',
                 '_run_vertex',
                 '_was_run',
                 '_dep_failed',
                 '_success',
                 '_result',
                 '_schedules_to_run',
                 '_dependencies_to_add',
                 '_on_dep_run_depth',
                 '__weakref__'
                 ]

    def __init__(self):
        self._dependency_graph_weakref = None
        self._full_vertex = DependencyVertex(self)
        self._run_vertex = DependencyVertex(self)
        self._was_run = False
        self._dep_failed = False
        self._schedules_to_run = False
        self._dependencies_to_add = OrderedSet()
        self._on_dep_run_depth = 0

    def run(self, environment): # pragma: no cover
        """Override this method to do something useful.

        This method will be called in an atomic way, and possibly in a separate thread or process.
        """
        pass

    @property
    def dependency_graph(self):
        """Returns the dependency graph of this dependency."""
        return self._dependency_graph_weakref() if self._dependency_graph_weakref else None

    def is_my_dependency(self, dependency):
        """Checks whether the given dependency is my dependency.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> c = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> dg.add_dependency(c)
        >>> a.depend_on(b)
        >>> a.is_my_dependency(b)
        True
        >>> a.is_my_dependency(c)
        False
        >>> b.is_my_dependency(a)
        False

        """
        return Vertex.are_connected(dependency._full_vertex, self._full_vertex)

    def is_dependency_of(self, dependency):
        """Checks whether self is a dependency of the given one.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> c = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> dg.add_dependency(c)
        >>> a.depend_on(b)
        >>> b.is_dependency_of(a)
        True
        >>> c.is_dependency_of(a)
        False
        >>> a.is_dependency_of(b)
        False

        """
        return dependency.is_my_dependency(self)

    @property
    def may_run(self):
        """Checks whether the dependency may run"""
        return not self._was_run and self._run_vertex.is_source and not self._dependencies_to_add

    def depend_on(self, dependency):
        """Makes self depend on another dependency.
        If self already depends on it, this method does nothing.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> a.is_my_dependency(b)
        False
        >>> a.depend_on(b)
        >>> a.is_my_dependency(b)
        True
        >>> a.depend_on(b)
        >>> a.is_my_dependency(b)
        True

        """

        assert self._dependency_graph_weakref is not None, 'The Dependency must belong to a Dependency Graph before depending'
        assert dependency._dependency_graph_weakref is not None, 'The Dependency must belong to a Dependency Graph before depending'

        # if already dependency, do nothing
        if self.is_my_dependency(dependency):
            return

        if not self._on_dep_run_depth:
            self._depend_on(dependency)
        else:
            self._dependencies_to_add.add(dependency)

    def _depend_on(self, dependency):
        
        Vertex.connect(dependency._full_vertex, self._full_vertex)
        Vertex.connect(dependency._run_vertex, self._run_vertex)
        nrsw = self._dependency_graph_weakref()._not_run_source_wrs
        my_wr = wr(self)
        if my_wr in nrsw:
            nrsw.remove(my_wr)
        if dependency._was_run:
            self._on_dep_run(dependency)

    def revdepend_on(self, dependant):
        """Makes dependant depend on self.
        If dependant depends on self already, this method does nothing.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> a.is_my_dependency(b)
        False
        >>> b.revdepend_on(a)
        >>> a.is_my_dependency(b)
        True
        >>> b.revdepend_on(a)
        >>> a.is_my_dependency(b)
        True
        
        """
        dependant.depend_on(self)

    @property
    def dependencies(self):
        """Returns a view of my dependencies.

        >>> dg = DependencyGraph()
        >>> a = Dependency()
        >>> b = Dependency()
        >>> c = Dependency()
        >>> dg.add_dependency(a)
        >>> dg.add_dependency(b)
        >>> dg.add_dependency(c)
        >>> a.depend_on(b)
        >>> a.depend_on(c)
        >>> b.depend_on(c)
        >>> set(a.dependencies) == set([b,c])
        True
        >>> set(b.dependencies) == set([c])
        True
        >>> set(c.dependencies)
        set([])

        """
        return DependencyView(self._full_vertex.prev_vertices)

    @property
    def not_yet_run_deps(self):
        """Returns a view of my dependencies that have not run yet"""
        return DependencyView(self._run_vertex.prev_vertices)

    def _schedule_to_run(self):
        if self._schedules_to_run:
            raise RuntimeError('Dependency loop detected')
        self._schedules_to_run = True

    def _unschedule_to_run(self):
        self._schedules_to_run = False
        return False

    def _on_dep_run(self, dependency):
        self._on_dep_run_depth = self._on_dep_run_depth + 1
        try:
            self.on_dep_run(dependency)
        finally:
            self._on_dep_run_depth = self._on_dep_run_depth - 1
            if not self._on_dep_run_depth:
                while self._dependencies_to_add:
                    self._depend_on(self._dependencies_to_add.popfirst())

    def on_dep_run(self, dependency):
        """Called each time one of dependencies was run"""
        if Vertex.are_connected(dependency._run_vertex, self._run_vertex):
            Vertex.disconnect(dependency._run_vertex, self._run_vertex)
        if self._run_vertex.is_source:
            self._dependency_graph_weakref()._not_run_source_wrs.add(wr(self))
        if not dependency._success:
            self._dep_failed = True

    def _mark_as_run(self, success, result):
        """called to mark that self completed"""
        assert self.may_run
        self._was_run = True
        self._dependency_graph_weakref()._not_run_source_wrs.remove(wr(self))
        self._success = success
        self._result = result

        # alert my revdeps about this
        for revdep in list(self._run_vertex.next_vertices):
            revdep._wr_dep()._on_dep_run(self)

    @property
    def was_run(self):
        return self._was_run

    @property
    def success(self):
        return self._success

    @property
    def result(self):
        return self._result
