# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Block writer file-like type for file-like objects.

>>> import sys
>>> f = BlockedFile(sys.stdout)
>>> f.write('a\\n')
a
>>> f.start_block('A')
>>> f.write('b\\n')
A {
    b
>>> f.start_block('B')
>>> f.start_block('C')
>>> f.write('c\\n')
    B {
        C {
            c
>>> f.stop_block()
        }
>>> f.write('d\\n')
        d
>>> f.stop_block()
    }
>>> f.write('e\\n')
    e
>>> f.stop_block()
}
>>> f.write('a'); print
a
>>> f.start_block('A')
Traceback (most recent call last):
    ...
BlockedFileException: can start only on start of a line
>>> f.write('A\\nb\\n')
A
b
>>> f.start_block('B')
>>> f.write('c'); print
B {
    c
>>> f.stop_block()
Traceback (most recent call last):
    ...
BlockedFileException: can stop only on start of a line
>>> f.write('B\\nC\\n')
B
    C
>>> f.stop_block()
}
>>> f.stop_block()
Traceback (most recent call last):
    ...
BlockedFileException: no block to stop
>>> f.start_block('empty')
>>> f.stop_block()
>>> f.write('')
>>> f.flush()
>>> import tempfile, os
>>> temp = tempfile.NamedTemporaryFile(delete = False)
>>> name = temp.name
>>> f = BlockedFile(temp)
>>> f.start_block('A')
>>> f.write('a\\n')
>>> f.close()
>>> open(name, 'rb').read()
'A {\\n    a\\n}\\n'
>>> os.unlink(name)

"""

from .indentedfile import IndentedFile

class BlockedFileException(Exception): pass

class BlockedFile(object):
    """Block writer file-like type for file-like objects."""
    __slots__ = ['block_title_suffix',
                 'block_start',
                 'block_end',
                 'active_blocks',
                 'blocks_to_activate',
                 'indented_file',
                 ]
    def __init__(self,
                 internal_file,
                 block_title_suffix = ' ',
                 block_start = '{',
                 block_end = '}',
                 indentation = '    ',
                 ):
        self.block_title_suffix = block_title_suffix
        self.block_start = block_start
        self.block_end = block_end
        self.active_blocks = []
        self.blocks_to_activate = []
        self.indented_file = IndentedFile(internal_file, indentation = indentation)

    def start_block(self, block_title):
        if not self.indented_file.on_start_of_line:
            raise BlockedFileException('can start only on start of a line')

        self.blocks_to_activate.append(block_title)

    def stop_block(self):
        if not self.indented_file.on_start_of_line:
            raise BlockedFileException('can stop only on start of a line')
        
        if self.blocks_to_activate:
            del self.blocks_to_activate[-1]
            return

        if not self.active_blocks:
            raise BlockedFileException('no block to stop')

        del self.active_blocks[-1]
        self.indented_file.stop_indent()
        self.indented_file.write(self.block_end + '\n')

    def write(self, data):
        if not data:
            return

        for block_title in self.blocks_to_activate:
            self.indented_file.write(block_title + self.block_title_suffix)
            self.indented_file.write(self.block_start + '\n')
            self.indented_file.start_indent()

        self.active_blocks.extend(self.blocks_to_activate)
        del self.blocks_to_activate[:]

        self.indented_file.write(data)

    def flush(self):
        self.indented_file.flush()

    def close(self):
        while self.active_blocks:
            self.stop_block()
        self.indented_file.close()
