#!/usr/bin/env python

# Copyright (c) 2016-2020, Adam Karpierz
# Licensed under the BSD license
# https://opensource.org/licenses/BSD-3-Clause

# Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997, 2000
#  The Regents of the University of California.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

import sys
import os
import getopt
import ctypes as ct

import libpcap as pcap

#ifndef lint
copyright = "@(#) Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, "\
            "1995, 1996, 1997, 2000\n"\
            "The Regents of the University of California.  "\
            "All rights reserved.\n"
#endif

MAXIMUM_SNAPLEN = 262144

try:
    statustostr = pcap.statustostr
except AttributeError:
    statustostr = lambda status: str(status).encode("utf-8")


def main(argv=sys.argv):

    global program_name
    program_name = os.path.basename(argv[0])

    try:
        opts, args = getopt.getopt(argv[1:], "i:Ips:aB:")
    except getopt.GetoptError:
        usage()

    device = None
    dorfmon = False
    dopromisc = False
    snaplen = MAXIMUM_SNAPLEN
    bufsize = 0
    useactivate = False
    for opt, optarg in opts:
        if opt == '-i':
            device = optarg.encode("utf-8")
        elif opt == '-I':
            dorfmon = True
            useactivate = True  # required for rfmon
        elif opt == '-p':
            dopromisc = True
        elif opt == '-s':
            try:
                long_snaplen = int(optarg)
            except:
                error("invalid snaplen {}", optarg)
            if not (0 <= long_snaplen <= MAXIMUM_SNAPLEN):
                error("invalid snaplen {}", optarg)
            elif long_snaplen == 0:  # <AK> fix, was: snaplen == 0:
                snaplen = MAXIMUM_SNAPLEN
            else:
                snaplen = long_snaplen
        elif opt == '-B':
            try:
                bufsize = int(optarg) * 1024
            except:
                error("invalid packet buffer size {}", optarg)
            if bufsize <= 0:
                error("invalid packet buffer size {}", optarg)
            useactivate = True  # required for bufsize
        elif opt == '-a':
            useactivate = True
        else:
            usage()

    status = 0

    ebuf = ct.create_string_buffer(pcap.PCAP_ERRBUF_SIZE)

    if device is None:
        devlist = ct.POINTER(pcap.pcap_if_t)()
        if pcap.findalldevs(ct.byref(devlist), ebuf) == -1:
            error("{!s}", ebuf.value.decode("utf-8", "ignore"))
        if not devlist:
            error("no interfaces available for capture")
        device = devlist[0].name
        pcap.freealldevs(devlist)

    if useactivate:

        pd = pcap.create(device, ebuf)
        if not pd:
            error("{!s}: pcap.create failed: {!s}",
                  device.decode("utf-8"),
                  ebuf.value.decode("utf-8", "ignore"))

        status = pcap.set_snaplen(pd, snaplen)
        if status != 0:
            error("{!s}: pcap.set_snaplen failed: {!s}",
                  device.decode("utf-8"),
                  statustostr(status).decode("utf-8", "ignore"))

        if dopromisc:
            status = pcap.set_promisc(pd, 1)
            if status != 0:
                error("{!s}: pcap.set_promisc failed: {!s}",
                      device.decode("utf-8"),
                      statustostr(status).decode("utf-8", "ignore"))

        if dorfmon:
            try:
                status = pcap.set_rfmon(pd, 1)
            except AttributeError:
                error("pcap.set_rfmon is not available on this platform")
            if status != 0:
                error("{!s}: pcap.set_rfmon failed: {!s}",
                      device.decode("utf-8"),
                      statustostr(status).decode("utf-8", "ignore"))

        status = pcap.set_timeout(pd, 1000)
        if status != 0:
            error("{!s}: pcap.set_timeout failed: {!s}",
                  device.decode("utf-8"),
                  statustostr(status).decode("utf-8", "ignore"))

        if bufsize != 0:
            status = pcap.set_buffer_size(pd, bufsize)
            if status != 0:
                error("{!s}: pcap.set_buffer_size failed: {!s}",
                      device.decode("utf-8"),
                      statustostr(status).decode("utf-8", "ignore"))

        status = pcap.activate(pd)
        if status < 0:
            # pcap.activate() failed.
            error("{!s}: {!s}\n({!s})",
                  device.decode("utf-8"),
                  statustostr(status).decode("utf-8", "ignore"),
                  pcap.geterr(pd).decode("utf-8", "ignore"))
        elif status > 0:
            # pcap.activate() succeeded, but it's warning us
            # of a problem it had.
            warning("{!s}: {!s}\n({!s})",
                    device.decode("utf-8"),
                    statustostr(status).decode("utf-8", "ignore"),
                    pcap.geterr(pd).decode("utf-8", "ignore"))
        else:
            print("{!s} opened successfully".format(device.decode("utf-8")))

        pcap.close(pd)

    else:

        ebuf.value = b""
        pd = pcap.open_live(device, 65535, 0, 1000, ebuf)
        if not pd:
            error("{!s}", ebuf.value.decode("utf-8", "ignore"))
        elif ebuf.value:
            warning("{!s}", ebuf.value.decode("utf-8", "ignore"))
        else:
            print("{!s} opened successfully".format(device.decode("utf-8")))

        pcap.close(pd)

    return 1 if status < 0 else 0


def usage():
    global program_name
    print("Usage: {} [ -Ipa ] [ -i interface ] [ -s snaplen ] "
          "[ -B bufsize ]".format(program_name), file=sys.stderr)
    sys.exit(1)


def error(fmt, *args):
    global program_name
    print("{}: ".format(program_name), end="", file=sys.stderr)
    print(fmt.format(*args), end="", file=sys.stderr)
    if fmt and fmt[-1] != '\n':
        print(file=sys.stderr)
    sys.exit(1)


def warning(fmt, *args):
    global program_name
    print("{}: WARNING: ".format(program_name), end="", file=sys.stderr)
    print(fmt.format(*args), end="", file=sys.stderr)
    if fmt and fmt[-1] != '\n':
        print(file=sys.stderr)


sys.exit(main() or 0)
