# Copyright (c) gocept gmbh & co. kg
# See also LICENSE.txt

import gocept.net.configure.dhcpd
import gocept.net.dhcp
import gocept.net.directory
import mock
import netaddr
import tempfile
import unittest

from gocept.net.configure.dhcpd import DHCPd, HostsFormatter, NetworkFormatter


class DHCPdDirectoryAccessTest(unittest.TestCase):

    disclaimer = '# auto-generated by localconfig-dhcpd\n\n'

    def setUp(self):
        self.p_directory = mock.patch('gocept.net.configure.dhcpd.Directory')
        self.fake_directory = self.p_directory.start()

    def tearDown(self):
        self.p_directory.stop()

    def test_complete_config_acceptance(self):
        """This tests tries to compile most of the nasty cases."""
        self.fake_directory().lookup_networks_details.return_value = {
            'fe': [dict(cidr='172.20.2.0/24', dhcp=True)],
            'srv': [dict(cidr='172.20.3.0/24', dhcp=True),
                    dict(cidr='172.30.3.0/24', dhcp=True)]}
        self.fake_directory().list_nodes_addresses.return_value = [
            {'ip': '172.20.3.7/24',
             'mac': '00:13:72:4f:e8:6b',
             'name': 'barney',
             'vlan': 'srv'},
            {'ip': '172.30.3.3/24',
             'mac': '00:13:72:4f:e8:6b',
             'name': 'barney',
             'vlan': 'srv'},
            {'ip': '172.30.3.5/24',
             'mac': '02:00:00:03:10:4e',
             'name': 'test04',
             'vlan': 'srv'},
            {'ip': '172.30.3.6/24',
             'mac': '02:00:00:03:10:4e',
             'name': 'test04',
             'vlan': 'srv'},
            {'ip': '172.30.3.8/24',
             'mac': '02:00:00:03:10:4e',
             'name': 'test04',
             'vlan': 'srv'},
            {'ip': '172.30.3.2/24',
             'mac': '02:00:00:03:10:4e',
             'name': 'test03',
             'vlan': 'srv'},
            {'ip': '172.20.2.5/24',
             'mac': '02:00:00:02:10:4e',
             'name': 'test03',
             'vlan': 'fe'}]
        d = DHCPd('loc', 4)
        d.query_directory()
        self.maxDiff = None
        self.assertMultiLineEqual("""\
# auto-generated by localconfig-dhcpd

shared-network fe {
    subnet 172.20.2.0 netmask 255.255.255.0 {
        range 172.20.2.4 172.20.2.4;
        range 172.20.2.6 172.20.2.254;
        option subnet-mask 255.255.255.0;
        option routers 172.20.2.1;
        authoritative;
    }
}

shared-network srv {
    subnet 172.20.3.0 netmask 255.255.255.0 {
        range 172.20.3.4 172.20.3.6;
        range 172.20.3.8 172.20.3.254;
        option subnet-mask 255.255.255.0;
        option routers 172.20.3.1;
        authoritative;
    }
    subnet 172.30.3.0 netmask 255.255.255.0 {
        range 172.30.3.4 172.30.3.4;
        range 172.30.3.7 172.30.3.7;
        range 172.30.3.9 172.30.3.254;
        option subnet-mask 255.255.255.0;
        option routers 172.30.3.1;
        authoritative;
    }
}

host barney {
    hardware ethernet 0:13:72:4f:e8:6b;
    fixed-address 172.20.3.7;
    option host-name "barney";
}

host test03-fe-0 {
    hardware ethernet 2:0:0:2:10:4e;
    fixed-address 172.20.2.5;
    option host-name "test03";
}

host test03-srv-1 {
    hardware ethernet 2:0:0:3:10:4e;
    fixed-address 172.30.3.2;
    option host-name "test03";
}
""", d.render())

    def test_query_directory_should_tolerate_missing_keys(self):
        self.fake_directory().list_nodes_addresses.return_value = [
            {'ip': '172.30.3.2/24',
             'name': 'test01'},
            {'mac': '02:00:00:03:12:95',
             'name': 'test02'}]
        d = DHCPd('loc')
        d.query_directory()
        self.assertEqual(d.render(), self.disclaimer)

    def test_query_directory_should_tolerate_malformed_addresses(self):
        self.fake_directory().list_nodes_addresses.return_value = [
            {'ip': '',
             'mac': '02:00:00:03:12:94',
             'name': 'test01'},
            {'ip': '172.30.3.3/24',
             'mac': '',
             'name': 'test02'}]
        d = DHCPd('loc')
        d.query_directory()
        self.assertEqual(d.render(), self.disclaimer)

    def test_render_should_read_includefile(self):
        d = DHCPd('location')
        with tempfile.NamedTemporaryFile(prefix='inc1') as tf1, \
             tempfile.NamedTemporaryFile(prefix='inc2') as tf2:
            tf1.write('# static include 1\n')
            tf1.flush()
            tf1.seek(0)
            tf2.write('# static include 2\n')
            tf2.flush()
            tf2.seek(0)
            self.assertMultiLineEqual(
                d.render([tf1.name, tf2.name]), """\
# auto-generated by localconfig-dhcpd

# static include 1
# static include 2
""")


class HostFormatterTest(unittest.TestCase):

    mac1 = netaddr.EUI('00:11:43:d7:8a:70', dialect=netaddr.mac_unix)

    def test_namechooser_should_pick_hostname_vlan(self):
        pass

    def test_ipv4(self):
        hosts = gocept.net.dhcp.Hosts().add(gocept.net.dhcp.HostAddr(
            'host1', 'vlan', self.mac1, netaddr.IPNetwork('10.0.1.1/24')))
        formatter = HostsFormatter.new(4, hosts)
        self.assertMultiLineEqual(str(formatter), """\
host host1 {
    hardware ethernet 0:11:43:d7:8a:70;
    fixed-address 10.0.1.1;
    option host-name "host1";
}
""")

    def test_ipv6(self):
        hosts = gocept.net.dhcp.Hosts().add(gocept.net.dhcp.HostAddr(
            'h1', 'vlan', self.mac1, netaddr.IPNetwork('2001:db8:1::1/64')))
        formatter = HostsFormatter.new(6, hosts)
        self.assertMultiLineEqual(str(formatter), """\
host h1 {
    hardware ethernet 0:11:43:d7:8a:70;
    fixed-address6 2001:db8:1::1;
    option host-name "h1";
}
""")

    def test_namechooser_one_host(self):
        self.assertListEqual(['myhost'], list(HostsFormatter.choose_names(
            [gocept.net.dhcp.HostAddr('myhost', 'n1', None, None)])))

    def test_namechooser_multiple_hosts(self):
        self.assertListEqual(
            ['hosta-n1-0', 'hostb-n1-1'],
            list(HostsFormatter.choose_names(
                [gocept.net.dhcp.HostAddr('hosta', 'n1', None, None),
                 gocept.net.dhcp.HostAddr('hostb', 'n1', None, None)])))


class NetworkFormatterTest(unittest.TestCase):

    def test_ipv4_subnet(self):
        shnet = gocept.net.dhcp.SharedNetwork()
        shnet.register(gocept.net.dhcp.Subnet(
            netaddr.IPNetwork('172.20.3.0/24'), True))
        shnet.register(gocept.net.dhcp.Subnet(
            netaddr.IPNetwork('172.20.4.0/24'), False))
        formatter = NetworkFormatter.new(4, shnet, 'srv')
        self.assertMultiLineEqual(str(formatter), """\
shared-network srv {
    subnet 172.20.3.0 netmask 255.255.255.0 {
        range 172.20.3.4 172.20.3.254;
        option subnet-mask 255.255.255.0;
        option routers 172.20.3.1;
        authoritative;
    }
    subnet 172.20.4.0 netmask 255.255.255.0 {
        option subnet-mask 255.255.255.0;
        option routers 172.20.4.1;
        authoritative;
    }
}

""")

    def test_ipv6_subnet(self):
        shnet = gocept.net.dhcp.SharedNetwork()
        shnet.register(gocept.net.dhcp.Subnet(
            netaddr.IPNetwork('2001:db8:2::/64'), True))
        shnet.register(gocept.net.dhcp.Subnet(
            netaddr.IPNetwork('2001:db8:3::/64'), False))
        formatter = NetworkFormatter.new(6, shnet, 'fe')
        self.assertMultiLineEqual(str(formatter), """\
shared-network fe {
    subnet6 2001:db8:2::/64 {
        range6 2001:db8:2:0:d1c0::/80;
        authoritative;
    }
    subnet6 2001:db8:3::/64 {
        authoritative;
    }
}

""")
