# setup.py
# Copyright 2013 Roger Marsh
# Licence: See LICENCE (BSD licence)
"""DPT API setup script
"""
# There is a Makefile in the DPT distribuion file with multi-language support
# in mind.
#
# This script works directly on the zipped DPT distribution file.
#
# The attempt to use the Mingw32CCompiler class fails in three ways.
# C++ #include statements refer to stdafx.h but the file name is StdAfx.h
# On fixing this by renaming the file, the first few *.cpp files compile but
# bmset.cpp fails at line 53 of du1step.h "error: forward declaration of 'const
# struct dpt::DU1FieldIndex'".  Eight other files fail to compile for the same
# reason.
# It is not clear how to persuade build_ext to use the mingw toolset to do the
# SWIG part of the build.  The nine compilation failures were ignored while
# trying to do this.
#
# No difference attempting to force the command to look like the one used in
# dptMakefile by calls to set_executables().
#
# This setup script does the C++ part of the build by driving the make in
# dptMakefile (the nine modules above are compiled successfully here) and
# then completes the package build using distutils.core.setup.

# The change in status of distutils to legacy prompted some work on switching
# to setuptools, concurrent with upgrade to FreeBSD 10.1 and a portsnap from
# mid-November 2014.  pyNN-setuptoolsNN-5.5.1 was installed as a dependency of
# something so I used that.  NN values are (27 33 34), with 335 being the main
# version for development work.

# With distutils.core.setup(...) 'python3.3 setup.py bdist' succeeds.
# With setuptools.setup(...) 'python3.3 setup.py bdist' fails:
# running build_py
# error: can't copy 'icence.txt': doesn't exist or not a regular file.

# The relevant clauses of setup() call are:
#        package_dir={'dptdb':''},
#        packages=['dptdb'],
#        package_data={'': ['licence.txt', 'LICENCE', '_dptapi.pyd']},

# I had already downloaded the latest version of setuptools to install it on
# Microsoft Windows XP before noticing that the installation had been done for
# me on FreeBSD, so the XP box got setuptools-12.0.5 instead.  January 2015.

# Here setuptools.setup(...) 'python setup.py bdist' fails as well:
# running build_py
# error: can't copy 'icence.txt': doesn't exist or not a regular file.

# The swig command is run as 'wine swig ...' on FreeBSD, but just 'swig ...' on
# Microsoft Windows.  Setup is run as setup(...) function call on Microsoft
# Windows, but as 'wine python _wine_setup.py ...' process call on FreeBSD.

# Reverting to distutils.core.setup(...) gets both commands to succeed.

# Changing directory structure and relevant clauses of setup() call to:
#        include_package_data=True,
#        packages=['dptdb'],
#        package_data={'': ['licence.txt', '_dptapi.pyd']},
# given the setup job is run in a svn checkout directory avoids the problem.

# The Microsoft Windows version of Python 3.4.2 from the msi file includes
# setuptools 2.1 which I found out on installing this Python.  February 2015.

# Moved from swigwin-2.0.8 to swigwin-4.0.1 as default when trying out the next
# MinGW after 6.3.0 that I dowmloaded: 8.2.0 which introduced a warning and a
# note into the compilation of the code generated by swig 2.0.8.  These went
# away at 4.0.1.  November 2019.

# Changed the use of PATH_TO_PYTHON and PYTHON_VERSON to be flexible enough to
# follow the change in default Python locations between 3.4 and 3.5.
# November 2019.


import os
import posixpath
import sys
import subprocess
import zipfile
import shutil
import setuptools
import platform
_urllib_available = True
try:
    import urllib.request
except ImportError:
    _urllib_available = False

# Used to check a previously extracted version is the one being extracted.
_EXTRACT = os.path.join('dptdb', 'dptextract')

# Edits to _EXTRACT required for successful build in environment with the
# version of mingw32-g++ used.
_BUILD = os.path.join('dptdb', 'dptbuild')


def setup(
    dpt_distribution_file='DPT_V3R0_DBMS.ZIP',
    dpt_documentation_file='DPT_V3R0_DOCS.ZIP',
    dpt_downloads_from='http://www.solentware.co.uk/files/',
    path_to_swig=os.path.join('C:', 'swigwin-4.0.1'),
    allow_36_and_later_hack=False,
    **attrs):
    """Extract DPT source code from distribution and call distutils.setup

    dpt_distribution_file is the default DPT distribution
    dpt_documentation_file is the default DPT documentation
    dpt_downloads_from is the site from which DPT files are downloaded
    path_to_swig is the default location of the swig command
    allow_36_and_later_hack enables the hack copying to site-packages

    The following command line arguments override these defaults:
    DPT_DIST       dpt_distribution_file
    DPT_DOCS       dpt_documentation_file
    PATH_TO_SWIG   path_to_swig

    The other command line options are described below.

    MINGW345
    gcc-3.4.5 is the most recent version supported by MinGW known to compile
    the DPT API in dpt_distribution_file on Microsoft Windows or Wine without
    any modifications.  The MINGW345 keyword is given on the command line if
    the job should assume mingw32-g++ version 3.4.5 is available.

    When cross-compiling on *nix some changes from '#include "dir\file.h' to
    '#include "dir/file.h' are needed whatever version is available.  Starting
    with an unknown version after MinGW-3.4.5 the header file names <string.h>,
    <limits.h>, and <stdlib.h> must be included, in stdafx.h for convenience,
    to enable successful builds.  MinGW-4.7.2 is known to need this change.

    Another instance of this kind was spotted at MinGW-6.3.0 related to the
    C++11 standard it seems.

    PATH_TO_PYTHON
    Use this when building under Microsoft Windows if Python is not installed
    directly in the default location.  C:\Python34 for 3.4 versions of Python.
    If the D: drive were used instead then PATH_TO_PYTHON=D:\Python is the
    setting, without the version number which goes in PYTHON_VERSION if needed.

    PYTHON_VERSION
    Used to pick out the python<major minor version>.lib filewhen creating the
    _dptapi.pyd file: PYTHON_VERSION=27 for example.
    Use this when cross-compiling if the build under Wine is to be done for a
    version of Python other than the one running this script.  Assumption is
    all the Pythons are at same version.
    On Microsoft Windows the swig build is done by the Python running this
    setup.py script, and PYTHON_VERSION for the makefile is figured from that.

    clean_extract
    Clear out dptapi_python_wrap.cxx, dptapi_python_wrap.o, _dptapi_pyd and
    dptapi.py, generated by SWIG, and the version and licence files, and all
    stuff created from the DPT distribution while doing this.  The files made
    by setuptools.setup() are left alone.

    If present clean_extract is the only command done.

    clean_swig
    Clear out dptapi_python_wrap.cxx, dptapi_python_wrap.o, _dptapi_pyd and
    dptapi.py, generated by SWIG, and the version and licence files.

    If present, and clean_extract is not, clean_swig is the only command done.

    """
    make_arguments = (
        'PATH_TO_PYTHON=',
        'MINGW345',
        'PATH_TO_SWIG=',
        'DPT_DIST=',
        'DPT_DOCS=',
        'PYTHON_VERSION=',
        )

    # In dptMakefile clean_extract should imply clean_swig without it being a
    # pre-requisite relationship.
    clean_up_arguments = ('clean_extract', 'clean_swig')
    
    # Determine the WINE command line option for the 'make' job
    wine = 'WINE=wine'
    if sys.platform == 'win32':
        wine = False

        # The problem here is finding a reliable test to distinguish running
        # under Wine from running under Microsoft Windows.
        # So just state the MSYS requirement for the record.
        sys.stdout.write(
            'On Microsoft Windows this setup must be run in an MSYS shell.\n')
        sys.stdout.write(
            'Otherwise it is assumed the build will be done under Wine.\n')

    if len(sys.argv) < 2:
        sys.stdout.write('Please specify a setup command to setup.py\n')
        return

    # If a clean up argument is present just do the first one found and exit.
    for clean_up in clean_up_arguments:
        for a in sys.argv[1:]:
            if a.lower() == clean_up:

                # GNU make is called gmake in BSD but usually make otherwise.
                # Note wine is not available on OpenBSD.
                if platform.system() in ('FreeBSD', 'OpenBSD', 'NetBSD'):
                    command = 'gmake'
                else:
                    command = 'make'

                sp = subprocess.Popen(
                    [command, '-f', 'dptMakefile', clean_up], cwd='dptdb')
                
                r = sp.wait()
                if r != 0:
                    sys.stdout.write(
                        ''.join(('dptMakefile ', clean_up, ' command fails\n')))
                else:
                    sys.stdout.write(
                        ''.join(('dptMakefile ', clean_up, ' command done\n')))
                return
    
    for exclude in make_arguments:
        if sys.argv[1].upper().startswith(exclude):
            sys.stdout.write(
                ' '.join((sys.argv[1], 'is not a setup command\n')))
            return
    
    for a in sys.argv[2:]:
        if a.startswith('DPT_DIST='):
            dpt_distribution_file = a.split('=')[-1].strip()
    
    for a in sys.argv[2:]:
        if a.startswith('DPT_DOCS='):
            dpt_documentation_file = a.split('=')[-1].strip()
    
    for a in sys.argv[2:]:
        if a.startswith('PATH_TO_SWIG='):
            path_to_swig = a.split('=')[-1].strip()

    mingw345 = False
    for a in sys.argv[2:]:
        if a.lower() == 'mingw345':
            mingw345 = 'GCC_VERSION_PATCHING=false'

    path_to_python = None
    for a in sys.argv[2:]:
        if a.startswith('PATH_TO_PYTHON='):
            path_to_python = a

    python_version = None
    for a in sys.argv[2:]:
        if a.startswith('PYTHON_VERSION='):
            python_version = a

    downloads = (dpt_distribution_file, dpt_documentation_file)
    if not _urllib_available:
        sys.stdout.write(
            ''.join((
                'Module urllib.request is not in this Python.\n',
                'You will have to download:\n ',
                '\n '.join(downloads),
                '\nmanually if not already present.\n')))
    for distfile in downloads:
        dfile = os.path.join('dptdb', distfile)
        if not os.path.exists(dfile):
            durl = ''.join((dpt_downloads_from, distfile))
            try:
                ddata = urllib.request.urlopen(durl).read()
                dfw = open(dfile, 'wb')
                try:
                    dfw.write(ddata)
                finally:
                    dfw.close()
                sys.stdout.write(' '.join((durl, 'downloaded\n')))
            except:
                sys.stdout.write(
                    ''.join((
                        'Download ',
                        durl,
                        ' failed.\n',
                        'You may have to download this file manually.\n')))
                return

    distfile = os.path.join('dptdb', dpt_distribution_file)
    if not os.path.exists(distfile):
        sys.stdout.write(''.join(('setup cannot find ', distfile, '\n')))
        return
    if not zipfile.is_zipfile(distfile):
        sys.stdout.write(' '.join((distfile, 'is not a zipped file\n')))
        return
    zf = zipfile.ZipFile(distfile)
    ok = False
    try:
        zipbase = os.path.join(_EXTRACT, 'zipcompare')
        present = False
        absent = False
        matched = True
        for n in zf.namelist():
            if os.path.exists(os.path.join(_EXTRACT, n)):
                if os.path.isfile(os.path.join(_EXTRACT, n)):
                    present = True
                    f = open(os.path.join(_EXTRACT, n), 'rb')
                    # 'rb' is not needed on MS Windows Python 3.3.0
                    # 'rb' is needed on Python 3.3.0 built from Python sources
                    # on FreeBSD.
                    # Not known what the FreeBSD port of Python 3.3.n does.
                    # This port is available now, so will find out shortly.
                    try:
                        pft = f.read()
                    finally:
                        f.close()
                    zf.extract(n, path=zipbase)
                    f = open(os.path.join(zipbase, n), 'rb')
                    try:
                        eft = f.read()
                    finally:
                        f.close()
                    del f
                    if eft != pft:
                        sys.stdout.write(' '.join(('file', n, 'changed\n')))
                        matched = False
            else:
                absent = True
            if n == 'licence.txt':
                if os.path.exists(os.path.join(_EXTRACT, n)):
                    if os.path.isfile(os.path.join(_EXTRACT, n)):
                        present = True
                        f = open(os.path.join(_EXTRACT, n), 'rb')
                        try:
                            pft = f.read()
                        finally:
                            f.close()
                        f = open(os.path.join(zipbase, n), 'rb')
                        try:
                            eft = f.read()
                        finally:
                            f.close()
                        del f
                        if eft != pft:
                            sys.stdout.write(' '.join(('file', n, 'changed\n')))
                            matched = False
                else:
                    absent = True
        if present and absent:
            # error
            pass # ok initialised False
        elif absent:
            # extract files
            zf.extractall(path=_EXTRACT)
            zf.extract('licence.txt', path='dptdb') # for ease of redistribution
            ok = True
        elif present:
            ok = matched # all existing files must be unchanged
        if present:
            shutil.rmtree(zipbase)
    finally:
        zf.close()
    if not ok:
        sys.stdout.write(
            ' '.join(('setup abandonned because existing extracted',
                      'files do not match zipped distribution file.\n')))
        return
    
    builddir = os.path.join(os.getcwd(), _BUILD)
    for bd in (
        builddir,
        os.path.join(builddir, 'stdafx'),
        os.path.join(builddir, 'source'),
        os.path.join(builddir, 'include'),
        os.path.join(builddir, 'source', 'dbapi'),
        os.path.join(builddir, 'include', 'dbapi'),
        ):
        try:
            os.mkdir(bd)
        except:
            if not os.path.isdir(bd):
                sys.stdout.write('Create build directory fails\n')
                return
    stdafx_copy = (
        # C++ #include statements refer to stdafx.h
        # dptMakefile references to stdafx.cpp and stdafx.h are awkward because
        # of the punctuation characters in the path names.
        (os.path.join(
            _EXTRACT, 'sample projects', 'HelloWorld! (MSVC)', 'StdAfx.h'),
         os.path.join(_EXTRACT, 'stdafx', 'stdafx.h')),
        (os.path.join(
            _EXTRACT, 'sample projects', 'HelloWorld! (MSVC)', 'StdAfx.cpp'),
         os.path.join(_EXTRACT, 'stdafx', 'stdafx.cpp')),
        )
    for inp, outp in stdafx_copy:
        if os.path.isfile(outp):
            f = open(inp)
            try:
                pft = f.read()
            finally:
                f.close()
            f = open(outp)
            try:
                eft = f.read()
            finally:
                f.close()
            del f
            if eft != pft:
                ok = False
        else:
            f = open(inp)
            try:
                pft = f.read()
                if len(os.path.dirname(outp)):
                    try:
                        os.makedirs(os.path.dirname(outp))#, exist_ok=True)
                    except OSError:
                        pass # assume target directory already exists
                fo = open(outp, 'w')
                try:
                    fo.write(pft)
                finally:
                    fo.close()
                del fo
            finally:
                f.close()
            del f
    dptapi_python_copy = (
        # dptMakefile references to dptapi_python.i are awkward because of the
        # punctuation characters in the path name.
        (os.path.join(
            _EXTRACT, 'sample projects', 'DPT with Python', 'dptapi_python.i'),
         os.path.join(_BUILD, 'dptapi_python.i'),
         os.path.join('dptdb', 'extend_dptapi_python'),
         '%extend dpt::APIRoundedDouble {'),
        )
    for inp, outp, merge, splitter in dptapi_python_copy:
        f = open(merge)
        try:
            mft = f.read()
        finally:
            f.close()
        del f
        if os.path.isfile(outp):
            f = open(inp)
            try:
                pft = f.read()
            finally:
                f.close()
            f = open(outp)
            try:
                eft = f.read()
            finally:
                f.close()
            spft = pft.split(splitter, 1)
            if len(spft) != 2:
                ok = False
            elif ''.join((spft[0], mft, splitter, spft[-1])) != eft:
                ok = False
        else:
            f = open(inp)
            try:
                spft = f.read().split(splitter)
                if len(spft) == 2:
                    if len(os.path.dirname(outp)):
                        try:
                            os.makedirs(os.path.dirname(outp))#, exist_ok=True)
                        except OSError:
                            pass # assume target directory already exists
                    fo = open(outp, 'w')
                    try:
                        fo.write(''.join((spft[0], mft, splitter, spft[-1])))
                    finally:
                        fo.close()
                    del fo
                else:
                    ok = False
            finally:
                f.close()
            del f


    def get_source_files(directory):
        """Return list of *.cpp source files without extension.""" 
        files =[]
        for f in os.listdir(directory):
            if os.path.isfile(os.path.join(directory, f)):
                p, e = os.path.splitext(f)
                if e in ('.cpp',):
                    files.append(p)
        return files


    def get_include_files(directory):
        """Return list of *.h source files without extension.""" 
        files =[]
        for f in os.listdir(directory):
            if os.path.isfile(os.path.join(directory, f)):
                p, e = os.path.splitext(f)
                if e in ('.h',):
                    files.append(p)
        return files


    def default_path_to_python(python_version):
        """Return default location of 32-bit python.exe for user install.

        Before Python 3.5 series this is C:/PythonNM.
        After Python 3.4 series this is:
        C:/Users/<user>/AppData/Local/Programs/Python/PythonNM-32 on Windows
        and:
        C:/Users/<user>/AppData/PythonNM-32 on Wine for two reasons.

        On Wine the ../Local hierarchy may not exist.  Perhaps it should not,
        or perhaps it is created first time something is installed.  Installing
        Python on Wine does not work for Python 3.5 and later, and one option
        is to install Python for user-only on Windows and copy PythonNM-32 to
        Wine.

        """
        # posixpath, not os.path, to handle the 'C:' components correctly.
        # On Windows the job is running in a msys shell, so it is posix rules
        # which must be used.

        if python_version is None:
            major_minor_version = ''.join(
                [str(vi) for vi in sys.version_info[:2]])
        else:
            major_minor_version = python_version.split('=', 1)[-1]
        if major_minor_version < '35':
            return posixpath.join('C:', 'Python' + major_minor_version)

        # 'USERNAME' is on Windows but not Wine, unless explicitly created.
        user = os.getenv('USERNAME') or os.getenv('USER')

        # 'users' not 'Users' because it is 'users' on Wine, where the case
        # matters, and 'Users' on Windows where the case does not matter.
        p = posixpath.join('C:',
                           'users',
                           user,
                           'AppData',
                           'Local',
                           'Programs',
                           'Python')
        if os.path.exists(os.path.dirname(p)):
            return posixpath.join(
                p, major_minor_version.join(('Python', '-32')))
        return posixpath.join('C:',
                              'users',
                              user,
                              'AppData',
                              major_minor_version.join(('Python', '-32')))

    
    def python_library_version(python_version):
        """Return version number for python<major minor version>.lib.

        The swig build in the makefile uses this to construct the file name.

        When python_version is None assume the same version of Python will do
        the build as is running this script.

        """
        if python_version is None:
            return '='.join((
                'PYTHON_VERSION',
                ''.join([str(vi) for vi in sys.version_info[:2]])))
        else:
            return python_version


    # Use make -f dptMakefile python ... for C++ build of DPT API.
    # The arguments to this wrapper of the setuputils.setup() call allow some
    # flexibility to the build without using command line options.

    # GNU make is called gmake in BSD but usually make otherwise.
    # Note wine is not available on OpenBSD.
    if platform.system() in ('FreeBSD', 'OpenBSD', 'NetBSD'):
        command = 'gmake'
    else:
        command = 'make'

    job = [
        command,
        '-f',
        '/'.join(os.path.join(
            '..', 'dptMakefile').split('\\')),
        'python']

    job.append(
        ''.join(
            ('DPT_NAMES=',
             ' '.join(get_source_files(
                 os.path.join(_EXTRACT, 'source'))),
             )))
    job.append(
        ''.join(
            ('DPTAPI_NAMES=',
             ' '.join(get_source_files(
                 os.path.join(_EXTRACT, 'source', 'dbapi'))),
             )))

    job.append(
        ''.join(
            ('DPT_INC_NAMES=',
             ' '.join(get_include_files(
                 os.path.join(_EXTRACT, 'include'))),
             )))
    job.append(
        ''.join(
            ('DPTAPI_INC_NAMES=',
             ' '.join(get_include_files(
                 os.path.join(_EXTRACT, 'include', 'dbapi'))),
             )))
    job.append(''.join(('OPTIONS=', '-O3')))
    job.append(''.join(('DEFINES=', '-DNDEBUG')))

    # Swig not expected to be on PATH.
    job.append(''.join(('PATH_TO_SWIG=', path_to_swig)))

    if mingw345:
        job.append(mingw345)
    if wine:
        job.append(wine)
    if path_to_python:
        job.append(path_to_python)
        job.append(python_library_version(python_version))
    elif wine:
        job.append(''.join((
            'PATH_TO_PYTHON=',
            default_path_to_python(python_version))))
        job.append(python_library_version(python_version))
    else:

        # On Microsoft Windows force make to use the Python version running
        # this job because this job will do the setuptools.setup() call.
        if python_version:
            sys.stdout.write(''.join((
                python_version, ' ignored on Microsoft Windows', '\n')))

        job.append(''.join((
            'PATH_TO_PYTHON=',
            default_path_to_python(None))))
        job.append(python_library_version(None))

    # For creating version number module only: not used in the swig build.
    job.append(''.join(('PYTHON_RUNNING_MAKE=', sys.executable)))

    sp = subprocess.Popen(job, cwd=builddir)
    
    r = sp.wait()
    if r != 0:
        sys.stdout.write('Build C++ extension module fails\n')
        return

    version = release = '0'
    dptdb_version = ('0', '0')
    version_file = os.path.join('..', 'version.py')
    for nv in open(os.path.join('dptdb', 'version.py')):
        nv = [v.strip() for v in nv.split('=')]
        if len(nv) == 2:
            n, v = nv
            if n == '_dpt_version':
                v = v[1:-1].split('.')
                if len(v) == 2:
                    version, release = v
            elif n == '_dptdb_version':
                v = v[1:-1].split('.')
                if len(v) in (2, 3):
                    dptdb_version = v

    # Default is version of Python running this script.
    if python_version is None:
        python_version = ''.join(
            [str(vi) for vi in sys.version_info[:2]])
    else:
        python_version = python_version.split('=')[-1]
    name = '-'.join((''.join(('dpt', version, '.', release)), 'dptdb'))

    # Remove 'make' arguments from sys.argv before setup() call.
    # Let setup complain if anything is wrong.
    argv = sys.argv[:]
    for e in range(len(sys.argv)-1, 1, -1):
        for exclude in make_arguments:
            if sys.argv[e].upper().startswith(exclude):
                del sys.argv[e]
                break

    if sys.platform == 'win32':
        long_description = open('README.txt').read()
        setuptools.setup(
            name=name,
            version='.'.join(dptdb_version),
            long_description=long_description,
            **attrs)

    # With the Microsoft Windows version of Python 3.6 and later the Wine job
    # will fail in some way, for example:
    # Unimplemented function api-ms-win-core-path-l1-1-0.dll.PathCchCan...izeEx
    # on a 64bit box running FreeBSD 12.1 amd64 with emulators/i386-wine,
    # without reaching if __name__ == '__main__' in the _wine_setup module.
    # or:
    # Creating tar archive
    # 0009:fixme:file:GetFileInformationByHandleEx ...
    # error: [WinError 120] Call not implemented: 'dpt3.0-dpt-0.7.2
    # after entering the setuptools.setup(..) call im the _wine_setup module.
    # This on a 64bit box running FreeBSD 11.3 i386 with emulators/wine.
    # It does not matter which Python does the sdist command, so let the one
    # running this script do it.
    elif sys.argv[1] == 'sdist':
        long_description = open('README.txt').read()
        setuptools.setup(
            name=name,
            version='.'.join(dptdb_version),
            long_description=long_description,
            **attrs)

    # The Wine job has not been known to fail for the reasons above on Pythons
    # earlier than 3.6 so let it do the install command.
    # 'python3.6 setup.py install' with MinGW-8.2.0
    # works at FreeBSD 11.3 i386 with emulators/wine
    # but not at FreeBSD 12.1 amd64 with emulators/i386-wine.
    elif sys.argv[1] == 'install' and python_version < '36':
        job = [
            'wine',
            os.path.join(
                default_path_to_python(python_version),
                'python.exe',
                ),
            '_wine_setup.py']
        job.extend(sys.argv[1:])
        job.append(name)
        job.append('.'.join(dptdb_version))
        sp = subprocess.Popen(job)
        r = sp.wait()
        if r != 0:
            sys.stdout.write('wine python setup.py ... fails\n')
            return

    # Assume the Wine job will fail for Python 3.6 and later.
    # If this hack is enabled 'import dptdb' works, but 'import dptdb.dptapi'
    # and equivalents fail stating 'libgcc_s_dw2-1.dll' and 'libstdc++-6.dll'
    # cannot be found.  In Python27 the import finds them.
    # Copy __init__.py, dptapi.py, _dptapi.pyd, and version.py from dptdb in
    # the distribution to dptdb in <path_to_python>/Lib/site-packages.
    # Check that dptdb does not exist in site-packages and that all four files
    # are in the distribution before copying anything.
    # All the potential destinations managed by setuptools are ignored, so
    # there may be a 'dptdb' in an 'egg' which has not been spotted.
    elif sys.argv[1] == 'install' and allow_36_and_later_hack:
        site_packages = os.path.join(default_path_to_python(python_version),
                                     'Lib',
                                     'site-packages')
        site_packages = os.path.expanduser(os.path.join(
            '~', '.wine', 'drive_c',
            os.path.join(*site_packages.split('/')[1:])))
        if 'dptdb' in os.getcwd() and 'dptdb' not in os.listdir(site_packages):
            source = os.path.join(os.getcwd(), 'dptdb')
            destination = os.path.join(site_packages, 'dptdb')
            files = set(os.listdir(source))
            copy = True
            copyfiles = '__init__.py', 'dptapi.py', '_dptapi.pyd', 'version.py'
            for f in copyfiles:
                if f not in files:
                    copy = False
                    break
            if copy:
                os.mkdir(destination)
                for f in copyfiles:
                    shutil.copy(os.path.join(source, f), destination)
                sys.stdout.write(
                    'Essential files in dptdb copied to site-packages.\n')
                sys.stdout.write(
                    "Existence of a 'dptdb' in an 'egg' was not checked.\n")
            else:
                sys.stdout.write(
                    'Some expected files missing: nothing copied.\n')
        else:
            sys.stdout.write(
                'Source dptdb missing or destination is in site-packages:\n')
            sys.stdout.write('Nothing copied.\n')

    # Assume the Wine job will fail for Python 3.6 and later.
    # Refuse to do the install command.
    elif sys.argv[1] == 'install':
        sys.stdout.write(
            '\nThe setup install command will fail at Python 3,6 and later.\n')
        sys.stdout.write(
            'The package build has probably succeeded, but there is little\n')
        sys.stdout.write(
            "point in copying to site-packages because 'import dptdb.dptapi'\n")
        sys.stdout.write(
            "will fail anyway.  The reports are 'libgcc_s_dw2-1.dll' and\n")
        sys.stdout.write(
            "'libstdc++-6.dll' cannot be found.  They are found when\n")
        sys.stdout.write(
            "Python 2.7, for example, does the import.\n")

    # Not trying to mend the world.
    # Let it succeed or fail as it wishes.
    # For example:
    # On a 64bit box running FreeBSD 11.3 i386 and emulators/wine the build
    # command succeeds. The setup was run from 'unix' Python 3.6 targetting
    # 'windows' Python 3.8.
    # On a 64bit box running FreeBSD 12.1 amd64 and emulators/i386-wine the
    # build command fails, but all the stuff before starting the job to run
    # setuptools.setup() has been done. The setup was run from 'unix' Python
    # 3.7 targetting 'windows' Python 3.7.
    else:
        job = [
            'wine',
            os.path.join(
                default_path_to_python(python_version),
                'python.exe',
                ),
            '_wine_setup.py']
        job.extend(sys.argv[1:])
        job.append(name)
        job.append('.'.join(dptdb_version))
        sp = subprocess.Popen(job)
        r = sp.wait()
        if r != 0:
            sys.stdout.write('wine python setup.py ... fails\n')
            return

    sys.argv[:] = argv


if __name__ == '__main__':
    
    setup(
        description='DPT database API wrappers built using SWIG',
        author='Roger Marsh',
        author_email='roger.marsh@solentware.co.uk',
        url='http://www.solentware.co.uk',
        packages=[
            'dptdb',
            'dptdb.test',
            ],
        include_package_data=True,
        package_data={
            '': ['licence.txt',
                 '_dptapi.pyd',
                 'DPT_V3R0_DOCS.ZIP',
                 'CONTACT',
                 ],
            },
        platforms='Microsoft Windows',
        license='BSD',
        classifiers=[
            'License :: OSI Approved :: BSD License',
            'Programming Language :: Python :: 2.7',
            'Programming Language :: Python :: 3.3',
            'Programming Language :: Python :: 3.4',
            'Programming Language :: Python :: 3.5',
            'Programming Language :: Python :: 3.6',
            'Programming Language :: Python :: 3.7',
            'Programming Language :: Python :: 3.8',
            'Operating System :: Microsoft :: Windows',
            'Topic :: Database',
            'Topic :: Software Development',
            'Intended Audience :: Developers',
            'Development Status :: 7 - Inactive',
            ],
        )
