# --------------------------------------------------------------------------
# Source file provided under Apache License, Version 2.0, January 2004,
# http://www.apache.org/licenses/
# (c) Copyright IBM Corp. 2015, 2016
# --------------------------------------------------------------------------


class ModelingObjectBase(object):

    __array_priority__ = 100

    """ModelingObjectBase()

    Parent class for all modeling objects (variables and constraints).

    This class is not intended to be instantiated directly.
    """
    __slots__ = ("_name", "_model", "_has_automatic_name")

    # @profile
    def __init__(self, model, name=None, has_automatic_name=False):
        self._name = name
        self._model = model
        self._has_automatic_name = has_automatic_name

    def _get_name(self):
        """ This property is used to get or set the name of the modeling object.

        """
        return self._name

    def set_name(self, name):
        self.check_name(name)
        self._name = name
        if name:
            self._has_automatic_name = False

    def check_name(self, new_name):
        # INTERNAL: basic method for checking names.
        pass  # pragma: no cover

    name = property(_get_name, set_name)

    def has_name(self):
        """ Checks whether the object has a name.

        Returns:
            True if the object has a name.

        """
        return self.name is not None

    def has_automatic_name(self):
        return self._has_automatic_name

    def has_user_name(self):
        """ Checks whether the object has a valid name given by the user.

        Returns:
            True if the object has a valid name given by the user.

        """
        return self.has_name() and not self._has_automatic_name

    def notify_automatic_name(self):
        self._has_automatic_name = True

    @property
    def model(self):
        """
        This property returns the :class:`docplex.mp.model.Model` to which the object belongs.
        """
        return self._model

    def _get_model(self):
        return self._model

    def is_in_model(self, model):
        return model and self._model == model

    def _check_model_has_solution(self):
        self.model.check_has_solution()

    def fatal(self, msg, *args):
        self.error_handler.fatal(msg, args)

    def error(self, msg, *args):
        self.error_handler.error(msg, args)

    def warning(self, msg, *args):
        self.error_handler.warning(msg, args)

    def trace(self, msg, *args):
        self.error_handler.trace(msg, args)

    @property
    def error_handler(self):
        return self._model.error_handler


class ModelingObject(ModelingObjectBase):

    __slots__ = ("_index", "_origin")

    @staticmethod
    def is_valid_index(idx):
        # INTERNAL: This is where the valid index check is performed
        return idx >= 0

    # @profile
    def __init__(self, model, name=None, index=-1, is_automatic_name=False):
        ModelingObjectBase.__init__(self, model, name, has_automatic_name=is_automatic_name)
        self._index = index
        self._origin = None

    def is_generated(self):
        """ Checks whether this object has been generated by another modeling object.

        If so, the origin object is stored in the ``_origin`` attribute.

        Returns:
            True if the objects has been generated.
        """
        try:
            is_generated = self._origin is not None
            return is_generated

        except AttributeError:
            return False

    def notify_origin(self, origin):
        if origin is not None:
            self._origin = origin

    def origin(self):
        return self._origin

    def __hash__(self):
        return id(self)

    @property
    def unchecked_index(self):
        return self._index

    def get_index(self):
        return self._index

    def set_index(self, idx):
        self._index = idx

    def has_valid_index(self):
        return self._index >= 0

    def _get_index_or_raise(self):
        ''' Returns the index if valid, otherwise raises an exception.'''
        if not self.has_valid_index():
            self.fatal("Modeling object {0!s} has invalid index: {1:d}", self, self._index)  # pragma: no cover
        return self._index

    index = property(get_index, set_index)
    safe_index = property(_get_index_or_raise)


class Expr(ModelingObjectBase):
    """Expr()

    Parent class for all expression classes.
    """
    __slots__ = ()

    def __init__(self, model, name=None):
        ModelingObjectBase.__init__(self, model, name)

    def clone(self):
        raise NotImplementedError  # pragma: no cover

    def copy(self, target_model, var_mapping):
        # internal
        raise NotImplementedError  # pragma: no cover

    def number_of_variables(self):
        """
        Returns:
            The number of variables in the expression.
        """
        return sum(1 for _ in self.iter_variables())  # pragma: no cover

    def iter_variables(self):
        """
        Iterates over all variables present in the solution.

        Returns:
            An iterator over all variables present in the expression.
        """
        raise NotImplementedError  # pragma: no cover

    def contains_var(self, dvar):
        """ Checks whether a variable is present in the expression.

        :param: dvar (:class:`docplex.mp.linear.Var`): A decision variable.

        Returns:
            True if the variable is present in the expression, else False.
        """
        raise NotImplementedError  # pragma: no cover

    def __contains__(self, dvar):
        """Overloads operator `in` for an expression and a variable.

        :param: dvar (:class:`docplex.mp.linear.Var`): A decision variable.

        Returns:
            True if the variable is present in the expression, else False.
        """
        return self.contains_var(dvar)

    def _num_to_stringio(self, oss, num, ndigits=None):
        # INTERNAL
        if ndigits is None:
            ndigits = self.model.float_precision
        if num == int(num):
            oss.write('%d' % num)
        else:
            # use second arg as nb digits:
            oss.write("{0:.{1}f}".format(num, ndigits))

    def __pos__(self):
        # + e is identical to e
        return self

