
# -*- coding: utf-8 -*-

u'''Test ellipsoidal function with the C(GeodTest.dat} from
U{C.F.F. Karney, GeographicLib<https://GeographicLib.SourceForge.io>},
also available U{here<https://Zenodo.org/record/32156>}, file C{GeodTest.dat}.
See also U{Geodesic calculations for an ellipsoid done right
<https://GeographicLib.SourceForge.io/scripts/geod-calc.html>}.
'''

__all__ = ('Tests',)
__version__ = '19.04.08'

from base import geographiclib, TestsBase

from pygeodesy import ellipsoidalKarney, ellipsoidalVincenty, \
                      Ellipsoids, VincentyError

_LLK = ellipsoidalKarney.LatLon
_LLV = ellipsoidalVincenty.LatLon


class Tests(TestsBase):

    def testGeodTest(self, test, line, fmtV, epsV, fmtK, epsK):
        # format: lat1 lon1 azimuth1 lat2 lon2 azimuth2 s12 as12 m12 S12
        lat1, lon1, azi1, lat2, lon2, azi2, s12, _, _, _ = map(float, test.split())

        try:  # compute Vincenty destination lat, lon and final bearing
            ll, fb = _LLV(lat1, lon1).destination2(s12, azi1)
            self.test(line + 'lat', ll.lat, lat2, fmt=fmtV, known=abs(ll.lat - lat2) < epsV)
            self.test(line + 'lon', ll.lon, lon2, fmt=fmtV, known=abs(ll.lon - lon2) < epsV)
            self.test(line + 'fb',  fb,     azi2, fmt=fmtV, known=abs(fb - azi2) < epsV)
        except VincentyError as x:
            self.test(line + 'VincentyError', x, None, known=True)

        if geographiclib:  # compute Karney destination lat, lon and final bearing
            ll, b2 = _LLK(lat1, lon1).destination2(s12, azi1)
            self.test(line + 'lat2', ll.lat, lat2, fmt=fmtK, known=abs(ll.lat - lat2) < epsK)
            self.test(line + 'lon2', ll.lon, lon2, fmt=fmtK, known=abs(ll.lon - lon2) < epsK)
            self.test(line + 'azi2', b2,     azi2, fmt=fmtK, known=abs(b2 - azi2) < epsK)


if __name__ == '__main__':

    import sys

    if len(sys.argv) > 1:  # get "GeodTest.dat" file
        tests = open(sys.argv[1], 'rb')
        v = False
        fmtV, epsV = '%.3f', 1.9e-3
        fmtK, epsK = '%.8f', 1.9e-8

    else:
        try:
            from StringIO import StringIO
        except ImportError:
            from io import StringIO
        v = True
        fmtV, epsV = '%.6f',  1.5e-6
        fmtK, epsK = '%.12f', 1.5e-12

        # first 500 of the 500,000 lines in file "GeodTest.dat"
        tests = StringIO('''
36.530042355041 0 176.125875162171 -48.164270779097768864 5.762344694676510456 175.334308316285410561 9398502.0434687 84.663858149358862201 6333544.7732452481809 -559418252332.321555
62.912967770911 0 119.074519410562 -23.186512533703375483 68.567247430960081525 154.287114123871723086 11254667.2007643 101.34948410954772013 6243126.4334319566678 24901551594881.448587
2.881248229541 0 27.763592972746 53.997072295385487038 44.520619105667619923 52.159486739947740473 6958264.1576889 62.658669064582460176 5654035.862840240277 17238566899063.427934
15.543058059808 0 165.300914097354 -49.416672064705806337 166.82808321068892367 22.033030848248991914 16063336.69802 144.535997930981394845 3730158.2076640481146 -101426349516912.422958
7.656410023405 0 151.07456745694 -60.405136414529743836 110.581139538776836414 75.518295964648694469 11865846.7919421 106.813570461088588378 6105060.7173597274207 -53439722800180.53675
35.588015734358 0 61.748037768251 -11.212600121556132671 144.177076511359789425 133.027766301029387437 15515045.597482 139.719563133296925693 4130006.1801038544218 50359093425263.7649
65.656162297631 0 176.971135321064 -6.529066987956306169 2.895923948124535977 178.740350145953804927 8009999.3798375 72.122872435008850655 6059140.1956432440007 1251055292541.359652
30.533518318799 0 85.971354917522 -8.855512357256652639 112.749196494484880359 119.512003891281849601 12687461.941296 114.310919010718276127 5799408.2575653090308 23675630576744.267289
29.238894557327 0 8.285992223016 29.866967483293077534 171.726892560962686504 171.662244609397133176 13404610.3533163 120.539419025426201571 5526417.3852308035093 115700414594360.174805
4.731224406747 0 124.670505428083 -29.238404337998337136 133.459813533283710564 69.814124320813157633 14430782.9853097 129.988518636556796002 4884051.6874703858557 -38732501718055.656275
75.51148228351 0 83.078727908415 55.600487151982554087 75.128743229495482358 153.896688535571761996 3723062.6140266 33.461326117854864555 3516199.9499673388967 50144837813997.001372
49.434096553168 0 121.492426162955 -22.858434825398832954 67.207513019331651092 142.939529056687250447 10390945.2847473 93.600913797433934271 6350213.8030348145819 15153811930312.44578
69.431271600219 0 160.836505874559 -32.634475461067263829 22.238523021935408944 172.112992582333446069 11462378.9330848 103.216641036098150218 6198216.8141054193122 7978005675822.946256
9.010849979097 0 22.595570363789 62.224984867045180852 47.119133304525191631 54.318105291965165296 7015795.1742086 63.154487812129182241 5682148.7125026896869 22429007854320.616993
14.64372994794 0 15.026799504039 33.069613195449914273 5.796995960185562748 17.403844642345699255 2123472.323569 19.128777244884873422 2084286.7481321837012 1677849152931.016632
49.267261230396 0 34.70953369568 53.760899500984702142 119.20430589851807476 141.069793254983383767 7247849.6976504 65.158126022505373806 5791304.0472384722252 75280816233222.014561
37.673622543409 0 83.929705275651 -3.225632004244734605 103.742354347526275933 127.880328388523563663 11440592.6041698 103.050946575205294017 6202435.800752108356 31036080066856.977947
9.903006408449 0 42.161189985253 47.108712008352567441 99.93410735838654628 104.103491123037808204 9951338.2587212 89.586452691503851389 6372686.0386036167927 43773919694178.531425
4.81766601028 0 77.585971920787 5.399363208348655124 135.250063670789734431 102.177017677369766623 14934241.1292402 134.592258677281558105 4529846.4052708146851 17345996058004.905562
23.997257867127 0 154.652103205115 -58.669224697402902434 54.875014559502078324 131.345308028427832621 10456532.7345179 94.175020977359294291 6347953.2379287200598 -16476912634806.11821
13.320487305712 0 52.885328556715 9.184228666560197378 151.302636871911434271 128.175105446048411097 15992616.9668738 144.02859296572290717 3758113.7445070520803 53168532477896.602091
11.426121149489 0 36.661518374479 53.467939714246953208 67.724192258332804312 78.863492168727103044 7523817.2236513 67.733947111963973 5893483.7949206756713 29825655567886.074557
12.450771501443 0 51.833024203378 39.699455912449005615 67.156324625647345425 85.276829539877646927 7185925.8370823 64.716088373368477583 5754809.8287294847493 23617185395399.468387
25.637954641151 0 7.771604073213 70.942602144324344714 17.379213644124078423 21.869008256311540091 5157420.0236623 46.39931828516223762 4613884.6645766322919 9973895471403.256837
64.532563494597 0 32.386111613387 69.633349004538138083 110.601933261211357292 138.575192608514502852 4182294.0443743 37.581418393486804033 3890590.6602304707689 75200736365895.735752
43.490825907507 0 79.577181597122 16.190814826165394745 87.606993293114071408 131.92940950703932379 8599025.6283719 77.416522018614342411 6216720.016880406038 36987353763638.421293
36.959641322378 0 46.115014139012 54.774904542264254729 52.952506835522642928 85.904050867675986538 4429667.3228199 39.852623320342363961 4082314.7451962365681 28133848636636.247529
49.335855077049 0 164.340861233836 -62.008568677306543298 172.274876714259906956 21.992601380428880091 18515195.0943815 166.621466455094483308 1492687.4253517237052 -100808961041389.130228
12.671683150446 0 26.486476107039 27.303729660018309396 159.184451093392926517 150.697920369702518187 15048952.4710535 135.40445670641063035 4509443.0049480220123 87864289836267.171987
16.893963000253 0 62.39454334819 31.979897342657852696 55.946758945497236631 87.398292964766602599 5847962.731732 52.674538121314982322 5059357.1492701126785 17650959117477.641033
31.139604566062 0 27.565873937078 49.991467522545417544 133.96241119146678476 142.016540226251410949 9939788.0275973 89.389011831585247038 6387271.3190976276149 80988552229838.549731
34.599988294336 0 166.387927615662 2.81855884227782685 7.234774421858760424 168.804029404993467036 3597526.7300894 32.412464450271497792 3408826.3447636112217 1705331315160.228814
34.925303445343 0 121.241074862675 -38.118070550305277383 93.395971367720076098 117.017376985227903935 12559533.5645849 113.151779835803174813 5848371.0945372793911 -2983532967248.145458
47.085699953661 0 106.166306945526 3.377824468034521099 65.294745963254786023 138.982543784559539308 7881773.4010645 70.982274831566017912 6017543.6972735839468 23180884012977.226796
42.42109773843 0 90.251828651418 4.631157299023652512 84.335044694540457806 132.120804319473515361 9200537.9686874 82.85963942287029988 6317015.4565433254978 29571787161260.21281
64.666501974273 0 36.879255647851 38.95221521950675959 133.416463103112854127 160.690585690748686425 7821879.0965592 70.316709797109643535 6010733.479471329554 87660636995618.022007
44.031384767694 0 77.492914985025 -39.013296805131325253 160.302849365986273941 115.363714797610358389 18289653.4563314 164.720955132452512122 1681182.1705990921178 26766218367872.015159
8.844075925912 0 153.067452138574 -48.456093143589437295 38.613821719885969569 137.651668464815052952 7389056.0000217 66.559126437158172284 5837083.1191384870412 -10889258553010.5852
52.476185025541 0 110.636930432732 -13.861956131519484445 74.250571564775901948 143.968262419704497829 10183890.4042073 91.721691232349206524 6362507.0001123906965 23553657256478.321412
8.976365449267 0 63.898816738176 25.289201968371266546 47.063578519727353671 78.672532588106343923 5290497.6026118 47.668902224382218871 4701326.1651246356267 10424717010585.798424
63.08923513506 0 33.398602909575 49.498121110507357467 131.993365399320304741 157.425605424889536215 6830237.4756285 61.390109967914399041 5603408.3062655532341 87819384410169.976167
33.794130897525 0 111.850293446187 -2.104241718493654396 56.532079408825217729 129.406954892035424479 7119055.4336308 64.141898435245081481 5723367.1586376420029 12392075259496.328122
24.491583454564 0 143.756427949791 -39.997315471628972475 49.004258662720912374 135.432994751523173741 8762329.3850261 78.946839983586964774 6241916.1765066750873 -5878181475070.129716
29.713553114443 0 77.49059468183 9.333271992381416199 98.670857333200301844 120.694280236304113488 10330974.8552083 93.056223109496083789 6356992.0673499047907 30500018608229.417566
48.48043688093 0 32.947418626872 38.653646179484441622 136.060467518677213323 152.489415748331057845 9426008.3631053 84.754261565900787363 6360864.0848639650457 84604545216811.02796
13.896748175492 0 54.52053350004 -4.564994956240214845 166.894968092835703734 127.520036113640947625 18254889.7571327 164.425950954215566403 1729849.2634206553164 51546736056865.855262
51.661592917381 0 4.452157407826 -38.522500159619912054 178.679306959466138844 176.468684587437611312 18540185.6161794 166.829588271464169283 1483380.7914904332674 121848454457387.193167
56.3269751734 0 24.658607700525 45.529795503687501491 145.048737846272554942 160.706226714169064006 8253949.1895893 74.191839148385005478 6145057.5333656163073 96330381402702.393275
43.333911233427 0 173.019242079634 -2.968924563433199128 5.044082955484587151 174.913527615338350788 5153026.4694777 46.421337494700817662 4607924.2654487386697 1337595627933.543705
39.001678538466 0 24.591382667984 1.618184178387125103 163.262955108027678135 161.09772971386841808 15194490.3704939 136.702262971171421382 4406640.7966952381381 96607345224486.64928
63.758775485865 0 63.327049113388 25.17974033943720596 107.504991932891908801 154.05555223257211081 8337896.7811702 74.995657826681196357 6161977.3081946679644 64201492089310.286611
21.399072590463 0 34.661527757273 57.391251345243967151 89.082271206721964021 101.27611455375290594 7974419.4482318 71.7540793160630804 6053926.9977463336317 47107642231231.85856
47.967022403547 0 130.016325821124 -11.470672507979396723 48.285094684224854882 148.389695684750002628 8123586.3046895 73.177040076110441065 6089944.9618503324637 12978551999357.690171
18.912843152349 0 65.312622112164 27.356639349122835344 84.391681754671188822 104.661818309238396464 8535264.7666499 76.875500330038308305 6198696.1360798827935 27779190421936.317364
53.91858081059 0 107.773420301503 -40.533597900970847686 103.488381204008843713 132.398259207370627207 14330915.6890974 129.067367666996053091 4942326.792354242388 17410890728273.067115
55.496983044446 0 19.104260736838 52.337949466456523592 149.908278798430181246 162.333474334205108746 7734671.6724793 69.515528979665798782 5982247.5566080689881 101429347549104.453629
33.506365611803 0 146.134838860662 -58.474075605819738855 78.7186055605329346 117.457565115569101509 12505869.2696334 112.628139192634925213 5875851.2066026199438 -20277605247299.091325
11.583629854123 0 175.405841079194 -35.840568348239475134 4.157534199469788225 174.451066901814232803 5267069.8227735 47.459028881819562988 4685304.7307793504905 -673981751831.778245
10.229295216134 0 72.127910020525 15.629163409997943457 102.677417064289464034 103.482102722183029137 11045968.5159604 99.529901910481163606 6274136.6891374252135 22122583562740.955366
11.385616722429 0 80.876668493759 -1.256179633150734234 133.625254783633314458 104.470489013738508911 14791911.8158261 133.308468233706530664 4628542.1997959734114 16643260535437.672496
18.878721716993 0 126.506486559251 -40.558252601871319412 112.89705386708456919 90.226134266466775247 13270232.7345542 119.542488500691307921 5538080.3481474545731 -25624590642023.493226
11.680590160491 0 3.454617381719 86.193595821107165314 116.956880430710651346 117.618329340274182003 8905185.9740981 80.109437920394771759 6284171.2594118929493 80865573549985.871467
4.624672945954 0 84.859617539498 6.874181093561844144 54.891198619260574194 90.820789878786191112 6082040.5178468 54.817510011410934784 5195786.0148775669096 4204393012831.821438
14.284440635946 0 97.201927360601 -2.554657243384560971 71.539549368441047665 105.722921401612977296 8102681.6827271 73.027278326066566303 6080520.3700239492982 6010449723957.98058
49.245126292946 0 48.890604376082 1.329645321136461702 138.134968538140002028 150.465118219878877889 13122810.6665654 118.096979975653016515 5638390.7267605087887 71841688490301.198818
52.567535980701 0 101.333917966987 -44.026426573445515261 121.367591215509636529 123.975227063717588285 15689524.9613062 141.298586118482115749 3980700.7745432290519 16009030273045.213412
8.476800870747 0 85.411846946378 -3.399080708673397422 138.655764270174507286 98.994343217446874323 15427033.4589312 139.042850810432243181 4167848.4504067070037 9580100905319.310826
21.211133111746 0 123.430174457287 -26.776984673364351569 67.020054548341436624 119.394664651635683914 8963095.9449195 80.771003338217735705 6275995.4660582282258 -2848216384103.329073
49.305010396359 0 17.378266228641 52.05723926904983978 151.887270484247142456 161.537826425477283376 8460859.2421616 76.047924642468289003 6199507.4289227565853 102083422298403.492909
25.292124359247 0 80.920772697985 3.784724165483273182 102.727744934480926468 116.4531554054392155 11111090.2510494 100.105662533436915162 6264762.7576923090295 25076833177041.55528
36.359393471816 0 125.907547808617 -40.409298772796866713 86.033195335656813313 121.080297615268009517 12215268.8786999 110.046045092174023741 5975525.541582626796 -3410358215533.164124
7.035197972016 0 16.840493763342 73.013460186925778849 76.188125151657360753 78.87776945962910291 8811577.1302192 79.293644827840365714 6263784.8066368778234 43902133541180.638829
47.898956763857 0 61.708136542095 52.870131799666999619 51.020900069545151078 102.113351369331051692 3592351.5930099 32.311353423297759242 3405733.7347661413862 28576280361218.564189
31.371144087006 0 95.038829595627 28.909631785856762476 17.071526465331372406 103.660711362255135363 1665550.2846815 14.999405550550820854 1646623.4338249408166 6087592163895.266734
51.949999288565 0 171.138506435006 -58.257854831091299448 15.812862089412024943 169.606337169069613737 12301852.480392 110.785163709269884489 5949768.146617839051 -1083846771649.999162
3.88996347756 0 155.167945553097 -58.869445004288913981 132.036794965085585441 53.946549037654148249 12656393.8352475 113.899510718180677934 5839566.5953753128131 -71615731394785.034635
39.819854185972 0 133.177550503651 -45.54704259850543869 77.540663933413047906 126.917177154676469019 12204135.1604139 109.93495493047832773 5980542.951958147291 -4424330141143.762382
53.906238419729 0 21.597565540644 45.046384823603570871 149.38242876945976636 162.117474447248029643 8661174.050408 77.853005964076328921 6245080.6602695035391 99499568772183.376624
5.23823083584 0 164.789687149715 -74.853338324199100781 96.07209672095543853 85.425493305288860246 10739064.5751586 96.650133285379783498 6334500.1981856381806 -56177012697341.75533
23.104516713391 0 116.646256493613 -13.205775181751514372 57.532285422139942929 122.358040028963864846 7443018.8017836 67.077197356965430078 5855749.9166494779297 4030187796005.440987
5.083671277342 0 93.251069157874 -5.135863872649337681 115.048832990096880586 93.168399576995632604 12838857.2071642 115.720221429087154784 5727027.7367091550769 -58305437622.558448
34.446980587849 0 135.570387658353 13.355452912309759641 19.253945880531200378 143.568718864193056249 3037507.0168011 27.362197843972206395 2923486.7717393421138 5645935337551.662125
36.984116640248 0 45.842558309494 .766426443891580595 147.485816830606268876 144.982273873382287301 14657580.5791576 131.931957265463190485 4762176.0433370080845 70089798919624.006087
70.488815458652 0 20.631606843012 10.228416314659243099 159.204118025562126299 173.11197302748756288 10910936.3186733 98.136363991670042994 6324848.9149942362075 107993197784347.702844
21.559194636621 0 12.683213040733 73.178089048973846609 131.762699581940471607 135.271161440271341775 8915869.2337664 80.181673874994059337 6289437.6709855756404 86803032925623.741404
15.067806173607 0 97.772651083729 -16.90871723132077799 149.240493850337138082 90.725082091931811281 16739117.6455835 150.858112213263132668 3096362.6025953445956 -4972094990362.972061
66.681040053372 0 91.646561698604 -23.440025289756829252 98.819595131758846212 154.388319751570359184 12761250.2849649 114.891652148460828936 5782187.3765512770437 44389048061716.756132
39.409146973655 0 148.595915265879 -51.344465279370613269 54.299399477395513333 139.903236261795425743 11353780.7523553 102.268031975689130807 6217028.7971459613554 -6144832443236.672814
6.808768903883 0 37.871880481974 33.922319712554458178 143.463070326988010311 132.788229856712967155 14088307.285157 126.809782407729828799 5121098.8953441382978 67092008884046.605626
36.982703444905 0 179.616543962684 -83.333068057523963159 177.611924577957562406 2.633768744581642783 14840467.8507368 133.573048432790773514 4625571.5733430764618 -125378040351703.190501
41.813673151676 0 16.405331944003 20.587333492521720117 164.201962479604314468 166.991083245940221296 12907703.7913101 116.078517467142846426 5756219.5876817085708 106616373108773.02114
81.833023825449 0 60.123995474598 52.61281538614421497 110.766184475279468991 168.281485715810706354 4562323.774583 41.001650648692565486 4185005.4981187706769 76606470734405.894829
16.105294911471 0 131.904797964935 -41.681372899642136145 82.150183233451312234 107.012959072513907181 10551884.584959 95.061019405522182528 6337396.0570977477936 -17579107088322.070597
54.526345137566 0 7.773653323206 27.082807052701802982 171.332530862400505556 174.934339881142140869 10924581.4061629 98.219051475416375929 6331219.9699918788988 118399839865594.378517
52.125418790586 0 49.90176846866 57.992299464190689001 15.128356773546692481 62.34298208473670643 1162588.691752 10.455170176021143192 1156176.5423144634564 8800889303749.603922
64.249804270848 0 146.084092210402 -10.526000344959011434 33.778701722506370788 165.687803859940836515 8766416.1986483 78.938067290322435929 6248251.4645329798913 13861682613984.33775
4.438849098677 0 63.524406400047 25.802394934786791927 98.50253119930099371 97.837270592495947254 10650547.1250328 95.956003632299222222 6328729.9540432635349 24216219386914.607583
32.167545281004 0 15.730179181123 47.16325546576891 156.720176105541609702 160.291021678920564135 10915214.3981797 98.139454328650182919 6331904.7878083723758 102349755159756.684479
61.559300187203 0 108.39023739083 47.932062978964284053 35.09471409221675257 137.54506511542545457 2671035.2541507 24.020276800965954505 2593830.8799447825539 20625676702539.705017
14.532835548091 0 150.5977117649 -60.204339807983975744 117.726217548610045108 72.585010656772232104 12916193.0499398 116.274856236698632908 5720282.5486311684862 -55180537881083.787064
10.921226035849 0 55.357395703568 35.247410984329374119 59.471562388985018642 81.179468094302121466 6552967.9476886 59.026613241850537737 5455151.8726335506024 18229548126795.283291
40.365735022656 0 15.602963749057 19.21826804770563416 165.507915017642527586 167.452074301866923668 13240327.1407986 119.073105167344706813 5603790.7624722589778 107511961644445.782225
67.106835222583 0 38.870657572171 55.428809300888078528 121.956541206928638419 154.501481394520737888 5588483.9031275 50.223999449827840619 4903862.3869554005331 81879433121718.383126
68.949380257798 0 44.637588242057 49.294374468687378483 119.661419832788623041 157.209488754087441819 5994407.3092052 53.877977665233808091 5154021.6690928913524 79709275018380.095811
1.353978214987 0 143.143813031951 -32.534872085863344049 152.280609797481380461 45.281812386148205805 15506396.1708873 139.585530346496334075 4154788.4116020964393 -69176257602485.369557
35.642635084621 0 82.062035338344 -1.097899218946765715 104.69735995962855758 126.298488917783092077 11410935.6779242 102.785102079990524485 6209023.5626441682063 31235245213772.235728
7.404369017575 0 132.280122049852 -18.034366293950019065 28.460387638998924211 129.520363751000799537 4206924.3248979 37.915395784357547146 3906558.565535026669 -1946840315544.076395
62.842322572224 0 29.800334622272 76.310832995825354922 80.11052468362523739 106.655921818336268914 3124291.703171 28.075211318363466718 3001506.7846166596468 54427194420994.499345
15.605731483474 0 48.465411695648 39.838033717901157535 42.98961047786086566 69.69528146422232451 4949263.4800926 44.573460317963126322 4465688.8802502842359 14991101724557.790153
1.639018345309 0 151.496138693428 -56.710165877845243147 125.242287231675295254 60.1187560131680823 12225494.1272409 110.030229844958291103 5997654.8328508075296 -64634264432590.982506
15.855026952648 0 34.163809923765 55.796061639165187987 59.736598787303052224 73.536727826062621888 6685140.5029947 60.175745123672763956 5525119.7667624793437 27830360940392.300639
10.614568238758 0 144.238884596913 -36.553675319552677119 38.656806212281346306 134.414836606695490374 6605623.8477879 59.51820208378900084 5480564.5289019885395 -6935125637851.11415
51.608950756674 0 44.069317622873 -36.798455994877477331 163.588375017283169921 147.3242040214355212 17915905.8775998 161.26730811753694571 2065230.6163431670175 73060222647740.064061
34.499131948124 0 124.110195748763 -15.952937085210558866 55.126807113657539101 134.743426025962705864 8071189.326103 72.726433538546497313 6072442.6036576163566 7506281015765.37646
11.76588215561 0 128.445444885656 -34.573086586710321228 69.304064202465949565 111.515973223075667684 8918849.31472 80.361617315826578917 6270189.6437997324485 -11950617682064.851317
16.215716057056 0 7.408439705129 26.643498959840949855 1.50048092488194166 7.958966071177607606 1165004.8285076 10.495858240867719311 1158505.7837001120876 388508924209.933632
1.642674834724 0 29.177853976299 60.891878271763423716 91.543409983652637537 92.2730950362455758 9931149.7293326 89.394385728172346965 6373272.0974046722251 44619364205291.457241
43.411472221121 0 19.001333854403 64.148186913821551876 136.552579389352694652 147.19322870602970949 7487842.5085465 67.306472707759703075 5889589.8112492686309 90768937553608.503236
25.453887730695 0 24.584520071297 62.31167774869338713 39.238629619778292526 53.78589332444363108 5019894.7199129 45.170419099317422141 4517468.5078418338855 20649721775477.865914
64.993189876625 0 38.018885135551 27.239733033114883155 136.649227320518020072 162.935700444146301212 9121093.480273 82.01662442154982575 6323566.8863524016788 88438048496728.95224
60.889795730578 0 10.532570098252 6.562844367534484798 170.15900709434291393 174.85134517458345533 12475973.5475584 112.21207404928834131 5924515.2700090767267 116383691517868.400405
57.04127718334 0 94.919868984137 -29.537259231420067439 105.350044044461756901 141.393876478961542963 13620924.838441 122.660987593183621986 5362850.9396325683362 32860882432328.071311
60.978643537156 0 140.474222105121 -66.447699831143443296 83.741960628420151773 129.422490051758175534 15687934.8126056 141.240336312857354519 3986753.1305105185906 -7822542535263.158911
32.123115445334 0 23.38510368802 56.926817599968096449 20.1971170588961402 37.960006049295442046 3162677.875824 28.458924606506385286 3034650.8379222131861 10304054040768.542958
58.797840343884 0 56.928302334159 55.018265995959122076 83.735441278690692548 130.766514657995262908 4780352.4346638 42.975526529806668021 4346832.7995004770906 52256821636368.033286
22.202879028004 0 169.770590151956 -78.131053729786100635 56.09817635932642261 127.132203471256548782 11686060.1389879 105.203427352166662752 6149894.0761225089597 -30182597923715.406871
25.106481366095 0 107.940945983295 -18.223027345294129929 86.221345156045042597 114.875019395395574661 10491611.8298357 94.547312536853968611 6338285.7543086794512 4893411938681.754777
24.960880065837 0 121.91120003799 -16.100412777564627162 54.195259657284100046 126.748473963924217284 7429091.1457004 66.95008254306999708 5850376.8124583278694 3413506579258.832199
43.821476474756 0 104.211675347897 -15.192944143353673943 85.057605312625081443 133.468780913302085511 10776666.0281384 97.0794317961459894 6314834.5537843399664 20664608634683.413062
38.638954341593 0 88.705960434571 -1.279756550060796468 93.430943123745488266 128.545661704819814661 10399099.9308788 93.669942428268920937 6352403.3642702965607 28132549918372.043798
16.453346963544 0 121.094998098232 -29.465712645249150927 79.050045796724820227 109.482097420925889895 9881221.6365467 89.04162431784249498 6358034.3284215258137 -8196337701325.977217
2.313140808847 0 58.3221781927 31.819596255581056403 88.251997328019101892 91.173935030537722958 9714162.0920061 87.513623648833368947 6356990.7974976661945 23189359246371.094253
.384481324443 0 153.330159683047 -63.260477387264792299 96.535465277414860409 84.208873088520732131 10370985.9514813 93.348756322476606702 6364367.9140288482187 -48890763759081.178893
13.219213385707 0 69.880456686163 12.908849199388031348 116.734308245854221967 110.313865884978004382 12477283.6158877 112.416686577439777585 5884415.4780549774435 28532330489542.852483
27.356362633595 0 161.130977136877 -73.345839080077979694 99.753937450114115582 89.074641755278530858 13203937.8736131 118.86523513738713513 5584330.1599414098813 -51002388849589.749997
5.488011725705 0 168.723562733006 -58.777033293076301197 161.963989905489443137 21.998888971547690234 13893751.4889353 124.996491754660006134 5246149.7102100822342 -103891882373900.000985
31.027138293626 0 105.713678101658 5.093068996650957756 53.687921438754064179 124.015842231205040438 6289472.367629 56.665745021016531531 5313886.7617347639715 12917031047756.782877
39.561788005441 0 49.623608517651 35.659488664161195063 111.729315023485090819 133.697281357224465572 9137431.7060819 82.206381042489058605 6320521.5759506729685 59446568146194.189698
9.365703185509 0 117.514050405621 -24.330783755993681155 71.666567863169498133 106.278062394906532393 8622651.143068 77.705796305372042791 6212488.9022163606881 -7928047336405.021855
24.498938706676 0 133.05200462986 -46.218742288678102651 91.58007778869994691 106.279110700398463937 12050354.8421526 108.550776585488708262 6032881.3901185545233 -18915010007133.174413
71.268543687759 0 59.77806463364 50.143352562967794792 101.322502529614900453 154.314496467351817235 5203012.1247675 46.766538605908926541 4647341.6734580074077 66934534232642.400881
23.544845000338 0 89.38790851495 -16.604375537698709402 134.279201209116874988 106.895989785631793913 15208288.1076431 137.048601550764912644 4333806.0653446399799 12355167267362.268327
38.263951306103 0 19.916514668653 54.031289901242443439 144.862701667601447669 152.937009701639052462 9239174.511395 83.061498323359864433 6341903.3494873326423 94173621309373.222307
23.562635615291 0 80.573281377538 .616910921891635868 110.974662272004912692 115.206118431861710978 12120175.1033395 109.204058485726389675 6009298.5779596710809 24440198924826.338413
61.811416615593 0 5.210785345513 -6.243779047728270441 175.658894820005592479 177.520147377314802192 13828042.0566139 124.398710524397575281 5284463.913920494343 122056743536456.526391
13.669054148218 0 175.882921853998 -70.416698815768998398 169.642468357273528972 11.978943348082444009 13659776.1656234 122.907016807912071721 5369309.6642484363946 -116097163697384.947771
25.239863555674 0 72.660722517458 19.558616692495251553 88.645708539223010883 113.578814362496618689 8979040.4729696 80.873684947725551533 6284716.4309433480087 28886005362168.772019
42.574228047871 0 42.484577412642 -23.555114207891525381 162.457529431664418011 147.104330772141315424 17353287.2799453 156.208135240676972026 2595315.5095251938167 73996486653921.795261
45.948664491925 0 125.501505529871 -51.275737933996271494 103.747943009536247178 115.235373813929389854 14618302.2574747 131.65788987679485316 4754834.0114845676918 -7258666613688.728352
6.207840098865 0 46.932489817344 25.37460155252639202 143.135783283238833788 126.548735117002698534 14719464.3826153 132.537318807914037472 4709430.1347141916788 56241730227226.656007
23.293901585878 0 30.11522926978 23.967824782385457861 153.611115762586601851 149.71411642602454622 14054961.6046631 126.453949634480673884 5156108.9791990843407 84594284106270.037797
5.137007523384 0 112.450171526302 -22.787506689567668284 111.258646767306005306 86.335099650483121234 12416199.1495246 111.88414464780866759 5902652.0130660953839 -18427189745584.973914
13.49052916217 0 149.841874885262 -59.789841532293534065 114.0549834222546852 75.61260524272574761 12631733.3880672 113.717236654455340942 5839254.3875498760748 -52500201865716.183446
55.121886116223 0 60.310560136943 -2.802505438545373872 126.245955176418434911 150.100575968176290064 12480126.5458992 112.32747498677505816 5905357.2950359764575 63505624285836.478239
1.64634475425 0 53.4096539003 35.735653026164747412 102.768725495109981254 99.024701978368740318 11056439.1336091 99.587315232066780719 6278303.9703253250913 32209508293527.974629
33.220561601385 0 60.944147422931 15.906681603252465748 117.446653873859610414 130.449965412146623536 11441622.3283164 103.009208604710885088 6213040.5184963080271 49102753878996.129329
8.578624774979 0 159.881536052134 -54.375987804166438992 33.243362594038374877 144.361175098783120134 7638701.0986741 68.798505999362127026 5932815.9290899905514 -10966985471201.924372
5.77970608814 0 68.289732837182 3.588250783959711994 156.659583234995199502 112.153293017227595771 17238719.0136839 155.335091342470098071 2662081.4751240302938 30949183894332.264383
61.311328199397 0 89.57002169026 -51.719977004790021053 134.141060992499470161 129.170451102958259156 17067257.2707292 153.664139524494415219 2827433.5495539228509 28019376019642.107609
24.953400204545 0 60.873278298893 22.935153219965243145 109.461291078916797899 120.676555897335531052 10751192.9312926 96.808318888805579289 6327337.9979921328677 42235052306875.616295
37.28650419962 0 105.846589706801 -5.31468999017340053 70.995224692450001666 129.681272635793703055 8710146.2891495 78.471600399555701121 6233113.1687980880415 16826926825355.060259
30.160079821538 0 28.41535255758 49.147133873971183685 16.182514450690278252 38.926644196247833733 2512645.2482055 22.616141106216277157 2448080.2241452205773 7427690246558.576607
16.654648572154 0 143.543915057454 -55.079363368716652794 93.170898482120550441 97.017491592293405748 11709356.2799059 105.44761599839894611 6138920.2240006241971 -32888088228817.416906
4.770995501813 0 60.912246501358 26.410770923895313444 109.934003168640763715 103.660846029905783223 11744138.4659669 105.798312770756343481 6125663.2729206456382 30174174874452.667597
8.508198845478 0 84.082450769329 -2.539669792389350667 138.682325493918155242 100.016568432751889991 15413933.4624267 138.923296944556152637 4178196.1193601004861 11238903358508.795814
34.963300781157 0 105.181589969978 10.56329945264939349 50.532119480076461094 126.356065708364370669 5772408.2365698 51.997773511859326469 5012898.2468387978666 14947662873570.640371
66.856398707289 0 64.613916203622 51.443871023711154125 88.799161779252647479 145.239198135632494823 4864003.0488799 43.723231095619395689 4407953.7905619622338 57072879457610.955989
71.606214176302 0 169.717716110676 -80.240756380896471977 170.591260542729673958 19.402427357239077616 19009997.0303785 171.070474762228879474 993187.4139545125469 -106482196423531.556041
35.987827127174 0 9.144224039667 51.976039799073830497 165.045244285456340612 167.962110268433272754 10151272.2530231 91.255215269757186624 6392676.3850747860087 112477713884317.173701
48.726663942803 0 68.675518243798 29.36565487352353594 91.354349089561449156 135.101278715212414845 7711704.2763444 69.389390059319984968 5966166.3194273146691 46961810541538.294191
2.426983891456 0 17.906760480723 61.74548581736795016 142.333198773095603791 139.666208757007697169 12204647.4113768 109.8031285334024834 6013138.4261740722722 86184253361578.662134
47.588932093582 0 122.785380726288 13.414522997031928676 39.32299270673336857 144.276515542082538905 5251124.7295255 47.28481692553263005 4676075.0925451574847 15181161179377.134173
36.218460985676 0 139.813729934548 -49.831353332108057829 72.426737728886789983 126.251620412456127052 11888995.5719362 107.091841667139831045 6081472.7086567907992 -9585527230608.96039
7.49325833463 0 93.2200117328 1.822589208020720631 53.589207810480716054 97.925176229123591222 5974884.2101696 53.852322648130430534 5133264.5116104780945 3318504847229.409396
27.424486935629 0 119.079137094176 -7.987193900849071735 49.318809304075510174 128.386594768747861603 6599262.3694883 59.469165013574310527 5476855.0490182101887 6567819739076.891678
46.817096329661 0 177.443915970136 -56.807375049863995376 4.521801986352845996 176.806062785469997925 11493075.0687165 103.509540165812146964 6187150.3904802915624 -451111297660.535662
25.643558024411 0 124.334575817147 -25.969833883721214784 64.903749477058805498 124.103844353942396512 9010091.5311688 81.192887458513370173 6283434.5465613618597 -162862980761.25444
45.469368982526 0 80.428951351357 32.807359742666679411 65.107961864982896069 124.577990419073311227 5641968.0800627 50.775783354657025868 4934112.5540824352438 31201594130416.757862
24.830020284097 0 105.189491874788 -21.552137872353264451 102.466021945024658935 109.640197455957527371 12195822.147997 109.903219125927743748 5978671.561293141969 3141093116404.660866
64.616754960515 0 30.346600456016 -26.271741383677221182 158.321581345077227778 165.993649158304508871 15470363.2647919 139.22169265381116203 4181889.8460401249567 96046793984379.640557
16.643526079775 0 20.664942438895 41.023069113496231465 155.562542134178539359 153.407413608014235895 13143072.0128514 118.219854127721094761 5644586.001095604946 93941577448914.334362
36.15988228751 0 17.377755506413 16.445470954340165151 165.26146144887337155 165.424806257118547888 13980304.2458731 125.741378499215436476 5209462.1600587150062 104805730938815.247066
64.020000391978 0 96.562559689549 -3.222107095787784058 84.129726900596147365 154.083851495687961409 10038981.9782605 90.371142116639844169 6372416.6756594180745 40682391211347.531829
26.297529885507 0 66.29632339408 7.811709739637789565 123.143308309659900758 123.998105582524580755 12821755.1122628 115.481380978730787412 5753304.71525964774 40741066154353.01873
33.950423037177 0 121.878508697173 -45.235922276879247756 129.419282696457860108 91.493405946969508649 15577190.1037708 140.309065947199354219 4066164.7351090241787 -21471812656324.656815
34.821801199155 0 118.660107420752 -43.047563118771784209 121.565540573787556347 99.850542926816455273 14976561.6701469 134.91019258583362906 4506731.2497052672024 -13289977552176.004558
10.653174040903 0 125.3085108692 -22.881001889671246471 159.765030106649837101 60.474878214383859851 17484216.7762368 157.490274665028773195 2451723.9945256764603 -45780984648851.156399
20.927440693092 0 118.56019007428 -15.011946251300212797 55.594746074667419808 121.838839080254913211 7260211.8202706 65.431076273888427729 5782012.4643014015533 2313308490792.760384
12.968620426257 0 178.873012048075 -29.011142183886545518 .856639457160087712 178.744943180805444512 4645657.6699228 41.864441746005877275 4243253.5472574248942 -90381212163.717157
3.995380005482 0 93.974042050361 -.353000909366607426 48.401254652487494783 95.61255104490823643 5405250.0400016 48.719267867908325167 4777042.0489484093692 1155584717223.091651
54.038784174903 0 66.594110717849 18.843340962663756469 105.405456105476194179 145.215722323742016319 9294472.0516509 83.63245950315995792 6338269.6941880132925 55598406320675.241259
45.454618894443 0 93.71721846208 19.672224913225212102 64.132979510420318465 131.894863434272278286 6470407.2976449 58.25309351482810141 5414287.5676692738644 26971767396948.087224
.336229490133 0 142.938070562508 -44.315073617889112497 132.711860436436568992 57.236875169147836234 13275086.3473914 119.49956516270727856 5558321.8609412293493 -60583204684031.955483
21.284888626672 0 9.072053968952 71.956293555096475643 23.707587498942648352 28.235946630176998385 5831837.1680569 52.471077423216790538 5052859.5998623450064 13558927920791.20024
.602259171289 0 93.001011809839 -1.372652195171202924 37.704230798676011604 92.737862264367886216 4202587.4646708 37.879469363463621433 3903063.329056078769 -185588576882.842082
59.067914674476 0 26.151871253161 32.37714763436166481 148.638709383249201293 164.415210005459220238 9464287.9964605 85.089220373242280187 6365574.0409056093201 97896547928895.351466
29.497862233915 0 13.977178429685 77.876831887051061586 86.074590642863907859 93.017704501543092289 6726733.5760998 60.492200266898625167 5550209.193927527314 55961568397417.796143
.782571037546 0 124.483761380166 -7.405194804958325194 11.962468510681354308 123.78848440720781131 1607526.1239133 14.488987492121221826 1590448.0802016657555 -490368073619.326134
40.568797005828 0 176.967866854556 -75.001310556014873765 173.33414438551211889 8.916523271571969543 16154564.7242942 145.386525693016968361 3633564.0057558463737 -119044270879518.557787
37.515864873 0 152.481356079909 -33.145966631434870375 32.349240627000767625 154.035798269630554191 8511799.1436023 76.689215549976807557 6188802.7731351695221 1097929701265.75403
36.266630542565 0 58.536799059456 32.938605083647632908 98.042857137837771412 124.954750401989604459 8567178.3141464 77.101732738505688898 6213165.742940488778 46939376407596.820132
27.519233030584 0 109.792298153411 18.92431463111676364 20.712135699712837144 118.05845743732695548 2319388.9189003 20.894340302506120788 2268375.5123861668183 5834045463948.213373
63.796155583615 0 93.182114234024 34.281652047931346551 66.815398066146857636 147.693079659532831351 5518390.5006592 49.636338554904196403 4856318.2551449340177 38562822537571.238
25.227691428744 0 178.813615605026 -56.830634769512607124 2.137046322216981174 178.04156212630539394 9094559.8059179 81.91610670218562002 6299591.9970986857783 -545728404810.726411
80.099757106308 0 98.682899111772 -68.481892540173793233 107.03672012307358152 152.384476595411732823 17684467.3789704 159.168312123745299437 2267493.4082943149541 38034239037794.847683
38.658859285804 0 25.266656343288 -22.502042312255045681 171.827566033607920431 158.834327972406616761 18054702.8673914 162.480000165758506518 1956289.5902058901166 94527641930706.588127
26.534745770457 0 67.994988291617 -4.504199404774904959 138.544338028736198943 123.633526406204995735 14991568.9132533 135.043612032889634057 4505528.8317583773435 39282049882980.15504
51.011272934004 0 71.247417277266 -9.226343110711589378 120.310050148004273803 142.789186422067726679 12897778.0816525 116.123611534730993279 5726657.1784100932082 50574870424781.135254
13.180040781533 0 147.716474193955 -51.684044075581987823 137.764139808689058215 56.844890846790482976 14316708.9845049 128.87596871262001553 4974375.6094730806049 -64265937367385.09491
39.965140465287 0 72.42926319053 19.467357090193160592 93.87008736129341052 129.125490840369698403 8963746.5534881 80.697051559475716021 6287113.1682685379392 40055255246313.982956
29.466765674287 0 69.409093290345 33.594607695346900264 58.324143410893318197 101.975372004110701642 5485203.1798304 49.388625263026207875 4832095.192430150763 22999808033416.649242
44.294953125423 0 88.714323913792 -1.184701423910018391 92.83428503150369859 134.201097217000354924 10328096.7889515 93.015130795121029373 6358735.3931068816176 32131229570973.907217
67.097829740185 0 86.909743997283 5.878638801092361219 90.756172727681814092 156.936698928214965922 9438990.1361739 84.945005725412891256 6350709.5649934309439 49542633279613.149298
53.890042118759 0 32.376057952399 47.611646039058464701 131.482217984255263527 152.077667341137571425 7865067.9794542 70.702494306519023189 6025613.8285325688878 84736718077588.908738
34.789757845931 0 63.42481169811 13.906233945800986186 115.526238064488354512 130.769627488544942774 11347563.2291608 102.165944325835331781 6232494.4016590693762 47575101880209.725533
4.72901904238 0 87.043118490209 3.675485213595875123 80.781861923838053716 94.188426367881210167 8964224.5513394 80.795763076051664114 6275217.3953492340362 5039465675551.357098
69.698716382432 0 87.538022653363 68.665599857810000535 21.503123449290355915 107.667809564607514764 856384.3501512 7.696239047615556785 853826.461092201219 14252510465147.681251
39.948488487143 0 145.637497587456 -60.791732218313879334 82.476062801971208463 117.665952598924245826 13405430.2409911 120.721343212368831005 5473110.0993555517615 -19783779419729.652003
39.035294738812 0 116.764838248793 -33.122375734011885196 89.871476211442694565 124.06809844446510914 12223644.1264709 110.126186514023579671 5972229.2787322829147 5158772821818.065245
20.948231001797 0 117.645508125132 -34.085934037206664329 117.886548615401165992 93.366647127153261837 13813062.2669096 124.453790959515389161 5246247.9688980691646 -17141962605073.758059
28.809907142277 0 125.966112307426 -43.746811413059194078 107.199938541142910315 101.218434581372606303 13487276.3526636 121.497572499253957806 5425930.336422973476 -17483681754086.306
45.854751222656 0 166.121912963353 -44.514857277517285465 19.560360617265975499 166.449925135617216472 10194905.690224 91.833879375194211194 6358156.9272790740095 231854435435.001777
56.723060499888 0 139.889433661381 -18.049261796696935351 42.099193394607446737 158.127140901909437379 9166243.4046834 82.557413544422902008 6310306.2416606838506 12890345777904.509499
25.116265473386 0 97.732153963831 9.832901687662390248 51.448057387281271068 114.348702917675315571 5682772.0698146 51.202513998342462154 4956523.8772448572875 11725253544134.655152
49.501521808089 0 145.987668320635 -57.516443280706395811 64.700933124812600373 137.459801078684994967 13259299.3086349 119.408356062535898645 5544267.220189525554 -6031929440189.848234
53.266564553316 0 100.038376477179 -23.072599811854353085 95.392601883768613929 140.11914346910385612 12382089.9881726 111.516739526784857435 5923652.7824960531757 28329764857526.749437
17.445789085626 0 132.723368950062 -45.117253614099604179 97.361647352714237542 97.339040559114690863 11934818.5608586 107.505747003042553489 6070383.2512726435339 -24996274006675.432026
31.573304401912 0 153.033027244272 -11.277328262946838772 19.602358024470780923 156.779729902321278544 5182540.2850615 46.700537601065785847 4627608.9932721657291 2644340745153.302034
29.327071072276 0 33.242240347342 -4.823524348692753957 164.626841855534002015 151.314796841237986387 16852685.6378135 151.675333380703139089 3059999.9610315227422 83506349174823.137051
6.022332347551 0 53.954105388497 36.317087922282382191 89.326155365943332293 94.589051951234519364 9559434.3144611 86.101917299986323874 6350809.1072351244162 28692013100669.359526
34.652221222776 0 77.644152174059 3.975432760072988695 105.460358584149336847 126.257925455057776837 11166707.9668331 100.576353887770082607 6260090.4388825473774 34327375341963.004003
30.224344357247 0 64.629648041413 28.815983592510947588 89.807393686833652977 116.984374981602214054 8445468.5722465 76.035891694239835548 6181167.4235991487074 36980457842103.06652
44.252665137734 0 34.388821473524 66.161716765269504385 63.218655052917733089 88.869959091487584759 4432851.2703143 39.85981721964703309 4085503.3575441587363 38552852736172.495362
15.563224557173 0 13.350130962367 74.85997160109044556 53.523360738830844199 58.125008091801599692 7326398.1199733 65.918805466347528702 5818882.3370769574378 31687371830016.95685
22.647639492465 0 94.20425734835 -.265481234322245997 79.537677388980538159 112.945829781101506075 8953151.5409072 80.680116662942100169 6275257.0950497276402 13223166500174.913376
5.402299205336 0 107.265570452471 -12.09959802100401094 155.371356164844581053 76.454291606611546441 17227042.8066184 155.246679889291962611 2666440.1156538117343 -21736988641995.384318
24.121192655026 0 146.321029354121 -6.746464854112403829 19.102484309526166557 149.342236428404529347 3995518.558491 36.007542136972956859 3737706.1890470889727 2131625211649.079073
43.306862373969 0 81.564928015753 .663830251510330864 101.295720736944021701 133.862626019472877418 10873536.3261399 97.920226219622377044 6309003.3461924338641 36944375838362.290791
73.804557877394 0 59.949912110792 -61.764623993812864077 148.53752192208728032 149.298771734214187254 18159958.6714517 163.445192611156283581 1820464.5878393777439 63272857509579.724927
1.921877004399 0 73.748492545106 14.505121777043152035 111.665518774378904342 97.740221959573042714 12291054.6381762 110.765007480315596641 5946932.0845231932045 16924936200638.24623
3.948804680583 0 132.40993894891 -24.713205327374609041 34.15938414057724518 125.866132443874710561 4869336.048718 43.881215790548414839 4407195.1171290000183 -4617044336121.893254
71.627200626312 0 115.42153502291 50.480261762892572693 42.263849601327375299 153.395658919027889614 3154875.596851 28.362309497937146147 3028259.8804540495921 26879306379460.584027
31.021074891215 0 101.459335727786 11.787247309610298486 49.506459509701543725 120.836778496205652679 5503176.1550637 49.576278409709077984 4842401.13088711233 13676908468061.609813
81.79281339164 0 165.970058450758 -19.956673621926589926 14.599479227718158949 177.883855088778318382 11321260.954864 101.915530959552271329 6235718.9084302624417 8434983076476.759638
72.726413802613 0 166.750616723726 .738565517814632747 12.600972006926935366 176.085445031797102893 8039654.0648994 72.368616750393277947 6070621.8385983952494 6604259439771.353264
51.439832390754 0 160.000115262687 25.799128425384019151 9.800259258495130083 166.282764024264400608 2965475.0265668 26.693613362052638262 2859630.302493363658 4439612684967.478298
54.354223370964 0 160.588186514995 -3.89727713218443562 16.693318601969881579 168.780736094342670589 6639506.6673042 59.796994455964346898 5499571.8462943569844 5788740463156.197033
36.247533543427 0 98.4914109683 2.479585385319365366 72.317321261259084782 126.935932221710976786 8266459.0782741 74.467530507016830717 6130078.0518296575082 20081094936570.606127
3.614880856795 0 150.580074179974 -.836621274661653704 2.49019283667939344 150.640498065220872945 564843.1980769 5.091123837856573108 564100.2041594860821 42614819873.29359
36.424488021464 0 164.635704660421 -69.098226894675014267 43.929873562506536244 143.378403982035222146 12242383.2550208 110.238304067748422131 5974275.919114033839 -15040239270347.110266
31.364889533308 0 49.927470615537 19.156121137418312204 130.658473280948997903 136.204475163249889667 12336175.9095289 111.034494004828466811 5960150.1468693402492 60974566883427.925684
42.293886536084 0 28.795397560798 68.881948915820043124 79.375865154433475597 99.052731550674832787 5284525.2484557 47.514468639927731381 4702192.5845312310706 49726324205183.1115
13.505092314924 0 109.371728051151 -23.509015988874367192 122.232226283490703789 90.354603607380295663 13870946.820391 124.998160637120039647 5209954.2584601119676 -13419449242717.578046
34.834884359342 0 154.158885818845 -64.950635804576933586 70.184354384370296755 122.478937039089014982 12602578.6181395 113.485766746684273942 5840114.3824242856442 -22409623537465.370791
39.645470422142 0 5.489559572253 4.93110866954922869 176.100220910177652788 175.754221027085472992 15052461.9413137 135.391549382134450615 4517513.1715669204918 120596095845873.505965
24.964126463103 0 116.357131086195 -35.503236005726961968 122.767107174013856181 94.201020059323563953 14471307.4328492 130.380471136973720885 4846677.1473519545397 -15645354872172.656582
50.860078679137 0 48.549731553333 58.728016150447603431 76.607530844044723203 114.353508957613003404 4726685.9775161 42.499003292010857925 4307106.3509766754958 46562563907020.194799
17.956455420121 0 84.711349925556 13.725792757600825649 60.384502867485774326 102.783241874788952957 6458576.8866868 58.196036433606491796 5404595.2351554257642 12750535602216.113037
22.973477397103 0 56.009153846096 40.298523592676960029 59.95799345823100206 90.056365696932726789 5884951.5486095 52.986700724527881607 5082969.6664530213514 24048428828720.40175
20.052889949232 0 81.977578557245 21.526221960544430975 19.836419824898973673 89.081274273399774348 2070318.9073022 18.652543175684854109 2033974.1668489991053 5012810395227.100067
16.446670152465 0 159.114455786501 -65.26987212388837528 134.049514626723232179 54.613514542799105423 13606499.4308253 122.461602401628370918 5389401.1274961893973 -73960364925985.852815
32.667967014316 0 133.565559499199 -47.687068378110340646 86.855941174728367032 115.13353157269340785 12392913.4355562 111.634620039129593513 5914486.2977904466738 -13025109317242.52627
23.882278043017 0 159.775437976533 -56.637272907294450745 158.001120751267172822 35.013276119319165143 15956798.4167285 143.604184899951795803 3804025.179210481653 -88309437354762.051646
43.15371946164 0 127.900346484883 6.880771895177972878 36.289124705134547842 144.499557452454256019 5359696.9901114 48.27493093785239943 4748466.1243124221391 11721693873627.441743
8.758688136766 0 27.612955485812 59.205526512430480382 115.787267608171238769 116.793618855456045689 10585783.4004184 95.254975336052389573 6354066.499562712334 63085772302320.059662
34.19913109722 0 74.184795626387 29.463206450868118642 68.74271021898829455 113.906069374400258811 6404683.7468433 57.663399986051205475 5379274.0212839762604 28055890273446.469366
35.431227855473 0 174.170972642324 5.507959555354536301 2.91336720934712491 175.225891805561864953 3326839.5060205 29.972051306486833528 3177297.4580713237113 744618580162.659095
32.377611976934 0 53.514590790067 32.568215067951098953 107.814340389787643655 126.321687264202713653 9577205.7961999 86.193064266947696149 6362295.3613650268617 51454766646575.550898
24.207700330096 0 6.609151060411 48.941036400993861617 170.30937998956386908 170.816056872563568065 11845706.0301748 106.509638982176025204 6138336.1815344711374 116298551060848.19066
24.417791659942 0 5.578899154386 32.657810935795211782 174.401372292953739858 173.96702918983767907 13659908.7101205 122.837243875655863982 5393661.5377544350573 119262247494114.807732
64.847880581235 0 50.019810869132 -30.959341511189455321 144.596944855700431736 157.635538872114446568 15523526.9685424 139.724661369360329736 4133076.699118238321 76174390857721.328442
15.400682983076 0 39.192169982475 52.180060488771494126 68.363538073510894164 82.619819936898400609 7196316.6769513 64.781442337943824963 5761628.1808499252058 30691326905882.732025
20.30702371603 0 136.38662532312 -49.230654785600509329 119.08512299537121757 81.553785297210408241 13795285.7692325 124.236159441434755298 5268599.9008693565266 -38753908258931.603701
10.186836061964 0 82.267763928946 12.315308710130267896 21.800336803956232414 86.572316548512324727 2391377.9926337 21.551561937968063741 2335399.3024574214718 3036352548919.039197
3.29573414861 0 87.821100305774 .074874782625623256 122.092567299351798892 93.940323652995750961 13584065.3175424 122.437054156026470335 5365184.3694571431268 4315699135918.283976
33.407984806849 0 122.191972019229 -32.4206091341754167 80.147416945184135973 123.183256201587681687 11113016.2651722 100.128916693778627276 6260513.4919075679958 700034219239.659178
5.279623415292 0 4.274443210376 82.600418456713382013 34.418367124568620467 35.057122384191760799 8738697.0906736 78.634865028912825625 6249874.4065565247452 21795260816196.660079
25.649537747683 0 64.865015071073 26.274632977288270007 93.043031668414287215 114.480069066657040147 9072248.307782 81.693927420183926783 6301476.7783773057374 35036687408251.362941
4.880836497265 0 50.699961849204 28.433985598740278404 132.962041543077873195 118.817241997245908749 13782606.3928806 124.118422614546485114 5282222.5123124392186 48108349355089.425014
36.073476866209 0 73.628400722975 32.426458512822231104 65.171889395963000333 113.230923394172072087 5902892.2607144 53.139300318861954715 5094325.39455934645 27976343275747.535065
29.915591206012 0 134.128797614747 8.427284359102363151 20.342496251702238537 140.993359066103446665 3187240.7459052 28.716333877181775825 3055567.744709984454 4844415196350.739798
72.871537977431 0 134.839250817819 -16.412702228950326306 47.392541669265207368 167.389704666321769305 10489953.9978922 94.442513359784892197 6351168.2107231721558 23031878369755.607142
42.324699151203 0 138.820391875538 -54.082375801658559769 159.985465829683374144 56.023710499162035173 18043978.6883763 162.439167521351888741 1933967.5058662378746 -58576871824258.028678
28.138235511711 0 58.06531482097 38.704318557556548767 78.307550831333070691 106.58212572574282344 7159172.0372752 64.446029349201575506 5745915.4583241243153 34276794308422.751346
28.609257437314 0 90.030509538437 15.206274688848876938 59.878378252291937461 114.460428216582021499 6299658.8006871 56.749307952352034743 5320102.3785673802467 17243127446945.906709
19.733666783068 0 154.856576444873 -62.789549909135750677 66.649184622336982523 119.232671643970504603 10817351.452045 97.409962504693476326 6314544.4976207158121 -25191239264915.549604
49.049525107536 0 132.404361521493 31.235252170086746129 19.933467970516981453 145.488469796544444421 2590882.1620568 23.319583896484318269 2520140.6653193377237 9246026976106.850739
41.398488994315 0 38.869586543594 11.983236088753451682 145.307368849730022939 151.192640854621333388 13111827.751917 117.973024468856771262 5650844.4169478969142 79449096101197.210234
18.876424052196 0 56.57788712278 27.629693628282550328 16.148459400243926173 63.008012267520905097 1912944.6434413 17.232906042359234535 1884263.3299372670063 4538207998016.060009
8.472126606194 0 50.878449101086 34.077783768135457454 115.693209573426513587 112.252263721678805053 11780372.9525835 106.08168162466617219 6124216.993599705108 43348125159022.555394
34.58087139347 0 119.51608959783 17.10033952344412208 26.538642768874612543 131.391925094901741202 3272715.1419499 29.478401766123939966 3130417.1342838252635 8383714728409.53007
28.777948175329 0 138.670640161434 -8.128039572354068093 28.609615412805776968 144.189095481787942945 5109836.9832486 46.046774710295466595 4577410.3573931347948 3894258194940.962695
74.217609778402 0 27.833495488521 17.307081581667638884 150.748549863579923274 172.33461107925966369 9638187.1093334 86.679653797623678337 6374463.8500946867147 102340689522403.815213
74.625606660121 0 38.179620354476 4.13522957581125988 142.094814225629385106 170.513213518677842571 10899884.2917781 98.057936187066722584 6322113.6303125284246 93712795066095.583366
55.1109089628 0 139.3535538145 -61.544040804471924645 167.154866481811488887 51.41567550486556306 18972859.3911957 170.771247233237815402 1030682.2598597670973 -62247715688294.038368
6.883546430739 0 60.856949620007 27.782465760718321245 53.844549799795048045 78.357665271794591075 6115350.3427556 55.099353820695446294 5216038.7332613528719 12349979750997.426016
8.704861676299 0 160.416343798255 -30.157336788087704784 14.759635614931121259 157.487063280782991923 4581052.4219628 41.281055635081539709 4194977.1320337956234 -2067242933933.440725
57.872340625455 0 157.620535505255 -16.819268371138775019 22.734618221531407858 167.76166343605853696 8538971.7896995 76.907059340681482937 6198237.5466675123726 7167922125230.204612
54.057542230937 0 58.248576372303 58.190742593673139638 15.576363250416263395 71.213972172079197344 1069175.9728947 9.614541411629378521 1064188.4855863357847 9172357020545.922511
7.001775004218 0 9.698830599339 57.21498113382328257 163.530751680014219366 162.056064225922885446 12730617.315912 114.505247485635312888 5824642.2774952224618 107888264845417.263408
29.803563295095 0 64.763711892167 31.811817748935670162 81.663494432962086726 112.546403824976533742 7615365.0739802 68.561664017276336395 5927658.8598767413247 33751194708084.255678
18.595208426603 0 118.048935154494 -11.315726799011520833 48.40948959052526486 121.43457376458473811 6260126.4978565 56.419692455176718621 5296423.8627318135352 2388507113964.771943
6.584421624615 0 94.347343283992 6.119956010623282031 5.687908365969318786 94.977170961145542452 631401.2844154 5.690821603023266574 630363.7344763242265 444215273013.050083
36.346895046544 0 99.139989617843 -13.455262906770638952 92.797729354548470482 125.067595131171128739 11134607.274742 100.318325464792231615 6258978.3049725788861 18305534591621.745346
35.587449260284 0 164.722604537032 -61.258454070258775796 32.464613321885551221 153.57829958021641094 11130492.3456052 100.242755137348515087 6262840.651050322332 -7881139582706.149245
54.028875076888 0 9.614942916371 76.656194303754013886 147.658261698858636858 154.870276891566517021 5323412.218253 47.837321912938937505 4729758.9241662933205 102888349124477.028648
68.133629450931 0 145.898768795305 38.020470355438228516 22.502210018001561022 164.603876406614419546 3626103.4464053 32.61353374667064207 3434258.0617849549826 13234020436823.565094
.619589446719 0 123.234806771554 -28.475185707346800366 125.031589334957935063 71.946313988392631092 13427274.3110213 120.951885296499688727 5463815.1626488546803 -36209509102492.189124
2.625315406536 0 50.177742617223 33.123699771125018829 47.79687028128473857 66.233040505914673408 6008013.425711 54.129918576186182286 5153944.9401905905015 11331948802419.561859
27.698836668 0 55.852452085041 36.967605507924953528 91.462107367844151538 113.555931272413563554 8338486.1315818 75.058711083507797049 6155849.7000491676966 40769883821706.618024
55.955534476153 0 104.057603055208 -3.784112064078197904 75.465096433718344393 146.940895363525021973 9456097.3248103 85.146646569414239765 6345340.0513183878674 30310463421141.12883
79.456211214482 0 17.524286711813 10.772454234710489761 162.137694811371507816 176.774642770856609628 9932809.3214666 89.344032381790741509 6383374.2579114962558 112805261610233.591368
47.989018598438 0 125.990077952583 23.928739648982382998 28.95316949786064774 143.614709768380884364 3692843.4837281 33.244947832959740721 3489573.2092240761865 12452042327378.00208
19.905471446872 0 35.370449801259 5.199376594427973168 162.802668362269052957 146.85751976134435445 16659153.4769667 149.945207290191329828 3226965.6484825630968 78822071434035.453825
31.780420080524 0 24.208726233877 6.28533432228228802 164.204548022854726934 159.451359223110790001 15473670.5316307 139.215085581039906332 4202141.7275758376323 95701190002660.327434
18.214906062982 0 117.33880455805 -8.15187358030599678 43.923539410567657827 121.501272690157890238 5638077.7971539 50.813665975171177372 4927574.18022861238 2936406951863.381831
6.924055477767 0 69.757020762401 20.145480383316395405 92.310644735771801934 97.369333245026517217 9994019.5055092 90.05176029316491218 6360978.3301083286157 19482837269098.0598
14.294778748163 0 52.346759469283 38.715578802658041074 89.141426830359699218 100.831086703169735042 8958028.2289877 80.663300880571901498 6283840.8452036043355 34244632095789.186549
50.559653931278 0 59.052202219634 -25.760685020606670078 145.864001048186774336 142.714673928912201053 16011329.5848346 144.151376566626643661 3744227.8950163651084 59165140017740.67476
31.651311572009 0 42.297832577674 49.387283074154966931 28.91704365675353116 61.54523514840285809 3110546.3106303 27.995777823658382912 2988596.1386741315116 13602104872867.88147
45.596585431176 0 175.118535776544 10.120685293027048417 2.869737593886038748 176.526915822865195835 3941582.6108094 35.498840302484269788 3694564.2706561697467 994654238394.783463
80.39278445103 0 76.80964891084 2.463216066166428262 102.917599852201487682 170.609003610638667815 9972489.396536 89.731079490395534584 6378759.6137616510411 66419487765212.055961
6.740182520913 0 13.884298571596 58.201293688922668194 155.022079105500830851 153.181268692772793577 12464482.6632632 112.119618606382756843 5926582.4770486619974 98619343539422.228973
29.083044569381 0 138.914722020679 3.089607756264869197 20.686800266168889736 144.857820265584640885 3614550.3444458 32.569166427562063184 3423117.0018634644628 4193831244813.785825
.621092749897 0 16.102844264946 48.232274219514793775 18.567244222293248261 24.556437713520476934 5570557.1953373 50.174133621240939315 4886194.6341925733932 5970937312946.952477
37.653638828447 0 137.853808039161 30.265261486767069606 7.440737656545109234 142.022971576269979005 1069047.8799166 9.625621064868388049 1064036.7507147022245 2944500711417.582393
4.404606235786 0 156.123678419485 -11.166197413147649405 176.728933575636367164 24.287272842988926489 19179562.4156505 172.61941726095557154 871396.5132016561672 -93263941141010.612045
27.176701036956 0 6.030012719257 84.64553919608213361 90.023922499395014947 92.786261566827419868 7009435.8491467 63.034203638164144768 5684617.5267910179521 61445159223359.059902
3.302308911476 0 8.97551862912 61.618716728051320811 162.510958917359390587 160.923939760553143161 12650721.6919684 113.795211947181884636 5854325.1477374319804 107603405398599.282884
58.879758886177 0 114.824276652884 27.643242341173310281 45.376215094118427881 147.964522484664060996 4901772.5681154 44.108077963579631634 4433296.1609875820692 23431820444230.569935
9.475220278148 0 38.932383422044 2.218749985363731397 170.35834207829155104 141.658474129787309151 18339165.4163233 165.111514354156799642 1673744.7601118090214 72599278566241.718856
41.466841043791 0 109.578369879174 26.01914232966282241 32.534760195962826366 128.159784291644724597 3437725.6660353 30.952024150161768797 3273296.0616481287833 13124523626070.666893
53.888359009931 0 98.381516752494 10.049636247749444181 72.20534695581802854 143.602159725305979245 7945884.5383174 71.533315832132905006 6040973.6839650793269 31960523701308.405554
23.853889740799 0 118.196826582424 -31.030496743714609035 91.539195292896346896 109.891398221790066834 11478610.3454773 103.43140154442073546 6185290.750478418275 -5863203094428.693584
55.956731505923 0 57.706685040664 30.113795853670504753 109.09698163150170057 146.779343066540629328 8367728.2577391 75.264404165111183194 6169807.7347468452234 63013162955011.005236
28.505205913081 0 164.266676515654 -73.967996185770407079 65.944242524678311061 120.592998665528690691 12329069.5588533 111.002219397624849306 5948339.4981782590423 -30908991213226.098046
51.947055580729 0 7.529183101991 52.211149755493853953 168.054184827868318196 172.425991612466622102 8405690.9692773 75.544634537319494055 6186579.2297242596159 116799015637170.21588
19.603262241917 0 155.310241469883 -64.497146189687610562 72.073418936580752449 114.249128850218973268 11129803.9331653 100.217567223637470053 6268053.7393983095357 -29040226058653.862954
8.667276657619 0 170.401487964052 -25.711398997503429914 6.036423474639316832 169.46409829426344205 3859105.2080866 34.77770734969814879 3626505.4134608915443 -661424323766.964183
45.292351113841 0 104.755570801274 33.904613360449113425 30.989094387217340705 124.896016003639768267 2922442.9238597 26.304113184462176336 2821145.1426668718809 14231419374410.82222
35.328099152836 0 86.221109285317 -25.053079543515994032 137.07863866549488376 115.960750169085924019 15774272.4712622 142.112205171099396346 3909338.9347988234089 21001668281359.954247
37.793589573001 0 130.437271461167 10.298031202628896261 27.737373649972565409 142.265457288483062006 4118622.1932492 37.099758244179782669 3837117.033810311744 8350426299600.192198
77.858061800612 0 29.666112170968 -.738692377457131114 150.933627031610412238 174.004687728765283776 11266464.3885094 101.364099086660346358 6259756.2582398729698 102230672562011.00646
40.877645811776 0 54.126934551558 50.619289349009252109 67.178203328134562752 105.177353051106966565 5152942.0221439 46.356392328279722171 4610975.9831860985536 36097469097223.160205
17.450044814197 0 113.230674522015 -2.127739360728270606 38.604245449983240002 118.657849063154071354 4757178.5381671 42.87410925484394413 4325545.2438641988197 3828379824354.209612
17.678164508809 0 54.002097515254 37.188366752551397072 90.865255121713861749 104.826548310924443141 8918983.7043879 80.307058065303350956 6277988.5205254455472 35898387165952.114734
7.649776922473 0 21.234632447942 46.019256998096815647 20.402590686048474912 31.068587858373432921 4680601.0073264 42.155636243040333103 4270159.6609485451716 6945253734504.42889
49.493928171376 0 71.328984986837 9.023355647006460499 106.678645473545463878 141.375490245644784666 10430816.8172407 93.891969970093436229 6360959.7988278229791 49513070716361.390736
14.047904930908 0 97.048958862099 -1.321473235187342942 67.41887873159455547 105.590912857545573175 7621201.7997324 68.68769560651643038 5922664.0090450095101 6025188743388.939901
16.45111520522 0 103.259824693141 7.459338626176654053 30.150648477469862081 109.667751753912015087 3426099.5117818 30.87587618383336511 3262718.3697454248367 4520271029708.90057
11.695368374369 0 49.92711210542 16.34475503818505324 146.829159972333832736 128.666191953615620089 15231713.6153841 137.159351935786427165 4348092.3999257378273 55613455511136.303617
20.984641117615 0 160.333317255672 -55.785670347269857117 158.086173376809152059 33.903276835070866505 15736335.196457 141.615721780035742751 3980817.154754208854 -89489324626677.555073
31.084824113429 0 162.048136420535 -74.40960356104954391 111.28093498596546035 78.510830028460637658 13938106.5963352 125.468445120776849816 5195415.0780561726369 -59138231717845.733443
18.870663525454 0 58.180445349074 9.922203731557252276 138.538333703570730931 125.268473410633811544 14471227.2633508 130.332118290906566426 4865369.2173303833311 47370987321405.479702
66.02991541352 0 100.342754669946 -27.082389068807474432 91.42584373318543312 153.267616113652685304 12786765.2978041 115.130508724236588555 5768794.6090355894455 37440745112045.917347
70.284967296784 0 99.426968016059 14.627507744772528826 74.629604673487682213 159.825210987464350709 7906464.5403885 71.140498396889045993 6032133.8812390120941 42735958177528.847494
37.972632798984 0 51.109946721162 46.941086934168500825 18.833435620675305904 63.924916745147929433 1834454.575038 16.509125462248133188 1809253.6860104463296 9056719742547.616931
1.191438237168 0 41.096786511078 42.371665615781592609 126.270074354353137462 117.352106840235935033 12799722.4026313 115.234892115541379594 5772127.3295170764404 53890800135300.343996
9.537350360342 0 27.883681370449 60.888683732870369006 106.363656697521814635 108.973271101790304502 9953495.5052647 89.568654046446137096 6378032.1073367046561 57359656766635.469939
18.552975093978 0 24.392879944219 53.972089572794851906 136.000657911061631725 138.361588598637747362 10943757.2097338 98.43918262683151297 6319804.6218684734052 80645584263599.882025
38.039842019748 0 53.848666905753 47.850866585266756679 25.277933660113874242 71.284667717289698135 2318222.3005012 20.861931047147056453 2267498.4141262080266 12323276629978.341741
47.235493725582 0 21.915442545912 64.28061417115742798 16.447696559380283557 35.692750218622282227 2145577.2956871 19.294588483947561948 2105438.3811477489203 9747398544817.480495
6.067040272821 0 93.10609854649 -3.339044133284998898 91.646260438367775038 95.928000721064338725 10239343.5540606 92.289971215306380516 6351783.804067240863 1990237079120.228144
13.166671186654 0 156.9877056519 -46.029874698950134815 30.599196661458344065 146.81052520771938439 7238142.9773793 65.206181332385281853 5774710.8599206300839 -7188054550435.828277
31.072695366125 0 109.704067512395 25.605076391429388597 14.382002787797685973 116.566847403823664013 1533173.4468247 13.808532332300897289 1518397.1029942927585 4844992979348.218602
.02350686392 0 118.985409232472 -15.052412502533805435 150.673460237577267056 64.906720897603116626 16400518.515323 147.755333346962655155 3405195.9826022676341 -38168763228247.818173
12.885878052126 0 144.500189246031 -24.631881773331082027 170.403348095215713823 38.497783925432162186 18368927.3837289 165.365333320060938028 1644607.2769569684109 -74940093549172.812255
17.268835767565 0 3.700557637282 84.347730272741105718 37.3521300230326359 38.599794179850297196 7594086.835291 68.314024623064996945 5924761.3482346320039 24713679410214.626245
56.718530905299 0 9.897772522309 11.680772997528739585 170.55261711913522143 174.460435573782803412 12373686.2285536 111.28049139485288627 5964980.0706111166438 116554213838862.626748
43.093607402015 0 56.807564653343 -4.129057873299349744 136.743626644380405912 142.147603954035099233 13954561.6977207 125.624523770214740978 5192759.2194061199915 60325097259819.123459
28.085299755109 0 17.582954267573 22.19178209778295611 164.962391587579711277 163.266967353376055975 14213360.6768809 127.840170439906095765 5071368.7419537869414 103122501389921.510692
28.846720155831 0 92.000519155689 -20.517212185627749577 128.116443546305191548 110.769757212729769379 14738801.4630775 132.806927125116955592 4666764.840918191954 13248646836515.542398
43.843107846075 0 153.846400347841 -71.269218648518086463 117.744328537214035374 81.341005509003024961 15526314.4049277 139.771876578631718745 4119449.6980175601552 -51324354250566.084605
41.175109889346 0 175.73404687087 -28.381892766056997998 4.526028955176634461 176.348678015376635495 7714923.0870663 69.508645990850295256 5957365.6034066730878 434128324633.069356
17.144265589441 0 150.880568229333 -28.786089606638436195 25.937634058026934774 147.973102126478450604 5797835.3616146 52.247238798693200617 5027173.1875010444452 -2052005564979.092369
18.140008084981 0 123.779585448215 -37.872671755644909775 116.904192578456646427 88.568636418662019413 13572412.5466204 122.270517389846537266 5382144.1579172642762 -24865714493037.827716
22.821365629541 0 115.952557015238 -26.321446841940254784 85.206986852001494507 112.408108683680597384 10661023.0786034 96.07086930131082739 6322801.6727993400481 -2501697534369.466062
32.257990201783 0 150.869187174021 -54.824260467183555711 156.561573965852661863 45.534810497878492661 16898916.2234232 152.109817534890474575 2998828.9864221340382 -74535975683922.348712
39.49327314527 0 140.836566705476 -16.423550298805272605 36.685133578510657407 149.426789575292090701 7268775.0376729 65.491314370547717719 5786727.8920857071384 6065435176920.375666
23.767991890779 0 81.632828734311 -16.806590952462069709 149.738610286184802047 108.898472198506895209 16785010.6291059 151.247345211174036293 3061967.1783097333549 19242778189717.260658
42.496576764546 0 .165548738367 61.200095175201199603 179.666196014678657282 179.746893753529690115 8509305.8221784 76.480111135775561028 6211557.6845057436865 127219347433402.487683
23.37006646977 0 .910228532498 66.474418401287595733 1.552183188406980816 2.088868324926246289 4791481.5364996 43.115174479981862946 4353337.4030778568938 833605733052.19669
19.994795859237 0 173.593173660623 -75.177219719926159433 158.741596988184950467 24.127093846133804994 13742605.6264524 123.670307530519052626 5318046.9798840628751 -105863201806255.800978
1.269658479516 0 46.496047601802 43.592976578444478602 90.684525887486170424 91.539242394540461006 9969428.812229 89.783735466496703945 6367305.2324604843569 31816609569360.513062
18.645215730907 0 91.204763835331 -5.07541546422921436 101.127103351067474523 107.948021613622137757 11367937.1086597 102.449079092539422931 6209191.0456299706663 11811987332961.23841
68.391288883142 0 32.291360786422 50.406165949083117294 135.501938931444434125 162.003144394968907867 6326014.5139429 56.853780046931667658 5343595.603955821559 91858101945008.14683
28.459545317132 0 131.627116861212 -17.207732564305947062 43.634692466981953215 136.506792603896624589 6898743.7410256 62.168063753941591155 5622742.1728876008424 3443908545857.640803
7.686752352213 0 146.211145206052 -28.917723347931304784 163.486470248682233358 38.990273918754185336 17094911.9530533 153.881391878770278586 2837200.9925330907525 -75807485146464.610969
.64034186365 0 123.369802893649 -32.592414472935303475 76.048812057266849688 98.024248734061607106 8746660.6462652 78.803442938924070906 6240212.8442177494349 -17889922736138.424974
51.022969333621 0 70.928244094143 50.602735331531517447 49.7928930953542578 110.500760610103236022 3441686.4042094 30.95559583236947878 3277376.1483858760347 27987828749670.560419
22.288465298004 0 69.621519507085 -4.912231564840104562 142.681743549936929249 119.428358444442736532 15576793.1624633 140.333346714541782701 4069290.6074817798098 35156569187743.198103
38.278706194814 0 116.769886228164 -45.213514932524026697 131.987859404856965922 96.014100177112467728 16014157.4068109 144.244469214877104097 3719365.5049028169952 -14668254788536.750992
14.24969115701 0 113.619675867679 -27.151095883522850275 127.729463064735888941 85.960623495334244591 14441696.1273339 130.12857089315515138 4865013.9500414288711 -19521656365878.331201
32.973411269805 0 46.065698353431 8.000357843774115639 144.233746766534798714 142.364016793767431065 14108473.2364152 126.988325582496448135 5110004.2818679087806 68070157484700.31499
11.884027068704 0 59.246997449177 -6.431773707290350515 170.549196384305545959 122.18085060024305472 18827330.7069827 169.609961560850627963 1164341.0997014020218 44426770056644.922307
63.299659355513 0 4.961364308518 62.834146216572416716 171.218297548713166872 175.117432600361965738 5993185.7864619 53.850658370552939515 5154314.1874600872928 120536329850929.123524
37.104397046123 0 81.889793301424 22.902813652261998905 70.233438162142126967 120.935911762169260057 6805796.5407313 61.280195135154350172 5583008.2364739307334 27577131767888.997322
27.912201073082 0 93.335334979907 22.584853282541441188 31.616864803984472932 107.124038156544818919 3230177.5487443 29.096248304555476689 3093295.0927297065702 9732934051970.815943
66.035196665325 0 98.296318724173 31.269173344204243454 65.399305777763442733 151.892599739667467015 5765572.68568 51.861650382347039753 5012957.7356936473664 37918053866653.078372
58.443950266403 0 29.561905381646 35.519613515114951744 143.165121509798372763 161.479891246030836337 9044293.6999022 81.311310583576305245 6314213.1821767600409 93396249610742.871303
39.516194068199 0 144.535341584624 -53.741921506580303349 67.12140213504665522 130.871188003208945244 12163096.875085 109.551802866346940804 5996348.8914581477191 -9660403732413.237398
83.04262434686 0 143.179609854665 9.046227267580238248 35.933736921416898057 175.770788535406313053 8375679.5080332 75.361401388226560165 6168294.6149973241096 23076663392875.912425
47.852703462638 0 14.782448744043 75.917034235437853607 125.150808062635956586 135.354466840050929534 5723133.267717 51.438119183622424248 4988964.0162152511547 85390548876603.331405
18.352487745925 0 131.158495572002 -39.72482561850349812 77.372086855266511348 111.852577362107756209 10263734.2384338 92.471280284934150787 6355110.2838508335232 -13632930094129.292034
19.610027893484 0 61.148999531971 3.917212154430625183 142.62175176423617025 124.176952499321120883 15179268.8311452 136.724652045198704588 4375279.6200615356714 44498545040230.247792
37.890019024071 0 28.155454916229 60.230364486607121438 26.245688508506560554 48.509486474608027261 3095227.5260516 27.844494485709030885 2975285.8904934431719 14394482235439.926403
21.27400519437 0 66.518655015288 29.59732207051775542 29.129807944898327676 79.28853866782912214 3062027.7697048 27.58142914962496311 2945278.3863362862975 9013967294900.483375
18.73757565583 0 173.208006744922 -62.91375966956401215 14.845859018929689704 165.794535940950061082 9144588.3219893 82.353726643526844342 6308367.961877671709 -5241762726523.552269
42.702510624272 0 20.557404168888 15.360445685187465876 161.450680426048786046 164.457024354600450171 13302891.0974729 119.647390054436528639 5570480.3571510424877 101866160529438.855012
9.629511505319 0 170.171749157156 -58.330108507597165263 17.628133627704536353 161.349788081183968369 7700234.7132002 69.348168951861912878 5955182.5546470553986 -6235257597677.886041
28.045488507748 0 172.720784715037 -68.514276940652034598 19.958792438281018172 162.262811488702415433 10832170.1136121 97.543258542384409921 6311521.1346343046551 -7398003716798.313929
2.772765838802 0 114.5727676662 -23.686765736998649393 77.544158137101346033 97.517041033584958228 8873511.1191586 79.963279648143941131 6261669.4752752822571 -12033996241403.109894
13.491620630907 0 140.873701285728 -50.843574559524607691 118.404202153051101869 75.92895029648765785 13145525.078646 118.366744383000663652 5610326.7721114432559 -45907341190201.709936
31.502257150564 0 19.853899687231 44.438741945021011589 152.028260148130413326 156.092336139519375239 11131020.2526886 100.089362799001040705 6297321.982397689138 96438090550519.765353
19.686906931917 0 2.048670973611 33.820395378014357315 178.008341908010666323 177.679585823345407684 14078751.9392915 126.609273924748206813 5156765.338486156337 124410416867781.41462
22.762532924501 0 28.503393613681 27.506735286151725226 153.242452449881161736 150.262824421350724109 13748258.9509153 123.685962876758118408 5332567.538098162742 86131087379773.993898
50.340413863206 0 2.53859523901 -31.250436707467694216 179.014498940189705553 178.103045898455404295 17882405.7777759 160.905575858919580505 2119232.6694804685793 124366479736738.071608
2.98066520225 0 47.463675192493 40.341542234586241042 109.605984313079385169 105.401958406556345535 11441450.635794 103.030505859102173865 6207893.6379039073443 40927721811346.618862
29.247293372331 0 45.029410864491 48.318632112438986603 92.282799489207885044 111.987482299261451218 7801472.7770445 70.197565495082740953 5997336.2213959132983 47337640377558.832335
24.389388046744 0 163.414770950738 .675299289459258553 6.78930079807174955 164.921926385404543513 2723889.5881121 24.546497715284933339 2641353.8302269112429 1063357718624.287312
62.482652918877 0 8.252125941321 33.232811536179021506 170.163205247231471813 175.445424087909893808 9358158.9821303 84.124507549133482345 6356535.0438882536496 118428234240816.513899
34.590918562694 0 7.993291365237 51.42553215181805146 167.128188073882944443 169.43130654704308259 10392812.6123231 93.42837056060338675 6383899.2127347763746 114336917888267.914364
69.821651887204 0 62.92538659285 72.043794732817548879 23.320160820555760315 85.002463342036758801 879631.9531361 7.904642800990551886 876860.9138046876294 15632650536333.254176
38.376944784789 0 17.473863587495 55.279252586694670166 148.483684405587701783 155.614243012976590637 9203619.3049205 82.73829954433574547 6337809.4817681025881 97807801663396.399292
46.892014406907 0 67.39750819497 50.054601019930333772 43.609545475730982851 100.758450815354056981 3197345.7586115 28.76243158014969311 3065258.4797589822865 23589784072974.096025
28.353602081529 0 76.274827000925 31.194436445411213131 31.458476146983860433 92.204244235304787391 3048218.327439 27.451144274668761873 2933128.7134389620344 11247248294344.322481
5.326532548779 0 9.097780130462 57.758262113136848914 164.499274258052841383 162.877999232894481095 12873200.9722311 115.79055481691698242 5764031.5244371421305 108899008384970.564931
72.117921223457 0 6.152872172491 27.006897756515806363 173.173811781657280321 177.878073603765553327 8997693.7848986 80.896624818001908421 6306005.4355679725371 121647169141209.337038
64.177261656652 0 169.155628588387 -62.917580953803579439 18.990464058492882635 169.629168390979152429 14186020.8922706 127.726090058553197116 5035463.735536794077 335168376245.250453
37.947075857326 0 139.342238102087 -43.128166633901710655 61.680974170546238004 135.269008366092031214 10925408.6406507 98.422557317742166922 6292327.2717351662724 -2878139657337.864058
26.307941680393 0 147.225038832701 -49.314417365795352401 55.82607154156295034 131.974573587520471938 10034382.3145598 90.391387400182285975 6361835.7956457968674 -10775848973793.992569
53.631996624514 0 94.231206530873 -24.322411486443992751 103.867391100271214598 139.458674175181108442 13051226.0570234 117.538037317391998175 5647534.6296093432815 31970480317274.357164
44.004238156528 0 109.824798440887 -19.557318635270308642 81.358494523201842003 134.029609915777875955 10832509.5519374 97.586049107336685077 6306751.8120159249145 17096772259554.504432
49.639180896033 0 172.391914509103 7.83011525308658278 5.114172330857355993 175.025640902818529736 4659374.0988898 41.960826165673466071 4254522.1423640281091 1860437924352.772834
27.392085228067 0 163.92423380568 -65.834622858583744806 41.772796000787220511 143.178050120950638477 10931147.314304 98.437517504715533477 6297520.1924809563317 -14673671331667.488425
44.143562138878 0 142.681821455567 -7.997604143032335841 31.723396618973566196 153.896387549517793874 6597698.9193097 59.436679031786432857 5477238.5042466885471 7919560556324.465099
42.989475047544 0 35.485593498718 -22.173129586699477905 164.896770399382662366 152.675596916020677073 17310552.7872924 155.802389397994553144 2641260.9889039502104 82910772016290.623801
12.028199404842 0 6.750615276825 35.085785799083180955 173.892270533009265887 171.931230872909921274 14750382.7256983 132.666113046006373812 4728707.9536052384558 116980822519594.864843
56.167036940831 0 177.985750277647 -25.378865450268494417 2.197739985587264746 178.756775063163026945 9037584.4900647 81.404025659176216637 6291172.1765981184605 544981451044.363742
44.441940918959 0 57.104069418997 -13.782608118540132249 143.084798779238657385 141.818572490275480224 15143214.5322994 136.335988833293368798 4413419.1198004675552 59888368012974.924525
8.208033333575 0 105.258344953671 -16.657588778768321206 100.829440071396485963 94.804408628370802516 11423879.6577821 102.958197727251546693 6195905.1220307793426 -7374599359370.306865
68.774240952093 0 28.537069063729 23.629665922502092902 148.675530832607903311 169.091766635019720024 9447802.0379189 84.954583535924707864 6361877.220132852006 99534040798290.973825
17.64690117594 0 40.89679291018 4.022724409294159549 161.803702045621127057 141.272784545677804594 16896388.7566713 152.110851977692491599 3011397.5282767167562 70939153735974.58841
47.950793718575 0 147.354822294378 13.280167579164271578 20.120413486875134985 158.170551379188741332 4272286.6100659 38.471873067036243415 3958962.9208959494587 7639893821355.344163
12.497996919791 0 150.82261659966 -47.521449956879211038 150.549196046350859887 44.718791294774737051 15258577.9916899 137.336081323681884506 4340164.2319249751239 -75051437676344.019602
3.515785865095 0 59.263243201138 17.790615163216038928 141.461005551791370174 115.74678435231796389 15186653.965357 136.806978670866646778 4365588.3395028833446 39870534036456.562468
28.768517563908 0 110.322680357855 -26.519374288385967715 98.067906321718286668 113.25392864265308837 12113386.7220927 109.152345047190042437 6007048.7596179940492 2069281422608.337633
31.982822775287 0 109.290114539956 -12.401138571716524225 73.327969848986095205 124.879499241463509126 9212160.6522482 83.008447712202809834 6312229.7146436225717 11003469916752.864095
19.570432528161 0 28.740966951582 61.619396375101618346 99.785406498834552256 107.998379654880312637 8614755.089148 77.505251812796221543 6226311.3612559850593 56067851874192.170291
34.143048708981 0 69.51983962219 36.997252125040645904 56.01564263110565584 103.921662760483339202 5013801.5800171 45.134241459400238718 4512393.8659177969 24303198210303.780218
3.935957557567 0 52.436217226781 37.824446788544112769 83.438439603293535085 89.229164462126771199 9168789.8813277 82.586235168871286337 6311363.6927805446849 25979298394134.639755
61.129461056535 0 17.551253764348 78.387863550789787985 118.814812128944507196 133.70610813122942263 3997295.7892009 35.917884092904948979 3742170.8344936212443 82271032127693.16589
3.421155549617 0 117.603077191721 -21.219381398155866681 53.544399883561208716 108.462517491504855818 6439245.0997699 58.031001140391312493 5393641.4739003617397 -6448565537694.646061
32.199157039059 0 20.027269750465 63.304502042590373196 132.034925840673696992 139.912627269622829752 8604303.0175665 77.364120210701674249 6229815.5630395658899 84869733581589.295238
.822552758098 0 34.818378986644 13.515662875312775488 8.989765670729653024 35.949534634702059623 1718259.3202848 15.486296191256021634 1697416.6604600584153 797855641430.608891
22.093738274028 0 170.681472860323 -6.742624347861355679 4.531931315167888431 171.306863155411353935 3227863.7987681 29.090241409291797292 3090980.253994166485 441218726980.953197
39.48039894993 0 87.282048692256 -24.910509138681227713 128.137681811632628713 121.711631340407178135 14949490.8636683 134.670804855150454647 4527683.2952598058841 24318824920062.879679
23.69170905579 0 132.063863063217 -40.687590598372122427 76.389897347637172336 116.394432846310699826 10623821.7355492 95.714778422051439614 6329194.6885535600748 -11066522084844.935136
.813894017158 0 18.816974707058 50.593807692461305191 155.175629862287786372 149.533910451363380311 13842545.7518034 124.538169596131027243 5276254.0872247014538 92515652983842.460323
60.498135169978 0 14.402672810838 12.198221826511738496 165.83945638179899956 172.783459292832685266 11847843.0323106 106.552664683262988173 6131930.6747794226509 112168579559174.22495
56.613748278121 0 164.607893291612 34.436568298724291799 7.117337693395296031 169.786491850891132892 2523532.6367527 22.706702518236147407 2458214.2373751964518 3661190909306.800345
4.353450729152 0 103.233574040466 -3.145254788626592435 30.50229334501562488 103.563748974524958336 3492737.3971927 31.481180872375856418 3319631.297584571004 232863039233.953875
28.862498245296 0 77.616850437576 16.26472215669566619 85.497207496626466819 116.933862512763500344 8729542.7699654 78.626797302294860567 6239936.5749710824841 27756398721897.174066
24.304907152864 0 169.325325310051 1.433605151881832464 4.165940219922587965 170.272462365962871455 2570010.31357 23.159688911039333127 2500619.0065841359612 668241278182.719071
10.189698663604 0 149.730183652442 -43.326676023438175932 153.148366512831958335 42.92139385486819693 15514023.7287334 139.635728491944681302 4150994.8579423507827 -75541446904842.90808
49.848181590467 0 1.740157383446 20.405242225208311985 178.247549230809907989 178.801032702563268372 12220827.5344691 109.887118815892790351 6022565.4770274773585 125428624899323.659416
13.030866301142 0 109.893930639714 -22.482831544519822258 140.761210729738855165 82.371736722714851668 15755116.1820327 141.971040979357694774 3920414.5259027910152 -19421881186864.423462
33.193281467185 0 151.700748956165 -41.151672070525223539 38.419124284860247174 148.221356053664153845 9125842.4572481 82.217488953861158949 6301544.0399167709777 -2457933595572.730309
33.363702991722 0 172.898206162262 -63.212816746476843028 15.714998156903843357 166.777356149472787827 10796096.0063367 97.229060680347454537 6313972.1564420164324 -4328854162846.139344
81.209730724037 0 64.607034218371 -33.892680517408251506 120.971495805190110004 170.404813772446965722 14218639.870483 127.984344279961882842 5029362.0312811118308 74927506291674.605445
84.49313372567 0 44.062176316481 -17.570092516031215028 137.259693182063353824 175.973637230107021589 12392439.9503488 111.531809859488348303 5935680.7949336646973 93436915821997.576421
8.988389567023 0 135.591577170335 -24.221027716300497021 33.960051796000590159 130.753279335550387413 5207072.006276 46.926129485889807704 4644237.5060699699244 -3413776361754.360498
7.233769436208 0 176.654551320883 -48.849132604776785109 176.601435366476713421 5.037779120577946247 15380027.8289912 138.358281063797337733 4271593.333427371255 -121560228685539.24198
25.667301923648 0 165.865810332805 -50.620232529080889031 22.061010504617130536 159.730880799186332309 8724941.2419856 78.596084162726465716 6235905.5322179990845 -4335019959513.589124
11.393453958608 0 103.983428298864 -11.354875472482677189 76.258812283705016962 104.014356938494710659 8798950.2908855 79.304542647955827858 6246649.2853773449699 21816453964.55196
67.964135603447 0 82.943262082749 -19.640050782646093412 105.698978700603770784 156.650619021388568928 12667180.9117455 114.032689494338304427 5824218.7016903955541 52154384017978.331639
62.051119792364 0 11.128018862381 83.855266840097823611 112.674858066810490831 122.383784147680661628 3437930.4441712 30.891069868834740073 3274798.5715679876677 78807778935382.438339
25.102149869815 0 40.235857039559 48.296557545890560676 34.072462542900448281 61.41586730915713718 3931247.5339453 35.38932910307202416 3686615.2490819738547 14964859344184.471094
51.327128301361 0 157.507841386438 20.173772376862884207 12.682422408955768442 165.220897942890940421 3630369.8599353 32.683402669213993196 3437105.8869462066052 5449804328186.89709
23.474216147125 0 46.678628932975 47.501521345953050621 54.338775236772878081 80.572387690285547473 5464355.8057291 49.190120790076259041 4819009.3910381529155 23948341952063.846333
36.683051268707 0 59.9134220097 -34.023171607795531921 174.31025861217819742 123.138509441522237857 19425192.879691 174.935403818920014616 574068.1429244003429 44682327617138.469844
58.744150157638 0 136.522735331872 -31.540912691185800066 52.441923606881506406 155.194512958366995248 11118098.9573425 100.134652014767295626 6265005.0978975825826 13201781071499.890157
51.751281137642 0 33.757132989077 64.121648788158237473 21.295234110582796607 51.966943039533835852 1850116.5799348 16.635495527980984067 1824356.8330153547199 12884889301176.930698
19.773933963855 0 88.244372170064 -4.599612537189952262 107.72709070189264194 109.270070124629001544 12044598.4080381 108.543100332699285645 6029451.5407405566971 14834037742870.059892
39.200876320217 0 134.856615004185 -16.799514501718282638 43.675397103785494015 144.940350340820159936 7684240.5553611 69.234710761708416701 5946684.7247707816254 7119964311987.891899
38.105810689938 0 164.670289256197 -16.33741078518428335 13.110282576578816704 167.467077049712781311 6180176.6676345 55.685031363005101385 5252467.9074625082755 1974618932947.901788
41.521310479281 0 170.317403231942 -26.145378737930547576 9.977868070750335139 171.92933505080876208 7561784.4790375 68.129172489529052334 5902049.762281321437 1138495223655.47569
32.241265005248 0 102.500901349632 -3.245748416402291224 71.931980979619742952 124.120046609527451966 8518683.91191 76.753578689823740205 6191281.4857895716355 15259035122497.517915
41.5153048399 0 17.301852441557 72.97564440737551351 36.459312843012288941 49.41132449653871056 3998772.473891 35.956871650262591942 3742632.2852534434507 22725908076412.652383
.392629863796 0 25.902510436627 50.342417843480968873 35.445910446930489214 43.087650398316018744 6471966.798244 58.288878571911068624 5413333.9954409372486 12140102677564.2535
4.213823194191 0 115.396132131615 -22.878804028266716664 69.376306288000431687 102.211557813582238543 8102268.1662216 73.015921625809089974 6081089.6199742587348 -9302233372009.588337
20.887224275633 0 85.008601901305 21.254912121078436127 21.899214754867725164 92.966926161389850749 2274336.0154563 20.490374931706914196 2226208.2883082050546 5615982051465.404671
54.110966620489 0 145.266915564664 26.486640906235318411 19.097602913800180787 158.053446850266598587 3445564.6777048 31.011845390383106327 3280240.8171578636715 9037168974943.306119
67.353134259898 0 51.005098873193 43.041538374783853027 114.186949886940596532 155.795322500736167796 6582850.4474452 59.177622849039962867 5479215.4170259799255 74186400245646.212036
11.117043578156 0 97.379569600861 -10.570763915881842733 107.625834147203669139 98.142400502746669987 12143815.8415362 109.451796621369871879 5994270.3006119077512 538077628729.114299
42.020759525796 0 43.442582556896 12.624313141296751219 139.841076840612661427 148.38566192389802066 12699106.2476873 114.26751780414556216 5828586.3898407929424 74217002188639.046493
71.663254855522 0 149.800422195409 -25.169998746665320364 33.175749310361502176 169.905376778384264918 11039983.0605937 99.405807205381622094 6282838.8209678649497 14225027110652.887943
56.279982724539 0 68.279892030062 -41.360281944907619279 147.341948469936636537 136.551910060918047489 17139690.6662776 154.314282449997431034 2768502.0240323154614 48295107264106.958536
60.174647030691 0 22.901941065938 39.182099298590229418 150.590705311218945391 165.522950851634609237 8668548.8071412 77.922632897679367273 6246203.3558503090457 100993609555420.749449
64.60720493901 0 108.984728103641 -57.494440629886070709 113.054056725195484074 130.988276746435850126 16487654.3721395 148.441701929211374632 3333513.871827889334 15572115521240.964969
'''.strip())

    m = ellipsoidalKarney if geographiclib else ellipsoidalVincenty
    t = Tests(__file__, __version__, m, verbose=v)

    for n, test in enumerate(tests.readlines()):
        t.testGeodTest(test.rstrip(), 'line %d ' % (n + 1,), fmtV, epsV, fmtK, epsK)

    # XXX Pythonista run_path doesn't reload modules
    E = Ellipsoids.WGS84
    t.test(E.name + '.KsOrder', E.KsOrder, 8)

    tests.close()

    t.results()
