////////////////////////////////////////////////////////////////////////////////
// STEPS - STochastic Engine for Pathway Simulation
// Copyright (C) 2007-2014Okinawa Institute of Science and Technology, Japan.
// Copyright (C) 2003-2006University of Antwerp, Belgium.
//
// See the file AUTHORS for details.
//
// This file is part of STEPS.
//
// STEPSis free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// STEPSis distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////

// STL headers.
#include <cassert>
#include <sstream>
#include <string>
#include <iostream>

// STEPS headers.
#include "../common.h"
#include "../error.hpp"
#include "model.hpp"
#include "surfsys.hpp"
#include "ohmiccurr.hpp"
#include "chanstate.hpp"

////////////////////////////////////////////////////////////////////////////////

USING_NAMESPACE(std);
USING_NAMESPACE(steps::model);

////////////////////////////////////////////////////////////////////////////////

OhmicCurr::OhmicCurr(string const & id, Surfsys * surfsys,
		  ChanState * chanstate, double erev, double g)
: pID(id)
, pModel(0)
, pSurfsys(surfsys)
, pChanState(chanstate)
, pERev(erev)
, pG(g)
{
    if (pSurfsys == 0)
    {
        ostringstream os;
        os << "No surfsys provided to OhmicCurr initializer function";
        throw steps::ArgErr(os.str());
    }
    if (pChanState == 0)
    {
        ostringstream os;
        os << "No channel state provided to OhmicCurr initializer function";
        throw steps::ArgErr(os.str());
    }
	if (pG < 0.0)
	{
		ostringstream os;
		os << "Channel conductance can't be negative";
		throw steps::ArgErr(os.str());
	}

	pModel = pSurfsys->getModel();
	assert (pModel != 0);

    pSurfsys->_handleOhmicCurrAdd(this);
}

////////////////////////////////////////////////////////////////////////////////

OhmicCurr::~OhmicCurr(void)
{
    if (pSurfsys == 0) return;
    _handleSelfDelete();
}

////////////////////////////////////////////////////////////////////////////////

void OhmicCurr::setID(string const & id)
{
    assert(pSurfsys != 0);
    // The following might raise an exception, e.g. if the new ID is not
    // valid or not unique. If this happens, we don't catch but simply let
    // it pass by into the Python layer.
    pSurfsys->_handleOhmicCurrIDChange(pID, id);
    // This line will only be executed if the previous call didn't raise
    // an exception.
    pID = id;
}

////////////////////////////////////////////////////////////////////////////////

void OhmicCurr::setChanState(ChanState * chanstate)
{
	assert(chanstate != 0);
	pChanState = chanstate;
}

////////////////////////////////////////////////////////////////////////////////

void OhmicCurr::setERev(double erev)
{
	assert (pSurfsys != 0);
    pERev = erev;
}

////////////////////////////////////////////////////////////////////////////////

void OhmicCurr::setG(double g)
{
	assert (pSurfsys != 0);
	if(g < 0.0)
	{
		ostringstream os;
		os << "Conductance provided to OhmicCurr::setG function can't be negative";
		throw steps::ArgErr(os.str());
	}
    pG = g;
}

////////////////////////////////////////////////////////////////////////////////

void OhmicCurr::_handleSelfDelete(void)
{
	pSurfsys->_handleOhmicCurrDel(this);
	pG = 0.0;
	pERev = 0;
	pSurfsys = 0;
	pModel = 0;
}

////////////////////////////////////////////////////////////////////////////////

// END

