use std::fmt;
use std::io::Cursor;

use failure::Error;
use insta;

use symbolic_common::ByteView;
use symbolic_debuginfo::Object;
use symbolic_symcache::{SymCache, SymCacheWriter};

/// Helper to create neat snapshots for symbol tables.
struct FunctionsDebug<'a>(&'a SymCache<'a>);

impl fmt::Debug for FunctionsDebug<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        for result in self.0.functions() {
            match result {
                Ok(function) => writeln!(f, "{:>16x} {}", &function.address(), &function.name())?,
                Err(error) => writeln!(f, "{:?}", error)?,
            }
        }

        Ok(())
    }
}

#[test]
fn test_write_header_linux() -> Result<(), Error> {
    let buffer = ByteView::open("../testutils/fixtures/linux/crash.debug")?;
    let object = Object::parse(&buffer)?;

    let mut buffer = Vec::new();
    SymCacheWriter::write_object(&object, Cursor::new(&mut buffer))?;
    let symcache = SymCache::parse(&buffer)?;
    insta::assert_debug_snapshot_matches!(symcache, @r###"
   ⋮SymCache {
   ⋮    debug_id: DebugId {
   ⋮        uuid: "c0bcc3f1-9827-fe65-3058-404b2831d9e6",
   ⋮        appendix: 0,
   ⋮    },
   ⋮    arch: Amd64,
   ⋮    has_line_info: true,
   ⋮    has_file_info: true,
   ⋮    functions: 1955,
   ⋮}
    "###);

    Ok(())
}

#[test]
fn test_write_functions_linux() -> Result<(), Error> {
    let buffer = ByteView::open("../testutils/fixtures/linux/crash.debug")?;
    let object = Object::parse(&buffer)?;

    let mut buffer = Vec::new();
    SymCacheWriter::write_object(&object, Cursor::new(&mut buffer))?;
    let symcache = SymCache::parse(&buffer)?;
    insta::assert_debug_snapshot_matches!("functions_linux", FunctionsDebug(&symcache));

    Ok(())
}

#[test]
fn test_write_header_macos() -> Result<(), Error> {
    let buffer =
        ByteView::open("../testutils/fixtures/macos/crash.dSYM/Contents/Resources/DWARF/crash")?;
    let object = Object::parse(&buffer)?;

    let mut buffer = Vec::new();
    SymCacheWriter::write_object(&object, Cursor::new(&mut buffer))?;
    let symcache = SymCache::parse(&buffer)?;
    insta::assert_debug_snapshot_matches!(symcache, @r###"
   ⋮SymCache {
   ⋮    debug_id: DebugId {
   ⋮        uuid: "67e9247c-814e-392b-a027-dbde6748fcbf",
   ⋮        appendix: 0,
   ⋮    },
   ⋮    arch: Amd64,
   ⋮    has_line_info: true,
   ⋮    has_file_info: true,
   ⋮    functions: 1863,
   ⋮}
    "###);

    Ok(())
}

#[test]
fn test_write_functions_macos() -> Result<(), Error> {
    let buffer =
        ByteView::open("../testutils/fixtures/macos/crash.dSYM/Contents/Resources/DWARF/crash")?;
    let object = Object::parse(&buffer)?;

    let mut buffer = Vec::new();
    SymCacheWriter::write_object(&object, Cursor::new(&mut buffer))?;
    let symcache = SymCache::parse(&buffer)?;
    insta::assert_debug_snapshot_matches!("functions_macos", FunctionsDebug(&symcache));

    Ok(())
}

#[test]
fn test_write_large_symbol_names() -> Result<(), Error> {
    let buffer = ByteView::open("../testutils/fixtures/regression/large_symbol.sym")?;
    let object = Object::parse(&buffer)?;

    let mut buffer = Vec::new();
    SymCacheWriter::write_object(&object, Cursor::new(&mut buffer))?;
    SymCache::parse(&buffer)?;

    Ok(())
}
