# Copyright (c) 2014, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.utils
from stix.utils import dates

import stix.bindings.exploit_target as exploit_target_binding
from stix.common.related import (GenericRelationshipList, RelatedCOA, 
                                RelatedExploitTarget, RelatedPackageRefs)
from stix.common import StructuredText, VocabString, InformationSource
from stix.coa import CourseOfAction
from stix.data_marking import Marking
from .vulnerability import Vulnerability
from .weakness import Weakness
from .configuration import Configuration

from datetime import datetime
from dateutil.tz import tzutc

class ExploitTarget(stix.Entity):
    _binding = exploit_target_binding
    _binding_class = _binding.ExploitTargetType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"
    _version = "1.1.1"

    def __init__(self, id_=None, idref=None, timestamp=None, title=None, description=None, short_description=None):
        self.id_ = id_ or stix.utils.create_id("et")
        self.idref = idref
        self.version = self._version
        self.title = title
        self.description = description
        self.short_description = short_description
        self.information_source = None
        self.handling = None
        self.potential_coas = PotentialCOAs()
        self.related_exploit_targets = RelatedExploitTargets()
        self.vulnerabilities = None
        self.weaknesses = None
        self.configuration = None
        self.related_packages = RelatedPackageRefs()
        
        if timestamp:
            self.timestamp = timestamp
        else:
            self.timestamp = datetime.now(tzutc()) if not idref else None
            
    @property
    def id_(self):
        return self._id
    
    @id_.setter
    def id_(self, value):
        if not value:
            self._id = None
        else:
            self._id = value
            self.idref = None
    
    @property
    def idref(self):
        return self._idref
    
    @idref.setter
    def idref(self, value):
        if not value:
            self._idref = None
        else:
            self._idref = value
            self.id_ = None # unset id_ if idref is present
    
    @property
    def timestamp(self):
        return self._timestamp

    @timestamp.setter
    def timestamp(self, value):
        self._timestamp = dates.parse_value(value)

    @property
    def title(self):
        return self._title

    @title.setter
    def title(self, value):
        self._title = value

    @property
    def description(self):
        return self._description

    @description.setter
    def description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._description = value
            else:
                self._description = StructuredText(value=value)
        else:
            self._description = None

    @property
    def short_description(self):
        return self._short_description

    @short_description.setter
    def short_description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._short_description = value
            else:
                self._short_description = StructuredText(value=value)
        else:
            self._short_description = None

    @property
    def information_source(self):
        return self._information_source
    
    @information_source.setter
    def information_source(self, value):
        if not value:
            self._information_source = None
        elif isinstance(value, InformationSource):
            self._information_source = value
        else:
            raise ValueError('value must be instance of InformationSource')

    @property
    def handling(self):
        return self._handling

    @handling.setter
    def handling(self, value):
        if value and not isinstance(value, Marking):
            raise ValueError('value must be instance of Marking')

        self._handling = value
 
    @property
    def vulnerabilities(self):
        return self._vulnerabilities
    
    @vulnerabilities.setter
    def vulnerabilities(self, value):
        self._vulnerabilities = []
        if not value:
            return
        elif isinstance(value, list):
            for v in value:
                self.add_vulnerability(v)
        else:
            self.add_vulnerability(value)
            
    def add_vulnerability(self, v):
        if not v:
            return
        elif isinstance(v, Vulnerability):
            self.vulnerabilities.append(v)
        else:
            raise ValueError('Cannot add type %s to vulnerabilities list' % type(v))
 
    @property
    def weaknesses(self):
        return self._weaknesses
    
    @weaknesses.setter
    def weaknesses(self, value):
        self._weaknesses = []
        if not value:
            return
        elif isinstance(value, list):
            for v in value:
                self.add_weakness(v)
        else:
            self.add_weakness(value)
            
    def add_weakness(self, v):
        if not v:
            return
        elif isinstance(v, Weakness):
            self.weaknesses.append(v)
        else:
            raise ValueError('Cannot add type %s to weakness list' % type(v))
 
    @property
    def configuration(self):
        return self._configuration
    
    @configuration.setter
    def configuration(self, value):
        self._configuration = []
        if not value:
            return
        elif isinstance(value, list):
            for v in value:
                self.add_configuration(v)
        else:
            self.add_configuration(value)
            
    def add_configuration(self, v):
        if not v:
            return
        elif isinstance(v, Configuration):
            self.configuration.append(v)
        else:
            raise ValueError('Cannot add type %s to configuration list' % type(v))
 
    def to_obj(self, return_obj=None):
        if not return_obj:
            return_obj = self._binding_class()

        return_obj.set_id(self.id_)
        return_obj.set_idref(self.idref)
        return_obj.set_timestamp(dates.serialize_value(self.timestamp))
        return_obj.set_version(self.version)
        return_obj.set_Title(self.title)

        if self.description:
            return_obj.set_Description(self.description.to_obj())
        if self.short_description:
            return_obj.set_Short_Description(self.short_description.to_obj())
        if self.information_source:
            return_obj.set_Information_Source(self.information_source.to_obj())
        if self.handling:
            return_obj.set_Handling(self.handling.to_obj())
        if self.potential_coas:
            return_obj.set_Potential_COAs(self.potential_coas.to_obj())
        if self.related_exploit_targets:
            return_obj.set_Related_Exploit_Targets(self.related_exploit_targets.to_obj())
        if self.vulnerabilities:
            return_obj.set_Vulnerability([x.to_obj() for x in self.vulnerabilities])
        if self.weaknesses:
            return_obj.set_Weakness([x.to_obj() for x in self.weaknesses])
        if self.configuration:
            return_obj.set_Configuration([x.to_obj() for x in self.configuration])
        if self.related_packages:
            return_obj.set_Related_Packages(self.related_packages.to_obj())
            
        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.id_ = obj.get_id()
        return_obj.idref = obj.get_idref()
        return_obj.timestamp = obj.get_timestamp() # not yet implemented

        if isinstance(obj, cls._binding_class): # TTPType properties
            return_obj.version = obj.get_version() or cls._version
            return_obj.title = obj.get_Title()
            return_obj.description = StructuredText.from_obj(obj.get_Description())
            return_obj.short_description = StructuredText.from_obj(obj.get_Short_Description())
            return_obj.information_source = InformationSource.from_obj(obj.get_Information_Source())
            return_obj.handling = Marking.from_obj(obj.get_Handling())
            return_obj.potential_coas = PotentialCOAs.from_obj(obj.get_Potential_COAs())
            return_obj.related_exploit_targets = RelatedExploitTargets.from_obj(obj.get_Related_Exploit_Targets())
            return_obj.vulnerabilities = [Vulnerability.from_obj(x) for x in obj.get_Vulnerability()]
            return_obj.weaknesses = [Weakness.from_obj(x) for x in obj.get_Weakness()]
            return_obj.configuration = [Configuration.from_obj(x) for x in obj.get_Configuration()]
            return_obj.related_packages = RelatedPackageRefs.from_obj(obj.get_Related_Packages())

        return return_obj

    def to_dict(self):
        d = {}
        if self.id_:
            d['id'] = self.id_
        if self.idref:
            d['idref'] = self.idref
        if self.timestamp:
            d['timestamp'] = dates.serialize_value(self.timestamp)
        if self.version:
            d['version'] = self.version or self._version
        if self.title:
            d['title'] = self.title
        if self.description:
            d['description'] = self.description.to_dict()
        if self.short_description:
            d['short_description'] = self.short_description.to_dict()
        if self.information_source:
            d['information_source'] = self.information_source.to_dict()
        if self.handling:
            d['handling'] = self.handling.to_dict()
        if self.potential_coas:
            d['potential_coas'] = self.potential_coas.to_dict()
        if self.related_exploit_targets:
            d['related_exploit_targets'] = self.related_exploit_targets.to_dict()
        if self.vulnerabilities:
            d['vulnerabilities'] = [x.to_dict() for x in self.vulnerabilities]
        if self.weaknesses:
            d['weaknesses'] = [x.to_dict() for x in self.weaknesses]
        if self.configuration:
            d['configuration'] = [x.to_dict() for x in self.configuration]
        if self.related_packages:
            d['related_packages'] = self.related_packages.to_dict()

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.id_ = dict_repr.get('id')
        return_obj.idref = dict_repr.get('idref')
        return_obj.timestamp = dict_repr.get('timestamp')
        return_obj.version = dict_repr.get('version', cls._version)
        return_obj.title = dict_repr.get('title')
        return_obj.description = StructuredText.from_dict(dict_repr.get('description'))
        return_obj.short_description = StructuredText.from_dict(dict_repr.get('short_description'))
        return_obj.information_source = InformationSource.from_dict(dict_repr.get('information_source'))
        return_obj.handling = Marking.from_dict(dict_repr.get('handling'))
        return_obj.potential_coas = PotentialCOAs.from_dict(dict_repr.get('potential_coas'))
        return_obj.related_exploit_targets = RelatedExploitTargets.from_dict(dict_repr.get('related_exploit_targets'))
        return_obj.vulnerabilities = [Vulnerability.from_dict(x) for x in dict_repr.get('vulnerabilities', [])]
        return_obj.weaknesses = [Weakness.from_dict(x) for x in dict_repr.get('weaknesses', [])]
        return_obj.configuration = [Configuration.from_dict(x) for x in dict_repr.get('configuration', [])]
        return_obj.related_packages = RelatedPackageRefs.from_dict(dict_repr.get('related_packages'))
        
        return return_obj

class PotentialCOAs(GenericRelationshipList):
    _namespace = "http://stix.mitre.org/ExploitTarget-1"
    _binding = exploit_target_binding
    _binding_class = exploit_target_binding.PotentialCOAsType
    _binding_var = "Potential_COA"
    _contained_type = RelatedCOA
    _inner_name = "coas"

    def __init__(self, coas=None, scope=None):
        if coas is None:
            coas = []
        super(PotentialCOAs, self).__init__(*coas, scope=scope)

class RelatedExploitTargets(GenericRelationshipList):
    _namespace = "http://stix.mitre.org/ExploitTarget-1"
    _binding = exploit_target_binding
    _binding_class = exploit_target_binding.RelatedExploitTargetsType
    _binding_var = "Related_Exploit_Target"
    _contained_type = RelatedExploitTarget
    _inner_name = "related_exploit_targets"

    def __init__(self, related_exploit_targets=None, scope=None):
        if related_exploit_targets is None:
            related_exploit_targets = []
        super(RelatedExploitTargets, self).__init__(*related_exploit_targets, scope=scope)
