#!/usr/bin/env python
# -*- coding: ascii -*-

r"""
Generate FORTRAN code to perform the curve fit calculation from XYmath
"""
from __future__ import print_function
from __future__ import absolute_import

import time
from .eqn_parse import get_const_list
from .helper_funcs import fortran_doubleStr
from .newtGreg2 import quadNG

def make_line_legal(srcLine, sepChar=' '):
    '''Assume that line is NOT numbered and that sepChar separates elements'''
    
    sL = srcLine.strip().split(sepChar)
    srcList = []
    
    src = '      '
    for s in sL:
        
        if len(src) + len(s) < 72:
            src += s + sepChar
        else:
            srcList.append( src )
            src = '     $ ' + s + sepChar
    
    src = src[:-1] # get rid of trailing sepChar
    srcList.append( src )
    return '\n'.join( srcList ) + '\n'


def make_fit_func_src( eqnObj ):
    '''Make FORTRAN source code for LinCurveFit, NonLinCurveFit and Spline objects.'''

    # make comment string for any type of eqnObj
    full_desc = eqnObj.get_full_description()
    sL = ['C  '+s for s in full_desc.split('\n')]
    sL.insert(0,'C  Curve Fit Results from XYmath '+ time.strftime('%m/%d/%Y') )
    

        
    # build test evaluation
    if eqnObj.ds:
        x_test = eqnObj.ds.xArr[-1]
        data_comment = eqnObj.ds.get_data_pair_comment_lines(com_char='C', len_max=80, sepChar=',')
        xmin = fortran_doubleStr(eqnObj.ds.xmin)
        xmax = fortran_doubleStr(eqnObj.ds.xmax)
    else:
        x_test = 1.0
        data_comment = ''
        xmin = '0.0D0'
        xmax = '0.0D0'


    if hasattr(eqnObj, 'get_eqn_str_w_numbs'):
        # Either LinCurveFit or NonLinCurveFit will work here
        
        # get any imports required from numpy module
        eqnStr = make_line_legal( eqnObj.get_fortran_eqn_str_w_numbs() )
        
        # setup string substitution dictionary
        substD = {'func_name':'curve_fit_sub', 'comment':'\n'.join(sL), 
                'eqnStr':eqnStr, 'x_test':fortran_doubleStr(x_test), 'y_xymath':fortran_doubleStr(eqnObj.eval_xrange(x_test)),
                'x_desc':eqnObj.ds.get_x_desc(), 'y_desc':eqnObj.ds.get_y_desc(), 'data_comment':data_comment,
                'xmin':xmin, 'xmax':xmax}
            
        return func_src_template%substD
    
    else: # spline here
        Ndim = 100
        Nminus2 = Ndim - 2
        xSpline, ySpline = eqnObj.get_xy_plot_arrays( Npoints=Ndim, logScale=0)
        quadTerp = quadNG(xSpline, ySpline)
        # setup string substitution dictionary
        xL = [fortran_doubleStr('%s'%x) for x in xSpline]
        aL = [fortran_doubleStr('%s'%a) for a in quadTerp.a]
        bL = [fortran_doubleStr('%s'%b) for b in quadTerp.b]
        cL = [fortran_doubleStr('%s'%c) for c in quadTerp.c]
        
        xArr = make_line_legal( '      DATA xArr/'+ ','.join(xL)+ '/', sepChar=',')
        aArr = make_line_legal( '      DATA aArr/'+ ','.join(aL)+ '/', sepChar=',')
        bArr = make_line_legal( '      DATA bArr/'+ ','.join(bL)+ '/', sepChar=',')
        cArr = make_line_legal( '      DATA cArr/'+ ','.join(cL)+ '/', sepChar=',')
        
        substD = {'func_name':'curve_fit_sub', 'comment':'\n'.join(sL), 'order':eqnObj.order,
                'x_test':fortran_doubleStr(x_test), 'y_xymath':eqnObj.eval_xrange(x_test), 'Ndim':Ndim, 'xArr':xArr,
                'aArr':aArr,  'bArr':bArr, 'cArr':cArr, 'Nminus2':Nminus2, 
                'yMax':fortran_doubleStr(ySpline[-1]), 'yMin':fortran_doubleStr(ySpline[0]),
                'x_desc':eqnObj.ds.get_x_desc(), 'y_desc':eqnObj.ds.get_y_desc(), 
                'data_comment':data_comment, 'xmin':xmin, 'xmax':xmax}
            
        return spline_template%substD
#==========================================================================
spline_template = """
C This FORTRAN Source Code Generated by XYmath

      SUBROUTINE %(func_name)s( x, y )
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      implicit integer (I-N)
      PARAMETER (N=%(Ndim)s, M=%(Nminus2)s)
      PARAMETER (yMin=%(yMin)s, yMax=%(yMax)s)
      DOUBLE PRECISION xArr(N), aArr(M), bArr(M), cArr(M)
      DOUBLE PRECISION x, y, dx, dx2
      INTEGER i
      
C  set up f2py comments just in case a python import is desired
cf2py        DOUBLE PRECISION,INTENT(IN)::  x      ! %(x_desc)s
cf2py        DOUBLE PRECISION,INTENT(OUT):: y      ! %(y_desc)s

C Return y value of spline curve fit performed by XYmath for y=f(x).
C If x is outside the original dataset x range, a warning will be printed
C and a limit value will be returned, either y=f(xmin) or y=f(xmax).

%(comment)s

%(data_comment)s

C xArr holds x values across dataset range
%(xArr)s

C aArr, bArr and cArr are quadratic Newton-Gregory Interpolation coefficients
%(aArr)s
%(bArr)s
%(cArr)s

C  If input value of x is out of range, print warning
      IF (x.lt.%(xmin)s .or. x.gt.%(xmax)s)THEN
        print*, 'WARNING... x is outside range in %(func_name)s'
        print*, '  x =',x,' x range = (%(xmin)s to %(xmax)s)'
      ENDIF

C  Find index into xArr to evaluate quadratic fit of original spline
      CALL find_index(xArr,N,x,i)
      
      if (i .le. 0)then
        print*,'WARNING... Index too low in %(func_name)s'
        y = yMin
      elseif (i .ge. N)then
        print*,'WARNING... Index too high in %(func_name)s'
        y = yMax
      else
        i = min(i, M)
        dx = x - xArr(i)
        dx2 = x-xArr(i+1)
        y = aArr(i) + dx*( bArr(i) + dx2*cArr(i) ) 
      endif

      RETURN
      END
C
C
      SUBROUTINE find_index(xArr,N,xVal,iSpot)
      INTEGER iSpot,N
      DOUBLE PRECISION xVal,xArr(N)
      
C        For array xArr, and value xVal, this subroutine finds iSpot
C        such that xVal is in range xArr(iSpot) to xArr(iSpot+1)
C
C        xArr must be monotonically increasing .
C        If xVal is out of range, iSpot=0 or iSpot=N is returned.
      INTEGER iLo,iMid,iHi
      iLo=0
      iHi=N+1
100   if(iHi-iLo.gt.1)then
        iMid=(iHi+iLo)/2
        if( xVal.ge.xArr(iMid) )then
          iLo=iMid
        else
          iHi=iMid
        endif
        goto 100
      endif
      if(xVal.eq.xArr(1))then
        iSpot=1
      elseif(xVal.eq.xArr(N))then
        iSpot=N-1
      else
        iSpot=iLo
      endif
      return
      END
C
      PROGRAM TEST
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      implicit integer (I-N)

      print*, '============================'
      call %(func_name)s( %(x_test)s, y_test )
      print*, 'y_test  =',y_test,'for x_test =',%(x_test)s
      print*, 'y_xymath=',%(y_xymath)s
      print*, ' '
      print*, 'y_test should equal y_xymath above.'
      STOP
      END      
"""
# ======================================================================
func_src_template = """
C This FORTRAN Source Code Generated by XYmath

      SUBROUTINE %(func_name)s( x, y )
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      implicit integer (I-N)
      DOUBLE PRECISION x, y
      
C  set up f2py comments just in case a python import is desired
cf2py        DOUBLE PRECISION,INTENT(IN)::  x      ! %(x_desc)s
cf2py        DOUBLE PRECISION,INTENT(OUT):: y      ! %(y_desc)s

%(comment)s    

%(data_comment)s

C  If input value of x is out of range, print warning
      IF (x.lt.%(xmin)s .or. x.gt.%(xmax)s)THEN
        print*, 'WARNING... x is outside range in %(func_name)s'
        print*, '  x =',x,' x range = (%(xmin)s to %(xmax)s)'
      ENDIF
    
%(eqnStr)s
      RETURN
      END
      PROGRAM TEST
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      implicit integer (I-N)

      print*, '============================'
      call %(func_name)s( %(x_test)s, y_test )
      print*, 'y_test  =',y_test,'for x_test =',%(x_test)s
      print*, 'y_xymath=',%(y_xymath)s
      print*, ' '
      print*, 'y_test should equal y_xymath above.'
      STOP
      END      
"""

if __name__=='__main__':
    from numpy import array, double
    from .dataset import DataSet
    from .linfit import LinCurveFit
    from .nonlinfit import NonLinCurveFit
    from .splines import Spline
    
    xArr = array( [1,2,3,4,5,6], dtype=double)
    yArr = array( [1.2,3.1,9.2,15.8,24.6,36.5], dtype=double)
    
    DS = DataSet(xArr, yArr, xName='fiddle', yName='faddle')
    lf = LinCurveFit( DS, ['const', 'x', 'x**2'])
    
    src = make_fit_func_src( lf )
    print(src)
    print('='*44)
    
    lf = LinCurveFit( DS, ['const', 'x'], ytran='log(y)')
    src = make_fit_func_src( lf )
    print(src)
    
    print('='*44)
    nlf = NonLinCurveFit(DS, rhs_eqnStr='A * exp(c*x/pi)')
    src = make_fit_func_src( nlf )
    print(src)
        
    print('='*44)
    nlf = NonLinCurveFit(DS, rhs_eqnStr='A * pi*x + c')
    src = make_fit_func_src( nlf )
    print(src)
        
    print('='*44, ' Spline')
    sp = Spline(DS, order=2, smoothing=0.0)
    src = make_fit_func_src( sp )
    print(src)
    