use thiserror::Error;

/// Errors returned while loading/parsing a serialized SymCache.
///
/// After a SymCache was successfully parsed via [`SymCache::parse`](crate::SymCache::parse), an Error that occurs during
/// access of any data indicates either corruption of the serialized file, or a bug in the
/// converter/serializer.
#[derive(Debug, Error, Clone, Copy, PartialEq, Eq)]
#[non_exhaustive]
pub enum ErrorKind {
    /// The buffer is not correctly aligned.
    ///
    /// This variant is currently unused.
    #[error("source buffer is not correctly aligned")]
    BufferNotAligned,
    /// The header's size doesn't match our expected size.
    ///
    /// This variant is currently unused.
    #[error("header is too small")]
    HeaderTooSmall,
    /// The file was generated by a system with different endianness.
    #[error("endianness mismatch")]
    WrongEndianness,
    /// The file magic does not match.
    #[error("wrong format magic")]
    WrongFormat,
    /// The format version in the header is wrong/unknown.
    #[error("unknown SymCache version")]
    WrongVersion,
    /// The self-advertised size of the buffer is not correct.
    ///
    /// This variant is currently unused.
    #[error("incorrect buffer length")]
    BadFormatLength,
    /// The debug file could not be converted to a symcache.
    #[error("bad debug file")]
    BadDebugFile,
    /// Header could not be parsed from the cache file.
    #[error("could not read header")]
    InvalidHeader,
    /// File data could not be parsed from the cache file.
    #[error("could not read files")]
    InvalidFiles,
    /// Function data could not be parsed from the cache file.
    #[error("could not read functions")]
    InvalidFunctions,
    /// Source location data could not be parsed from the cache file.
    #[error("could not read source locations")]
    InvalidSourceLocations,
    /// Range data could not be parsed from the cache file.
    #[error("could not read ranges")]
    InvalidRanges,
    /// The header claimed an incorrect number of string bytes.
    #[error("expected {expected} string bytes, found {found}")]
    UnexpectedStringBytes {
        /// Expected number of string bytes.
        expected: usize,
        /// Number of string bytes actually found in the cache file.
        found: usize,
    },
}

/// An error returned when handling a [`SymCache`](crate::SymCache).
#[derive(Debug, Error)]
#[error("{kind}")]
pub struct Error {
    pub(crate) kind: ErrorKind,
    #[source]
    pub(crate) source: Option<Box<dyn std::error::Error + Send + Sync + 'static>>,
}

impl Error {
    /// Creates a new SymCache error from a known kind of error as well as an
    /// arbitrary error payload.
    pub(crate) fn new<E>(kind: ErrorKind, source: E) -> Self
    where
        E: Into<Box<dyn std::error::Error + Send + Sync>>,
    {
        let source = Some(source.into());
        Self { kind, source }
    }

    /// Returns the corresponding [`ErrorKind`] for this error.
    pub fn kind(&self) -> ErrorKind {
        self.kind
    }
}

impl From<ErrorKind> for Error {
    fn from(kind: ErrorKind) -> Self {
        Self { kind, source: None }
    }
}
