/* Copyright 2017 - 2022 R. Thomas
 * Copyright 2017 - 2022 Quarkslab
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "pyPE.hpp"
#include "pyIterators.hpp"

#include "LIEF/PE/hash.hpp"
#include "LIEF/PE/RichHeader.hpp"

#include <string>
#include <sstream>

namespace LIEF {
namespace PE {

template<class T>
using getter_t = T (RichHeader::*)(void) const;

template<class T>
using setter_t = void (RichHeader::*)(T);

template<class T>
using no_const_getter = T (RichHeader::*)(void);


template<>
void create<RichHeader>(py::module& m) {
  py::class_<RichHeader> rich(m, "RichHeader",
      R"delim(
      Class which represents the not-so-documented rich header

      This structure is usually located at the end of the :attr:`~lief.PE.Binary.dos_stub`
      and contains information about the build environment.

      It is generated by the Microsoft linker `link.exe` and there are no options to disable
      or remove this information.
      )delim");

  init_ref_iterator<RichHeader::it_entries>(rich, "it_entries");

  rich
    .def(py::init<>())
    .def_property("key",
        static_cast<getter_t<uint32_t>>(&RichHeader::key),
        static_cast<setter_t<uint32_t>>(&RichHeader::key),
        "Key used to encode the header (xor operation)")

    .def_property_readonly("entries",
        static_cast<no_const_getter<RichHeader::it_entries>>(&RichHeader::entries),
        "Return an iterator over the " RST_CLASS_REF(lief.PE.RichEntry) " within the header",
        py::return_value_policy::reference)

    .def("add_entry",
        static_cast<void (RichHeader::*)(const RichEntry&)>(&RichHeader::add_entry),
        "Add a new " RST_CLASS_REF(lief.PE.RichEntry) "",
        "entry"_a)

    .def("add_entry",
        static_cast<void (RichHeader::*)(uint16_t, uint16_t, uint32_t)>(&RichHeader::add_entry),
        "Add a new " RST_CLASS_REF(lief.PE.RichEntry) " given its "
        ":attr:`~lief.PE.RichEntry.id`, "
        ":attr:`~lief.PE.RichEntry.build_id`, "
        ":attr:`~lief.PE.RichEntry.count`",
        "id"_a, "build_id"_a, "count"_a)

    .def("raw",
        py::overload_cast<>(&RichHeader::raw, py::const_),
        R"delim(
        The raw structure of the Rich header without xor-encoding.

        This function is equivalent as calling the other raw function with a `xor_key` set to 0
        )delim")

    .def("raw",
        py::overload_cast<uint32_t>(&RichHeader::raw, py::const_),
        R"delim(
        Given this rich header, this function re-computes
        the raw bytes of the structure with the provided xor-key.

        You can access the decoded data's structure with the `xor_key` set to 0
        )delim",
        "xor_key"_a)

    .def("hash",
        py::overload_cast<ALGORITHMS>(&RichHeader::hash, py::const_),
        R"delim(
        Compute the hash of the decoded rich header structure with the given hash :class:`~lief.PE.ALGORITHMS`
        )delim",
        "algo"_a)

    .def("hash",
        py::overload_cast<ALGORITHMS, uint32_t>(&RichHeader::hash, py::const_),
        R"delim(
        Compute the hash of the rich header structure encoded with the provided key and the given hash
        :class:`~lief.PE.ALGORITHMS`
        )delim",
        "algo"_a, "xor_key"_a)

    .def("__eq__", &RichHeader::operator==)
    .def("__ne__", &RichHeader::operator!=)
    .def("__hash__",
        [] (const RichHeader& rich_header) {
          return Hash::hash(rich_header);
        })

    .def("__str__", [] (const RichHeader& rich_header)
        {
          std::ostringstream stream;
          stream << rich_header;
          std::string str = stream.str();
          return str;
        });
}

}
}
