# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
from weakref import ref as wr

__all__ = ['WrView']

class WrView(object):
    """
    A view of objects weak-refed by elements of a sequence.

    >>> class A(): pass
    ...
    >>> a = A()
    >>> b = A()
    >>> from weakref import ref as wr
    >>> v = WrView([wr(a), wr(b)])
    >>> list(v) == [a, b]
    True
    >>> b in v
    True
    >>> A() in v
    False
    >>> len(v)
    2
    
    """
    ___slots__ = ['_internal_sequence',]
    def __init__(self, internal_sequence):
        self._internal_sequence = internal_sequence

    def __len__(self):
        return len(self._internal_sequence)

    def __iter__(self):
        for x_wr in self._internal_sequence:
            yield x_wr()

    def __contains__(self, x):
        return wr(x) in self._internal_sequence
