# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Simple, but powerfull, shell-like testing.

A test file consists of:
- non-test lines - any line that does not start with 2 spaces.
- test lines - all lines that start with 2 spaces.

A single test is a single shell command. It can have multiple lines.
After each shell command, the expected output is to be specified.
If the shell command is expected to return a non-0 exit value,
the correct one is to be specified.

>>> from cStringIO import StringIO
>>> execute(StringIO('''
... My first comment
...   $ echo Hello
...   Hello
...   $ foo() {
...   >   echo bar
...   > }
...   $ foo
...   bar
...   $ cat <<EOF | grep blah
...   > abfddf
...   > bfblahdf
...   > skfablaha
...   > blk
...   > blah
...   > EOF
...   bfblahdf
...   skfablaha
...   blah
...   $ (echo bar; exit 5)
...   ? 5
...   bar
... '''))
>>> try:
...     execute(StringIO('  > unexpected\\n  ? 5\\n  unexpected'))
... except CompilationFailed, e:
...     for test, line in e.args[0]:
...         print '%d:%s' % (line, test)
0:No command to continue
1:No command to check exit code
2:No command to check output
>>> try:
...     execute(StringIO('  $ foo() {\\n  bar\\n  > }\\n  ? 0'))
... except CompilationFailed, e:
...     for text, line in e.args[0]:
...         print '%d:%s' % (line, text)
2:Extra command line was found after an expected output line
3:Exit code was found after an expected output line
>>> try:
...     execute(StringIO('  $ foo() {\\n  ? 0\\n  > echo bar }'))
... except CompilationFailed, e:
...     for text, line in e.args[0]:
...         print '%d:%s' % (line, text)
2:Extra command line was found after expected exit code
>>> try:
...     execute(StringIO('  $ (exit 5)\\n  ? 4\\n  ? 3'))
... except CompilationFailed, e:
...     for text, line in e.args[0]:
...         print '%d:%s' % (line, text)
2:Exit code redifinition found
>>> try:
...     execute(StringIO('  $ (exit 5)\\n  ? \\n  ? 5f'))
... except CompilationFailed, e:
...     for test, line in e.args[0]:
...         print '%d:%s' % (line, test)
1:Exit code is not an integer
2:Exit code is not an integer
>>> try:
...    execute(StringIO('  $ echo bar\\n  ? 5\\n  foo'))
... except TestFailed, e:
...    e.args[0].write_full_diff()
 **** Command 0 ****
 *** echo bar ***
-bar
-Exit Value: 0
+foo
+Exit Value: 5
>>> execute_re(StringIO('  $ echo Hello World!\\n  H.*d!'))

"""

__all__ = ['execute', 'execute_re','CompilationFailed', 'TestFailed',]

import sys, subprocess, operator, re

from bfc.diff import Diff, nop

class TstException(Exception): pass

class CompilationFailed(TstException): pass

class TestFailed(TstException): pass

class TestCommand(object):
    __slots__ = ['command',
                 'expected_result',
                 'expected_exit_code',
                 ]
    def __init__(self, command):
        self.command = [command]
        self.expected_result = []
        self.expected_exit_code = None

def _handle_new_command(tests, errors, index, tst_line):
    tests.append(TestCommand(tst_line[2:].rstrip()))

def _handle_extra_line(tests, errors, index, tst_line):
    if not tests:
        errors.append(('No command to continue', index))
        return
    prev_command = tests[-1]
    if prev_command.expected_result:
        errors.append(('Extra command line was found after an expected output line', index))
        return
    if prev_command.expected_exit_code is not None:
        errors.append(('Extra command line was found after expected exit code', index))
        return
    prev_command.command.append(tst_line[2:].rstrip())
    
def _handle_expected_exit_code(tests, errors, index, tst_line):
    if not tests:
        errors.append(('No command to check exit code', index))
        return
    prev_command = tests[-1]
    if prev_command.expected_result:
        errors.append(('Exit code was found after an expected output line', index))
        return
    if prev_command.expected_exit_code is not None:
        errors.append(('Exit code redifinition found', index))
        return
    try:
        prev_command.expected_exit_code = int(tst_line[2:].rstrip())
    except ValueError:
        errors.append(('Exit code is not an integer', index))

def _handle_expected_output(tests, errors, index, tst_line):
    if not tests:
        errors.append(('No command to check output', index))
        return
    prev_command = tests[-1]
    prev_command.expected_result.append(tst_line.rstrip())

_prefix_to_handler = {
    '$ ' : _handle_new_command,
    '> ' : _handle_extra_line,
    '? ' : _handle_expected_exit_code,
    }

def _parse_tst(fileobj):
    tests = []
    errors = []

    for index, line in enumerate(fileobj):
        if line[:2] != '  ':
            continue
        tst_line = line[2:]
        handler = _prefix_to_handler.get(tst_line[:2], _handle_expected_output)
        handler(tests, errors, index, tst_line)

    return tests, errors

def shell_escape(text):
    for c in "\\'\"` *?{}[]()|<>=;&":
        text = text.replace(c, "\\" + c)
    return text

def write_echo(shell_script_lines, expected_output_lines, text, plain_text):
    shell_script_lines.append('echo %s' % (shell_escape(text),))
    expected_output_lines.append(plain_text(text))

def execute(fileobj, diff_eq = operator.eq, plain_text = nop, shell = 'bash'):
    tests, errors = _parse_tst(fileobj)
    if errors:
        raise CompilationFailed(errors)
    
    shell_script_lines = []
    expected_output_lines = []

    for index, test in enumerate(tests):

        write_echo(shell_script_lines, expected_output_lines, '**** Command %d ****' % (index,), plain_text)
        for command in test.command:
            write_echo(shell_script_lines, expected_output_lines, '*** %s ***' % (command,), plain_text)

        shell_script_lines.extend(test.command)
        expected_output_lines.extend(test.expected_result)

        shell_script_lines.append('echo Exit Value: $?')
        exit_value = test.expected_exit_code if test.expected_exit_code is not None else 0
        expected_output_lines.append(plain_text('Exit Value: %d' % (exit_value,)))

    p = subprocess.Popen([shell],
                         stdin = subprocess.PIPE,
                         stdout = subprocess.PIPE)
    out, err = p.communicate('\n'.join(shell_script_lines))
    out_lines = out.split('\n')[:-1]

    d = Diff(out_lines, expected_output_lines, diff_eq)
    if not d.are_same:
        raise TestFailed(d)

def re_eq(output, expected):
    return re.match(expected, output)

def execute_re(fileobj, shell = 'bash'):
    execute(fileobj, re_eq, re.escape, shell = shell)

if __name__ == '__main__': # pragma: no cover
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option('-r', dest='re', help='treat expected output as Python regexp', action = 'store_true')
    options, args = parser.parse_args()
    if not args:
        parser.print_help()
    executer = execute_re if options.re else execute
    for fn in args:
        try:
            executer(open(fn, 'rb'))
        except CompilationFailed, e:
            for test, line in e.args[0]:
                print '%s:%d:%s' % (fn, line, test)
        except TestFailed, e:
            print '%s:failed' % (fn,)
            e.args[0].write_full_diff()
