# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""

Introducing a picklable partial.

Note: in Python 3.1 and up, this module won't be needed

>>> p = PicklablePartial(int, base=16)
>>> p
PicklablePartial(<type 'int'>, base = 16)
>>> p('0x10')
16
>>> p.func
<type 'int'>
>>> p.args
()
>>> p.keywords
{'base': 16}
>>> import cPickle as pickle
>>> s = pickle.dumps(p, -1)
>>> del p
>>> p = pickle.loads(s)
>>> p('0x10')
16
>>> p.func
<type 'int'>
>>> p.args
()
>>> p.keywords
{'base': 16}

"""

from functools import partial

class PicklablePartial(object):
    __slots__ = ['_partial','_func', '_args', '_kws']

    def __init__(self, func, *args, **kws):
        self._init(func, *args, **kws)

    def _init(self, func, *args, **kws):
        self._func = func
        self._args = args
        self._kws = kws
        self._partial = partial(func, *args, **kws)

    def __call__(self, *args, **kws):
        return self._partial(*args, **kws)

    def __repr__(self):
        args = [repr(self._func)] + [repr(a) for a in self._args] + ['%s = %r' % (k, v) for k, v in self._kws.iteritems()]
        return '%s(%s)' % (self.__class__.__name__, ', '.join(args))

    def __getstate__(self):
        return {'f':self._func, 'a':self._args, 'k':self._kws}

    def __setstate__(self, s):
        self._init(s['f'], *s['a'], **s['k'])

    # functools.partial like properties
    @property
    def func(self): return self._func

    @property
    def args(self): return self._args

    @property
    def keywords(self): return self._kws
