# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""

>>> d = FrozenOrderedDict([(1,2),(3,4), ('foo', 'bar')])
>>> d
FrozenOrderedDict({1:2, 3:4, 'foo':'bar'})
>>> d[1]
2
>>> d[4]
Traceback (most recent call last):
    ...
KeyError: 4
>>> len(d)
3
>>> list(d)
[1, 3, 'foo']
>>> d.keys()
[1, 3, 'foo']
>>> d.values()
[2, 4, 'bar']
>>> d.items()
[(1, 2), (3, 4), ('foo', 'bar')]
>>> 1 in d
True
>>> 2 in d
False
>>> d.get('foo')
'bar'
>>> d.get('bar')
>>> d.get('bar',5)
5
>>> cmp(d, FrozenOrderedDict([(1,2)]))
1
>>> cmp(d, FrozenOrderedDict([(1,2),(3,4),('foo','barx')]))
-1
>>> cmp(d, FrozenOrderedDict([(1,2),(3,4),('foo','baa')]))
1
>>> d.copy()
FrozenOrderedDict({1:2, 3:4, 'foo':'bar'})
>>> FrozenOrderedDict.from_keys((1,2,3),'bar')
FrozenOrderedDict({1:'bar', 2:'bar', 3:'bar'})
>>> d = OrderedDict([(1,2),(3,4),('foo','bar')])
>>> d
OrderedDict({1:2, 3:4, 'foo':'bar'})
>>> d[1]
2
>>> d[4]
Traceback (most recent call last):
    ...
KeyError: 4
>>> len(d)
3
>>> list(d)
[1, 3, 'foo']
>>> d.keys()
[1, 3, 'foo']
>>> d.values()
[2, 4, 'bar']
>>> d.items()
[(1, 2), (3, 4), ('foo', 'bar')]
>>> 1 in d
True
>>> 2 in d
False
>>> d.get('foo')
'bar'
>>> d.get('bar')
>>> d.get('bar',5)
5
>>> cmp(d, OrderedDict([(1,2)]))
1
>>> cmp(d, OrderedDict([(1,2),(3,4),('foo','barx')]))
-1
>>> cmp(d, OrderedDict([(1,2),(3,4),('foo','baa')]))
1
>>> d.copy()
OrderedDict({1:2, 3:4, 'foo':'bar'})
>>> OrderedDict.from_keys((1,2,3),'bar')
OrderedDict({1:'bar', 2:'bar', 3:'bar'})
>>> d[3] = 5
>>> d
OrderedDict({1:2, 3:5, 'foo':'bar'})
>>> d.update([(1,'b')])
>>> d.update(xip='a')
>>> d
OrderedDict({1:'b', 3:5, 'foo':'bar', 'xip':'a'})
>>> del d['foo']
>>> del d[2]
Traceback (most recent call last):
    ...
KeyError: 2
>>> d.pop(1, 2, 3)
Traceback (most recent call last):
    ...
TypeError: pop expected at most 2 arguments
>>> d.pop(1)
'b'
>>> d.pop(2)
Traceback (most recent call last):
    ...
KeyError: 2
>>> d.pop(2, 5)
5
>>> d.popitem()
('xip', 'a')
>>> d.clear()
>>> d
OrderedDict({})
>>> d.popitem()
Traceback (most recent call last):
    ...
KeyError: 'popitem(): OrderedDict is empty'
>>> d.setdefault(5)
>>> d
OrderedDict({5:None})
>>> d.setdefault(5, 6)
>>> d
OrderedDict({5:None})
>>> d.setdefault(6, 6)
6
>>> d
OrderedDict({5:None, 6:6})
>>> class Foo(object):
...     def __iter__(self):
...         return iter((1,2))
...     def __getitem__(self, key):
...         return key * 10
>>> from collections import Mapping
>>> Mapping.register(Foo)
>>> o = OrderedDict(Foo())
>>> o
OrderedDict({1:10, 2:20})
>>> o = o.freeze()
>>> o
FrozenOrderedDict({1:10, 2:20})
>>> set([o])
set([FrozenOrderedDict({1:10, 2:20})])
>>> OrderedDict((), foo = 5)
Traceback (most recent call last):
    ...
TypeError: OrderedDict expected at most 1 non-keyword argument or only keyword arguments
>>> o = OrderedDict([(1,2),(3,4)])
>>> import cPickle as pickle
>>> s = pickle.dumps(o, -1)
>>> O = pickle.loads(s)
>>> O == o
True

"""

__all__ = ['FrozenOrderedDict', 'OrderedDict']

from bfc.linkedlist import LinkedList, BaseDataNode
from bfc.freeze import freeze

from itertools import chain
from collections import Mapping

def _no_items_dict_items(sequence):
    for key in sequence:
        yield key, sequence[key]

def sequence_to_key_value_pairs(sequence):
    for attr in ['iteritems', 'items']:
        a = getattr(sequence, attr, None)
        if a is not None:
            return a()
    if isinstance(sequence, Mapping):
        return _no_items_dict_items(sequence)
    return sequence

class KeyValueNode(BaseDataNode):
    __slots__ = ['key', 'value']

    def __init__(self, key, value):
        BaseDataNode.__init__(self)
        self.key = key
        self.value = value

class BaseOrderedDict(object):
    __slots__ = [
        '_list',
        '_dict',
        ]

    def __init__(self, *args, **kws):
        self._init()
        self._update(*args, **kws)

    def _init(self):
        self._list = LinkedList()
        self._dict = {}

    def _set(self, key, value):
        kvn = self._dict.get(key, None)
        if kvn is not None:
            kvn.value = value
        else:
            kvn = KeyValueNode(key, value)
            self._dict[key] = kvn
            kvn.insert_before(self._list)
        
    def _update_from_sequence(self, sequence):
        for key, value in sequence_to_key_value_pairs(sequence):
            self._set(key, value)

    def __getitem__(self, key):
        return self._dict[key].value

    def __len__(self):
        return len(self._dict)

    def __iter__(self):
        return self.iterkeys()

    # Py3k TODO: use the new approach
    def iterkeys(self):
       for kvn in self._list:
            yield kvn.key

    def keys(self):
        return self._dict.keys()

    def iteritems(self):
        for kvn in self._list:
            yield kvn.key, kvn.value

    def items(self):
        return list(self.iteritems())

    def itervalues(self):
        for kvn in self._list:
            yield kvn.value

    def values(self):
        return list(self.itervalues())

    def __contains__(self, key):
        return key in self._dict

    def get(self, key, default = None):
        kvn = self._dict.get(key, None)
        return default if kvn is None else kvn.value

    __hash__ = None

    def __repr__(self):
        return '%s({%s})' % (self.__class__.__name__,
                            ', '.join('%r:%r' % (key, value)
                                      for key, value in self.iteritems()))

    def __cmp__(self, other):
        if not isinstance(other, BaseOrderedDict): # pragma: no cover
            return NotImplemented

        iter_left = self.iteritems()
        iter_right = other.iteritems()

        while True:
            try:
                left = iter_left.next()
            except StopIteration:
                left_exists = False
            else:
                left_exists = True
            try:
                right = iter_right.next()
            except StopIteration:
                right_exists = False
            else:
                right_exists = True

            if not left_exists:
                return -int(right_exists)
            if not right_exists:
                return 1
            r = cmp(left, right)
            if r:
                return r

    def copy(self):
        return self.__class__(self)

    @classmethod
    def from_keys(cls, sequence, value = None):
        return cls((key, value) for key in sequence)

    has_key = __contains__
    
    def freeze(self):
        return FrozenOrderedDict((key, freeze(value)) for key, value in self.iteritems())

    def _update(self, *args, **kws):
        if len(args) > 1 or (len(args) == 1 and kws):
            raise TypeError('%s expected at most 1 non-keyword argument or only keyword arguments' % (self.__class__.__name__, ))
        if args:
            self._update_from_sequence(args[0])
        else:
            self._update_from_sequence(kws)

    def __getstate__(self):
        return {'_':tuple(self.iteritems())}

    def __setstate__(self, state):
        self._init()
        self._update(state['_'])


class FrozenOrderedDict(BaseOrderedDict):
    __slots__ = ['_hash']

    def __init__(self, *args, **kws):
        BaseOrderedDict.__init__(self, *args, **kws)
        self._hash = None

    def __hash__(self):
        if self._hash is None:
            self._hash = hash(tuple(self.iteritems()))
        return self._hash

class OrderedDict(BaseOrderedDict):
    __slots__ = []

    __setitem__ = BaseOrderedDict._set

    def __delitem__(self, key):
        kvn = self._dict.pop(key)
        kvn.detach()

    def clear(self):
        while self:
            item, kvn = self._dict.popitem()
            kvn.detach()

    def update(self, *args, **kws):
        self._update(*args, **kws)
    
    def pop(self, key, *default):
        if len(default) > 1:
            raise TypeError('pop expected at most 2 arguments')
        try:
            kvn = self._dict.pop(key)
        except KeyError:
            if default:
                return default[0]
            raise
        else:
            value = kvn.value
            kvn.detach()
            return value

    def popitem(self):
        if not self:
            raise KeyError('popitem(): %s is empty' % (self.__class__.__name__,))
        kvn = self._list.prev
        kvn.detach()
        del self._dict[kvn.key]
        return kvn.key, kvn.value

    def setdefault(self, key, default = None):
        kvn = self._dict.get(key, None)
        if kvn is not None:
            return kvn.value

        kvn = KeyValueNode(key, default)
        self._dict[key] = kvn
        kvn.insert_before(self._list)
        return default
        
