# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
An indentation wrapper file-like type for file-like objects.

>>> import sys
>>> f = IndentedFile(sys.stdout)
>>> f.write('')
>>> f.write('bar\\n')
bar
>>> f.start_indent()
>>> f.write('foo\\n')
    foo
>>> f.start_indent()
>>> f.write('bas\\n')
        bas
>>> f.stop_indent()
>>> f.write('xip\\n')
    xip
>>> f.write('blah\\n')
    blah
>>> f.stop_indent()
>>> f.write('a'); print
a
>>> f.start_indent()
Traceback (most recent call last):
    ....
IndentedFileException: can start only on start of a line
>>> f.write('A\\nb\\n')
A
b
>>> f.start_indent()
>>> f.write('c'); print
    c
>>> f.stop_indent()
Traceback (most recent call last):
    ...
IndentedFileException: can stop only on start of a line
>>> f.write('B\\n')
B
>>> f.stop_indent()
>>> f.write('d\\n')
d
>>> f.stop_indent()
Traceback (most recent call last):
    ...
IndentedFileException: cannot stop an indentation that was not started

"""
class IndentedFileException(Exception): pass

class IndentedFile(object):
    __slots__ = ['indentation',
                 'indentation_level',
                 'internal_file',
                 'on_start_of_line']

    def __init__(self,
                 internal_file,
                 indentation = '    '):
        self.indentation = indentation
        self.indentation_level = 0
        self.internal_file = internal_file
        self.on_start_of_line = True

    def start_indent(self):
        if not self.on_start_of_line:
            raise IndentedFileException('can start only on start of a line')
        self.indentation_level = self.indentation_level + 1

    def stop_indent(self):
        if not self.on_start_of_line:
            raise IndentedFileException('can stop only on start of a line')
        if not self.indentation_level:
            raise IndentedFileException('cannot stop an indentation that was not started')
        self.indentation_level = self.indentation_level - 1
        
    def write(self, data):
        if not data:
            return

        if self.on_start_of_line:
            self.internal_file.write(self.indentation * self.indentation_level)
            self.on_start_of_line = False

        if data[-1] != '\n':
            data = data.replace('\n', '\n' + (self.indentation * self.indentation_level))
            self.internal_file.write(data)
        else:
            data = data[:-1].replace('\n', '\n' + (self.indentation * self.indentation_level))
            self.internal_file.write(data + '\n')
            self.on_start_of_line = True

    def flush(self):
        self.internal_file.flush()

    def close(self):
        self.internal_file.close()

    # this is not covered due to the test environment itself
    def fileno(self): # pragma: no cover
        return self.internal_file.fileno()
