# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.

from ..threadpool import ThreadPool

from functools import partial

class StopThreadedRunner(Exception): pass

class ThreadedRunner(object):
    __slots__ = ['depgraph',
                 'environment',
                 'pool',
                 'total_success',
                 ]

    def __init__(self, depgraph, environment, workers):
        self.depgraph = depgraph
        self.environment = environment
        self.pool = ThreadPool(workers)
        self.total_success = True

    def run(self):
        try:
            if self.schedule_not_run_sources():
                self.pool.flush()
        except StopThreadedRunner:
            pass
        finally:
            self.pool.join()
        return not set(dep for dep in self.depgraph.ready_to_run_dependencies if not dep._schedules_to_run)
        
    def _callback(self, dep, success, result):
        dep._mark_as_run(success, result)
        self.environment.on_threaded_done(dep, success, result)

        if not success:
            if not self.environment.keep_on_error:
                raise StopThreadedRunner()
            return

        self.schedule_not_run_sources()

    def schedule_to_run(self, dep):
        self.environment.on_threaded_start(dep)
        dep._schedules_to_run = True
        self.pool.apply_async(dep.run, (self.environment,), {}, callback = partial(self._callback, dep))

    def schedule_not_run_sources(self):
        ready_to_run = set(dep for dep in self.depgraph.ready_to_run_dependencies if not dep._schedules_to_run and not dep._dep_failed)
        for dep in ready_to_run:
            self.schedule_to_run(dep)
        return not not ready_to_run

def threaded_run_all(depgraph, environment, workers):
    return ThreadedRunner(depgraph, environment, workers).run()
