# Copyright (c) 2020-2022, Manfred Moitzi
# License: MIT License
from __future__ import annotations
from typing import cast
import pytest

from ezdxf.lldxf.tags import Tags
from ezdxf.lldxf.tagwriter import TagCollector
from ezdxf.entities import Hatch
from ezdxf.proxygraphic import (
    load_proxy_graphic,
    export_proxy_graphic,
    ProxyGraphic,
    ProxyGraphicError,
)
from ezdxf.protocols import SupportsVirtualEntities, query_virtual_entities


def test_load_proxy_graphic():
    binary_data = load_proxy_graphic(Tags.from_text(DATA))
    assert len(binary_data) == 968


def test_export_proxy_graphic():
    tagwriter = TagCollector()
    binary_data = load_proxy_graphic(Tags.from_text(DATA))
    export_proxy_graphic(binary_data, tagwriter)
    s = "".join(tag.dxfstr() for tag in tagwriter.tags)
    assert s == DATA


class TestProxyGraphic:
    @pytest.fixture(scope="class")
    def data(self) -> bytes:
        return load_proxy_graphic(Tags.from_text(DATA))

    @pytest.fixture
    def parser(self, data: bytes) -> ProxyGraphic:
        return ProxyGraphic(data)

    def test_info(self, parser: ProxyGraphic):
        indices = list(parser.info())
        assert len(indices) == 13
        index, size, type_ = indices[0]
        assert (index, size, type_) == (8, 84, "POLYLINE_WITH_NORMALS")

    def test_supports_virtual_entities_protocol(self, parser):
        assert isinstance(parser, SupportsVirtualEntities) is True
        assert len(query_virtual_entities(parser)) > 0

    def test_multi_leader_entities(self):
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # UNKNOWN_TYPE_51; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # UNICODE_TEXT2; size: 200
        # UNKNOWN_TYPE_51; size: 12
        # ATTRIBUTE_LAYER; size: 12
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # ATTRIBUTE_FILL; size: 12
        # POLYGON; size: 84
        # ATTRIBUTE_MARKER; size: 12
        # POLYLINE; size: 60
        # ATTRIBUTE_MARKER; size: 12
        # POLYLINE; size: 60
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_LINEWEIGHT; size: 12
        # ATTRIBUTE_MARKER; size: 12
        # ATTRIBUTE_TRUE_COLOR; size: 12
        # ATTRIBUTE_LINETYPE; size: 12
        # ATTRIBUTE_LINEWEIGHT; size: 12
        # UNKNOWN_TYPE_51; size: 12
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(MULITILEADER)))
        indices = list(parser.info())
        assert len(indices) == 23
        entities = list(parser.virtual_entities())
        assert len(entities) == 4
        text = entities[0]
        assert text.dxftype() == "TEXT"
        assert text.dxf.text == "W410"
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available
        assert text.dxf.true_color is None

        hatch = entities[1]  # POLYGON
        assert hatch.dxftype() == "HATCH"
        assert len(hatch.paths[0].vertices) == 3
        assert hatch.dxf.layer == "0"  # no DXF document available
        assert hatch.dxf.color == 256  # by layer
        assert hatch.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[2]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 2
        assert polyline.dxf.layer == "0"  # no DXF document available
        assert polyline.dxf.color == 256  # by layer
        assert polyline.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[3]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 2
        assert polyline.dxf.layer == "0"  # no DXF document available
        assert polyline.dxf.color == 256  # by layer
        assert polyline.dxf.linetype == "BYLAYER"  # no DXF document available

    def test_image_entities(self):
        # UNICODE_TEXT2; size: 204
        # POLYLINE; size: 132
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(IMAGE)))
        indices = list(parser.info())
        assert len(indices) == 2

        entities = list(parser.virtual_entities())
        assert len(indices) == 2

        text = entities[0]
        assert text.dxftype() == "TEXT"
        assert text.dxf.text == "AcDbRasterImage"
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available

        polyline = entities[1]
        assert polyline.is_closed is False
        assert polyline.dxftype() == "POLYLINE"
        assert len(polyline.vertices) == 5
        assert text.dxf.layer == "0"  # no DXF document available
        assert text.dxf.color == 256  # by layer
        assert text.dxf.linetype == "BYLAYER"  # no DXF document available

    def test_circular_arc_entities(self):
        # example "aec_doors_walls.dxf" of issue #497; ACAD_PROXY_ENTITY <59D>
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(ARC_AEC_DOOR)))
        indices = list(parser.info())
        # [
        # ...
        # (96, 108, 'POLYGON'),
        # (204, 108, 'POLYGON'),
        # (312, 100, 'CIRCULAR_ARC'),
        # (412, 100, 'CIRCULAR_ARC'),
        # (512, 108, 'POLYGON'),
        # (620, 108, 'POLYGON')
        # ]
        assert len(indices) == 13
        entities = list(parser.virtual_entities())
        assert len(entities) == 6
        # 1. ARC - clockwise oriented
        arc1 = entities[2]
        assert arc1.dxf.extrusion.isclose(
            (0, 0, -1)
        ), "expected inverted extrusion vector"

        assert arc1.dxf.center.isclose(
            (-9333.398890018769, 10110.240054080468, 0.0)  # OCS!
        )
        assert arc1.dxf.start_angle == pytest.approx(90.0003859699524)
        assert arc1.dxf.end_angle == pytest.approx(-179.9996140300476)
        assert arc1.dxf.radius == pytest.approx(850)

        # 2. ARC - regular counter clockwise oriented arc
        arc2 = entities[3]
        assert arc2.dxf.extrusion.isclose(
            (0, 0, 1)
        ), "expected WCS aligned extrusion vector"

        assert arc2.dxf.center.isclose(
            (9333.41034197779, 11810.240054041897, 0.0)  # OCS == WCS
        )
        assert arc2.dxf.start_angle == pytest.approx(-90.0003859699524)
        assert arc2.dxf.end_angle == pytest.approx(-0.0003859699524042526)
        assert arc2.dxf.radius == pytest.approx(850)

    def test_lwpolyline_still_raises_consolidated_exception(self):
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(LWPOLYLINE_DATA)))
        with pytest.raises(ProxyGraphicError):
            list(parser.virtual_entities())
        with pytest.raises(ProxyGraphicError):
            list(parser.__virtual_entities__())

    def test_raw_exception_in_lwpolyline_parser(self):
        # TODO: fix error in LWPOLYLINE proxy graphic entity
        parser = ProxyGraphic(load_proxy_graphic(Tags.from_text(LWPOLYLINE_DATA)))
        with pytest.raises(IndexError):
            list(parser.unsafe_virtual_entities())

    def test_polygon_creation(self):
        parser = ProxyGraphic(
            load_proxy_graphic(Tags.from_text(POLYGON_IN_MULTILEADER))
        )
        hatch = cast(
            Hatch,
            [e for e in parser.unsafe_virtual_entities() if e.dxftype() == "HATCH"][0],
        )
        assert (
            hatch.dxf.elevation.isclose((0, 0, -235.49623891538792)) is True
        ), "expected elevation value to be set"
        polygon_path = hatch.paths[0]
        bulge_values = [bulge for x, y, bulge in polygon_path.vertices]
        assert bulge_values == [0, 0, 0], "expected all bulge values to be 0"


DATA = """160
968
310
C80300000D000000540000002000000002000000033E695D8B227240B00D3CF1FB7B5540000000000000000082C85BAC2FDE7240FB1040429FB05740000000000000000000000000000000000000000000000000000000000000F03F5400000020000000020000004AF9442AE7FA60405A2D686189715A4000000000000000
310
00C0DC003571AE5F40043422DDA4515D40000000000000000000000000000000000000000000000000000000000000F03F64000000040000001EA72DF9806A69402CE3B4E7B59D34400000000000000000770FBC9D50855E4000000000000000000000000000000000000000000000F03FB634003D352CE93FB1DDE561C5C1
310
E33F00000000000000000418DC3967E1F83F000000000C0000001200000000000000D0000000260000001F8BC5F8B8B46A40197732241FF06140000000000000000000000000000000000000000000000000000000000000F03F0943D77B25BDEF3F417457E0C451C0BF00000000000000003100370032002C003400320000
310
00000006000000010000000000000000000440000000000000F03F0000000000000000000000000000F03F00000000000000000000000000000000000000000000000000000000000000000000000041007200690061006C00000061007200690061006C002E007400740066000000000000000C00000012000000FF7F0000
310
6400000004000000813C33FBB3606A400278BF21B8F4614000000000000000009AEFA7C64B37034000000000000000000000000000000000000000000000F03F0943D77B25BDEF3F437457E0C451C0BF0000000000000000182D4454FB210940000000000C00000010000000010000000C0000001700000000000000540000
310
0020000000020000001EA72DF9806A69402CE3B4E7B59D344000000000000000001EA72DF9806A69402CE3B4E7B59D3440000000000000000000000000000000000000000000000000000000000000F03F540000002000000002000000B296839B8D1A724001000000F06355400000000000000000B296839B8D1A72400100
310
0000F0635540000000000000000000000000000000000000000000000000000000000000F03F540000002000000002000000632D073753076140FFFFFFFF2F525A400000000000000000632D073753076140FFFFFFFF2F525A40000000000000000000000000000000000000000000000000000000000000F03F5400000020
310
000000020000000960E446A3456F405AF2DBF448AB604000000000000000000960E446A3456F405AF2DBF448AB6040000000000000000000000000000000000000000000000000000000000000F03F
"""

MULITILEADER = """160
640
310
80020000170000000C00000016000000000000C00C00000033000000000000000C00000013000000993A0000C800000026000000EC2335956D6D9440F0AEEB8E7B766840000000000000000000000000000000000000000000000000000000000000F03F000000000000F03F00000000000000000000000000000000570034
310
00310030000000000004000000010000000000000000000840626666666666F23F0000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000001000000000000000000000072006F006D0061006E0073002E0073006800780000005C00000000000C000000330000000000
310
00000C000000100000001D0000000C00000016000000000000C00C00000012000000FF7F00000C00000013000000010000000C00000014000000010000005400000007000000030000000EA778BBEFF9934072D00E24B4FE694000000000000000005E00F4D6D4EB9340264DF666FD0B6B40000000000000000028C9814D17
310
04944016652748DB316A4000000000000000000C00000013000000891300003C00000006000000020000001B387D8403FF9340C41A1BB647186A400000000000000000FB0DD6A357189440F0AEEB8E7BD6684000000000000000000C00000013000000112700003C0000000600000002000000FB0DD6A357189440F0AEEB8E
310
7BD66840000000000000000080F9275C765D9440F0AEEB8E7BD6684000000000000000000C00000016000000000000C00C00000012000000FF7F00000C00000017000000000000000C000000130000009A3A00000C00000016000000000000C00C00000012000000FF7F00000C00000017000000FFFFFFFF0C000000330000
310
0000000000
"""

IMAGE = """160
344
310
5801000002000000CC0000002600000092B5D7AAA19916C0BF88551BB606F83F000000000000000000000000000000000000000000000000000000000000F03F000000000000F03F00000000000000000000000000000000410063004400620052006100730074006500720049006D0061006700650000000F000000000000
310
000000000000000000000000000000F03F0000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000000000000000000000000000074007800740000000000000084000000060000000500000092B5D7AAA19916C0BF88551BB606F83F00000000000000008A21ADA701E6
310
01C0BF88551BB606F83F00000000000000008A21ADA701E601C06095C5228F990740000000000000000092B5D7AAA19916C06095C5228F990740000000000000000092B5D7AAA19916C0BF88551BB606F83F0000000000000000
"""

ARC_AEC_DOOR = """160
728
310
D80200000D0000000C00000010000000050000000C0000000E000000000100000C00000012000000FF7F0000100000002200000000000000000000000C00000010000000050000000C00000012000000000000000C0000000E000000010000006C00000007000000040000008705D40EB308C2404F852AD01EBFC340000000
310
0000000000188DCA03B308C240BF872AD01EA6C34000000000000000003988CA03B33AC240E19617BA1EA6C3400000000000000000A800D40EB33AC240719417BA1EBFC34000000000000000006C000000070000000400000043041686B408C2407A322AD01E11C7400000000000000000B27C1F91B408C2400A302AD01E2A
310
C7400000000000000000D3771F91B43AC2402C3F17BA1E2AC740000000000000000064FF1586B43AC2409C4117BA1E11C74000000000000000006400000004000000A800D40EB33AC240719417BA1EBFC34000000000000000000000000000908A4000000000000000000000000000000000010000000000F0BFD0A43C4F34
310
41DC3EADE1FCFFFFFFEF3F0000000000000000182D4454FB21F93F00000000640000000400000064FF1586B43AC2409C4117BA1E11C74000000000000000000000000000908A4000000000000000000000000000000000010000000000F03FD0A43C4F3441DCBEADE1FCFFFFFFEFBF0000000000000000182D4454FB21F93F
310
000000006C0000000700000004000000A800D40EB33AC240719417BA1EBFC3400000000000000000A36A8A18B33AC2404D9217BA1ED5C340000000000000000038418A18B3E3C340EE9276FE1DD5C34000000000000000003DD7D30EB3E3C340139576FE1DBFC34000000000000000006C000000070000000400000064FF15
310
86B43AC2409C4117BA1E11C740000000000000000069955F7CB43AC240C14317BA1EFBC6400000000000000000FE6B5F7CB4E3C340634476FE1DFBC6400000000000000000F9D51586B4E3C3403E4276FE1D11C7400000000000000000
"""

LWPOLYLINE_DATA = """160
3732
310
940E00001B0000000C000000130000009C3A0000880000001D000000000000000000F03F000000000000000000000000000000006ED29ACAFF2775400000000000000000AB4C58E87AB6EB3FFFFFFFFFFFFFDFBF65C7F9EC72B270400000000000000000FFFFFFFFFFFFDF3FAB4C58E87AB6EB3FB2B5E05EB5476340000000
310
000000000000000000000000000000000000000000000000000000F03F3C000000210000002D000000000090400000000000000000000000000002440C000000000000910030000000000000000C0000000000000000000000080000001F000000880000001D000000000000000000F03F0000000000000000000000000000
310
000000000000000000000000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000000000000000000000000F03F0000000000000000000000000000000000000000000000000000000000000000000000000000F03FC800000026000000086C346499317540D0267628AC
310
1071407F82AD2B82B463400000000000000000FFFFFFFFFFFFDFBFAB4C58E87AB6EB3F000000000000F03F00000000000000000000000000000000580031000000000002000000000000000000000000000440000000000000F03F0000000000000000000000000000F03F0000000000000000000000000000000000000000
310
0000000000000000000000000000000041007200690061006C00000061007200690061006C002E00740074006600000000000000C800000026000000C08AB94FEB767540EC7982DEC8C770403EE49A47586063400000000000000000FFFFFFFFFFFFDFBFAB4C58E87AB6EB3F000000000000F03F0000000000000000000000
310
0000000000580032000000000002000000000000000000000000000440000000000000F03F0000000000000000000000000000F03F00000000000000000000000000000000000000000000000000000000000000000000000041007200690061006C00000061007200690061006C002E007400740066000000000000000800
310
00001F0000000C00000013000000112700000C00000013000000010000000C0000001400000001000000540000000700000003000000A85B09A87FAF7640E422BB3E3C1370407F91D54CDD8F62406ED29ACAFFCF76404C6256A634C56F40B2B5E05EB55762401EFF06EE0A9F764081C9CD5F7A0770402605FACC498262400C
310
00000013000000891300001C0200000600000016000000632D084B45A776403276444F5B0D704053CBE78C138962407E0B8A8CC89976409C032806DD1C70405533BC61FB9A62406F54701C1B8D76409127BF98DB2D7040D4CF85FD9AAE6240198DB7230D8176403F841B23FF3F704077E395E38CC362405E3A5CCB6E757640
310
D6BB4EC1EF527040E8B03D976BD962403F06B09FC15E7640774480A9D8797040D183997459066340A22E581E53537640DED9A12B218D7040990EF0A49D1C634028DF4FE194477640EAD2E031D79F7040CE5D23B038326340B39C9311573B7640C6D14ED8A2B170401AB48419C546634028EC1FD8692E7640A578FD3A2CC270
310
4023546564DD59634020349080A2277640C28DB29BDCC97040A3C46E47BE6263406752F15D9D207640B269FE751BD17040928016141C6B634099D7627554197640B94063CDDDD7704045B0C61AEA726340525404CCC11176401E4763A518DE7040107CE9AB1B7A63403059F566DF09764025B18001C1E37040480CE917A480
310
6340CD76554BA701764016B33DE5CBE8704043892FAF76866340C63D447E13F9754037811C542EED7040551B27C2868B6340B93EE1041EF07540CC4F9F51DDF07040D4EA39A1C78F6340400A4CE4C0E675401B5348E1CDF370401420D29C2C936340F930A421F6DC75406BBF9906F5F570406CE35905A99563407E4309C2B7
310
D2754002C915C547F770402F5D3B2B309763406ED29ACAFFC7754023A43E20BBF77040B2B5E05EB59763400C0000001300000002000000540000000700000003000000454C49A25AB277408EEDBBBA2FEC70400FD7CCC5608A6340532FC02AA1D9774068D66BE475BF7040EDDDB7A3BB566340A90EE82FBAA3774084AB7AC5
310
BCDE70407056EA2FD97A63400C000000130000008A1300007C020000060000001A00000077AD18690AAB7740894C1B4076E57040BF96DBFA9C82634038C0706FBB9C7740F1305150E7F170402C0035C6FA906340D2487B73638E77400B9514A7B5FF704048A696E0EBA06340A2D88CC5A0717740113FC1BF271E714050BF7E
310
B913C4634023BEE5C70A387740A4CD1FC2905D7140DAC336FD4B0D64401388C1BB581B7740273DC7097E797140630A37448B2D644045BC23390D0D77405FB5C6C368857140A1D22AF14D3B6440F82E757ECDFE764022240DBF8A8F71407010C536014764407127EF9F9BF0764024E279619397714081C9C80248506440F1EC
310
CAB179E276401148EC10329D71408403F942C5566440C0C641C869D476409AAE433316A0714029C418E51B5A64401FFC8CF76DC67640706E5F2EEF9F71402011EBD6EE596440FC409A5141B87640DBF97595739C71400DD8C64FE9556440E68DB53C1DAA7640C19B7818EE95714011EF7A8D614E644020F44640F39B764022
310
0914FDD18C714050A8DDAADC436440EE84B6E3B48D764001F7F48892817140EA55C5C2DF3664404E6BD027C170764048283AAD76667140BCD67C4D921764401DCB4344CE527640ABD7ADB23448714097025404A2F46340702E51746833764042ADB5C6662A71408C6AFDBD37D2634093CC354606237640E8EA6084CB1C7140
310
65C2F89E81C263405F1F39F31B1276402451B716A7107140AC9F2B517CB463401938C3029B007640F99465C36C06714082546CEFACA8634002283CFC74EE7540686B18D08FFE704009339194989F63405D000C679BDB754076897C8283F97040638D705BC49963401C0945C1E6D17540E80B979F2FF87040AA22E9E33B9863
310
406ED29ACAFFC7754024A43E20BBF77040B2B5E05EB59763400C00000013000000122700000C00000013000000030000005400000007000000030000001C3184C00F58734021D65C59376E7140025A46018620644020FC8CF76D267340E98FF54A064A7140EEDDB7A3BBF66340AE217236D3477340BF8C9D3F337A71406EBE
310
95825C2E64400C000000130000008B1300000C030000060000002000000065297B7BF14F734070317D4C35747140380CEE4171276440B98A088232697340C97576CD41907140114BD296D447644006B65CA015837340945B1A185CB07140D4223D41E66C64409E527C1E3DB77340D5549559FFF271404A613096D8B96440EE
310
3759FBBED07340F50C06F8AA1072401BAA7D401BDC6440BAC196E430DD7340EBD354DB861D72405E4A8C65F4EA644048CF1FEA5DE97340DDAF54AFA92872400B50DB3FD0F76440D877C5E339F57340163B7F9EC5317240440967EF54026540ACD258A9B8007440E20F4ED38C3872402BC42B94280A654003F7AA12CE0B7440
310
89C83A78B13C7240E1CE254EF10E654020FC8CF76D16744057FFBEB7E53D72408777513D551065409CB7C8D12E1E7440D579482AB33C7240B3775943F30E654023333AAAAC2574407DFA2C73BE39724053B7CA7B890B65407C5DB384EF2C74401E2E4ABB25357240E9F2EEB93A0665406E250665FF3374408CC17D2B072F72
310
40F8E60FD129FF64403A498046A641744018BB9E27B11E724082EA6ED74CEC6440BE14386DE14E7440AF4D7DAEAD097240F7A5352909D4644034FEBCF7F05B7440E3DF0607EEF0714061FDB16075B76440D67B9E041569744045D8287863D57140C9D43118A8976440810CB51F9B847440D395EBBFB2997140B2937370BB52
310
6440FB0B096B7D937440202867246F7B71404643D145C92F64408378F7B274A37440DEBA30BD255E7140FA026A04F80D644055C80F16C1B474409CB435D1C7427140D7B68B465EEE6340A671E1B2A2C77440EC7B63A7462A7140E44284A612D26340D29C7C80BFD17440B8DC7341641F71403F71E93181C56340B3EAFBA759
310
DC74405E77A786931571402B8BA1BE2BBA6340114A312D79E77440B0F8DB9FF20C7140274DF61F35B06340B3A9EE1326F37440810DEFB59F057140B4733129C0A763405FF8056068FF7440A462BEF1B8FF704055BB9CADEFA06340DD244915480C7540E9A4277C5CFB704089E08180E69B6340F51D8A37CD1975402381087E
310
A8F87040D29F2A75C79863406ED29ACAFF27754023A43E20BBF77040B2B5E05EB59763400C0000001300000004000000540000000700000003000000C0951AF269767440547E4BB7DF047040FC859302487F6240532FC02AA149744082D18E4F6EB86F408777513D555062407192913FC46474407C97343D410F7040E87C45
310
A7448B62400C000000130000008C1300001C02000006000000160000001814D618976D7440E80A407A100A70407281EC5446856240F9909B60B57874402C939BF4C9197040A619A58D6E97624016B8EC77E782744028D944A4052B7040B1DECDBA54AB6240D9751FE0638C7440649D3D696A3D7040FC14C3F291C06240A9B6
310
891A6195744066A087239F507040F500E14BBFD6624010735C0BB8A67440D66416F813787040A20B08BB4E04634077C770C47EAF744051A75ED2A18B70402EB3C9FDE21A63408C501455A0B87440AD2AFF219B9E70401A2225BBCB306340B6FA9C3E53C2744070AFF9C6A6B07040D29C7609A24563405FB26002CECC744020
310
F64FA16BC17040C1671AFFFE586340238D835E67D274405A0E7EA837C97040B000A8CFFF616340ED63B52147D8744043BF039190D0704056C76CB27B6A63404BF4201C74DE7440EB40E1366BD77040404434CA65726340CAFBF01DF5E4744062CB1676BCDD7040FBFFC939B1796340F63750F7D0EB7440B896A42A79E37040
310
1683F923518063405D6669780EF3744000DB8A3096E870401D568EAB388663408C446771B4FA74404AD0C96308ED7040A00154F35A8B63400F9074B2C9027540A7AE61A0C4F070402A0E161EAB8F63407406BC0B550B754027AE52C2BFF370404A04A04E1C9363404765684D5D147540DB069DA5EEF570408D6CBDA7A19563
310
40166AA447E91D7540D5F0402646F7704080CF394C2E9763406ED29ACAFF27754023A43E20BBF77040B3B5E05EB5976340
"""

POLYGON_IN_MULTILEADER = """160
912
310
900300001A0000000C00000016000000000000C00C00000033000000000000000C00000013000000993A0000C8000000260000009544FF764C1F9A40FA7DC5410E1B9040FF126F30E16F6DC000000000000000000000000000000000000000000000F03F000000000000F03F00000000000000000000000000000000520057
310
003A002000520053002D0055004B004400000000000A000000010000000000000000005940000000000000F03F0000000000000000000000000000F03F00000000000000000000000000000000000000000000000000000000010000002200000041007200690061006C0000000000722900000000C800000026000000FB5B
310
60691F2C9A40AF2B15AF22F38A40FF126F30E16F6DC000000000000000000000000000000000000000000000F03F000000000000F03F000000000000000000000000000000003D003600350030002F002B00310034002C003100350000000B000000010000000000000000005940000000000000F03F000000000000000000
310
0000000000F03F00000000000000000000000000000000000000000000000000000000010000002200000041007200690061006C00000000007229000000000C00000033000000000000000C00000010000000010000000C00000016000000000000C00C00000012000000FF7F00000C00000013000000010000000C000000
310
14000000010000005400000007000000030000002488873CFC05944054B2D39FE4389940FF126F30E16F6DC058836CEC5398934072ECAC7D4DBF9A40FF126F30E16F6DC0C9408E5DAD7E9440B3D25D958F719940FF126F30E16F6DC00C00000014000000010000000C00000013000000891300003C00000006000000020000
310
0077E40ACD5442944083C2981A3A559940FF126F30E16F6DC01718F2A9C9B99840E476AB02C1A38F40FF126F30E16F6DC00C00000013000000112700003C00000006000000020000001718F2A9C9B99840E476AB02C1A38F40FF126F30E16F6DC05422C94D3ABB9840E476AB02C1A38F40FF126F30E16F6DC00C0000001600
310
0000000000C00C00000012000000FF7F00000C00000017000000000000000C000000130000009A3A00003C00000006000000020000005422C94D3ABB9840E476AB02C1A38F40FF126F30E16F6DC0AD2EC6E9979CA340E476AB02C1A38F40FF126F30E16F6DC00C00000016000000000000C00C00000012000000FF7F00000C
310
00000017000000FFFFFFFF0C0000003300000000000000
"""

if __name__ == "__main__":
    pytest.main([__file__])
