"""
Reproduce the max likelihood for a codon model of Yang and Nielsen 2008.

Mitochondrial DNA data is from Masami Hasegawa.
AlgoPy provides the gradients and hessians for informing the
truncated Newton nonlinear optimization.
"""

import math

import numpy
import algopy
import scipy
import scipy.optimize

g_human_mtdna = """
CTACCCGCCGCAGTACTGATCATTCTATTTCCCCCTCTATTGATCCCCACCTCCAAATAT
CTCATCAACAACCGACTAATTACCACCCAACAATGACTAATCAAACTAACCTCAAAACAA
ATGATAGCCATACACAACACTAAAGGACGAACCTGATCTCTTATACTAGTATCCTTAATC
ATTTTTATTGCCACAACTAACCTCCTCGGACTCCTGCCTCACTCATTTACACCAACCACC
CAACTATCTATAAACCTAGCCATGGCCATCCCCTTATGAGCGGGCGCAGTGATTATAGGC
TTTCGCTCTAAGATTAAAAATGCCCTAGCCCACTTCTTACCACAAGGCACACCTACACCC
CTTATCCCCATACTAGTTATTATCGAAACCATCAGCCTACTCATTCAACCAATAGCCCTG
GCCGTACGCCTAACCGCTAACATTACTGCAGGCCACCTACTCATGCATCTAATTGGAAGC
GCCACCCTAGCAATATCAACCATTAACCTTCCCTCTACACTTATCATCTTCACAATTCTA
ATTCTACTGACTATCCTAGAAATCGCTGTCGCCTTAATCCAAGCCTACGTTTTCACACTT
CTAGTAAGCCTCTACCTGCACGACAACACACCCCAACTAAATACTACCGTATGGCCCACC
ATAATTACCCCCATACTCCTTACACTATTTCTCATCACCCAACTAAAAATATTAAACACA
AGCTACCACTTACCTCCCTCACCAAAGCCCATAAAAATAAAAAATTATAACAAACCCTGA
GAACCAACCCCAATACGCAAAATTAACCCCCTAATAAAATTAATTAACCACTCATTCATC
GACCTCCCCACCCCATCCAACATCTCCGCATGATGAAACTTCGGCTCACTCCTTGGCGCC
TGCCTGATCCTCCAAATCACCACAGGACTATTCCTAGCCATGCACTACTCACCAGACGCC
TCAACCGCCTTTTCATCAATCGCCCACATCACTCGAGACGTAAATTATGGCTGAATCATC
CGCTACCTTCACGCCAATGGCGCCTCAATATTCTTTATCTGCCTCTTCCTACACATCGGG
CGAGGCCTATATTACGGATCATTTCTCTACTCAGAAACCTGAAACATCGGCATTATCCTC
CTGCTTGCAACTATAGCAACAGCCTTCATAGGTTATGTCCTCCCGTGAGGCCAAATATCA
TTCTGAGGGGCCACAGTAATTACAAACTTACTATCCGCCATCCCATACATTGGGACAGAC
CTAGTTCAATGAATCTGAGGAGGCTACTCAGTAGACAGTCCCACCCTCACACGATTCTTT
ACCTTTCACTTCATCTTGCCCTTCATTATTGCAACCCTAGCAGCACTCCACCTCCTATTC
TTGCACGAAACGGGATCAAACAACCCCCTAGGAATCACCTCCCATTCCGATAAAATCACC
TTCCACCCTTACTACACAATCAAAGACACCCTCGGCTTACTTCTCTTCCTTCTCTCCTTA
ATGACATTAACACTATTCTCACCAGACCTCCTAGGCGACCCAGACAATTATACCCTAGCC
AACCCCTTAAACACCCCTCCCCACATCAAGCCCGAATGATATTTCCTATTCGCCTACACA
ATTCTCCGATCCGTCCCTAACAAACTAGGAGGCGTCCTTGCCCTATTACTATCCATCCTC
ATCCTAGCAATAATCCCCATCCTCCATATATCCAAACAACAAAGCATAATATTTCGCCCA
CTAAGCCAATCACTTTATTGACTCCTAGCCGCAGACCTCCTCATTCTAACCTGAATCGGA
GGACAACCAGTAAGCTACCCTTTTACCATCATTGGACAAGTAGCATCCGTACTATACTTC
ACAACAATCCTAATCCTAATACCAACTATCTCCCTAATTGAAAACAAAATACTCAAATGG
GCCTTCGCCGACCGTTGACTATTCTCTACAAACCACAAAGACATTGGAACACTATACCTA
TTATTCGGCGCATGAGCTGGAGTCCTAGGCACAGCTCTAAGCCTCCTTATTCGAGCCGAG
CTGGGCCAGCCAGGCAACCTTCTAGGTAACGACCACATCTACAACGTTATCGTCACAGCC
CATGCATTTGTAATAATCTTCTTCATAGTAATACCCATCATAATCGGAGGCTTTGGCAAC
TGACTAGTTCCCCTAATAATCGGTGCCCCCGATATGGCGTTCCCCCGCATAAACAACATA
AGCTTCTGACTCTTACCTCCCTCTCTCCTACTCCTGCTCGCATCTGCTATAGTAGAGGCC
GGAGCAGGAACAGGTTGAACAGTCTACCCTCCCTTAGCAGGGAACTACTCCCACCCTGGA
GCCTCCGTAGACCTAACCATCTTCTCCTTACACCTAGCAGGTGTCTCCTCTATCTTAGGG
GCCATCAATTTCATCACAACAATTATCAATATAAAACCCCCTGCCATAACCCAATACCAA
ACGCCCCTCTTCGTCTGATCCGTCCTAATCACAGCAGTCCTACTTCTCCTATCTCTCCCA
GTCCTAGCTGCTGGCATCACTATACTACTAACAGACCGCAACCTCAACACCACCTTCTTC
GACCCCGCCGGAGGAGGAGACCCCATTCTATACCAACACCTATCCTGATTTTTCGGTCAC
CCTGAAGTTTATATTCTTATCCTACCAGGCTTCGGAATAATCTCCCATATTGTAACTTAC
TACTCCGGAAAAAAAGAACCATTTGGATACATAGGTATGGTCTGAGCTATGATATCAATT
GGCTTCCTAGGGTTTATCGTGTGAGCACACCATATATTTACAGTAGGAATAGACGTAGAC
ACACGAGCATATTTCACCTCCGCTACCATAATCATCGCTATCCCCACCGGCGTCAAAGTA
TTTAGCTGACTCGCCACACTCCACGGAAGCAATATGAAATGATCTGCTGCAGTGCTCTGA
GCCCTAGGATTCATCTTTCTTTTCACCGTAGGTGGCCTGACTGGCATTGTATTAGCAAAC
TCATCACTAGACATCGTACTACACGACACGTACTACGTTGTAGCTCACTTCCACTATGTC
CTATCAATAGGAGCTGTATTTGCCATCATAGGAGGCTTCATTCACTGATTTCCCCTATTC
TCAGGCTACACCCTAGACCAAACCTACGCCAAAATCCATTTCGCTATCATATTCATCGGC
GTAAATCTAACTTTCTTCCCACAACACTTTCTCGGCCTATCCGGAATGCCCCGACGTTAC
TCGGACTACCCCGATGCATACACCACATGAAATATCCTATCATCTGTAGGCTCATTCATT
TCTCTAACAGCAGTAATATTAATAATTTTCATGATTTGAGAAGCCTTCGCTTCGAAGCGA
AAAGTCCTAATAGTAGAAGAACCCTCCATAAACCTGGAGTGACTATATGGATGCCCCCCA
CCCTACCACACATTCGAAGAACCCGTATACATAAAATCTGCACATGCAGCGCAAGTAGGT
CTACAAGACGCTACTTCCCCTATCATAGAAGAGCTTATCACCTTTCATGATCACGCCCTC
ATAATCATTTTCCTTATCTGCTTCCTAGTCCTGTATGCCCTTTTCCTAACACTCACAACA
AAACTAACTAATACTAACATCTCAGACGCTCAGGAAATAGAAACCGTCTGAACTATCCTG
CCCGCCATCATCCTAGTCCTCATCGCCCTCCCATCCCTACGCATCCTTTACATAACAGAC
GAGGTCAACGATCCCTCCCTTACCATCAAATCAATTGGCCACCAATGGTACTGAACCTAC
GAGTACACCGACTACGGCGGACTAATCTTCAACTCCTACATACTTCCCCCATTATTCCTA
GAACCAGGCGACCTGCGACTCCTTGACGTTGACAATCGAGTAGTACTCCCGATTGAAGCC
CCCATTCGTATAATAATTACATCACAAGACGTCTTGCACTCATGAGCTGTCCCCACATTA
GGCTTAAAAACAGATGCAATTCCCGGACGTCTAAACCAAACCACTTTCACCGCTACACGA
CCGGGGGTATACTACGGTCAATGCTCTGAAATCTGTGGAGCAAACCACAGTTTCATGCCC
ATCGTCCTAGAATTAATTCCCCTAAAAATCTTTGAAATAGGACCCGTATTTACCCTAACC
CACCAATCACATGCCTATCATATAGTAAAACCCAGCCCATGACCCCTAACAGGGGCCCTC
TCAGCCCTCCTAATGACCTCCGGCCTAGCCATGTGATTTCACTTCCACTCCATAACGCTC
CTCATACTAGGCCTGCTAACCAACACACTAACCATATACCAATGATGGCGCGATGTAACA
CGAGAAAGCACATACCAAGGCCACCACACACCACCTGTCCAAAAAGGCCTTCGATACGGG
ATAATCCTATTTATTACCTCAGAAGTTTTTTTCTTCGCAGGATTTTTCTGAGCCTTTTAC
CACTCCAGCCTAGCCCCTACCCCCCAACTAGGAGGGCACTGGCCCCCAACAGGCATCACC
CCGCTAAATCCCCTAGAAGTCCCACTCCTAAACACATCCGTATTACTCGCATCAGGAGTA
TCAATCACCTGAGCTCACCATAGTCTAATAGAAAACAACCGAAACCAAATAATTCAAGCA
CTGCTTATTACAATTTTACTGGGTCTCTATTTTACCCTCCTACAAGCCTCAGAGTACTTC
GAATCTCCCTTCACCATTTCCGACGGCATCTACGGCTCAACATTTTTTGTAGCCACAGGC
TTCCATGGACTTCACGTCATTATTGGCTCAACTTTCCTCACTATCTGCTTCATCCGCCAA
CTAATATTTCACTTTACATCCAAACATCACTTTGGCTTCGAAGCCGCCGCCTGATACTGG
CATTTTGTAGATGTGGTTTGACTATTTCTGTATGTCTCCATCTATTGATGAGGGTCTCCC
ATGGCCAACCTCCTACTCCTCATTGTACCCATTCTAATCGCAATGGCATTCCTAATGCTT
ACCGAACGAAAAATTCTAGGCTATATACAACTACGCAAAGGCCCCAACGTTGTAGGCCCC
TACGGGCTACTACAACCCTTCGCTGACGCCATAAAACTCTTCACCAAAGAGCCCCTAAAA
CCCGCCACATCTACCATCACCCTATACATCACCGCCCCGACCTTAGCTCTCACCATCGCT
CTTCTACTATGAGCCCCCCTCCCCATACCCAACCCCCTGGTTAACCTCAACCTAGGCCTC
CTATTTATTCTAGCCACCTCTAGCCTAGCCGTTTACTCAATCCTCTGATCAGGGTGAGCA
TCAAACTCAAACTACGCCCTGATCGGCGCACTGCGAGCAGTAGCCCAAACAATCTCATAT
GAAGTCACCCTAGCCATCATTCTACTATCAACATTACTAATAAGTGGCTCCTTTAACCTC
TCCACCCTTATCACAGCACAAGAACACCTCTGATTACTCCTGCCATCATGACCCTTGGCC
ATAATATGATTTATCTCCACACTAGCAGAGACCAACCGAACCCCCTTCGACCTTGCCGAA
GGGGAGTCCGAACTAGTCTCAGGCTTCAACATCGAATACGCCGCAGGCCCCTTCGCCCTA
TTCTTCATAGCCGAATACACAAACATTATTATAATAAACACCCTCACCACTACAATCTTC
CTAGGAACAACATATGACGCACTCTCCCCTGAACTCTACACAACATATTTTGTCACCAAG
ACCCTACTTCTGACCTCCCTGTTCTTATGAATTCGAACAGCATACCCCCGATTCCGCTAC
GACCAACTCATACACCTCCTATGAAAAAACTTCCTACCACTCACCCTAGCATTACTTATA
TGATATGTCTCCATACCCATTACAATCTCCAGCATTCCCCCTCAAACCAATCCCCTGGCC
CAACCCGTCATCTACTCTACCATCTTTGCAGGCACACTCATCACAGCGCTAAGCTCGCAC
TGATTTTTTACCTGAGTAGGCCTAGAAATAAACATGCTAGCCTTTATTCCAATTCTAACC
AAAAAAATAAACCCTCGTTCCACAGAAGCTGCCATCAAGTATTTCCTCACGCAAGCAACC
GCATCCATAATCCTTCTAATAGCTATCCTCTTCAACAATATACTCTCCGGACAATGAACC
ATAACCAATACTACCAATCAATACTCATCATTAATAATCATAATGGCTATAGCAATAAAA
CTAGGAATAGCCCCCTTTCACTTCTGAGTCCCAGAGGTTACCCAAGGCACCCCTCTGACA
TCCGGCCTGCTTCTTCTCACATGACAAAAACTAGCCCCCATCTCAATCATATACCAAATC
TCTCCCTCACTAAACGTAAGCCTTCTCCTCACTCTCTCAATCTTATCCATCATAGCAGGC
AGTTGAGGTGGATTAAACCAAACCCAGCTACGCAAAATCTTAGCATACTCCTCAATTACC
CACATAGGATGAATAATAGCAGTTCTACCGTACAACCCTAACATAACCATTCTTAATTTA
ACTATTTATATTATCCTAACTACTACCGCATTCCTACTACTCAACTTAAACTCCAGCACC
ACAACCCTACTACTATCTCGCACCTGAAACAAGCTAACATGACTAACACCCTTAATTCCA
TCCACCCTCCTCTCCCTAGGAGGCCTGCCCCCGCTAACCGGCTTTTTGCCCAAATGGGCC
ATTATCGAAGAATTCACAAAAAACAATAGCCTCATCATCCCCACCATCATAGCCACCATC
ACCCTCCTTAACCTCTACTTCTACCTACGCCTAATCTACTCCACCTCAATCACACTACTC
CCCATATCTAACAACGTAAAAATAAAATGACAGTTTGAACATACAAAACCCACCCCACTC
CTCCCCACACTCATCACCCTTACCACGCTACTCCTACCTATCTCCCCTTTTATACTAATA
ATCTTAAACTTCGCCTTAATTTTAATAATCAACACCCTCCTAGCCTTACTACTAATAATT
ATTACATTTTGACTACCACAACTCAACGGCTACATAGAAAAATCCACCCCTTACGAGTGC
GGCTTCGACCCTATATCCCCCGCCCGCGTCCCTTTCTCCATAAAATTCTTCTTAGTAGCT
ATTACCTTCTTATTATTTGATCTAGAAATTGCCCTCCTTTTACCCCTACCATGAGCCCTA
CAAACAACTAACCTGCCACTAATAGTTATGTCATCCCTCTTATTAATCATCATCCTAGCC
CTAAGTCTGGCCTATGAGTGACTACAAAAAGGATTAGACTGAGCCGAACTAATCGTCCCA
ACAATTATATTACTACCACTGACATGACTCTCCAAAAAACACATAATTTGAATCAACACA
ACCACCCACAGCCTAATTATTAGCATCATCCCCCTACTATTTTTTAACCAAATCAACAAC
AACCTATTTAGCTGCTCCCCAACCTTTTCCTCCGACCCCCTAACAACCCCCCTCCTAATA
CTAACTACCTGACTCCTACCCCTGACAATCATGGCAAGCCAACGCCACTTATCCAGTGAA
CCACTATCACGAAAAAAACTCTACCTCTCTATACTAATCTCCCTACAAATCTCCTTAATT
ATAACATTCACAGCCACAGAACTAATCATATTTTATATCTTCTTCGAAACCACACTTATC
CCCACCTTGGCTATCATCACCCGATGAGGCAGCCAACCAGAACGCCTGAACGCAGGCACA
TACTTCCTATTCTACACCCTAGTAGGCTCCCTTCCCCTACTCATCGCACTAATTTACACT
CACAACACCCTAGGCTCACTAAACATTCTACTACTCACTCTCACTGCCCAAGAACTATCA
AACTCCTGAGCCAACAACTTAATATGACTAGCTTACACAATAGCTTTTATAGTAAAGATA
CCTCTTTACGGACTCCACTTATGACTCCCTAAAGCCCATGTCGAAGCCCCCATCGCTGGG
TCAATAGTACTTGCCGCAGTACTCTTAAAACTAGGCGGCTATGGTATAATACGCCTCACA
CTCATTCTCAACCCCCTGACAAAACACATAGCCTACCCCTTCCTTGTACTATCCCTATGA
GGCATAATTATAACAAGCTCCATCTGCCTACGACAAACAGACCTAAAATCGCTCATTGCA
TACTCTTCAATCAGCCACATGGCCCTCGTAGTAACAGCCATTCTCATCCAAACCCCCTGA
AGCTTCACCGGCGCAGTCATTCTCATAATCGCCCACGGACTTACATCCTCATTACTATTC
TGCCTAGCAAACTCAAACTACGAACGCACTCACAGTCGCATCATAATCCTCTCTCAAGGA
CTTCAAACTCTACTCCCACTAATAGCTTTTTGATGACTTCTAGCAAGCCTCGCTAACCTC
GCCTTACCCCCCACTATTAACCTACTGGGAGAACTCTCTGTGCTAGTAACCACATTCTCC
TGATCAAATATCACTCTCCTACTTACAGGACTCAACATACTAGTCACAGCCCTATACTCC
CTCTACATATTTACCACAACACAATGAGGCTCACTCACCCACCACATTAACAACATAAAA
CCCTCATTCACACGAGAAAACACCCTCATGTTCATACACCTATCCCCCATTCTCCTCCTA
TCCCTCAACCCCGACATCATTACCGGGTTTTCCTCTCCCCTCATTTACATAAATATTATA
CTAGCATTTACCATCTCACTTCTAGGAATACTAGTATATCGCTCACACCTCATATCCTCC
CTACTATGCCTAGAAGGAATAATACTATCGCTATTCATTATAGCTACTCTCATAACCCTC
AACACCCACTCCCTCTTAGCCAATATTGTGCCTATTGCCATACTAGTTTTTGCCGCCTGC
GAAGCAGCGGTAGGCCTAGCCCTACTAGTCTCAATCTCCAACACATATGGCCTAGACTAC
GTACATAACCTAAACCTACTCACCATGCACACTACTATAACCACCCTAACCCTGACTTCC
CTAATTCCCCCCATCCTTACCACCCTCGTTAACCCTAACAAAAAAAACTCATACCCCCAT
TATGTAAAATCCATTGTCGCATCCACCTTTATTATCAGTCTCTTCCCCACAACAATATTC
ATGTGCCTAGACCAAGAAGTTATTATCTCGAACTGACACTGAGCCACAACCCAAACAACC
CAGCTCTCCCTAAGCTTCAAACTAGACTACTTCTCCATAATATTCATCCCTGTAGCATTG
TTCGTTACATGGTCCATCATAGAATTCTCACTGTGATATATAAACTCAGACCCAAACATT
AATCAGTTCTTCAAATATCTACTCATTTTCCTAATTACCATGCTAATCTTAGTTACCGCT
AACAACCTATTCCAACTGTTCATCGGCTGAGAGGGCGTAGGAATTATATCCTTCTTGCTC
ATCAGTTGATGATACGCCCGAGCAGATGCCAACACAGCAGCCATTCAAGCAATCCTATAC
AACCGTATCGGCGATATCGGTTTCATCCTCGCCTTAGCATGATTTATCCTACACTCCAAC
TCATGAGACCCACAACAAATAGCCCTTCTAAACGCTAATCCAAGCCTCCCCCCACTACTA
GGCCTCCTCCTAGCAGCAGCAGGCAAATCAGCCCAATTAGGTCTCCACCCCTGACTCCCC
TCAGCCATAGAAGGCCCCACCCCAGTCTCAGCCCTACTCCACTCAAGCACTATAGTTGTA
GCAGGAGTCTTCTTACTCATCCGCTTCCACCCCCTAGCAGAAAATAGCCCACTAATCCAA
ACTCTAACACTATGCTTAGGCGCTATCACCACTCTGTTCGCAGCAGTCTGCGCCCTTACA
CAAAATGACATCAAAAAAATCGTAGCCTTCTCCACTTCAAGTCAACTAGGACTCATAGTA
GTTACAATCGGCATCAACCAACCACACCTAGCATTCCTGCACATCTGTACCCACGCCTTC
TTCAAAGCCATACTATTTATGTGCTCCGGGTCCATCATCCACAACCTTAACAATGAACAA
GATATTCGAAAAATAGGAGGACTACTCAAAACCATACCTCTCACTTCAACCTCCCTCACC
ATTGGCAGCCTAGCATTAGCAGGAATACCTTTCCTCACAGGTTTCTATTCCAAAGACCAC
ATCATCGAAACCGCAAACATATCATACACAAACGCCTGAGCCCTATCTATTACTCTCATC
GCTACCTCCCTGACAAGCGCCTATAGCACTCGAATAATTCTTCTCACCCTAACAGGTCAA
CCTCGCTTCCCTACCCTTACTAACATTAACGAAAATAACCCCACCCTACTAAACCCCATT
AAACGCCTGGCAGCCGGAAGCCTATTCGCAGGATTTCTCATTACTAACAACATTTCCCCC
GCATCCCCCTTCCAAACAACAATCCCCCTCTACCTAAAACTCACAGCCCTCGCTGTCACT
TTCCTAGGACTTCTAACAGCCCTAGACCTCAACTACCTAACCAACAAACTTAAAATAAAA
TCCCCACTATGCACATTTTATTTCTCCAACATACTCGGATTCTACCCTAGCATCACACAC
CGCACAATCCCCTATCTAGGCCTTCTTACGAGCCAAAACCTGCCCCTACTCCTCCTAGAC
CTAACCTGACTAGAAAAGCTATTACCTAAAACAATTTCACAGCACCAAATCTCCACCTCC
ATCATCACCTCAACCCAAAAAGGCATAATTAAACTTTACTTCCTCTCTTTCTTCTTCCCA
CTCATCCTAACCCTACTCCTAATCACA
"""

g_chimp_mtdna = """
TTACCCGCCGCAGTACTAATCATTCTATTCCCCCCTCTACTGGTCCCCACTTCTAAACAT
CTCATCAACAACCGACTAATTACCACCCAACAATGACTAATTCAACTGACCTCAAAACAA
ATAATAACTATACACAGCACTAAAGGACGAACCTGATCTCTCATACTAGTATCCTTAATC
ATTTTTATTACCACAACCAATCTTCTTGGGCTTCTACCCCACTCATTCACACCAACCACC
CAACTATCTATAAACCTAGCCATGGCTATCCCCCTATGAGCAGGCGCAGTAGTCATAGGC
TTTCGCTTTAAGACTAAAAATGCCCTAGCCCACTTCTTACCGCAAGGCACACCTACACCC
CTTATCCCCATACTAGTTATCATCGAAACTATTAGCCTACTCATTCAACCAATAGCCTTA
GCCGTACGTCTAACCGCTAACATTACTGCAGGCCACCTACTCATGCACCTAATTGGAAGC
GCCACACTAGCATTATCAACTATCAATCTACCCTATGCACTCATTATCTTCACAATTCTA
ATCCTACTGACTATTCTAGAGATCGCCGTCGCCTTAATCCAAGCCTACGTTTTTACACTT
CTAGTGAGCCTCTACCTGCACGACAACACACCCCAACTAAATACCGCCGTATGACCCACC
ATAATTACCCCCATACTCCTGACACTATTTCTCGTCACCCAACTAAAAATATTAAATTCA
AATTACCATCTACCCCCCTCACCAAAACCCATAAAAATAAAAAACTACAATAAACCCTGA
GAACCAACCCCGACACGCAAAATTAACCCACTAATAAAATTAATTAATCACTCATTTATC
GACCTCCCCACCCCATCCAACATTTCCGCATGATGGAACTTCGGCTCACTTCTCGGCGCC
TGCCTAATCCTTCAAATTACCACAGGATTATTCCTAGCTATACACTACTCACCAGACGCC
TCAACCGCCTTCTCGTCGATCGCCCACATCACCCGAGACGTAAACTATGGTTGGATCATC
CGCTACCTCCACGCTAACGGCGCCTCAATATTTTTTATCTGCCTCTTCCTACACATCGGC
CGAGGTCTATATTACGGCTCATTTCTCTACCTAGAAACCTGAAACATTGGCATTATCCTC
TTGCTCACAACCATAGCAACAGCCTTTATGGGCTATGTCCTCCCATGAGGCCAAATATCC
TTCTGAGGAGCCACAGTAATTACAAACCTACTGTCCGCTATCCCATACATCGGAACAGAC
CTGGTCCAGTGAGTCTGAGGAGGCTACTCAGTAGACAGCCCTACCCTTACACGATTCTTC
ACCTTCCACTTTATCTTACCCTTCATCATCACAGCCCTAACAACACTTCATCTCCTATTC
TTACACGAAACAGGATCAAATAACCCCCTAGGAATCACCTCCCACTCCGACAAAATTACC
TTCCACCCCTACTACACAATCAAAGATATCCTTGGCTTATTCCTTTTCCTCCTTATCCTA
ATGACATTAACACTATTCTCACCAGGCCTCCTAGGCGATCCAGACAACTATACCCTAGCT
AACCCCCTAAACACCCCACCCCACATTAAACCCGAGTGATACTTTCTATTTGCCTACACA
ATCCTCCGATCCATCCCCAACAAACTAGGAGGCGTCCTCGCCCTACTACTATCTATCCTA
ATCCTAACAGCAATCCCTGTCCTCCACACATCCAAACAACAAAGCATAATATTTCGCCCA
CTAAGCCAACTGCTTTACTGACTCCTAGCCACAGACCTCCTCATCCTAACCTGAATCGGA
GGACAACCAGTAAGCTACCCCTTCATCACCATCGGACAAATAGCATCCGTATTATACTTC
ACAACAATCCTAATCCTAATACCAATCGCCTCTCTAATCGAAAACAAAATACTTGAATGA
ACCTTCACCGACCGCTGACTATTCTCTACAAACCACAAAGATATTGGAACACTATACCTA
CTATTCGGTGCATGAGCTGGAGTCCTGGGCACAGCCCTAAGTCTCCTTATTCGGGCTGAA
CTAGGCCAACCAGGCAACCTCCTAGGTAATGACCACATCTACAATGTCATCGTCACAGCC
CATGCATTCGTAATAATCTTCTTCATAGTAATGCCTATTATAATCGGAGGCTTTGGCAAC
TGGCTAGTTCCCTTGATAATTGGTGCCCCCGACATGGCATTCCCCCGCATAAACAACATA
AGCTTCTGGCTCCTGCCCCCTTCTCTCCTACTTCTACTTGCATCTGCCATAGTAGAAGCC
GGCGCGGGAACAGGTTGAACAGTCTACCCTCCCTTAGCGGGAAACTACTCGCATCCTGGA
GCCTCCGTAGACCTAACCATCTTCTCCTTACATCTGGCAGGCATCTCCTCTATCCTAGGA
GCCATTAACTTCATCACAACAATTATTAATATAAAACCTCCTGCCATGACCCAATACCAA
ACACCCCTCTTCGTCTGATCCGTCCTAATCACAGCAGTCTTACTTCTCCTATCCCTCCCA
GTCCTAGCTGCTGGCATCACCATACTATTGACAGATCGTAACCTCAACACTACCTTCTTC
GACCCAGCCGGGGGAGGAGACCCTATTCTATATCAACACTTATTCTGATTTTTTGGCCAC
CCCGAAGTTTATATTCTTATCCTACCAGGCTTCGGAATAATTTCCCACATTGTAACTTAT
TACTCCGGAAAAAAAGAACCATTTGGATATATAGGCATGGTTTGAGCTATAATATCAATT
GGCTTCCTAGGGTTTATCGTGTGAGCACACCATATATTTACAGTAGGGATAGACGTAGAC
ACCCGAGCCTATTTCACCTCCGCTACCATAATCATTGCTATTCCTACCGGCGTCAAAGTA
TTCAGCTGACTCGCTACACTTCACGGAAGCAATATGAAATGATCTGCCGCAGTACTCTGA
GCCCTAGGGTTTATCTTTCTCTTCACCGTAGGTGGCCTAACCGGCATTGTACTAGCAAAC
TCATCATTAGACATCGTGCTACACGACACATACTACGTCGTAGCCCACTTCCACTACGTT
CTATCAATAGGAGCTGTATTCGCCATCATAGGAGGCTTCATTCACTGATTCCCCCTATTC
TCAGGCTATACCCTAGACCAAACCTATGCCAAAATCCAATTTGCCATCATGTTCATTGGC
GTAAACCTAACCTTCTTCCCACAGCACTTCCTTGGCCTATCTGGGATGCCCCGACGTTAC
TCGGACTACCCCGATGCATACACCACATGAAATGTCCTATCATCCGTAGGCTCATTTATC
TCCCTGACAGCAGTAATATTAATAATTTTCATGATTTGAGAAGCCTTTGCTTCAAAACGA
AAAGTCCTAATAGTAGAAGAGCCCTCCGCAAACCTGGAATGACTATATGGATGCCCCCCA
CCCTACCACACATTCGAAGAACCCGTATACATAAAATCTGCACATGCAGCGCAAGTAGGT
CTACAAGATGCTACTTCCCCTATCATAGAAGAACTTATTATCTTTCACGACCATGCCCTC
ATAATTATCTTTCTCATCTGCTTTCTAGTCCTATACGCCCTTTTCCTAACACTCACAACA
AAACTAACTAATACTAGTATTTCAGACGCCCAGGAAATAGAAACCGTCTGAACTATCCTG
CCCGCCATCATCCTAGTCCTTATTGCCCTACCATCCCTGCGTATCCTTTACATAACAGAC
GAGGTCAACGACCCCTCCTTTACTATTAAATCAATCGGCCATCAATGATATTGAACCTAC
GAATACACCGACTACGGCGGGCTAATCTTCAACTCCTACATACTCCCCCCATTATTTCTA
GAACCAGGTGATCTACGACTCCTTGACGTTGATAACCGAGTGGTCCTCCCAGTTGAAGCC
CCCGTTCGTATAATAATTACATCACAAGATGTTCTACACTCATGAGCTGTTCCCACATTA
GGCCTAAAAACAGACGCAATTCCCGGACGCCTAAACCAAACCACTTTCACCGCCACACGA
CCAGGAGTATACTACGGCCAATGCTCAGAAATCTGTGGAGCAAACCACAGTTTTATACCC
ATCGTCCTAGAATTAATCCCTCTAAAAATCTTTGAAATAGGACCCGTATTCACTCTAACC
CACCAATCACATGCCTACCACATAGTAAAACCCAGCCCATGACCCCTAACAGGGGCCCTC
TCGGCCCTCCTAATAACCTCCGGCCTGGCCATATGATTCCACTTCTACTCCACAACACTA
CTCACACTAGGCTTACTAACTAACACATTGACCATATATCAATGATGACGCGATGTTATA
CGAGAAGGCACATACCAAGGCCACCACACACCACCCGTCCAAAAAGGTCTCCGATATGGG
ATAATTCTTTTTATTACCTCAGAAGTTTTTTTCTTTGCAGGATTTTTTTGAGCTTTCTAC
CACTCCAGCCTAGCCCCTACCCCCCAGCTAGGAGGACACTGGCCCCCAACAGGTATTACC
CCACTAAATCCCCTAGAAGTCCCACTCCTAAACACATCTGTATTACTCGCATCAGGAGTA
TCAATTACTTGAGCCCATCACAGCTTAATAGAAAATAACCGAAACCAAATAATTCAAGCA
CTGCTTATTACGATTCTACTAGGTCTTTATTTTACCCTCCTACAAGCCTCAGAATATTTC
GAATCCCCTTTTACCATTTCCGATGGCATCTACGGCTCAACATTCTTTGTAGCCACAGGC
TTCCACGGACTCCACGTCATTATTGGATCAACTTTCCTCACTATCTGCCTCATCCGCCAA
CTAATATTTCACTTCACATCCAAACATCACTTCGGCTTTCAAGCCGCCGCCTGATACTGA
CACTTCGTAGATGTAGTCTGACTATTTCTATATGTCTCTATTTACTGATGAGGATCTCCC
ATGACCAACCTCCTACTCCTCATTGTACCCATCCTAATCGCAATAGCATTCCTAATGCTA
ACCGAACGAAAAATTCTAGGCTACATACAACTACGCAAAGGTCCCAACATTGTAGGTCCT
TACGGGCTATTACAGCCCTTCGCTGACGCCATAAAACTCTTCACTAAAGAACCCTTAAAA
CCCTCCACTTCAACCATTACCCTCTACATCACCGCCCCAACCCTAGCCCTCACCATTGCC
CTCTTACTATGAACCCCCCTCCCCATACCCAACCCCCTAGTCAATCTTAACTTAGGCCTC
CTATTTATTCTAGCCACCTCCAGCCTAGCCGTTTACTCAATCCTCTGATCAGGGTGAGCA
TCAAACTCGAACTACGCCTTAATCGGTGCACTACGAGCAGTAGCCCAAACAATCTCATAC
GAAGTCACTCTAGCCATTATCCTACTGTCAACGCTACTAATAAGTGGCTCCTTCAATCTC
TCTACCCTTGTCACAACACAAGAGCACCTCTGACTAATCCTGCCAACATGACCCCTGGCC
ATAATATGATTTATCTCTACACTAGCAGAGACCAACCGAACTCCCTTCGACCTTACTGAA
GGAGAATCTGAACTAGTCTCAGGCTTTAATATCGAGTATGCCGCAGGCCCCTTTGCCCTA
TTTTTCATAGCCGAATACATAAACATTATTATAATAAACACCCTCACTGCTACAATCTTC
CTAGGAGCAACATACAATACTCACTCCCCTGAACTCTACACGACATATTTTGTCACCAAA
GCTCTACTTCTAACCTCCCTGTTCCTATGAATTCGAACAGCATATCCCCGATTTCGCTAC
GACCAGCTCATACACCTCCTATGAAAAAACTTCCTACCACTCACCCTAGCATCACTCATG
TGATATATCTCCATACCCACTACAATCTCCAGCATCCCCCCTCAAACCAATCCCCTAGCC
CAACCCATCATCTACTCTACCATCCTTACAGGCACGCTCATTACAGCGCTAAGCTCACAC
TGATTTTTCACCTGAGTAGGCCTAGAAATAAATATACTAGCTTTTATCCCAATCCTAACC
AAAAAAATAAGCCCCCGCTCCACAGAAGCCGCCATCAAATACTTTCTCACACAAGCAACT
GCGTCCATAATTCTCCTGATAGCTATCCTCTCCAACAGCATACTCTCCGGACAATGAACC
ATAACCAATACTACCAATCAATACTCATCATTAATAATTATAATAGCAATGGCAATAAAA
CTAGGAATAGCCCCCTTTCACTTTTGAGTTCCAGAAGTTACCCAAGGCACCCCCCTAATA
TCCGGCCTACTCCTCCTCACATGACAAAAATTAGCCCCTATTTCAATTATATACCAAATC
TCCTCATCACTGAACGTAAACCTTCTCCTCACCCTTTCAATCTTGTCCATTATAGCAGGC
AGCTGAGGCGGACTAAACCAAACCCAACTACGCAAAATCCTAGCATACTCCTCAATCACC
CACATAGGCTGAATAATAGCAGTCCTACCATATAACCCTAACATAACCATTCTTAATTTA
ACCATTTACATCATCCTAACTACTACCGCATTTCTGCTACTCAACTTAAACTCCAGCACC
ACAACCCTACTACTATCTCGCACCTGAAACAAGCTAACATGATTAACTCCCCTAATTCCA
TCCACCCTCCTCTCCCTAGGAGGCCTACCCCCACTAACTGGCTTCTTACCCAAATGAGTT
ATCATCGAAGAATTCACAAAAAATAATAGCCTCATCATCCCCACCATCATAGCCATCATC
ACTCTCCTTAACCTCTATTTCTACCTACGCCTAATCTACTCCACCTCAATTACACTACTT
CCCATATCTAATAACGTAAAAATAAAATGACAATTCGAACATACAAAACCCACCCCCTTC
CTCCCTACACTCATCACCCTTACCACACTGCTTCTACCCATCTCCCCCTTCATACTAATA
ATCTTAAACTTCGTCCTAATTTTAATAACCAATACCCTTCTAGCCCTACTACTGATAATT
ATCACATTCTGACTACCACAACTCAACAGCTACATAGAAAAATCTACCCCTTACGAATGT
GGCTTCGACCCTATATCCCCCGCCCGCGTCCCCTTCTCCATAAAATTTTTCCTAGTAGCC
ATCACCTTCCTATTATTTGACCTAGAAATTGCCCTCCTATTGCCCTTACCTTGAGCCCTA
CAAACGGCCAACCTACCACTAATAGTCACATCATCCCTCTTATTAATTACTATCCTAGCC
CTAAGCCTCGCCTACGAATGATTACAAAAAGGGTTAGACTGAACCGAACTAATCATCCCG
ACAATTATATTACTACCACTAACATGATTCTCTAAAAAACGTATAATTTGAATCAACACA
ACCACTCACAGCCTAATTATCAGCACCATTCCCTTACTATTTTTTAACCAAATTAACAAC
AACCTATTCAGCTGTTCCCTGCCCTTCTCCTCCGACCCCTTAACAACTCCCCTCCTAATA
TTAACTGCTTGACTTCTACCCCTCACAATCATAGCAAGCCAGCGCCACCTATCCAACGAA
CCACTATCACGAAAAAAACTCTACCTCTCCATGCTAATTTCCCTCCAAATCTCCTTAATT
ATAACATTCTCGGCCACAGAGCTAATTATATTTTATATCTTCTTCGAAACCACACTTATC
CCCACCCTGGCTATCATCACCCGATGGGGTAACCAACCAGAACGCCTGAACGCAGGTACA
TACTTCCTATTCTATACCCTAGTAGGCTCCCTCCCCCTACTCATCGCACTAATCTATACC
CACAACACCCTAGGCTCACTAAATATCCTATTACTCACTCTTACAACCCAAGAACTATCA
AACACCTGAGCCAACAACTTAATATGACTAGCGTACACGATGGCTTTCATGGTAAAAATA
CCCCTTTACGGACTCCACCTATGACTCCCTAAAGCCCATGTCGAAGCCCCTATTGCCGGG
TCAATGGTACTTGCTGCAGTACTCTTAAAATTAGGTGGCTATGGCATAATACGCCTCACA
CTCATCCTCAACCCCCTAACAAAACATATAGCCTATCCCTTCCTCATGTTGTCCTTATGA
GGTATAATCATAACAAGCTCCATCTGCCTGCGACAAACAGACCTAAAATCGCTCATTGCA
TACCCTTCAGTCAGCCACATAGCCCTCGTAGTAACAGCCATTCTCATCCAAACCCCCTGA
AGCTTCACCGGCGCAATTATCCTCATAATCGCCCACGGACTTACATCCTCATTATTATCC
TGCCTAGCAAACTCAAATTATGAACGCACCCACAGTCGCATCATAATTCTCTCCCAAGGA
CTTCAAACTCTACTCCCACTAATAGCCTTTTGATGACTCCTGGCAAGCCTCGCTAACCTC
GCCCTACCCCCTACCATTAATCTCCTAGGGGAACTCTCCGTGCTAGTAACCTCATTCTCC
TGATCAAATACCACTCTCCTACTCACAGGATTCAACATACTAATCACAGCCCTGTACTCC
CTCTACATGTTTACCACAACACAATGAGGCTCACTCACCCACCACATTAATAGCATAAAG
CCCTCATTCACACGAGAAAACACTCTCATATTTTTACACCTATCCCCCATCCTCCTTCTA
TCCCTCAATCCTGATATCATCACTGGATTCACCTCCCCCCTTATTTATATAAATATTATA
CTAGCATTTACCATCTCACTTCTAGGAATACTAGTATATCGCTCACACCTAATATCTTCC
CTACTATGCCTAGAAGGAATAATACTATCACTGTTCATCATAGCCACCCTCATAACCCTC
AATACTCACTCCCTCTTAGCCAATATTGTACCCATCACCATACTAGTCTTTGCTGCCTGC
GAAGCAGCAGTAGGTCTAGCACTACTAGTTTCAATCTCTAACACATATGGCTTAGACTAC
GTACATAACCTAAACCTACTCACCATGTATGCTACCATAACCACCTTAGCCCTAACTTCC
TTAATTCCCCCCATCCTCGGCGCCCTCATTAACCCTAACAAAAAAAACTCATACCCCCAT
TACGTGAAATCCATTATCGCATCCACCTTTATCATTAGCCTTTTCCCCACAACAATATTC
ATATGCCTAGACCAAGAAACTATTATCTCGAACTGACACTGAGCAACAACCCAAACAACC
CAACTCTCCCTGAGCTTTAAACTAGACTATTTCTCCATAACATTTATCCCCGTAGCACTG
TTCGTTACATGATCCATCATAGAATTCTCACTATGATATATAGACTCAGACCCCAACATC
AACCAATTCTTCAAATACTTACTTATCTTCCTAATTACTATACTAATCCTAGTCACCGCT
AACAACCTATTCCAACTCTTCATCGGCTGAGAAGGCGTAGGAATTATATCCTTTCTACTC
ATTAGCTGATGGTACGCCCGAACAGATGCCAACACAGCAGCCATCCAAGCAATCCTATAT
AACCGTATCGGTGATATTGGTTTTGTCCTAGCCCTAGCATGATTTCTCCTACACTCCAAC
TCATGAGATCCACAACAAATAATCCTCCTAAGTACTAATACAGACCTTACTCCACTACTA
GGCTTCCTCCTAGCAGCAGCAGGCAAATCAGCTCAACTAGGCCTTCACCCCTGACTCCCC
TCAGCCATAGAAGGCCCTACCCCTGTTTCAGCCCTACTCCACTCAAGCACCATAGTCGTA
GCAGGAATCTTCCTACTCATCCGCTTCTACCCCCTAGCAGAGAATAACCCACTAATCCAA
ACTCTCACGCTATGCCTAGGCGCTATCACCACCCTATTCGCAGCAGTCTGCGCCCTCACA
CAAAATGACATCAAAAAAATCGTGGCCTTCTCCACTTCAAGCCAACTAGGACTCATAATA
GTTACAATCGGTATCAACCAACCACACCTAGCATTCCTTCACATCTGCACCCACGCTTTC
TTCAAAGCCATACTATTCATATGCTCCGGATCCATTATTCACAACCTCAATAATGAGCAA
GACATTCGAAAAATAGGAGGATTACTCAAAACCATACCCCTCACTTCAACCTCCCTCACC
ATTGGGAGCCTAGCATTAGCAGGAATACCCTTCCTCACAGGTTTCTACTCCAAAGACCTC
ATCATCGAAACCGCTAACATATCATACACAAACGCCTGAGCCCTATCTATTACTCTCATC
GCCACCTCTCTGACAAGCGCCTACAGCACCCGAATAATCCTCCTCACCCTAACAGGTCAA
CCTCGCTTCCCAACCCTCACCAACATTAACGAAAACAACCCCACTCTGTTAAATCCCATT
AAACGCCTAACCATTGGAAGCTTATTTGCAGGATTTCTCATTACCAACAACATTCTCCCC
ATATCTACTCCCCAAGTGACAATTCCCCTTTACTTAAAACTTACAGCCCTAGGCGTTACT
TCCCTAGGACTTCTAACAGCCCTAGACCTCAATTACCTAACCAGCAAGCTCAAAATAAAA
TCCCCACTATATACATTTCACTTCTCTAATATACTCGGATTCTACCCTAACATTATACAC
CGCTCGATCCCCTATCTAGGCCTTCTTACAAGCCAAAACCTACCCCTACTTCTTCTAGAC
CTGACCTGACTAGAGAAACTATTACCTAAAACAATTTCACAGTACCAAATCTCCGCTTCC
ATTACCACCTCAACCCAAAAAGGCATGATCAAACTTTATTTCCTCTCTTTTTTCTTCCCT
CTCATCTTAACCTTACTCCTAATCACA
"""

# http://en.wikipedia.org/wiki/File:Transitions-transversions-v3.png
g_ts = {'ag', 'ga', 'ct', 'tc'}
g_tv = {'ac', 'ca', 'gt', 'tg', 'at', 'ta', 'cg', 'gc'}

# http://en.wikipedia.org/wiki/Human_mitochondrial_genetics
g_stop_mito = {'tag', 'taa', 'aga', 'agg'}
g_code_mito = (
        ('gct', 'gcc', 'gca', 'gcg'),
        ('cgt', 'cgc', 'cga', 'cgg'),
        ('aat', 'aac'),
        ('gat', 'gac'),
        ('tgt', 'tgc'),
        ('caa', 'cag'),
        ('gaa', 'gag'),
        ('ggt', 'ggc', 'gga', 'ggg'),
        ('cat', 'cac'),
        ('att', 'atc'),
        ('tta', 'ttg', 'ctt', 'ctc', 'cta', 'ctg'),
        ('aaa', 'aag'),
        ('atg', 'ata'),
        ('ttt', 'ttc'),
        ('cct', 'ccc', 'cca', 'ccg'),
        ('tct', 'tcc', 'tca', 'tcg', 'agt', 'agc'),
        ('act', 'acc', 'aca', 'acg'),
        ('tgg', 'tga'),
        ('tat', 'tac'),
        ('gtt', 'gtc', 'gta', 'gtg'),
        ('tag', 'taa', 'aga', 'agg'),
        )


############################################################
# These functions precompute some matrices encoding
# some simple models of how molecular biology works.
# Speed is not important for any of these functions,
# because they are all outside of the maximization.
# All of these functions use numpy and not algopy.


def get_hamming(codons):
    """
    Get the hamming distance between codons, in {0, 1, 2, 3}.
    @param codons: sequence of lower case codon strings
    @return: matrix of hamming distances
    """
    ncodons = len(codons)
    ham = numpy.zeros((ncodons, ncodons), dtype=int)
    for i, ci in enumerate(codons):
        for j, cj in enumerate(codons):
            ham[i, j] = sum(1 for a, b in zip(ci, cj) if a != b)
    return ham

def get_ts_tv(codons):
    """
    Get binary matrices defining codon pairs differing by single changes.
    @param codons: sequence of lower case codon strings
    @return: two binary numpy arrays
    """
    ncodons = len(codons)
    ts = numpy.zeros((ncodons, ncodons), dtype=int)
    tv = numpy.zeros((ncodons, ncodons), dtype=int)
    for i, ci in enumerate(codons):
        for j, cj in enumerate(codons):
            nts = sum(1 for p in zip(ci,cj) if ''.join(p) in g_ts)
            ntv = sum(1 for p in zip(ci,cj) if ''.join(p) in g_tv)
            if nts == 1 and ntv == 0:
                ts[i, j] = 1
            if nts == 0 and ntv == 1:
                tv[i, j] = 1
    return ts, tv

def get_syn_nonsyn(code, codons):
    """
    Get binary matrices defining synonymous or nonynonymous codon pairs.
    @return: two binary matrices
    """
    ncodons = len(codons)
    inverse_table = dict((c, i) for i, cs in enumerate(code) for c in cs)
    syn = numpy.zeros((ncodons, ncodons), dtype=int)
    for i, ci in enumerate(codons):
        for j, cj in enumerate(codons):
            if inverse_table[ci] == inverse_table[cj]:
                syn[i, j] = 1
    return syn, 1-syn

def get_compo(codons):
    """
    Get a matrix defining site-independent nucleotide composition of codons.
    @return: integer matrix
    """
    ncodons = len(codons)
    compo = numpy.zeros((ncodons, 4), dtype=int)
    for i, c in enumerate(codons):
        for j, nt in enumerate('acgt'):
            compo[i, j] = c.count(nt)
    return compo

def get_asym_compo(codons):
    """
    This is an ugly function.
    Its purpose is to help determine which is the mutant nucleotide type
    given an ordered pair of background and mutant codons.
    This determination is necessary if we want to follow
    the mutation model of Yang and Nielsen 2008.
    Entry [i, j, k] of the returned matrix gives the number of positions
    for which the nucleotides are different between codons i and j and
    the nucleotide type of codon j is 'acgt'[k].
    @return: a three dimensional matrix
    """
    ncodons = len(codons)
    asym_compo = numpy.zeros((ncodons, ncodons, 4), dtype=int)
    for i, ci in enumerate(codons):
        for j, cj in enumerate(codons):
            for k, nt in enumerate('acgt'):
                asym_compo[i, j, k] = sum(1 for a, b in zip(ci, cj) if (
                    a != b and b == nt))
    return asym_compo

def get_lb_expected_subs(ham, subs_counts):
    """
    Get the lower bound of expected substitutions.
    This is expected minimum possible number of substitutions
    between aligned codons.
    """
    return numpy.sum(ham * subs_counts) / float(numpy.sum(subs_counts))


############################################################
# These functions turn the hardcoded DNA strings
# at the beginning of this file into something more
# friendly for a computer to use.
# Speed is not important for these functions.


def dna_to_codon_ord(codons, dna_string):
    codon_dict = dict((c, i) for i, c in enumerate(codons))
    dna = ''.join(dna_string.lower().split())
    n = len(dna)
    if n % 3:
        raise Exception('expected DNA length to be a multiple of 3')
    observed_codons = [dna[i:i+3] for i in range(0, n, 3)]
    return [codon_dict[c] for c in observed_codons]

def get_empirical_summary(ncodons, human_codon_ord, chimp_codon_ord):
    """
    Get codon counts and substitution counts.
    @param ncodons: allowed number of codons
    @param human_mtdna_ord: human codons converted into indices
    @param chimp_mtdna_ord: chimp codons converted into indices
    @return: codon_counts, subs_counts
    """
    codon_counts = numpy.zeros(ncodons, dtype=int)
    subs_counts = numpy.zeros((ncodons, ncodons), dtype=int)
    for i, j in zip(human_codon_ord, chimp_codon_ord):
        codon_counts[i] += 1
        codon_counts[j] += 1
        subs_counts[i, j] += 1
    return codon_counts, subs_counts


############################################################
# These functions use the precomputed matrices
# through AlgoPy and within the max likelihood search.
# Everything in here should use algopy and not numpy.


def fixation_h(x):
    """
    This is a fixation h function in the notation of Yang and Nielsen.
    """
    return 1. / algopy.special.hyp1f1(1., 2., -x)

def get_selection_F(log_counts, compo, log_nt_weights):
    """
    The F and S notation is from Yang and Nielsen 2008.
    Note that three of the four log nt weights are free parameters.
    One of the four log weights is zero and the other three
    are free parameters to be estimated jointly in the
    maximimum likelihood search,
    so this function is inside the optimization loop.
    Speed matters.
    @param log_counts: logs of empirical codon counts
    @param compo: codon composition as defined in the get_compo function
    @param log_nt_weights: un-normalized log mutation process probabilities
    @return: a log selection for each codon, up to an additive constant
    """
    return log_counts - algopy.dot(compo, log_nt_weights)

def get_selection_S(F):
    """
    The F and S notation is from Yang and Nielsen 2008.
    Speed matters.
    @param F: a selection value for each codon, up to an additive constant
    @return: selection differences F_j - F_i, also known as S_ij
    """
    e = algopy.ones_like(F)
    return algopy.outer(e, F) - algopy.outer(F, e)

def get_Q(
        ts, tv, syn, nonsyn, compo, asym_compo,
        h,
        log_counts,
        log_mu, log_kappa, log_omega, log_nt_weights):
    """
    Notation is from Yang and Nielsen 2008.
    The first group of args consists of precomputed ndarrays.
    The second group is only the fixation function.
    The third group consists of empirically (non-free) estimated parameters.
    The fourth group depends only on free parameters.
    Speed matters.
    @param ts: indicator for transition
    @param tv: indicator for transversion
    @param syn: indicator for synonymous codons
    @param nonsyn: indicator for nonsynonymous codons
    @param compo: site independent nucleotide composition per codon
    @param asym_compo: tensor from get_asym_compo function
    @param h: fixation function
    @param log_counts: empirically counted codons in the data set
    @param log_mu: free param for scaling
    @param log_kappa: free param for transition transversion rate distinction
    @param log_omega: free param for syn nonsyn rate distinction
    @param log_nt_weights: mostly free param array for mutation equilibrium
    @return: rate matrix
    """
    mu = algopy.exp(log_mu)
    kappa = algopy.exp(log_kappa)
    omega = algopy.exp(log_omega)
    F = get_selection_F(log_counts, compo, log_nt_weights)
    S = get_selection_S(F)
    pre_Q = mu * (kappa * ts + tv) * (omega * nonsyn + syn) * algopy.exp(
            algopy.dot(asym_compo, log_nt_weights)) * h(S)
    Q = pre_Q - algopy.diag(algopy.sum(pre_Q, axis=1))
    return Q

def slow_part(A, B):
    return A*B

def get_log_likelihood(P, v, subs_counts):
    """
    The stationary distribution of P is empirically derived.
    It is proportional to the codon counts by construction.
    @param P: a transition matrix using codon counts and free parameters
    @param v: stationary distribution proportional to observed codon counts
    @param subs_counts: observed substitution counts
    """
    A = subs_counts
    B = algopy.log(P.T * v)
    log_likelihoods = slow_part(A, B)
    return algopy.sum(log_likelihoods)

def eval_f(
        theta,
        subs_counts, log_counts, v,
        h,
        ts, tv, syn, nonsyn, compo, asym_compo,
        ):
    """
    @param theta: length six unconstrained vector of free variables
    """

    # unpack theta
    log_mu = theta[0]
    log_kappa = theta[1]
    log_omega = theta[2]
    log_nt_weights = algopy.zeros(4, dtype=theta)
    log_nt_weights[0] = theta[3]
    log_nt_weights[1] = theta[4]
    log_nt_weights[2] = theta[5]
    log_nt_weights[3] = 0

    # construct the transition matrix
    Q = get_Q(
            ts, tv, syn, nonsyn, compo, asym_compo,
            h,
            log_counts,
            log_mu, log_kappa, log_omega, log_nt_weights)
    P = algopy.expm(Q)
    
    # return the neg log likelihood
    return -get_log_likelihood(P, v, subs_counts)

def eval_grad_f(theta, *args):
    """
    compute the gradient of f in the forward mode of AD
    """
    theta = algopy.UTPM.init_jacobian(theta)
    retval = eval_f(theta, *args)
    return algopy.UTPM.extract_jacobian(retval)

def eval_hess_f(theta, *args):
    """
    compute the hessian of f in the forward mode of AD
    """
    n = len(theta)
    if n != 6:
        raise Exception(n)
    theta = algopy.UTPM.init_hessian(theta)
    retval = eval_f(theta, *args)
    return algopy.UTPM.extract_hessian(n, retval)


def main():
    code = g_code_mito
    stop = g_stop_mito

    # Precompute some numpy ndarrays
    # according to properties of DNA and the mitochondrial genetic code.
    all_codons = [c for codons in code for c in codons]
    codons = [c for codons in code[:-1] for c in codons]
    ts, tv = get_ts_tv(codons)
    syn, nonsyn = get_syn_nonsyn(code, codons)
    compo = get_compo(codons)
    asym_compo = get_asym_compo(codons)
    ham = get_hamming(codons)

    # Extract the codon counts and the substitution counts.
    # Then compute the empirical codon distribution and log codon counts.
    all_human_codons = dna_to_codon_ord(all_codons, g_human_mtdna)
    all_chimp_codons = dna_to_codon_ord(all_codons, g_chimp_mtdna)
    all_codon_counts, all_subs_counts = get_empirical_summary(
            64, all_human_codons, all_chimp_codons)
    if all_codon_counts[-len(stop):].tolist() != [0, 0, 0, 0]:
        raise Exception('unexpectedly saw stop codons')
    codon_counts = all_codon_counts[:len(codons)]
    subs_counts = all_subs_counts[:len(codons), :len(codons)]
    v = codon_counts / float(numpy.sum(codon_counts))
    log_counts = numpy.log(codon_counts)

    # predefine some plausible parameters but not the scaling parameter
    log_mu = 0
    log_kappa = 1
    log_omega = -3
    log_nt_weights = numpy.zeros(4)

    # get the rate matrix associated with the initial guess
    Q = get_Q(
            ts, tv, syn, nonsyn, compo, asym_compo,
            fixation_h,
            log_counts,
            log_mu, log_kappa, log_omega, log_nt_weights)

    # get the minimum expected number of substitutions between codons
    mu_empirical = get_lb_expected_subs(ham, subs_counts)
    mu_implied = -numpy.sum(numpy.diag(Q) * v)
    log_mu = math.log(mu_empirical) - math.log(mu_implied)

    # construct the initial guess vector
    theta = numpy.array([
        log_mu,
        log_kappa,
        log_omega,
        0,
        0,
        0,
        ])

    # construct the args to the neg log likelihood function
    fmin_args = (
            subs_counts, log_counts, v,
            fixation_h,
            ts, tv, syn, nonsyn, compo, asym_compo,
            )

    # do the search, using information about the gradient and hessian
    results = scipy.optimize.fmin_ncg(
            eval_f,
            theta,
            eval_grad_f,
            fhess_p=None,
            fhess=eval_hess_f,
            args=fmin_args,
            avextol=1e-05,
            epsilon=1.4901161193847656e-08,
            maxiter=100,
            full_output=True,
            disp=1,
            retall=0,
            callback=None,
            )

    # report a summary of the maximum likelihood search
    print(results)
    x = results[0]
    print(numpy.exp(x))


if __name__ == '__main__':
    main()

