# pylint: disable=line-too-long,invalid-name,redefined-builtin,too-many-lines
from __future__ import annotations
from typing import Tuple, Set, Dict, List, Optional, TYPE_CHECKING
import krpc.schema
from krpc.schema import KRPC_pb2
from krpc.types import TypeBase, ClassBase, WrappedClass, DocEnum
from krpc.event import Event
if TYPE_CHECKING:
    from krpc.services import Client


class AntennaState(DocEnum):
    """
    The state of an antenna. See SpaceCenter.Antenna#state.
    """
    deployed = 0, """
Antenna is fully deployed.
"""
    retracted = 1, """
Antenna is fully retracted.
"""
    deploying = 2, """
Antenna is being deployed.
"""
    retracting = 3, """
Antenna is being retracted.
"""
    broken = 4, """
Antenna is broken.
"""


class AutoStrutMode(DocEnum):
    """
    The state of an auto-strut. SpaceCenter.Part#autoStrutMode
    """
    off = 0, """
Off
"""
    root = 1, """
Root
"""
    heaviest = 2, """
Heaviest
"""
    grandparent = 3, """
Grandparent
"""
    force_root = 4, """
ForceRoot
"""
    force_heaviest = 5, """
ForceHeaviest
"""
    force_grandparent = 6, """
ForceGrandparent
"""


class CameraMode(DocEnum):
    """
    See SpaceCenter.Camera#mode.
    """
    automatic = 0, """
The camera is showing the active vessel, in "auto" mode.
"""
    free = 1, """
The camera is showing the active vessel, in "free" mode.
"""
    chase = 2, """
The camera is showing the active vessel, in "chase" mode.
"""
    locked = 3, """
The camera is showing the active vessel, in "locked" mode.
"""
    orbital = 4, """
The camera is showing the active vessel, in "orbital" mode.
"""
    iva = 5, """
The Intra-Vehicular Activity view is being shown.
"""
    map = 6, """
The map view is being shown.
"""


class CargoBayState(DocEnum):
    """
    The state of a cargo bay. See SpaceCenter.CargoBay#state.
    """
    open = 0, """
Cargo bay is fully open.
"""
    closed = 1, """
Cargo bay closed and locked.
"""
    opening = 2, """
Cargo bay is opening.
"""
    closing = 3, """
Cargo bay is closing.
"""


class CommLinkType(DocEnum):
    """
    The type of a communication link.
    See SpaceCenter.CommLink#type.
    """
    home = 0, """
Link is to a base station on Kerbin.
"""
    control = 1, """
Link is to a control source, for example a manned spacecraft.
"""
    relay = 2, """
Link is to a relay satellite.
"""


class ContractState(DocEnum):
    """
    The state of a contract. See SpaceCenter.Contract#state.
    """
    active = 0, """
The contract is active.
"""
    canceled = 1, """
The contract has been canceled.
"""
    completed = 2, """
The contract has been completed.
"""
    deadline_expired = 3, """
The deadline for the contract has expired.
"""
    declined = 4, """
The contract has been declined.
"""
    failed = 5, """
The contract has been failed.
"""
    generated = 6, """
The contract has been generated.
"""
    offered = 7, """
The contract has been offered to the player.
"""
    offer_expired = 8, """
The contract was offered to the player, but the offer expired.
"""
    withdrawn = 9, """
The contract has been withdrawn.
"""


class ControlInputMode(DocEnum):
    """
    See SpaceCenter.Control#inputMode.
    """
    additive = 0, """
Control inputs are added to the vessels current control inputs.
"""
    override = 1, """
Control inputs (when they are non-zero) override the vessels current control inputs.
"""


class ControlSource(DocEnum):
    """
    The control source of a vessel.
    See SpaceCenter.Control#source.
    """
    kerbal = 0, """
Vessel is controlled by a Kerbal.
"""
    probe = 1, """
Vessel is controlled by a probe core.
"""
    none = 2, """
Vessel is not controlled.
"""


class ControlState(DocEnum):
    """
    The control state of a vessel.
    See SpaceCenter.Control#state.
    """
    full = 0, """
Full controllable.
"""
    partial = 1, """
Partially controllable.
"""
    none = 2, """
Not controllable.
"""


class CrewMemberGender(DocEnum):
    """
    A crew member's gender.
    See SpaceCenter.CrewMember#gender.
    """
    male = 0, """
Male.
"""
    female = 1, """
Female.
"""


class CrewMemberType(DocEnum):
    """
    The type of a crew member.
    See SpaceCenter.CrewMember#type.
    """
    applicant = 0, """
An applicant for crew.
"""
    crew = 1, """
Rocket crew.
"""
    tourist = 2, """
A tourist.
"""
    unowned = 3, """
An unowned crew member.
"""


class DockingPortState(DocEnum):
    """
    The state of a docking port. See SpaceCenter.DockingPort#state.
    """
    ready = 0, """
The docking port is ready to dock to another docking port.
"""
    docked = 1, """
The docking port is docked to another docking port, or docked to
another part (from the VAB/SPH).
"""
    docking = 2, """
The docking port is very close to another docking port,
but has not docked. It is using magnetic force to acquire a solid dock.
"""
    undocking = 3, """
The docking port has just been undocked from another docking port,
and is disabled until it moves away by a sufficient distance
(SpaceCenter.DockingPort#reengageDistance).
"""
    shielded = 4, """
The docking port has a shield, and the shield is closed.
"""
    moving = 5, """
The docking ports shield is currently opening/closing.
"""


class DrainMode(DocEnum):
    """
    Resource drain mode.
    See SpaceCenter.ResourceDrain#drainMode.
    """
    part = 0, """
Drains from the parent part.
"""
    vessel = 1, """
Drains from all available parts.
"""


class EditorFacility(DocEnum):
    """
    Editor facility.
    See SpaceCenter.LaunchSite#editorFacility.
    """
    vab = 1, """
Vehicle Assembly Building.
"""
    sph = 2, """
Space Plane Hanger.
"""
    none = 0, """
None.
"""


class GameMode(DocEnum):
    """
    The game mode.
    Returned by SpaceCenter.GameMode
    """
    sandbox = 0, """
Sandbox mode.
"""
    career = 1, """
Career mode.
"""
    science = 2, """
Science career mode.
"""
    science_sandbox = 3, """
Science sandbox mode.
"""
    mission = 4, """
Mission mode.
"""
    mission_builder = 5, """
Mission builder mode.
"""
    scenario = 6, """
Scenario mode.
"""
    scenario_non_resumable = 7, """
Scenario mode that cannot be resumed.
"""


class LegState(DocEnum):
    """
    The state of a landing leg. See SpaceCenter.Leg#state.
    """
    deployed = 0, """
Landing leg is fully deployed.
"""
    retracted = 1, """
Landing leg is fully retracted.
"""
    deploying = 2, """
Landing leg is being deployed.
"""
    retracting = 3, """
Landing leg is being retracted.
"""
    broken = 4, """
Landing leg is broken.
"""


class MapFilterType(DocEnum):
    """
    The set of things that are visible in map mode.
    These may be combined with bitwise logic.
    """
    all = -1, """
Everything.
"""
    none = 0, """
Nothing.
"""
    debris = 1, """
Debris.
"""
    unknown = 2, """
Unknown.
"""
    space_objects = 4, """
SpaceObjects.
"""
    probes = 8, """
Probes.
"""
    rovers = 16, """
Rovers.
"""
    landers = 32, """
Landers.
"""
    ships = 64, """
Ships.
"""
    stations = 128, """
Stations.
"""
    bases = 256, """
Bases.
"""
    ev_as = 512, """
EVAs.
"""
    flags = 1024, """
Flags.
"""
    plane = 2048, """
Planes.
"""
    relay = 4096, """
Relays.
"""
    site = 8192, """
Launch Sites.
"""
    deployed_science_controller = 16384, """
Deployed Science Controllers.
"""


class MotorState(DocEnum):
    """
    The state of the motor on a powered wheel. See SpaceCenter.Wheel#motorState.
    """
    idle = 0, """
The motor is idle.
"""
    running = 1, """
The motor is running.
"""
    disabled = 2, """
The motor is disabled.
"""
    inoperable = 3, """
The motor is inoperable.
"""
    not_enough_resources = 4, """
The motor does not have enough resources to run.
"""


class ParachuteState(DocEnum):
    """
    The state of a parachute. See SpaceCenter.Parachute#state.
    """
    stowed = 0, """
The parachute is safely tucked away inside its housing.
"""
    armed = 1, """
The parachute is armed for deployment.
"""
    semi_deployed = 2, """
The parachute has been deployed and is providing some drag,
but is not fully deployed yet. (Stock parachutes only)
"""
    deployed = 3, """
The parachute is fully deployed.
"""
    cut = 4, """
The parachute has been cut.
"""


class RadiatorState(DocEnum):
    """
    The state of a radiator. SpaceCenter.Radiator#state
    """
    extended = 0, """
Radiator is fully extended.
"""
    retracted = 1, """
Radiator is fully retracted.
"""
    extending = 2, """
Radiator is being extended.
"""
    retracting = 3, """
Radiator is being retracted.
"""
    broken = 4, """
Radiator is broken.
"""


class ResourceConverterState(DocEnum):
    """
    The state of a resource converter. See SpaceCenter.ResourceConverter#state.
    """
    running = 0, """
Converter is running.
"""
    idle = 1, """
Converter is idle.
"""
    missing_resource = 2, """
Converter is missing a required resource.
"""
    storage_full = 3, """
No available storage for output resource.
"""
    capacity = 4, """
At preset resource capacity.
"""
    unknown = 5, """
Unknown state. Possible with modified resource converters.
In this case, check SpaceCenter.ResourceConverter#statusInfo for more information.
"""


class ResourceFlowMode(DocEnum):
    """
    The way in which a resource flows between parts. See SpaceCenter.Resources#flowMode.
    """
    vessel = 0, """
The resource flows to any part in the vessel. For example, electric charge.
"""
    stage = 1, """
The resource flows from parts in the first stage, followed by the second,
and so on. For example, mono-propellant.
"""
    adjacent = 2, """
The resource flows between adjacent parts within the vessel. For example,
liquid fuel or oxidizer.
"""
    none = 3, """
The resource does not flow. For example, solid fuel.
"""


class ResourceHarvesterState(DocEnum):
    """
    The state of a resource harvester. See SpaceCenter.ResourceHarvester#state.
    """
    deploying = 0, """
The drill is deploying.
"""
    deployed = 1, """
The drill is deployed and ready.
"""
    retracting = 2, """
The drill is retracting.
"""
    retracted = 3, """
The drill is retracted.
"""
    active = 4, """
The drill is running.
"""


class RosterStatus(DocEnum):
    """
    A crew member's roster status.
    See SpaceCenter.CrewMember#rosterStatus.
    """
    available = 0, """
Available.
"""
    assigned = 1, """
Assigned.
"""
    dead = 2, """
Dead.
"""
    missing = 3, """
Missing.
"""


class SASMode(DocEnum):
    """
    The behavior of the SAS auto-pilot. See SpaceCenter.AutoPilot#sASMode.
    """
    stability_assist = 0, """
Stability assist mode. Dampen out any rotation.
"""
    maneuver = 1, """
Point in the burn direction of the next maneuver node.
"""
    prograde = 2, """
Point in the prograde direction.
"""
    retrograde = 3, """
Point in the retrograde direction.
"""
    normal = 4, """
Point in the orbit normal direction.
"""
    anti_normal = 5, """
Point in the orbit anti-normal direction.
"""
    radial = 6, """
Point in the orbit radial direction.
"""
    anti_radial = 7, """
Point in the orbit anti-radial direction.
"""
    target = 8, """
Point in the direction of the current target.
"""
    anti_target = 9, """
Point away from the current target.
"""


class SolarPanelState(DocEnum):
    """
    The state of a solar panel. See SpaceCenter.SolarPanel#state.
    """
    extended = 0, """
Solar panel is fully extended.
"""
    retracted = 1, """
Solar panel is fully retracted.
"""
    extending = 2, """
Solar panel is being extended.
"""
    retracting = 3, """
Solar panel is being retracted.
"""
    broken = 4, """
Solar panel is broken.
"""


class SpeedMode(DocEnum):
    """
    The mode of the speed reported in the navball.
    See SpaceCenter.Control#speedMode.
    """
    orbit = 0, """
Speed is relative to the vessel's orbit.
"""
    surface = 1, """
Speed is relative to the surface of the body being orbited.
"""
    target = 2, """
Speed is relative to the current target.
"""


class SuitType(DocEnum):
    """
    A crew member's suit type.
    See SpaceCenter.CrewMember#suitType.
    """
    default = 0, """
Default.
"""
    vintage = 1, """
Vintage.
"""
    future = 2, """
Future.
"""
    slim = 3, """
Slim.
"""


class VesselSituation(DocEnum):
    """
    The situation a vessel is in.
    See SpaceCenter.Vessel#situation.
    """
    pre_launch = 0, """
Vessel is awaiting launch.
"""
    orbiting = 1, """
Vessel is orbiting a body.
"""
    sub_orbital = 2, """
Vessel is on a sub-orbital trajectory.
"""
    escaping = 3, """
Escaping.
"""
    flying = 4, """
Vessel is flying through an atmosphere.
"""
    landed = 5, """
Vessel is landed on the surface of a body.
"""
    splashed = 6, """
Vessel has splashed down in an ocean.
"""
    docked = 7, """
Vessel is docked to another.
"""


class VesselType(DocEnum):
    """
    The type of a vessel.
    See SpaceCenter.Vessel#type.
    """
    base = 0, """
Base.
"""
    debris = 1, """
Debris.
"""
    lander = 2, """
Lander.
"""
    plane = 3, """
Plane.
"""
    probe = 4, """
Probe.
"""
    relay = 5, """
Relay.
"""
    rover = 6, """
Rover.
"""
    ship = 7, """
Ship.
"""
    station = 8, """
Station.
"""
    space_object = 9, """
SpaceObject.
"""
    unknown = 10, """
Unknown.
"""
    eva = 11, """
EVA.
"""
    flag = 12, """
Flag.
"""
    deployed_science_controller = 13, """
DeployedScienceController.
"""
    deployed_science_part = 14, """
DeploedSciencePart.
"""
    dropped_part = 15, """
DroppedPart.
"""
    deployed_ground_part = 16, """
DeployedGroundPart.
"""


class WarpMode(DocEnum):
    """
    The time warp mode.
    Returned by SpaceCenter.WarpMode
    """
    rails = 0, """
Time warp is active, and in regular "on-rails" mode.
"""
    physics = 1, """
Time warp is active, and in physical time warp mode.
"""
    none = 2, """
Time warp is not active.
"""


class WheelState(DocEnum):
    """
    The state of a wheel. See SpaceCenter.Wheel#state.
    """
    deployed = 0, """
Wheel is fully deployed.
"""
    retracted = 1, """
Wheel is fully retracted.
"""
    deploying = 2, """
Wheel is being deployed.
"""
    retracting = 3, """
Wheel is being retracted.
"""
    broken = 4, """
Wheel is broken.
"""


class Alarm(ClassBase):
    """
    An alarm. Can be accessed using SpaceCenter#alarmManager.
    """
    @property
    def description(self) -> str:
        """
        Description of the alarm.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_Description",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    def _return_type_description(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_description(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_Description",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    @property
    def event_offset(self) -> float:
        """
        Seconds between the alarm going off and the event it references.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_EventOffset",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    def _return_type_event_offset(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_event_offset(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_EventOffset",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    @property
    def id(self) -> Optional[int]:
        """
        Unique identifier of the alarm.
        KSP destroys and recreates an alarm when it is edited.
        This id will remain constant between the old and new alarms.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_ID",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.uint32_type
        )

    def _return_type_id(self) -> TypeBase:
        return self._client._types.uint32_type

    def _build_call_id(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_ID",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.uint32_type
        )

    @property
    def time(self) -> float:
        """
        Time the alarm will trigger.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_Time",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    def _return_type_time(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_Time",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    @property
    def time_until(self) -> float:
        """
        Time until the alarm triggers.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_TimeUntil",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    def _return_type_time_until(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_until(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_TimeUntil",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.double_type
        )

    @property
    def title(self) -> str:
        """
        Title of the alarm
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    @property
    def type(self) -> str:
        """
        Type of alarm
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    def _return_type_type(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.string_type
        )

    @property
    def vessel(self) -> Optional[Vessel]:
        """
        Vessel the alarm references. ``None`` if it does not reference a vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Alarm_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    def _return_type_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Alarm_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Alarm")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )



class AlarmManager(ClassBase):
    """
    Alarm manager.
    Obtained by calling SpaceCenter#alarmManager.
    """
    @property
    def alarms(self) -> List[Alarm]:
        """
        A list of all alarms.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AlarmManager_get_Alarms",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AlarmManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Alarm"))
        )

    def _return_type_alarms(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Alarm"))

    def _build_call_alarms(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_get_Alarms",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AlarmManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Alarm"))
        )

    @classmethod
    def add_alarm(cls, time: float, title: str = 'Alarm', description: str = '') -> Alarm:
        """
        Create an alarm.

        :param time: Number of seconds from now that the alarm should trigger.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddAlarm",
            [time, title, description],
            ["time", "title", "description"],
            [self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_alarm(cls, time: float, title: str = 'Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddAlarm",
            [time, title, description],
            ["time", "title", "description"],
            [self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def add_apoapsis_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'Apoapsis Alarm', description: str = '') -> Alarm:
        """
        Create an alarm for the given vessel's next apoapsis.

        :param vessel: The vessel.

        :param offset: Time in seconds to offset the alarm by.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddApoapsisAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_apoapsis_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_apoapsis_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'Apoapsis Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddApoapsisAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def add_maneuver_node_alarm(cls, vessel: Vessel, node: Node, offset: float = 60.0, add_burn_time: bool = True, title: str = 'Maneuver Node Alarm', description: str = '') -> Alarm:
        """
        Create an alarm for the given vessel and maneuver node.

        :param vessel: The vessel.

        :param node: The maneuver node.

        :param offset: Time in seconds to offset the alarm by.

        :param add_burn_time: Whether the node's burn time should be included in the alarm.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddManeuverNodeAlarm",
            [vessel, node, offset, add_burn_time, title, description],
            ["vessel", "node", "offset", "add_burn_time", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type, self._client._types.bool_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_maneuver_node_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_maneuver_node_alarm(cls, vessel: Vessel, node: Node, offset: float = 60.0, add_burn_time: bool = True, title: str = 'Maneuver Node Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddManeuverNodeAlarm",
            [vessel, node, offset, add_burn_time, title, description],
            ["vessel", "node", "offset", "add_burn_time", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type, self._client._types.bool_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def add_periapsis_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'Periapsis Alarm', description: str = '') -> Alarm:
        """
        Create an alarm for the given vessel's next periapsis.

        :param vessel: The vessel.

        :param offset: Time in seconds to offset the alarm by.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddPeriapsisAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_periapsis_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_periapsis_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'Periapsis Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddPeriapsisAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def add_soi_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'SOI Change Alarm', description: str = '') -> Alarm:
        """
        Create an alarm for the given vessel's next sphere of influence change.

        :param vessel: The vessel.

        :param offset: Time in seconds to offset the alarm by.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddSOIAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_soi_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_soi_alarm(cls, vessel: Vessel, offset: float = 60.0, title: str = 'SOI Change Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddSOIAlarm",
            [vessel, offset, title, description],
            ["vessel", "offset", "title", "description"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type, self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def add_vessel_alarm(cls, time: float, vessel: Vessel, title: str = 'Vessel Alarm', description: str = '') -> Alarm:
        """
        Create an alarm linked to a vessel.

        :param time: Number of seconds from now that the alarm should trigger.

        :param vessel: Vessel to link the alarm to.

        :param title: Title for the alarm.

        :param description: Description for the alarm.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "AlarmManager_static_AddVesselAlarm",
            [time, vessel, title, description],
            ["time", "vessel", "title", "description"],
            [self._client._types.double_type, self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )

    @classmethod
    def _return_type_add_vessel_alarm(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "Alarm")

    @classmethod
    def _build_call_add_vessel_alarm(cls, time: float, vessel: Vessel, title: str = 'Vessel Alarm', description: str = '') -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "AlarmManager_static_AddVesselAlarm",
            [time, vessel, title, description],
            ["time", "vessel", "title", "description"],
            [self._client._types.double_type, self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.string_type, self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Alarm")
        )



class Antenna(ClassBase):
    """
    An antenna. Obtained by calling SpaceCenter.Part#antenna.
    """
    @property
    def allow_partial(self) -> bool:
        """
        Whether partial data transmission is permitted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_AllowPartial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @allow_partial.setter
    def allow_partial(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_set_AllowPartial",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Antenna"), self._client._types.bool_type],
            None
        )

    def _return_type_allow_partial(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_allow_partial(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_AllowPartial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @property
    def can_transmit(self) -> bool:
        """
        Whether data can be transmitted by this antenna.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_CanTransmit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    def _return_type_can_transmit(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_transmit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_CanTransmit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @property
    def combinable(self) -> bool:
        """
        Whether the antenna can be combined with other antennae on the vessel
        to boost the power.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_Combinable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    def _return_type_combinable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_combinable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_Combinable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @property
    def combinable_exponent(self) -> float:
        """
        Exponent used to calculate the combined power of multiple antennae on a vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_CombinableExponent",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    def _return_type_combinable_exponent(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_combinable_exponent(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_CombinableExponent",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    @property
    def deployable(self) -> bool:
        """
        Whether the antenna is deployable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    def _return_type_deployable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the antenna is deployed.

        Fixed antennas are always deployed.
        Returns an error if you try to deploy a fixed antenna.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Antenna"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.bool_type
        )

    @property
    def packet_interval(self) -> float:
        """
        Interval between sending packets in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_PacketInterval",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.float_type
        )

    def _return_type_packet_interval(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_packet_interval(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_PacketInterval",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.float_type
        )

    @property
    def packet_resource_cost(self) -> float:
        """
        Units of electric charge consumed per packet sent.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_PacketResourceCost",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    def _return_type_packet_resource_cost(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_packet_resource_cost(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_PacketResourceCost",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    @property
    def packet_size(self) -> float:
        """
        Amount of data sent per packet in Mits.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_PacketSize",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.float_type
        )

    def _return_type_packet_size(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_packet_size(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_PacketSize",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this antenna.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def power(self) -> float:
        """
        The power of the antenna.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_Power",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    def _return_type_power(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_power(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_Power",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.double_type
        )

    @property
    def state(self) -> AntennaState:
        """
        The current state of the antenna.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.enumeration_type("SpaceCenter", "AntennaState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "AntennaState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Antenna")],
            self._client._types.enumeration_type("SpaceCenter", "AntennaState")
        )

    def cancel(self) -> None:
        """
        Cancel current transmission of data.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_Cancel",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Antenna"), ],
            None
        )

    def _return_type_cancel(self) -> TypeBase:
        return None

    def _build_call_cancel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_Cancel",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Antenna"), ],
            None
        )

    def transmit(self) -> None:
        """
        Transmit data.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Antenna_Transmit",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Antenna"), ],
            None
        )

    def _return_type_transmit(self) -> TypeBase:
        return None

    def _build_call_transmit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Antenna_Transmit",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Antenna"), ],
            None
        )



class AutoPilot(ClassBase):
    """
    Provides basic auto-piloting utilities for a vessel.
    Created by calling SpaceCenter.Vessel#autoPilot.

    If a client engages the auto-pilot and then closes its connection to the server,
    the auto-pilot will be disengaged and its target reference frame, direction and roll
    reset to default.
    """
    @property
    def attenuation_angle(self) -> Tuple[float,float,float]:
        """
        The angle at which the autopilot considers the vessel to be pointing
        close to the target.
        This determines the midpoint of the target velocity attenuation function.
        A vector of three angles, in degrees, one for each of the pitch, roll and yaw axes.
        Defaults to 1° for each axis.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_AttenuationAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @attenuation_angle.setter
    def attenuation_angle(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_AttenuationAngle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_attenuation_angle(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_attenuation_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_AttenuationAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def auto_tune(self) -> bool:
        """
        Whether the rotation rate controllers PID parameters should be automatically tuned
        using the vessels moment of inertia and available torque. Defaults to ``True``.
        See SpaceCenter.AutoPilot#timeToPeak and SpaceCenter.AutoPilot#overshoot.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_AutoTune",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.bool_type
        )

    @auto_tune.setter
    def auto_tune(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_AutoTune",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.bool_type],
            None
        )

    def _return_type_auto_tune(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_auto_tune(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_AutoTune",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.bool_type
        )

    @property
    def deceleration_time(self) -> Tuple[float,float,float]:
        """
        The time the vessel should take to come to a stop pointing in the target direction.
        This determines the angular acceleration used to decelerate the vessel.
        A vector of three times, in seconds, one for each of the pitch, roll and yaw axes.
        Defaults to 5 seconds for each axis.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_DecelerationTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @deceleration_time.setter
    def deceleration_time(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_DecelerationTime",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_deceleration_time(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_deceleration_time(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_DecelerationTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def error(self) -> float:
        """
        The error, in degrees, between the direction the ship has been asked
        to point in and the direction it is pointing in. Throws an exception if the auto-pilot
        has not been engaged and SAS is not enabled or is in stability assist mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_Error",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    def _return_type_error(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_error(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_Error",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def heading_error(self) -> float:
        """
        The error, in degrees, between the vessels current and target heading.
        Throws an exception if the auto-pilot has not been engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_HeadingError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    def _return_type_heading_error(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_heading_error(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_HeadingError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def overshoot(self) -> Tuple[float,float,float]:
        """
        The target overshoot percentage used to autotune the PID controllers.
        A vector of three values, between 0 and 1, for each of the pitch, roll and yaw axes.
        Defaults to 0.01 for each axis.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_Overshoot",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @overshoot.setter
    def overshoot(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_Overshoot",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_overshoot(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_overshoot(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_Overshoot",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def pitch_error(self) -> float:
        """
        The error, in degrees, between the vessels current and target pitch.
        Throws an exception if the auto-pilot has not been engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_PitchError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    def _return_type_pitch_error(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_pitch_error(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_PitchError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def pitch_pid_gains(self) -> Tuple[float,float,float]:
        """
        Gains for the pitch PID controller.

        When SpaceCenter.AutoPilot#autoTune is true, these values are updated automatically,
        which will overwrite any manual changes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_PitchPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @pitch_pid_gains.setter
    def pitch_pid_gains(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_PitchPIDGains",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_pitch_pid_gains(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_pitch_pid_gains(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_PitchPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame for the target direction (SpaceCenter.AutoPilot#targetDirection).

        An error will be thrown if this property is set to a reference frame that rotates with
        the vessel being controlled, as it is impossible to rotate the vessel in such a
        reference frame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @reference_frame.setter
    def reference_frame(self, value: ReferenceFrame) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_ReferenceFrame",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            None
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def roll_error(self) -> float:
        """
        The error, in degrees, between the vessels current and target roll.
        Throws an exception if the auto-pilot has not been engaged or no target roll is set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_RollError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    def _return_type_roll_error(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_roll_error(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_RollError",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def roll_pid_gains(self) -> Tuple[float,float,float]:
        """
        Gains for the roll PID controller.

        When SpaceCenter.AutoPilot#autoTune is true, these values are updated automatically,
        which will overwrite any manual changes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_RollPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @roll_pid_gains.setter
    def roll_pid_gains(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_RollPIDGains",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_roll_pid_gains(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_roll_pid_gains(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_RollPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def roll_threshold(self) -> float:
        """
        The threshold at which the autopilot will try to match the target roll angle, if any.
        Defaults to 5 degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_RollThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.double_type
        )

    @roll_threshold.setter
    def roll_threshold(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_RollThreshold",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.double_type],
            None
        )

    def _return_type_roll_threshold(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_roll_threshold(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_RollThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.double_type
        )

    @property
    def sas(self) -> bool:
        """
        The state of SAS.

        Equivalent to SpaceCenter.Control#sAS
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_SAS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.bool_type
        )

    @sas.setter
    def sas(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_SAS",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.bool_type],
            None
        )

    def _return_type_sas(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_sas(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_SAS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.bool_type
        )

    @property
    def sas_mode(self) -> SASMode:
        """
        The current SpaceCenter.SASMode.
        These modes are equivalent to the mode buttons to the left of the navball that appear
        when SAS is enabled.

        Equivalent to SpaceCenter.Control#sASMode
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_SASMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.enumeration_type("SpaceCenter", "SASMode")
        )

    @sas_mode.setter
    def sas_mode(self, value: SASMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_SASMode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.enumeration_type("SpaceCenter", "SASMode")],
            None
        )

    def _return_type_sas_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "SASMode")

    def _build_call_sas_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_SASMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.enumeration_type("SpaceCenter", "SASMode")
        )

    @property
    def stopping_time(self) -> Tuple[float,float,float]:
        """
        The maximum amount of time that the vessel should need to come to a complete stop.
        This determines the maximum angular velocity of the vessel.
        A vector of three stopping times, in seconds, one for each of the pitch, roll
        and yaw axes. Defaults to 0.5 seconds for each axis.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_StoppingTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @stopping_time.setter
    def stopping_time(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_StoppingTime",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_stopping_time(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_stopping_time(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_StoppingTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def target_direction(self) -> Tuple[float,float,float]:
        """
        Direction vector corresponding to the target pitch and heading.
        This is in the reference frame specified by SpaceCenter.ReferenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_TargetDirection",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @target_direction.setter
    def target_direction(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_TargetDirection",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_target_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_target_direction(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_TargetDirection",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def target_heading(self) -> float:
        """
        The target heading, in degrees, between 0° and 360°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_TargetHeading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @target_heading.setter
    def target_heading(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_TargetHeading",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.float_type],
            None
        )

    def _return_type_target_heading(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_heading(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_TargetHeading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def target_pitch(self) -> float:
        """
        The target pitch, in degrees, between -90° and +90°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_TargetPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @target_pitch.setter
    def target_pitch(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_TargetPitch",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.float_type],
            None
        )

    def _return_type_target_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_TargetPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def target_roll(self) -> float:
        """
        The target roll, in degrees. ``NaN`` if no target roll is set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_TargetRoll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @target_roll.setter
    def target_roll(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_TargetRoll",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.float_type],
            None
        )

    def _return_type_target_roll(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_roll(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_TargetRoll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.float_type
        )

    @property
    def time_to_peak(self) -> Tuple[float,float,float]:
        """
        The target time to peak used to autotune the PID controllers.
        A vector of three times, in seconds, for each of the pitch, roll and yaw axes.
        Defaults to 3 seconds for each axis.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_TimeToPeak",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @time_to_peak.setter
    def time_to_peak(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_TimeToPeak",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_time_to_peak(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_time_to_peak(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_TimeToPeak",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def yaw_pid_gains(self) -> Tuple[float,float,float]:
        """
        Gains for the yaw PID controller.

        When SpaceCenter.AutoPilot#autoTune is true, these values are updated automatically,
        which will overwrite any manual changes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_get_YawPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @yaw_pid_gains.setter
    def yaw_pid_gains(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_set_YawPIDGains",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_yaw_pid_gains(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_yaw_pid_gains(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_get_YawPIDGains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def disengage(self) -> None:
        """
        Disengage the auto-pilot.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_Disengage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )

    def _return_type_disengage(self) -> TypeBase:
        return None

    def _build_call_disengage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_Disengage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )

    def engage(self) -> None:
        """
        Engage the auto-pilot.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_Engage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )

    def _return_type_engage(self) -> TypeBase:
        return None

    def _build_call_engage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_Engage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )

    def target_pitch_and_heading(self, pitch: float, heading: float) -> None:
        """
        Set target pitch and heading angles.

        :param pitch: Target pitch angle, in degrees between -90° and +90°.

        :param heading: Target heading angle, in degrees between 0° and 360°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_TargetPitchAndHeading",
            [self, pitch, heading],
            ["self", "pitch", "heading"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.float_type, self._client._types.float_type],
            None
        )

    def _return_type_target_pitch_and_heading(self) -> TypeBase:
        return None

    def _build_call_target_pitch_and_heading(self, pitch: float, heading: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_TargetPitchAndHeading",
            [self, pitch, heading],
            ["self", "pitch", "heading"],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), self._client._types.float_type, self._client._types.float_type],
            None
        )

    def wait(self) -> None:
        """
        Blocks until the vessel is pointing in the target direction and has
        the target roll (if set). Throws an exception if the auto-pilot has not been engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "AutoPilot_Wait",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )

    def _return_type_wait(self) -> TypeBase:
        return None

    def _build_call_wait(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "AutoPilot_Wait",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "AutoPilot"), ],
            None
        )



class Camera(ClassBase):
    """
    Controls the game's camera.
    Obtained by calling SpaceCenter#camera.
    """
    @property
    def default_distance(self) -> float:
        """
        Default distance from the camera to the subject, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_DefaultDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    def _return_type_default_distance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_default_distance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_DefaultDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def distance(self) -> float:
        """
        The distance from the camera to the subject, in meters.
        A value between SpaceCenter.Camera#minDistance and SpaceCenter.Camera#maxDistance.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_Distance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @distance.setter
    def distance(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_Distance",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.float_type],
            None
        )

    def _return_type_distance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_distance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_Distance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def focussed_body(self) -> Optional[CelestialBody]:
        """
        In map mode, the celestial body that the camera is focussed on.
        Returns ``None`` if the camera is not focussed on a celestial body.
        Returns an error is the camera is not in map mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_FocussedBody",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @focussed_body.setter
    def focussed_body(self, value: CelestialBody) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_FocussedBody",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.class_type("SpaceCenter", "CelestialBody")],
            None
        )

    def _return_type_focussed_body(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CelestialBody")

    def _build_call_focussed_body(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_FocussedBody",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @property
    def focussed_node(self) -> Optional[Node]:
        """
        In map mode, the maneuver node that the camera is focussed on.
        Returns ``None`` if the camera is not focussed on a maneuver node.
        Returns an error is the camera is not in map mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_FocussedNode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "Node")
        )

    @focussed_node.setter
    def focussed_node(self, value: Node) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_FocussedNode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.class_type("SpaceCenter", "Node")],
            None
        )

    def _return_type_focussed_node(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Node")

    def _build_call_focussed_node(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_FocussedNode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "Node")
        )

    @property
    def focussed_vessel(self) -> Optional[Vessel]:
        """
        In map mode, the vessel that the camera is focussed on.
        Returns ``None`` if the camera is not focussed on a vessel.
        Returns an error is the camera is not in map mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_FocussedVessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @focussed_vessel.setter
    def focussed_vessel(self, value: Vessel) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_FocussedVessel",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.class_type("SpaceCenter", "Vessel")],
            None
        )

    def _return_type_focussed_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_focussed_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_FocussedVessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @property
    def heading(self) -> float:
        """
        The heading of the camera, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_Heading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @heading.setter
    def heading(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_Heading",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.float_type],
            None
        )

    def _return_type_heading(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_heading(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_Heading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def max_distance(self) -> float:
        """
        Maximum distance from the camera to the subject, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_MaxDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    def _return_type_max_distance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_distance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_MaxDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def max_pitch(self) -> float:
        """
        The maximum pitch of the camera.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_MaxPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    def _return_type_max_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_MaxPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def min_distance(self) -> float:
        """
        Minimum distance from the camera to the subject, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_MinDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    def _return_type_min_distance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_min_distance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_MinDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def min_pitch(self) -> float:
        """
        The minimum pitch of the camera.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_MinPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    def _return_type_min_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_min_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_MinPitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @property
    def mode(self) -> CameraMode:
        """
        The current mode of the camera.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_Mode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.enumeration_type("SpaceCenter", "CameraMode")
        )

    @mode.setter
    def mode(self, value: CameraMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_Mode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.enumeration_type("SpaceCenter", "CameraMode")],
            None
        )

    def _return_type_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "CameraMode")

    def _build_call_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_Mode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.enumeration_type("SpaceCenter", "CameraMode")
        )

    @property
    def pitch(self) -> float:
        """
        The pitch of the camera, in degrees.
        A value between SpaceCenter.Camera#minPitch and SpaceCenter.Camera#maxPitch
        """
        return self._client._invoke(
            "SpaceCenter",
            "Camera_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )

    @pitch.setter
    def pitch(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Camera_set_Pitch",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Camera"), self._client._types.float_type],
            None
        )

    def _return_type_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Camera_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Camera")],
            self._client._types.float_type
        )



class CargoBay(ClassBase):
    """
    A cargo bay. Obtained by calling SpaceCenter.Part#cargoBay.
    """
    @property
    def open(self) -> bool:
        """
        Whether the cargo bay is open.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CargoBay_get_Open",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.bool_type
        )

    @open.setter
    def open(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CargoBay_set_Open",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CargoBay"), self._client._types.bool_type],
            None
        )

    def _return_type_open(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_open(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CargoBay_get_Open",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this cargo bay.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CargoBay_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CargoBay_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> CargoBayState:
        """
        The state of the cargo bay.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CargoBay_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.enumeration_type("SpaceCenter", "CargoBayState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "CargoBayState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CargoBay_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CargoBay")],
            self._client._types.enumeration_type("SpaceCenter", "CargoBayState")
        )



class CelestialBody(ClassBase):
    """
    Represents a celestial body (such as a planet or moon).
    See SpaceCenter#bodies.
    """
    @property
    def atmosphere_depth(self) -> float:
        """
        The depth of the atmosphere, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_AtmosphereDepth",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_atmosphere_depth(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_atmosphere_depth(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_AtmosphereDepth",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def biomes(self) -> Set[str]:
        """
        The biomes present on this body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_Biomes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.set_type(self._client._types.string_type)
        )

    def _return_type_biomes(self) -> TypeBase:
        return self._client._types.set_type(self._client._types.string_type)

    def _build_call_biomes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_Biomes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.set_type(self._client._types.string_type)
        )

    @property
    def equatorial_radius(self) -> float:
        """
        The equatorial radius of the body, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_EquatorialRadius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_equatorial_radius(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_equatorial_radius(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_EquatorialRadius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def flying_high_altitude_threshold(self) -> float:
        """
        The altitude, in meters, above which a vessel is considered to be
        flying "high" when doing science.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_FlyingHighAltitudeThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.float_type
        )

    def _return_type_flying_high_altitude_threshold(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_flying_high_altitude_threshold(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_FlyingHighAltitudeThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.float_type
        )

    @property
    def gravitational_parameter(self) -> float:
        """
        The `standard gravitational parameter <https://en.wikipedia.org/wiki/Standard_gravitational_parameter>`_ of the body in ``m^3s^{-2}``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_GravitationalParameter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_gravitational_parameter(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_gravitational_parameter(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_GravitationalParameter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def has_atmosphere(self) -> bool:
        """
        ``True`` if the body has an atmosphere.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_HasAtmosphere",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    def _return_type_has_atmosphere(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_atmosphere(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_HasAtmosphere",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    @property
    def has_atmospheric_oxygen(self) -> bool:
        """
        ``True`` if there is oxygen in the atmosphere, required for air-breathing engines.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_HasAtmosphericOxygen",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    def _return_type_has_atmospheric_oxygen(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_atmospheric_oxygen(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_HasAtmosphericOxygen",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    @property
    def has_solid_surface(self) -> bool:
        """
        Whether or not the body has a solid surface.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_HasSolidSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    def _return_type_has_solid_surface(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_solid_surface(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_HasSolidSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    @property
    def initial_rotation(self) -> float:
        """
        The initial rotation angle of the body (at UT 0), in radians.
        A value between 0 and ``2\pi``
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_InitialRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_initial_rotation(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_initial_rotation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_InitialRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def is_star(self) -> bool:
        """
        Whether or not the body is a star.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_IsStar",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    def _return_type_is_star(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_star(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_IsStar",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.bool_type
        )

    @property
    def mass(self) -> float:
        """
        The mass of the body, in kilograms.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_mass(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def name(self) -> str:
        """
        The name of the body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.string_type
        )

    @property
    def non_rotating_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to this celestial body, and
        orientated in a fixed direction (it does not rotate with the body).
         * The origin is at the center of the body.
         * The axes do not rotate.
         * The x-axis points in an arbitrary direction through the
           equator.
         * The y-axis points from the center of the body towards
           the north pole.
         * The z-axis points in an arbitrary direction through the
           equator.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_NonRotatingReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_non_rotating_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_non_rotating_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_NonRotatingReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def orbit(self) -> Optional[Orbit]:
        """
        The orbit of the body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    def _return_type_orbit(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Orbit")

    def _build_call_orbit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    @property
    def orbital_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to this celestial body, but
        orientated with the body's orbital prograde/normal/radial directions.
         * The origin is at the center of the body.
         * The axes rotate with the orbital prograde/normal/radial
           directions.
         * The x-axis points in the orbital anti-radial direction.
         * The y-axis points in the orbital prograde direction.
         * The z-axis points in the orbital normal direction.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_orbital_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_orbital_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the celestial body.
         * The origin is at the center of the body.
         * The axes rotate with the body.
         * The x-axis points from the center of the body
           towards the intersection of the prime meridian and equator (the
           position at 0° longitude, 0° latitude).
         * The y-axis points from the center of the body
           towards the north pole.
         * The z-axis points from the center of the body
           towards the equator at 90°E longitude.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def rotation_angle(self) -> float:
        """
        The current rotation angle of the body, in radians.
        A value between 0 and ``2\pi``
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_RotationAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_rotation_angle(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_rotation_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_RotationAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def rotational_period(self) -> float:
        """
        The sidereal rotational period of the body, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_RotationalPeriod",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_rotational_period(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_rotational_period(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_RotationalPeriod",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def rotational_speed(self) -> float:
        """
        The rotational speed of the body, in radians per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_RotationalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_rotational_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_rotational_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_RotationalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def satellites(self) -> List[CelestialBody]:
        """
        A list of celestial bodies that are in orbit around this celestial body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_Satellites",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CelestialBody"))
        )

    def _return_type_satellites(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CelestialBody"))

    def _build_call_satellites(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_Satellites",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CelestialBody"))
        )

    @property
    def space_high_altitude_threshold(self) -> float:
        """
        The altitude, in meters, above which a vessel is considered to be
        in "high" space when doing science.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_SpaceHighAltitudeThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.float_type
        )

    def _return_type_space_high_altitude_threshold(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_space_high_altitude_threshold(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_SpaceHighAltitudeThreshold",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.float_type
        )

    @property
    def sphere_of_influence(self) -> float:
        """
        The radius of the sphere of influence of the body, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_SphereOfInfluence",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_sphere_of_influence(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_sphere_of_influence(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_SphereOfInfluence",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    @property
    def surface_gravity(self) -> float:
        """
        The acceleration due to gravity at sea level (mean altitude) on the body,
        in ``m/s^2``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_get_SurfaceGravity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def _return_type_surface_gravity(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_surface_gravity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_get_SurfaceGravity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            self._client._types.double_type
        )

    def altitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> float:
        """
        The altitude, in meters, of the given position in the given reference frame.

        :param position: Position as a vector.

        :param reference_frame: Reference frame for the position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_AltitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def _return_type_altitude_at_position(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_altitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_AltitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def angular_velocity(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The angular velocity of the body in the specified reference frame.

        :returns: The angular velocity as a vector. The magnitude of the vector is the rotational
                  speed of the body, in radians per second. The direction of the vector indicates the axis
                  of rotation, using the right-hand rule.

        :param reference_frame: The reference frame the returned
                                angular velocity is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_AngularVelocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_angular_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_angular_velocity(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_AngularVelocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def atmospheric_density_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> float:
        """
        The atmospheric density at the given position, in ``kg/m^3``,
        in the given reference frame.

        :param position: The position vector at which to measure the density.

        :param reference_frame: Reference frame that the position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_AtmosphericDensityAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def _return_type_atmospheric_density_at_position(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_atmospheric_density_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_AtmosphericDensityAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def bedrock_height(self, latitude: float, longitude: float) -> float:
        """
        The height of the surface relative to mean sea level, in meters,
        at the given position. When over water, this is the height
        of the sea-bed and is therefore  negative value.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_BedrockHeight",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_bedrock_height(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_bedrock_height(self, latitude: float, longitude: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_BedrockHeight",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.double_type
        )

    def bedrock_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the surface at the given latitude and longitude, in the given
        reference frame. When over water, this is the position at the bottom of the sea-bed.

        :returns: Position as a vector.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.

        :param reference_frame: Reference frame for the returned position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_BedrockPosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_bedrock_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_bedrock_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_BedrockPosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def biome_at(self, latitude: float, longitude: float) -> str:
        """
        The biome at the given latitude and longitude, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_BiomeAt",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.string_type
        )

    def _return_type_biome_at(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_biome_at(self, latitude: float, longitude: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_BiomeAt",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.string_type
        )

    def density_at(self, altitude: float) -> float:
        """
        Gets the air density, in ``kg/m^3``, for the specified
        altitude above sea level, in meters.

        This is an approximation, because actual calculations, taking sun exposure into account
        to compute air temperature, require us to know the exact point on the body where the
        density is to be computed (knowing the altitude is not enough).
        However, the difference is small for high altitudes, so it makes very little difference
        for trajectory prediction.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_DensityAt",
            [self, altitude],
            ["self", "altitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_density_at(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_density_at(self, altitude: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_DensityAt",
            [self, altitude],
            ["self", "altitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type],
            self._client._types.double_type
        )

    def direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction in which the north pole of the celestial body is pointing,
        in the specified reference frame.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def latitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> float:
        """
        The latitude of the given position, in the given reference frame.

        :param position: Position as a vector.

        :param reference_frame: Reference frame for the position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_LatitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def _return_type_latitude_at_position(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_latitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_LatitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def longitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> float:
        """
        The longitude of the given position, in the given reference frame.

        :param position: Position as a vector.

        :param reference_frame: Reference frame for the position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_LongitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def _return_type_longitude_at_position(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_longitude_at_position(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_LongitudeAtPosition",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def msl_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position at mean sea level at the given latitude and longitude,
        in the given reference frame.

        :returns: Position as a vector.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.

        :param reference_frame: Reference frame for the returned position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_MSLPosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_msl_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_msl_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_MSLPosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the center of the body, in the specified reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def position_at_altitude(self, latitude: float, longitude: float, altitude: float, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position at the given latitude, longitude and altitude, in the given reference frame.

        :returns: Position as a vector.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.

        :param altitude: Altitude in meters above sea level.

        :param reference_frame: Reference frame for the returned position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_PositionAtAltitude",
            [self, latitude, longitude, altitude, reference_frame],
            ["self", "latitude", "longitude", "altitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position_at_altitude(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position_at_altitude(self, latitude: float, longitude: float, altitude: float, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_PositionAtAltitude",
            [self, latitude, longitude, altitude, reference_frame],
            ["self", "latitude", "longitude", "altitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def pressure_at(self, altitude: float) -> float:
        """
        Gets the air pressure, in Pascals, for the specified
        altitude above sea level, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_PressureAt",
            [self, altitude],
            ["self", "altitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_pressure_at(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_pressure_at(self, altitude: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_PressureAt",
            [self, altitude],
            ["self", "altitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type],
            self._client._types.double_type
        )

    def rotation(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float,float]:
        """
        The rotation of the body, in the specified reference frame.

        :returns: The rotation as a quaternion of the form ``(x, y, z, w)``.

        :param reference_frame: The reference frame that the returned
                                rotation is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_rotation(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_rotation(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def surface_height(self, latitude: float, longitude: float) -> float:
        """
        The height of the surface relative to mean sea level, in meters,
        at the given position. When over water this is equal to 0.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_SurfaceHeight",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_surface_height(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_surface_height(self, latitude: float, longitude: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_SurfaceHeight",
            [self, latitude, longitude],
            ["self", "latitude", "longitude"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type],
            self._client._types.double_type
        )

    def surface_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the surface at the given latitude and longitude, in the given
        reference frame. When over water, this is the position of the surface of the water.

        :returns: Position as a vector.

        :param latitude: Latitude in degrees.

        :param longitude: Longitude in degrees.

        :param reference_frame: Reference frame for the returned position vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_SurfacePosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_surface_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_surface_position(self, latitude: float, longitude: float, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_SurfacePosition",
            [self, latitude, longitude, reference_frame],
            ["self", "latitude", "longitude", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def temperature_at(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> float:
        """
        The temperature on the body at the given position, in the given reference frame.

        :param position: Position as a vector.

        :param reference_frame: The reference frame that the position is in.

        This calculation is performed using the bodies current position, which means that
        the value could be wrong if you want to know the temperature in the far future.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_TemperatureAt",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def _return_type_temperature_at(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_temperature_at(self, position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_TemperatureAt",
            [self, position, reference_frame],
            ["self", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.double_type
        )

    def velocity(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The linear velocity of the body, in the specified reference frame.

        :returns: The velocity as a vector. The vector points in the direction of travel,
                  and its magnitude is the speed of the body in meters per second.

        :param reference_frame: The reference frame that the returned
                                velocity vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CelestialBody_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_velocity(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CelestialBody_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class CommLink(ClassBase):
    """
    Represents a communication node in the network. For example, a vessel or the KSC.
    """
    @property
    def end(self) -> CommNode:
        """
        Start point of the link.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommLink_get_End",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.class_type("SpaceCenter", "CommNode")
        )

    def _return_type_end(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CommNode")

    def _build_call_end(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommLink_get_End",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.class_type("SpaceCenter", "CommNode")
        )

    @property
    def signal_strength(self) -> float:
        """
        Signal strength of the link.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommLink_get_SignalStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.double_type
        )

    def _return_type_signal_strength(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_signal_strength(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommLink_get_SignalStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.double_type
        )

    @property
    def start(self) -> CommNode:
        """
        Start point of the link.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommLink_get_Start",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.class_type("SpaceCenter", "CommNode")
        )

    def _return_type_start(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CommNode")

    def _build_call_start(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommLink_get_Start",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.class_type("SpaceCenter", "CommNode")
        )

    @property
    def type(self) -> CommLinkType:
        """
        The type of link.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommLink_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.enumeration_type("SpaceCenter", "CommLinkType")
        )

    def _return_type_type(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "CommLinkType")

    def _build_call_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommLink_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommLink")],
            self._client._types.enumeration_type("SpaceCenter", "CommLinkType")
        )



class CommNode(ClassBase):
    """
    Represents a communication node in the network. For example, a vessel or the KSC.
    """
    @property
    def is_control_point(self) -> bool:
        """
        Whether the communication node is a control point, for example a manned vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommNode_get_IsControlPoint",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    def _return_type_is_control_point(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_control_point(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommNode_get_IsControlPoint",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    @property
    def is_home(self) -> bool:
        """
        Whether the communication node is on Kerbin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommNode_get_IsHome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    def _return_type_is_home(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_home(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommNode_get_IsHome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    @property
    def is_vessel(self) -> bool:
        """
        Whether the communication node is a vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommNode_get_IsVessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    def _return_type_is_vessel(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommNode_get_IsVessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.bool_type
        )

    @property
    def name(self) -> str:
        """
        Name of the communication node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommNode_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommNode_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.string_type
        )

    @property
    def vessel(self) -> Vessel:
        """
        The vessel for this communication node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CommNode_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    def _return_type_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CommNode_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CommNode")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )



class Comms(ClassBase):
    """
    Used to interact with CommNet for a given vessel.
    Obtained by calling SpaceCenter.Vessel#comms.
    """
    @property
    def can_communicate(self) -> bool:
        """
        Whether the vessel can communicate with KSC.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_CanCommunicate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.bool_type
        )

    def _return_type_can_communicate(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_communicate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_CanCommunicate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.bool_type
        )

    @property
    def can_transmit_science(self) -> bool:
        """
        Whether the vessel can transmit science data to KSC.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_CanTransmitScience",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.bool_type
        )

    def _return_type_can_transmit_science(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_transmit_science(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_CanTransmitScience",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.bool_type
        )

    @property
    def control_path(self) -> List[CommLink]:
        """
        The communication path used to control the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_ControlPath",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CommLink"))
        )

    def _return_type_control_path(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CommLink"))

    def _build_call_control_path(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_ControlPath",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CommLink"))
        )

    @property
    def power(self) -> float:
        """
        The combined power of all active antennae on the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_Power",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )

    def _return_type_power(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_power(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_Power",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )

    @property
    def signal_delay(self) -> float:
        """
        Signal delay to KSC in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_SignalDelay",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )

    def _return_type_signal_delay(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_signal_delay(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_SignalDelay",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )

    @property
    def signal_strength(self) -> float:
        """
        Signal strength to KSC.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Comms_get_SignalStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )

    def _return_type_signal_strength(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_signal_strength(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Comms_get_SignalStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Comms")],
            self._client._types.double_type
        )



class Contract(ClassBase):
    """
    A contract. Can be accessed using SpaceCenter#contractManager.
    """
    @property
    def active(self) -> bool:
        """
        Whether the contract is active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def can_be_canceled(self) -> bool:
        """
        Whether the contract can be canceled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_CanBeCanceled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_can_be_canceled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_be_canceled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_CanBeCanceled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def can_be_declined(self) -> bool:
        """
        Whether the contract can be declined.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_CanBeDeclined",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_can_be_declined(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_be_declined(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_CanBeDeclined",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def can_be_failed(self) -> bool:
        """
        Whether the contract can be failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_CanBeFailed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_can_be_failed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_be_failed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_CanBeFailed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def description(self) -> str:
        """
        Description of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Description",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def _return_type_description(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_description(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Description",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    @property
    def failed(self) -> bool:
        """
        Whether the contract has been failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Failed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_failed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_failed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Failed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def funds_advance(self) -> float:
        """
        Funds received when accepting the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_FundsAdvance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_funds_advance(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds_advance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_FundsAdvance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def funds_completion(self) -> float:
        """
        Funds received on completion of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_FundsCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_funds_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_FundsCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def funds_failure(self) -> float:
        """
        Funds lost if the contract is failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_FundsFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_funds_failure(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds_failure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_FundsFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def keywords(self) -> List[str]:
        """
        Keywords for the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Keywords",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_keywords(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_keywords(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Keywords",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def notes(self) -> str:
        """
        Notes for the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Notes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def _return_type_notes(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_notes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Notes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    @property
    def parameters(self) -> List[ContractParameter]:
        """
        Parameters for the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Parameters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))
        )

    def _return_type_parameters(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))

    def _build_call_parameters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Parameters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))
        )

    @property
    def read(self) -> bool:
        """
        Whether the contract has been read.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Read",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_read(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_read(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Read",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def reputation_completion(self) -> float:
        """
        Reputation gained on completion of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_ReputationCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_reputation_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_reputation_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_ReputationCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def reputation_failure(self) -> float:
        """
        Reputation lost if the contract is failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_ReputationFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_reputation_failure(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_reputation_failure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_ReputationFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def science_completion(self) -> float:
        """
        Science gained on completion of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_ScienceCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    def _return_type_science_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_science_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_ScienceCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.double_type
        )

    @property
    def seen(self) -> bool:
        """
        Whether the contract has been seen.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Seen",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    def _return_type_seen(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_seen(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Seen",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.bool_type
        )

    @property
    def state(self) -> ContractState:
        """
        State of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.enumeration_type("SpaceCenter", "ContractState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ContractState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.enumeration_type("SpaceCenter", "ContractState")
        )

    @property
    def synopsis(self) -> str:
        """
        Synopsis for the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Synopsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def _return_type_synopsis(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_synopsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Synopsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    @property
    def title(self) -> str:
        """
        Title of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    @property
    def type(self) -> str:
        """
        Type of the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def _return_type_type(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Contract")],
            self._client._types.string_type
        )

    def accept(self) -> None:
        """
        Accept an offered contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_Accept",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )

    def _return_type_accept(self) -> TypeBase:
        return None

    def _build_call_accept(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_Accept",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )

    def cancel(self) -> None:
        """
        Cancel an active contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_Cancel",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )

    def _return_type_cancel(self) -> TypeBase:
        return None

    def _build_call_cancel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_Cancel",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )

    def decline(self) -> None:
        """
        Decline an offered contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Contract_Decline",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )

    def _return_type_decline(self) -> TypeBase:
        return None

    def _build_call_decline(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Contract_Decline",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Contract"), ],
            None
        )



class ContractManager(ClassBase):
    """
    Contracts manager.
    Obtained by calling SpaceCenter#contractManager.
    """
    @property
    def active_contracts(self) -> List[Contract]:
        """
        A list of all active contracts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_ActiveContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    def _return_type_active_contracts(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))

    def _build_call_active_contracts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_ActiveContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    @property
    def all_contracts(self) -> List[Contract]:
        """
        A list of all contracts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_AllContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    def _return_type_all_contracts(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))

    def _build_call_all_contracts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_AllContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    @property
    def completed_contracts(self) -> List[Contract]:
        """
        A list of all completed contracts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_CompletedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    def _return_type_completed_contracts(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))

    def _build_call_completed_contracts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_CompletedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    @property
    def failed_contracts(self) -> List[Contract]:
        """
        A list of all failed contracts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_FailedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    def _return_type_failed_contracts(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))

    def _build_call_failed_contracts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_FailedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    @property
    def offered_contracts(self) -> List[Contract]:
        """
        A list of all offered, but unaccepted, contracts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_OfferedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    def _return_type_offered_contracts(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))

    def _build_call_offered_contracts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_OfferedContracts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Contract"))
        )

    @property
    def types(self) -> Set[str]:
        """
        A list of all contract types.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractManager_get_Types",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.set_type(self._client._types.string_type)
        )

    def _return_type_types(self) -> TypeBase:
        return self._client._types.set_type(self._client._types.string_type)

    def _build_call_types(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractManager_get_Types",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractManager")],
            self._client._types.set_type(self._client._types.string_type)
        )



class ContractParameter(ClassBase):
    """
    A contract parameter. See SpaceCenter.Contract#parameters.
    """
    @property
    def children(self) -> List[ContractParameter]:
        """
        Child contract parameters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Children",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))
        )

    def _return_type_children(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))

    def _build_call_children(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Children",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ContractParameter"))
        )

    @property
    def completed(self) -> bool:
        """
        Whether the parameter has been completed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Completed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    def _return_type_completed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_completed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Completed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    @property
    def failed(self) -> bool:
        """
        Whether the parameter has been failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Failed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    def _return_type_failed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_failed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Failed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    @property
    def funds_completion(self) -> float:
        """
        Funds received on completion of the contract parameter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_FundsCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    def _return_type_funds_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_FundsCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    @property
    def funds_failure(self) -> float:
        """
        Funds lost if the contract parameter is failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_FundsFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    def _return_type_funds_failure(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds_failure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_FundsFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    @property
    def notes(self) -> str:
        """
        Notes for the parameter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Notes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.string_type
        )

    def _return_type_notes(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_notes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Notes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.string_type
        )

    @property
    def optional(self) -> bool:
        """
        Whether the contract parameter is optional.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Optional",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    def _return_type_optional(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_optional(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Optional",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.bool_type
        )

    @property
    def reputation_completion(self) -> float:
        """
        Reputation gained on completion of the contract parameter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_ReputationCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    def _return_type_reputation_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_reputation_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_ReputationCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    @property
    def reputation_failure(self) -> float:
        """
        Reputation lost if the contract parameter is failed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_ReputationFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    def _return_type_reputation_failure(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_reputation_failure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_ReputationFailure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    @property
    def science_completion(self) -> float:
        """
        Science gained on completion of the contract parameter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_ScienceCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    def _return_type_science_completion(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_science_completion(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_ScienceCompletion",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.double_type
        )

    @property
    def title(self) -> str:
        """
        Title of the parameter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ContractParameter_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ContractParameter_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ContractParameter")],
            self._client._types.string_type
        )



class Control(ClassBase):
    """
    Used to manipulate the controls of a vessel. This includes adjusting the
    throttle, enabling/disabling systems such as SAS and RCS, or altering the
    direction in which the vessel is pointing.
    Obtained by calling SpaceCenter.Vessel#control.

    Control inputs (such as pitch, yaw and roll) are zeroed when all clients
    that have set one or more of these inputs are no longer connected.
    """
    @property
    def abort(self) -> bool:
        """
        The state of the abort action group.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Abort",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @abort.setter
    def abort(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Abort",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_abort(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_abort(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Abort",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def antennas(self) -> bool:
        """
        Returns whether all antennas on the vessel are deployed,
        and sets the deployment state of all antennas.
        See SpaceCenter.Antenna#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Antennas",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @antennas.setter
    def antennas(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Antennas",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_antennas(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_antennas(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Antennas",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def brakes(self) -> bool:
        """
        The state of the wheel brakes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Brakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @brakes.setter
    def brakes(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Brakes",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_brakes(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_brakes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Brakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def cargo_bays(self) -> bool:
        """
        Returns whether any of the cargo bays on the vessel are open,
        and sets the open state of all cargo bays.
        See SpaceCenter.CargoBay#open.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CargoBays",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @cargo_bays.setter
    def cargo_bays(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_CargoBays",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_cargo_bays(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_cargo_bays(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CargoBays",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def current_stage(self) -> int:
        """
        The current stage of the vessel. Corresponds to the stage number in
        the in-game UI.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CurrentStage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.sint32_type
        )

    def _return_type_current_stage(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_current_stage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CurrentStage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.sint32_type
        )

    @property
    def custom_axis01(self) -> float:
        """
        The state of CustomAxis01.
        A value between -1 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CustomAxis01",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @custom_axis01.setter
    def custom_axis01(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_CustomAxis01",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_custom_axis01(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_custom_axis01(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CustomAxis01",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def custom_axis02(self) -> float:
        """
        The state of CustomAxis02.
        A value between -1 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CustomAxis02",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @custom_axis02.setter
    def custom_axis02(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_CustomAxis02",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_custom_axis02(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_custom_axis02(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CustomAxis02",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def custom_axis03(self) -> float:
        """
        The state of CustomAxis03.
        A value between -1 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CustomAxis03",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @custom_axis03.setter
    def custom_axis03(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_CustomAxis03",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_custom_axis03(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_custom_axis03(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CustomAxis03",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def custom_axis04(self) -> float:
        """
        The state of CustomAxis04.
        A value between -1 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_CustomAxis04",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @custom_axis04.setter
    def custom_axis04(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_CustomAxis04",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_custom_axis04(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_custom_axis04(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_CustomAxis04",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def forward(self) -> float:
        """
        The state of the forward translational control.
        A value between -1 and 1.
        Equivalent to the h and n keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Forward",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @forward.setter
    def forward(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Forward",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_forward(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_forward(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Forward",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def gear(self) -> bool:
        """
        The state of the landing gear/legs.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Gear",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @gear.setter
    def gear(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Gear",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_gear(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_gear(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Gear",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def input_mode(self) -> ControlInputMode:
        """
        Sets the behavior of the pitch, yaw, roll and translation control inputs.
        When set to additive, these inputs are added to the vessels current inputs.
        This mode is the default.
        When set to override, these inputs (if non-zero) override the vessels inputs.
        This mode prevents keyboard control, or SAS, from interfering with the controls when
        they are set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_InputMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlInputMode")
        )

    @input_mode.setter
    def input_mode(self, value: ControlInputMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_InputMode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.enumeration_type("SpaceCenter", "ControlInputMode")],
            None
        )

    def _return_type_input_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ControlInputMode")

    def _build_call_input_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_InputMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlInputMode")
        )

    @property
    def intakes(self) -> bool:
        """
        Returns whether all of the air intakes on the vessel are open,
        and sets the open state of all air intakes.
        See SpaceCenter.Intake#open.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Intakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @intakes.setter
    def intakes(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Intakes",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_intakes(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_intakes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Intakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def legs(self) -> bool:
        """
        Returns whether all landing legs on the vessel are deployed,
        and sets the deployment state of all landing legs.
        Does not include wheels (for example landing gear).
        See SpaceCenter.Leg#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Legs",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @legs.setter
    def legs(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Legs",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_legs(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_legs(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Legs",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def lights(self) -> bool:
        """
        The state of the lights.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Lights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @lights.setter
    def lights(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Lights",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_lights(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_lights(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Lights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def nodes(self) -> List[Node]:
        """
        Returns a list of all existing maneuver nodes, ordered by time from first to last.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Nodes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Node"))
        )

    def _return_type_nodes(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Node"))

    def _build_call_nodes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Nodes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Node"))
        )

    @property
    def parachutes(self) -> bool:
        """
        Returns whether all parachutes on the vessel are deployed,
        and sets the deployment state of all parachutes.
        Cannot be set to ``False``.
        See SpaceCenter.Parachute#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Parachutes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @parachutes.setter
    def parachutes(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Parachutes",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_parachutes(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_parachutes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Parachutes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def pitch(self) -> float:
        """
        The state of the pitch control.
        A value between -1 and 1.
        Equivalent to the w and s keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @pitch.setter
    def pitch(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Pitch",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def radiators(self) -> bool:
        """
        Returns whether all radiators on the vessel are deployed,
        and sets the deployment state of all radiators.
        See SpaceCenter.Radiator#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Radiators",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @radiators.setter
    def radiators(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Radiators",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_radiators(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_radiators(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Radiators",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def rcs(self) -> bool:
        """
        The state of RCS.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @rcs.setter
    def rcs(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_RCS",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_rcs(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_rcs(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def reaction_wheels(self) -> bool:
        """
        Returns whether all reactive wheels on the vessel are active,
        and sets the active state of all reaction wheels.
        See SpaceCenter.ReactionWheel#active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_ReactionWheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @reaction_wheels.setter
    def reaction_wheels(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_ReactionWheels",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_reaction_wheels(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_reaction_wheels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_ReactionWheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def resource_harvesters(self) -> bool:
        """
        Returns whether all of the resource harvesters on the vessel are deployed,
        and sets the deployment state of all resource harvesters.
        See SpaceCenter.ResourceHarvester#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_ResourceHarvesters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @resource_harvesters.setter
    def resource_harvesters(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_ResourceHarvesters",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_resource_harvesters(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_resource_harvesters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_ResourceHarvesters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def resource_harvesters_active(self) -> bool:
        """
        Returns whether any of the resource harvesters on the vessel are active,
        and sets the active state of all resource harvesters.
        See SpaceCenter.ResourceHarvester#active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_ResourceHarvestersActive",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @resource_harvesters_active.setter
    def resource_harvesters_active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_ResourceHarvestersActive",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_resource_harvesters_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_resource_harvesters_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_ResourceHarvestersActive",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def right(self) -> float:
        """
        The state of the right translational control.
        A value between -1 and 1.
        Equivalent to the j and l keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Right",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @right.setter
    def right(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Right",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_right(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_right(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Right",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def roll(self) -> float:
        """
        The state of the roll control.
        A value between -1 and 1.
        Equivalent to the q and e keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Roll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @roll.setter
    def roll(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Roll",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_roll(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_roll(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Roll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def sas(self) -> bool:
        """
        The state of SAS.

        Equivalent to SpaceCenter.AutoPilot#sAS
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_SAS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @sas.setter
    def sas(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_SAS",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_sas(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_sas(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_SAS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def sas_mode(self) -> SASMode:
        """
        The current SpaceCenter.SASMode.
        These modes are equivalent to the mode buttons to
        the left of the navball that appear when SAS is enabled.

        Equivalent to SpaceCenter.AutoPilot#sASMode
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_SASMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "SASMode")
        )

    @sas_mode.setter
    def sas_mode(self, value: SASMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_SASMode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.enumeration_type("SpaceCenter", "SASMode")],
            None
        )

    def _return_type_sas_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "SASMode")

    def _build_call_sas_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_SASMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "SASMode")
        )

    @property
    def solar_panels(self) -> bool:
        """
        Returns whether all solar panels on the vessel are deployed,
        and sets the deployment state of all solar panels.
        See SpaceCenter.SolarPanel#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_SolarPanels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @solar_panels.setter
    def solar_panels(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_SolarPanels",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_solar_panels(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_solar_panels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_SolarPanels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def source(self) -> ControlSource:
        """
        The source of the vessels control, for example by a kerbal or a probe core.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Source",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlSource")
        )

    def _return_type_source(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ControlSource")

    def _build_call_source(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Source",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlSource")
        )

    @property
    def speed_mode(self) -> SpeedMode:
        """
        The current SpaceCenter.SpeedMode of the navball.
        This is the mode displayed next to the speed at the top of the navball.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_SpeedMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "SpeedMode")
        )

    @speed_mode.setter
    def speed_mode(self, value: SpeedMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_SpeedMode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.enumeration_type("SpaceCenter", "SpeedMode")],
            None
        )

    def _return_type_speed_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "SpeedMode")

    def _build_call_speed_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_SpeedMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "SpeedMode")
        )

    @property
    def stage_lock(self) -> bool:
        """
        Whether staging is locked on the vessel.

        This is equivalent to locking the staging using Alt+L
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_StageLock",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @stage_lock.setter
    def stage_lock(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_StageLock",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_stage_lock(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_stage_lock(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_StageLock",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def state(self) -> ControlState:
        """
        The control state of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ControlState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.enumeration_type("SpaceCenter", "ControlState")
        )

    @property
    def throttle(self) -> float:
        """
        The state of the throttle. A value between 0 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Throttle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @throttle.setter
    def throttle(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Throttle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_throttle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_throttle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Throttle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def up(self) -> float:
        """
        The state of the up translational control.
        A value between -1 and 1.
        Equivalent to the i and k keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Up",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @up.setter
    def up(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Up",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_up(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_up(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Up",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def wheel_steering(self) -> float:
        """
        The state of the wheel steering.
        A value between -1 and 1.
        A value of 1 steers to the left, and a value of -1 steers to the right.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_WheelSteering",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @wheel_steering.setter
    def wheel_steering(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_WheelSteering",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_wheel_steering(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_wheel_steering(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_WheelSteering",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def wheel_throttle(self) -> float:
        """
        The state of the wheel throttle.
        A value between -1 and 1.
        A value of 1 rotates the wheels forwards, a value of -1 rotates
        the wheels backwards.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_WheelThrottle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @wheel_throttle.setter
    def wheel_throttle(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_WheelThrottle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_wheel_throttle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_wheel_throttle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_WheelThrottle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @property
    def wheels(self) -> bool:
        """
        Returns whether all wheels on the vessel are deployed,
        and sets the deployment state of all wheels.
        Does not include landing legs.
        See SpaceCenter.Wheel#deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Wheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @wheels.setter
    def wheels(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Wheels",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.bool_type],
            None
        )

    def _return_type_wheels(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_wheels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Wheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.bool_type
        )

    @property
    def yaw(self) -> float:
        """
        The state of the yaw control.
        A value between -1 and 1.
        Equivalent to the a and d keys.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_get_Yaw",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    @yaw.setter
    def yaw(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Control_set_Yaw",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.float_type],
            None
        )

    def _return_type_yaw(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_yaw(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_get_Yaw",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Control")],
            self._client._types.float_type
        )

    def activate_next_stage(self) -> List[Vessel]:
        """
        Activates the next stage. Equivalent to pressing the space bar in-game.

        :returns: A list of vessel objects that are jettisoned from the active vessel.

        When called, the active vessel may change. It is therefore possible that,
        after calling this function, the object(s) returned by previous call(s) to
        SpaceCenter#activeVessel no longer refer to the active vessel.
        Throws an exception if staging is locked.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_ActivateNextStage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Control"), ],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))
        )

    def _return_type_activate_next_stage(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))

    def _build_call_activate_next_stage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_ActivateNextStage",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Control"), ],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))
        )

    def add_node(self, ut: float, prograde: float = 0.0, normal: float = 0.0, radial: float = 0.0) -> Node:
        """
        Creates a maneuver node at the given universal time, and returns a
        SpaceCenter.Node object that can be used to modify it.
        Optionally sets the magnitude of the delta-v for the maneuver node
        in the prograde, normal and radial directions.

        :param ut: Universal time of the maneuver node.

        :param prograde: Delta-v in the prograde direction.

        :param normal: Delta-v in the normal direction.

        :param radial: Delta-v in the radial direction.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_AddNode",
            [self, ut, prograde, normal, radial],
            ["self", "ut", "prograde", "normal", "radial"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.double_type, self._client._types.float_type, self._client._types.float_type, self._client._types.float_type],
            self._client._types.class_type("SpaceCenter", "Node")
        )

    def _return_type_add_node(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Node")

    def _build_call_add_node(self, ut: float, prograde: float = 0.0, normal: float = 0.0, radial: float = 0.0) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_AddNode",
            [self, ut, prograde, normal, radial],
            ["self", "ut", "prograde", "normal", "radial"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.double_type, self._client._types.float_type, self._client._types.float_type, self._client._types.float_type],
            self._client._types.class_type("SpaceCenter", "Node")
        )

    def get_action_group(self, group: int) -> bool:
        """
        Returns ``True`` if the given action group is enabled.

        :param group: A number between 0 and 9 inclusive,
                      or between 0 and 250 inclusive when the `Extended Action Groups mod <https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/>`_ is installed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_GetActionGroup",
            [self, group],
            ["self", "group"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type],
            self._client._types.bool_type
        )

    def _return_type_get_action_group(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_get_action_group(self, group: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_GetActionGroup",
            [self, group],
            ["self", "group"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type],
            self._client._types.bool_type
        )

    def remove_nodes(self) -> None:
        """
        Remove all maneuver nodes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_RemoveNodes",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Control"), ],
            None
        )

    def _return_type_remove_nodes(self) -> TypeBase:
        return None

    def _build_call_remove_nodes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_RemoveNodes",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Control"), ],
            None
        )

    def set_action_group(self, group: int, state: bool) -> None:
        """
        Sets the state of the given action group.

        :param group: A number between 0 and 9 inclusive,
                      or between 0 and 250 inclusive when the `Extended Action Groups mod <https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/>`_ is installed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_SetActionGroup",
            [self, group, state],
            ["self", "group", "state"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type, self._client._types.bool_type],
            None
        )

    def _return_type_set_action_group(self) -> TypeBase:
        return None

    def _build_call_set_action_group(self, group: int, state: bool) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_SetActionGroup",
            [self, group, state],
            ["self", "group", "state"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type, self._client._types.bool_type],
            None
        )

    def toggle_action_group(self, group: int) -> None:
        """
        Toggles the state of the given action group.

        :param group: A number between 0 and 9 inclusive,
                      or between 0 and 250 inclusive when the `Extended Action Groups mod <https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/>`_ is installed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Control_ToggleActionGroup",
            [self, group],
            ["self", "group"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type],
            None
        )

    def _return_type_toggle_action_group(self) -> TypeBase:
        return None

    def _build_call_toggle_action_group(self, group: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Control_ToggleActionGroup",
            [self, group],
            ["self", "group"],
            [self._client._types.class_type("SpaceCenter", "Control"), self._client._types.uint32_type],
            None
        )



class ControlSurface(ClassBase):
    """
    An aerodynamic control surface. Obtained by calling SpaceCenter.Part#controlSurface.
    """
    @property
    def authority_limiter(self) -> float:
        """
        The authority limiter for the control surface, which controls how far the
        control surface will move.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_AuthorityLimiter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.float_type
        )

    @authority_limiter.setter
    def authority_limiter(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_AuthorityLimiter",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.float_type],
            None
        )

    def _return_type_authority_limiter(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_authority_limiter(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_AuthorityLimiter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.float_type
        )

    @property
    def available_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The available torque, in Newton meters, that can be produced by this control surface,
        in the positive and negative pitch, roll and yaw axes of the vessel. These axes
        correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the control surface has been fully deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @property
    def inverted(self) -> bool:
        """
        Whether the control surface movement is inverted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_Inverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @inverted.setter
    def inverted(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_Inverted",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.bool_type],
            None
        )

    def _return_type_inverted(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_inverted(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_Inverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this control surface.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def pitch_enabled(self) -> bool:
        """
        Whether the control surface has pitch control enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_PitchEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @pitch_enabled.setter
    def pitch_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_PitchEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.bool_type],
            None
        )

    def _return_type_pitch_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_pitch_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_PitchEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @property
    def roll_enabled(self) -> bool:
        """
        Whether the control surface has roll control enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_RollEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @roll_enabled.setter
    def roll_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_RollEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.bool_type],
            None
        )

    def _return_type_roll_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_roll_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_RollEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @property
    def surface_area(self) -> float:
        """
        Surface area of the control surface in ``m^2``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_SurfaceArea",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.float_type
        )

    def _return_type_surface_area(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_surface_area(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_SurfaceArea",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.float_type
        )

    @property
    def yaw_enabled(self) -> bool:
        """
        Whether the control surface has yaw control enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_get_YawEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )

    @yaw_enabled.setter
    def yaw_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ControlSurface_set_YawEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface"), self._client._types.bool_type],
            None
        )

    def _return_type_yaw_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_yaw_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ControlSurface_get_YawEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ControlSurface")],
            self._client._types.bool_type
        )



class CrewMember(ClassBase):
    """
    Represents crew in a vessel. Can be obtained using SpaceCenter.Vessel#crew.
    """
    @property
    def badass(self) -> bool:
        """
        Whether the crew member is a badass.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Badass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )

    @badass.setter
    def badass(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Badass",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.bool_type],
            None
        )

    def _return_type_badass(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_badass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Badass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )

    @property
    def career_log_flights(self) -> List[int]:
        """
        The flight IDs for each entry in the career flight log.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_CareerLogFlights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.sint32_type)
        )

    def _return_type_career_log_flights(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.sint32_type)

    def _build_call_career_log_flights(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_CareerLogFlights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.sint32_type)
        )

    @property
    def career_log_targets(self) -> List[str]:
        """
        The body name for each entry in the career flight log.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_CareerLogTargets",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_career_log_targets(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_career_log_targets(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_CareerLogTargets",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def career_log_types(self) -> List[str]:
        """
        The type for each entry in the career flight log.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_CareerLogTypes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_career_log_types(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_career_log_types(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_CareerLogTypes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def courage(self) -> float:
        """
        The crew members courage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Courage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @courage.setter
    def courage(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Courage",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.float_type],
            None
        )

    def _return_type_courage(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_courage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Courage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @property
    def experience(self) -> float:
        """
        The crew members experience.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Experience",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @experience.setter
    def experience(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Experience",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.float_type],
            None
        )

    def _return_type_experience(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_experience(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Experience",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @property
    def gender(self) -> CrewMemberGender:
        """
        The crew member's gender.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Gender",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "CrewMemberGender")
        )

    def _return_type_gender(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "CrewMemberGender")

    def _build_call_gender(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Gender",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "CrewMemberGender")
        )

    @property
    def name(self) -> str:
        """
        The crew members name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.string_type
        )

    @name.setter
    def name(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Name",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.string_type],
            None
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.string_type
        )

    @property
    def on_mission(self) -> bool:
        """
        Whether the crew member is on a mission.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_OnMission",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )

    def _return_type_on_mission(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_on_mission(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_OnMission",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )

    @property
    def roster_status(self) -> RosterStatus:
        """
        The crew member's current roster status.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_RosterStatus",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "RosterStatus")
        )

    def _return_type_roster_status(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "RosterStatus")

    def _build_call_roster_status(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_RosterStatus",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "RosterStatus")
        )

    @property
    def stupidity(self) -> float:
        """
        The crew members stupidity.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Stupidity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @stupidity.setter
    def stupidity(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Stupidity",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.float_type],
            None
        )

    def _return_type_stupidity(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_stupidity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Stupidity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.float_type
        )

    @property
    def suit_type(self) -> SuitType:
        """
        The crew member's suit type.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_SuitType",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "SuitType")
        )

    @suit_type.setter
    def suit_type(self, value: SuitType) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_SuitType",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.enumeration_type("SpaceCenter", "SuitType")],
            None
        )

    def _return_type_suit_type(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "SuitType")

    def _build_call_suit_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_SuitType",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "SuitType")
        )

    @property
    def trait(self) -> str:
        """
        The crew member's job.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Trait",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.string_type
        )

    def _return_type_trait(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_trait(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Trait",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.string_type
        )

    @property
    def type(self) -> CrewMemberType:
        """
        The type of crew member.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "CrewMemberType")
        )

    def _return_type_type(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "CrewMemberType")

    def _build_call_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.enumeration_type("SpaceCenter", "CrewMemberType")
        )

    @property
    def veteran(self) -> bool:
        """
        Whether the crew member is a veteran.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_get_Veteran",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )

    @veteran.setter
    def veteran(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "CrewMember_set_Veteran",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "CrewMember"), self._client._types.bool_type],
            None
        )

    def _return_type_veteran(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_veteran(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CrewMember_get_Veteran",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "CrewMember")],
            self._client._types.bool_type
        )



class Decoupler(ClassBase):
    """
    A decoupler. Obtained by calling SpaceCenter.Part#decoupler
    """
    @property
    def attached_part(self) -> Optional[Part]:
        """
        The part attached to this decoupler's explosive node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_AttachedPart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_attached_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_attached_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_AttachedPart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def decoupled(self) -> bool:
        """
        Whether the decoupler has fired.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_Decoupled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    def _return_type_decoupled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_decoupled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_Decoupled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    @property
    def impulse(self) -> float:
        """
        The impulse that the decoupler imparts when it is fired, in Newton seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_Impulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.float_type
        )

    def _return_type_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_Impulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.float_type
        )

    @property
    def is_omni_decoupler(self) -> bool:
        """
        Whether the decoupler is an omni-decoupler (e.g. stack separator)
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_IsOmniDecoupler",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    def _return_type_is_omni_decoupler(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_omni_decoupler(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_IsOmniDecoupler",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this decoupler.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def staged(self) -> bool:
        """
        Whether the decoupler is enabled in the staging sequence.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_get_Staged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    def _return_type_staged(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_staged(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_get_Staged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Decoupler")],
            self._client._types.bool_type
        )

    def decouple(self) -> Vessel:
        """
        Fires the decoupler. Returns the new vessel created when the decoupler fires.
        Throws an exception if the decoupler has already fired.

        When called, the active vessel may change. It is therefore possible that,
        after calling this function, the object(s) returned by previous call(s) to
        SpaceCenter#activeVessel no longer refer to the active vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Decoupler_Decouple",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Decoupler"), ],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    def _return_type_decouple(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_decouple(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Decoupler_Decouple",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Decoupler"), ],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )



class DockingPort(ClassBase):
    """
    A docking port. Obtained by calling SpaceCenter.Part#dockingPort
    """
    @property
    def can_rotate(self) -> bool:
        """
        Whether the docking port can be commanded to rotate while docked.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_CanRotate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    def _return_type_can_rotate(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_rotate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_CanRotate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @property
    def docked_part(self) -> Optional[Part]:
        """
        The part that this docking port is docked to. Returns ``None`` if this
        docking port is not docked to anything.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_DockedPart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_docked_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_docked_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_DockedPart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def has_shield(self) -> bool:
        """
        Whether the docking port has a shield.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_HasShield",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    def _return_type_has_shield(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_shield(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_HasShield",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @property
    def maximum_rotation(self) -> float:
        """
        Maximum rotation angle in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_MaximumRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    def _return_type_maximum_rotation(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_maximum_rotation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_MaximumRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    @property
    def minimum_rotation(self) -> float:
        """
        Minimum rotation angle in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_MinimumRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    def _return_type_minimum_rotation(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_minimum_rotation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_MinimumRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this docking port.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def reengage_distance(self) -> float:
        """
        The distance a docking port must move away when it undocks before it
        becomes ready to dock with another port, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_ReengageDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    def _return_type_reengage_distance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_reengage_distance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_ReengageDistance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to this docking port, and
        oriented with the port.
         * The origin is at the position of the docking port.
         * The axes rotate with the docking port.
         * The x-axis points out to the right side of the docking port.
         * The y-axis points in the direction the docking port is facing.
         * The z-axis points out of the bottom off the docking port.

        This reference frame is not necessarily equivalent to the reference frame
        for the part, returned by SpaceCenter.Part#referenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def rotation_locked(self) -> bool:
        """
        Lock rotation. When locked, allows auto-strut to work across the joint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_RotationLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @rotation_locked.setter
    def rotation_locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_set_RotationLocked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.bool_type],
            None
        )

    def _return_type_rotation_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_rotation_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_RotationLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @property
    def rotation_target(self) -> float:
        """
        Rotation target angle in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_RotationTarget",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    @rotation_target.setter
    def rotation_target(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_set_RotationTarget",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.float_type],
            None
        )

    def _return_type_rotation_target(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_rotation_target(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_RotationTarget",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.float_type
        )

    @property
    def shielded(self) -> bool:
        """
        The state of the docking ports shield, if it has one.

        Returns ``True`` if the docking port has a shield, and the shield is
        closed. Otherwise returns ``False``. When set to ``True``, the shield is
        closed, and when set to ``False`` the shield is opened. If the docking
        port does not have a shield, setting this attribute has no effect.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_Shielded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @shielded.setter
    def shielded(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_set_Shielded",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.bool_type],
            None
        )

    def _return_type_shielded(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_shielded(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_Shielded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.bool_type
        )

    @property
    def state(self) -> DockingPortState:
        """
        The current state of the docking port.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.enumeration_type("SpaceCenter", "DockingPortState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "DockingPortState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            self._client._types.enumeration_type("SpaceCenter", "DockingPortState")
        )

    def direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction that docking port points in, in the given reference frame.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the docking port, in the given reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def rotation(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float,float]:
        """
        The rotation of the docking port, in the given reference frame.

        :returns: The rotation as a quaternion of the form ``(x, y, z, w)``.

        :param reference_frame: The reference frame that the returned
                                rotation is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_rotation(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_rotation(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def undock(self) -> Vessel:
        """
        Undocks the docking port and returns the new SpaceCenter.Vessel that is created.
        This method can be called for either docking port in a docked pair.
        Throws an exception if the docking port is not docked to anything.

        When called, the active vessel may change. It is therefore possible that,
        after calling this function, the object(s) returned by previous call(s) to
        SpaceCenter#activeVessel no longer refer to the active vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "DockingPort_Undock",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), ],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    def _return_type_undock(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_undock(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "DockingPort_Undock",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "DockingPort"), ],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )



class Engine(ClassBase):
    """
    An engine, including ones of various types.
    For example liquid fuelled gimballed engines, solid rocket boosters and jet engines.
    Obtained by calling SpaceCenter.Part#engine.

    For RCS thrusters SpaceCenter.Part#rCS.
    """
    @property
    def active(self) -> bool:
        """
        Whether the engine is active. Setting this attribute may have no effect,
        depending on SpaceCenter.Engine#canShutdown and SpaceCenter.Engine#canRestart.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @active.setter
    def active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_Active",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.bool_type],
            None
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def auto_mode_switch(self) -> bool:
        """
        Whether the engine will automatically switch modes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_AutoModeSwitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @auto_mode_switch.setter
    def auto_mode_switch(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_AutoModeSwitch",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.bool_type],
            None
        )

    def _return_type_auto_mode_switch(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_auto_mode_switch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_AutoModeSwitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def available_thrust(self) -> float:
        """
        The amount of thrust, in Newtons, that would be produced by the engine
        when activated and with its throttle set to 100%.
        Returns zero if the engine does not have any fuel.
        Takes the engine's current SpaceCenter.Engine#thrustLimit and atmospheric conditions
        into account.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_available_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_available_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def available_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The available torque, in Newton meters, that can be produced by this engine,
        in the positive and negative pitch, roll and yaw axes of the vessel. These axes
        correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        Returns zero if the engine is inactive, or not gimballed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def can_restart(self) -> bool:
        """
        Whether the engine can be restarted once shutdown. If the engine cannot be shutdown,
        returns ``False``. For example, this is ``True`` for liquid fueled rockets
        and ``False`` for solid rocket boosters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_CanRestart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_can_restart(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_restart(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_CanRestart",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def can_shutdown(self) -> bool:
        """
        Whether the engine can be shutdown once activated. For example, this is
        ``True`` for liquid fueled rockets and ``False`` for solid rocket boosters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_CanShutdown",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_can_shutdown(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_shutdown(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_CanShutdown",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def gimbal_limit(self) -> float:
        """
        The gimbal limiter of the engine. A value between 0 and 1.
        Returns 0 if the gimbal is locked.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_GimbalLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @gimbal_limit.setter
    def gimbal_limit(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_GimbalLimit",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.float_type],
            None
        )

    def _return_type_gimbal_limit(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_gimbal_limit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_GimbalLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def gimbal_locked(self) -> bool:
        """
        Whether the engines gimbal is locked in place. Setting this attribute has
        no effect if the engine is not gimballed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_GimbalLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @gimbal_locked.setter
    def gimbal_locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_GimbalLocked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.bool_type],
            None
        )

    def _return_type_gimbal_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_gimbal_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_GimbalLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def gimbal_range(self) -> float:
        """
        The range over which the gimbal can move, in degrees.
        Returns 0 if the engine is not gimballed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_GimbalRange",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_gimbal_range(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_gimbal_range(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_GimbalRange",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def gimballed(self) -> bool:
        """
        Whether the engine is gimballed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Gimballed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_gimballed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_gimballed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Gimballed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def has_fuel(self) -> bool:
        """
        Whether the engine has any fuel available.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_HasFuel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_has_fuel(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_fuel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_HasFuel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def has_modes(self) -> bool:
        """
        Whether the engine has multiple modes of operation.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_HasModes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_has_modes(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_modes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_HasModes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def independent_throttle(self) -> bool:
        """
        Whether the independent throttle is enabled for the engine.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_IndependentThrottle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @independent_throttle.setter
    def independent_throttle(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_IndependentThrottle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.bool_type],
            None
        )

    def _return_type_independent_throttle(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_independent_throttle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_IndependentThrottle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def kerbin_sea_level_specific_impulse(self) -> float:
        """
        The specific impulse of the engine at sea level on Kerbin, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_kerbin_sea_level_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_kerbin_sea_level_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def max_thrust(self) -> float:
        """
        The amount of thrust, in Newtons, that would be produced by the engine
        when activated and fueled, with its throttle and throttle limiter set to 100%.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_max_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def max_vacuum_thrust(self) -> float:
        """
        The maximum amount of thrust that can be produced by the engine in a
        vacuum, in Newtons. This is the amount of thrust produced by the engine
        when activated, SpaceCenter.Engine#thrustLimit is set to 100%, the main
        vessel's throttle is set to 100% and the engine is in a vacuum.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_max_vacuum_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_vacuum_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def mode(self) -> str:
        """
        The name of the current engine mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Mode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.string_type
        )

    @mode.setter
    def mode(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_Mode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.string_type],
            None
        )

    def _return_type_mode(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Mode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.string_type
        )

    @property
    def modes(self) -> Dict[str, Engine]:
        """
        The available modes for the engine.
        A dictionary mapping mode names to SpaceCenter.Engine objects.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Modes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "Engine"))
        )

    def _return_type_modes(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "Engine"))

    def _build_call_modes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Modes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "Engine"))
        )

    @property
    def part(self) -> Part:
        """
        The part object for this engine.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def propellant_names(self) -> List[str]:
        """
        The names of the propellants that the engine consumes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_PropellantNames",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_propellant_names(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_propellant_names(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_PropellantNames",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def propellant_ratios(self) -> Dict[str, float]:
        """
        The ratio of resources that the engine consumes. A dictionary mapping resource names
        to the ratio at which they are consumed by the engine.

        For example, if the ratios are 0.6 for LiquidFuel and 0.4 for Oxidizer, then for every
        0.6 units of LiquidFuel that the engine burns, it will burn 0.4 units of Oxidizer.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_PropellantRatios",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)
        )

    def _return_type_propellant_ratios(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)

    def _build_call_propellant_ratios(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_PropellantRatios",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)
        )

    @property
    def propellants(self) -> List[Propellant]:
        """
        The propellants that the engine consumes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Propellants",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Propellant"))
        )

    def _return_type_propellants(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Propellant"))

    def _build_call_propellants(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Propellants",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Propellant"))
        )

    @property
    def specific_impulse(self) -> float:
        """
        The current specific impulse of the engine, in seconds. Returns zero
        if the engine is not active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def throttle(self) -> float:
        """
        The current throttle setting for the engine. A value between 0 and 1.
        This is not necessarily the same as the vessel's main throttle
        setting, as some engines take time to adjust their throttle
        (such as jet engines), or independent throttle may be enabled.

        When the engine's independent throttle is enabled
        (see SpaceCenter.Engine#independentThrottle), can be used to set the throttle percentage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Throttle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @throttle.setter
    def throttle(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_Throttle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.float_type],
            None
        )

    def _return_type_throttle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_throttle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Throttle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def throttle_locked(self) -> bool:
        """
        Whether the SpaceCenter.Control#throttle affects the engine. For example,
        this is ``True`` for liquid fueled rockets, and ``False`` for solid rocket
        boosters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_ThrottleLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    def _return_type_throttle_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_throttle_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_ThrottleLocked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.bool_type
        )

    @property
    def thrust(self) -> float:
        """
        The current amount of thrust being produced by the engine, in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Thrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Thrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def thrust_limit(self) -> float:
        """
        The thrust limiter of the engine. A value between 0 and 1. Setting this
        attribute may have no effect, for example the thrust limit for a solid
        rocket booster cannot be changed in flight.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_ThrustLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @thrust_limit.setter
    def thrust_limit(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Engine_set_ThrustLimit",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.float_type],
            None
        )

    def _return_type_thrust_limit(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thrust_limit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_ThrustLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    @property
    def thrusters(self) -> List[Thruster]:
        """
        The components of the engine that generate thrust.

        For example, this corresponds to the rocket nozzel on a solid rocket booster,
        or the individual nozzels on a RAPIER engine.
        The overall thrust produced by the engine, as reported by SpaceCenter.Engine#availableThrust,
        SpaceCenter.Engine#maxThrust and others, is the sum of the thrust generated by each thruster.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_Thrusters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))
        )

    def _return_type_thrusters(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))

    def _build_call_thrusters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_Thrusters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))
        )

    @property
    def vacuum_specific_impulse(self) -> float:
        """
        The vacuum specific impulse of the engine, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def _return_type_vacuum_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_vacuum_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Engine")],
            self._client._types.float_type
        )

    def available_thrust_at(self, pressure: float) -> float:
        """
        The amount of thrust, in Newtons, that would be produced by the engine
        when activated and with its throttle set to 100%.
        Returns zero if the engine does not have any fuel.
        Takes the given pressure into account.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_AvailableThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_available_thrust_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_available_thrust_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_AvailableThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def max_thrust_at(self, pressure: float) -> float:
        """
        The amount of thrust, in Newtons, that would be produced by the engine
        when activated and fueled, with its throttle and throttle limiter set to 100%.
        Takes the given pressure into account.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_MaxThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_max_thrust_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_thrust_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_MaxThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def specific_impulse_at(self, pressure: float) -> float:
        """
        The specific impulse of the engine under the given pressure, in seconds. Returns zero
        if the engine is not active.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_SpecificImpulseAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_specific_impulse_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_specific_impulse_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_SpecificImpulseAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Engine"), self._client._types.double_type],
            self._client._types.float_type
        )

    def toggle_mode(self) -> None:
        """
        Toggle the current engine mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Engine_ToggleMode",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Engine"), ],
            None
        )

    def _return_type_toggle_mode(self) -> TypeBase:
        return None

    def _build_call_toggle_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Engine_ToggleMode",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Engine"), ],
            None
        )



class Experiment(ClassBase):
    """
    Obtained by calling SpaceCenter.Part#experiment.
    """
    @property
    def available(self) -> bool:
        """
        Determines if the experiment is available given the current conditions.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Available",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    def _return_type_available(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_available(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Available",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    @property
    def biome(self) -> str:
        """
        The name of the biome the experiment is currently in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Biome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    def _return_type_biome(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_biome(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Biome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    @property
    def data(self) -> List[ScienceData]:
        """
        The data contained in this experiment.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Data",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ScienceData"))
        )

    def _return_type_data(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ScienceData"))

    def _build_call_data(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Data",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ScienceData"))
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the experiment has been deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    @property
    def has_data(self) -> bool:
        """
        Whether the experiment contains data.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_HasData",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    def _return_type_has_data(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_data(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_HasData",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    @property
    def inoperable(self) -> bool:
        """
        Whether the experiment is inoperable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Inoperable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    def _return_type_inoperable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_inoperable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Inoperable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    @property
    def name(self) -> str:
        """
        Internal name of the experiment, as used in
        `part cfg files <https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this experiment.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def rerunnable(self) -> bool:
        """
        Whether the experiment can be re-run.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Rerunnable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    def _return_type_rerunnable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_rerunnable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Rerunnable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.bool_type
        )

    @property
    def science_subject(self) -> ScienceSubject:
        """
        Containing information on the corresponding specific science result for the current
        conditions. Returns ``None`` if the experiment is unavailable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_ScienceSubject",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.class_type("SpaceCenter", "ScienceSubject")
        )

    def _return_type_science_subject(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ScienceSubject")

    def _build_call_science_subject(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_ScienceSubject",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.class_type("SpaceCenter", "ScienceSubject")
        )

    @property
    def title(self) -> str:
        """
        Title of the experiment, as shown on the in-game UI.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Experiment")],
            self._client._types.string_type
        )

    def dump(self) -> None:
        """
        Dump the experimental data contained by the experiment.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_Dump",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def _return_type_dump(self) -> TypeBase:
        return None

    def _build_call_dump(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_Dump",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def reset(self) -> None:
        """
        Reset the experiment.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_Reset",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def _return_type_reset(self) -> TypeBase:
        return None

    def _build_call_reset(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_Reset",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def run(self) -> None:
        """
        Run the experiment.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_Run",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def _return_type_run(self) -> TypeBase:
        return None

    def _build_call_run(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_Run",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def transmit(self) -> None:
        """
        Transmit all experimental data contained by this part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Experiment_Transmit",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )

    def _return_type_transmit(self) -> TypeBase:
        return None

    def _build_call_transmit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Experiment_Transmit",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Experiment"), ],
            None
        )



class Fairing(ClassBase):
    """
    A fairing. Obtained by calling SpaceCenter.Part#fairing.
    Supports both stock fairings, and those from the ProceduralFairings mod.
    """
    @property
    def jettisoned(self) -> bool:
        """
        Whether the fairing has been jettisoned.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Fairing_get_Jettisoned",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Fairing")],
            self._client._types.bool_type
        )

    def _return_type_jettisoned(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_jettisoned(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Fairing_get_Jettisoned",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Fairing")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this fairing.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Fairing_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Fairing")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Fairing_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Fairing")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def jettison(self) -> None:
        """
        Jettison the fairing. Has no effect if it has already been jettisoned.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Fairing_Jettison",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Fairing"), ],
            None
        )

    def _return_type_jettison(self) -> TypeBase:
        return None

    def _build_call_jettison(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Fairing_Jettison",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Fairing"), ],
            None
        )



class Flight(ClassBase):
    """
    Used to get flight telemetry for a vessel, by calling SpaceCenter.Vessel#flight.
    All of the information returned by this class is given in the reference frame
    passed to that method.
    Obtained by calling SpaceCenter.Vessel#flight.

    To get orbital information, such as the apoapsis or inclination, see SpaceCenter.Orbit.
    """
    @property
    def aerodynamic_force(self) -> Tuple[float,float,float]:
        """
        The total aerodynamic forces acting on the vessel,
        in reference frame SpaceCenter.ReferenceFrame.

        :returns: A vector pointing in the direction that the force acts,
                  with its magnitude equal to the strength of the force in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_AerodynamicForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_aerodynamic_force(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_aerodynamic_force(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_AerodynamicForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def angle_of_attack(self) -> float:
        """
        The pitch angle between the orientation of the vessel and its velocity vector,
        in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_AngleOfAttack",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_angle_of_attack(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_angle_of_attack(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_AngleOfAttack",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def anti_normal(self) -> Tuple[float,float,float]:
        """
        The direction opposite to the normal of the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_AntiNormal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_anti_normal(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_anti_normal(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_AntiNormal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def anti_radial(self) -> Tuple[float,float,float]:
        """
        The direction opposite to the radial direction of the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_AntiRadial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_anti_radial(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_anti_radial(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_AntiRadial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def atmosphere_density(self) -> float:
        """
        The current density of the atmosphere around the vessel, in ``kg/m^3``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_AtmosphereDensity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_atmosphere_density(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_atmosphere_density(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_AtmosphereDensity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def ballistic_coefficient(self) -> float:
        """
        The `ballistic coefficient <https://en.wikipedia.org/wiki/Ballistic_coefficient>`_.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_BallisticCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_ballistic_coefficient(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_ballistic_coefficient(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_BallisticCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def bedrock_altitude(self) -> float:
        """
        The altitude above the surface of the body, in meters. When over water, this is the altitude above the sea floor.
        Measured from the center of mass of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_BedrockAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_bedrock_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_bedrock_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_BedrockAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def center_of_mass(self) -> Tuple[float,float,float]:
        """
        The position of the center of mass of the vessel,
        in the reference frame SpaceCenter.ReferenceFrame

        :returns: The position as a vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_CenterOfMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_center_of_mass(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_center_of_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_CenterOfMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def direction(self) -> Tuple[float,float,float]:
        """
        The direction that the vessel is pointing in,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Direction",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Direction",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def drag(self) -> Tuple[float,float,float]:
        """
        The `aerodynamic drag <https://en.wikipedia.org/wiki/Aerodynamic_force>`_ currently acting on the vessel.

        :returns: A vector pointing in the direction of the force, with its magnitude
                  equal to the strength of the force in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Drag",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_drag(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_drag(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Drag",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def drag_coefficient(self) -> float:
        """
        The coefficient of drag. This is the amount of drag produced by the vessel.
        It depends on air speed, air density and wing area.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_DragCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_drag_coefficient(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_drag_coefficient(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_DragCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def dynamic_pressure(self) -> float:
        """
        The dynamic pressure acting on the vessel, in Pascals. This is a measure of the
        strength of the aerodynamic forces. It is equal to
        ``\frac{1}{2} . \mbox{air density} . \mbox{velocity}^2``.
        It is commonly denoted ``Q``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_DynamicPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_dynamic_pressure(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_dynamic_pressure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_DynamicPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def elevation(self) -> float:
        """
        The elevation of the terrain under the vessel, in meters. This is the height of the terrain above sea level,
        and is negative when the vessel is over the sea.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Elevation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_elevation(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_elevation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Elevation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def equivalent_air_speed(self) -> float:
        """
        The `equivalent air speed <https://en.wikipedia.org/wiki/Equivalent_airspeed>`_
        of the vessel, in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_EquivalentAirSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_equivalent_air_speed(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_equivalent_air_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_EquivalentAirSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def g_force(self) -> float:
        """
        The current G force acting on the vessel in ``g``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_GForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_g_force(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_g_force(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_GForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def heading(self) -> float:
        """
        The heading of the vessel (its angle relative to north), in degrees.
        A value between 0° and 360°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Heading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_heading(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_heading(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Heading",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def horizontal_speed(self) -> float:
        """
        The horizontal speed of the vessel in meters per second,
        in the reference frame SpaceCenter.ReferenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_HorizontalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_horizontal_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_horizontal_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_HorizontalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def latitude(self) -> float:
        """
        The `latitude <https://en.wikipedia.org/wiki/Latitude>`_ of the vessel for the body being orbited, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Latitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_latitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_latitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Latitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def lift(self) -> Tuple[float,float,float]:
        """
        The `aerodynamic lift <https://en.wikipedia.org/wiki/Aerodynamic_force>`_
        currently acting on the vessel.

        :returns: A vector pointing in the direction that the force acts,
                  with its magnitude equal to the strength of the force in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Lift",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_lift(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_lift(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Lift",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def lift_coefficient(self) -> float:
        """
        The coefficient of lift. This is the amount of lift produced by the vessel, and
        depends on air speed, air density and wing area.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_LiftCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_lift_coefficient(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_lift_coefficient(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_LiftCoefficient",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def longitude(self) -> float:
        """
        The `longitude <https://en.wikipedia.org/wiki/Longitude>`_ of the vessel for the body being orbited, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Longitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_longitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_longitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Longitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def mach(self) -> float:
        """
        The speed of the vessel, in multiples of the speed of sound.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Mach",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_mach(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_mach(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Mach",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def mean_altitude(self) -> float:
        """
        The altitude above sea level, in meters.
        Measured from the center of mass of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_MeanAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_mean_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mean_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_MeanAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def normal(self) -> Tuple[float,float,float]:
        """
        The direction normal to the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Normal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_normal(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_normal(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Normal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def pitch(self) -> float:
        """
        The pitch of the vessel relative to the horizon, in degrees.
        A value between -90° and +90°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_pitch(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_pitch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Pitch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def prograde(self) -> Tuple[float,float,float]:
        """
        The prograde direction of the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Prograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_prograde(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_prograde(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Prograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def radial(self) -> Tuple[float,float,float]:
        """
        The radial direction of the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Radial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_radial(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_radial(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Radial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def retrograde(self) -> Tuple[float,float,float]:
        """
        The retrograde direction of the vessels orbit,
        in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The direction as a unit vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Retrograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_retrograde(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_retrograde(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Retrograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def reynolds_number(self) -> float:
        """
        The vessels Reynolds number.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_ReynoldsNumber",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_reynolds_number(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_reynolds_number(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_ReynoldsNumber",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def roll(self) -> float:
        """
        The roll of the vessel relative to the horizon, in degrees.
        A value between -180° and +180°.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Roll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_roll(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_roll(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Roll",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def rotation(self) -> Tuple[float,float,float,float]:
        """
        The rotation of the vessel, in the reference frame SpaceCenter.ReferenceFrame

        :returns: The rotation as a quaternion of the form ``(x, y, z, w)``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Rotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_rotation(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_rotation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Rotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def sideslip_angle(self) -> float:
        """
        The yaw angle between the orientation of the vessel and its velocity vector, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_SideslipAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_sideslip_angle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_sideslip_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_SideslipAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def speed(self) -> float:
        """
        The speed of the vessel in meters per second,
        in the reference frame SpaceCenter.ReferenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def speed_of_sound(self) -> float:
        """
        The speed of sound, in the atmosphere around the vessel, in ``m/s``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_SpeedOfSound",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_speed_of_sound(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_speed_of_sound(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_SpeedOfSound",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def stall_fraction(self) -> float:
        """
        The current amount of stall, between 0 and 1. A value greater than 0.005 indicates
        a minor stall and a value greater than 0.5 indicates a large-scale stall.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_StallFraction",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_stall_fraction(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_stall_fraction(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_StallFraction",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def static_air_temperature(self) -> float:
        """
        The `static (ambient) temperature <https://en.wikipedia.org/wiki/Total_air_temperature>`_ of the atmosphere around the vessel, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_StaticAirTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_static_air_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_static_air_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_StaticAirTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def static_pressure(self) -> float:
        """
        The static atmospheric pressure acting on the vessel, in Pascals.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_StaticPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_static_pressure(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_static_pressure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_StaticPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def static_pressure_at_msl(self) -> float:
        """
        The static atmospheric pressure at mean sea level, in Pascals.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_StaticPressureAtMSL",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_static_pressure_at_msl(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_static_pressure_at_msl(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_StaticPressureAtMSL",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def surface_altitude(self) -> float:
        """
        The altitude above the surface of the body or sea level, whichever is closer, in meters.
        Measured from the center of mass of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_SurfaceAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_surface_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_surface_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_SurfaceAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    @property
    def terminal_velocity(self) -> float:
        """
        An estimate of the current terminal velocity of the vessel, in meters per second.
        This is the speed at which the drag forces cancel out the force of gravity.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_TerminalVelocity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_terminal_velocity(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_terminal_velocity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_TerminalVelocity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def thrust_specific_fuel_consumption(self) -> float:
        """
        The thrust specific fuel consumption for the jet engines on the vessel. This is a
        measure of the efficiency of the engines, with a lower value indicating a more
        efficient vessel. This value is the number of Newtons of fuel that are burned,
        per hour, to produce one newton of thrust.

        Requires `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_ThrustSpecificFuelConsumption",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_thrust_specific_fuel_consumption(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thrust_specific_fuel_consumption(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_ThrustSpecificFuelConsumption",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def total_air_temperature(self) -> float:
        """
        The `total air temperature <https://en.wikipedia.org/wiki/Total_air_temperature>`_
        of the atmosphere around the vessel, in Kelvin.
        This includes the SpaceCenter.Flight#staticAirTemperature and the vessel's kinetic energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_TotalAirTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_total_air_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_total_air_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_TotalAirTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def true_air_speed(self) -> float:
        """
        The `true air speed <https://en.wikipedia.org/wiki/True_airspeed>`_
        of the vessel, in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_TrueAirSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    def _return_type_true_air_speed(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_true_air_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_TrueAirSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.float_type
        )

    @property
    def velocity(self) -> Tuple[float,float,float]:
        """
        The velocity of the vessel, in the reference frame SpaceCenter.ReferenceFrame.

        :returns: The velocity as a vector. The vector points in the direction of travel,
                  and its magnitude is the speed of the vessel in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_Velocity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_velocity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_Velocity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def vertical_speed(self) -> float:
        """
        The vertical speed of the vessel in meters per second,
        in the reference frame SpaceCenter.ReferenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_get_VerticalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def _return_type_vertical_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_vertical_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_get_VerticalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Flight")],
            self._client._types.double_type
        )

    def simulate_aerodynamic_force_at(self, body: CelestialBody, position: Tuple[float,float,float], velocity: Tuple[float,float,float]) -> Tuple[float,float,float]:
        """
        Simulate and return the total aerodynamic forces acting on the vessel,
        if it where to be traveling with the given velocity at the given position in the
        atmosphere of the given celestial body.

        :returns: A vector pointing in the direction that the force acts,
                  with its magnitude equal to the strength of the force in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Flight_SimulateAerodynamicForceAt",
            [self, body, position, velocity],
            ["self", "body", "position", "velocity"],
            [self._client._types.class_type("SpaceCenter", "Flight"), self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_simulate_aerodynamic_force_at(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_simulate_aerodynamic_force_at(self, body: CelestialBody, position: Tuple[float,float,float], velocity: Tuple[float,float,float]) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Flight_SimulateAerodynamicForceAt",
            [self, body, position, velocity],
            ["self", "body", "position", "velocity"],
            [self._client._types.class_type("SpaceCenter", "Flight"), self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class Force(ClassBase):
    """
    Obtained by calling SpaceCenter.Part#addForce.
    """
    @property
    def force_vector(self) -> Tuple[float,float,float]:
        """
        The force vector, in Newtons.

        :returns: A vector pointing in the direction that the force acts,
                  with its magnitude equal to the strength of the force in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Force_get_ForceVector",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @force_vector.setter
    def force_vector(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Force_set_ForceVector",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Force"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_force_vector(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_force_vector(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Force_get_ForceVector",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def part(self) -> Part:
        """
        The part that this force is applied to.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Force_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Force_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def position(self) -> Tuple[float,float,float]:
        """
        The position at which the force acts, in reference frame SpaceCenter.ReferenceFrame.

        :returns: The position as a vector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Force_get_Position",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @position.setter
    def position(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Force_set_Position",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Force"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Force_get_Position",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame of the force vector and position.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Force_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @reference_frame.setter
    def reference_frame(self, value: ReferenceFrame) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Force_set_ReferenceFrame",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Force"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            None
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Force_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Force")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def remove(self) -> None:
        """
        Remove the force.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Force_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Force"), ],
            None
        )

    def _return_type_remove(self) -> TypeBase:
        return None

    def _build_call_remove(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Force_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Force"), ],
            None
        )



class Intake(ClassBase):
    """
    An air intake. Obtained by calling SpaceCenter.Part#intake.
    """
    @property
    def area(self) -> float:
        """
        The area of the intake's opening, in square meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Intake_get_Area",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )

    def _return_type_area(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_area(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Intake_get_Area",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )

    @property
    def flow(self) -> float:
        """
        The rate of flow into the intake, in units of resource per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Intake_get_Flow",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )

    def _return_type_flow(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_flow(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Intake_get_Flow",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )

    @property
    def open(self) -> bool:
        """
        Whether the intake is open.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Intake_get_Open",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.bool_type
        )

    @open.setter
    def open(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Intake_set_Open",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Intake"), self._client._types.bool_type],
            None
        )

    def _return_type_open(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_open(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Intake_get_Open",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this intake.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Intake_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Intake_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def speed(self) -> float:
        """
        Speed of the flow into the intake, in ``m/s``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Intake_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )

    def _return_type_speed(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Intake_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Intake")],
            self._client._types.float_type
        )



class LaunchClamp(ClassBase):
    """
    A launch clamp. Obtained by calling SpaceCenter.Part#launchClamp.
    """
    @property
    def part(self) -> Part:
        """
        The part object for this launch clamp.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchClamp_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchClamp")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchClamp_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchClamp")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def release(self) -> None:
        """
        Releases the docking clamp. Has no effect if the clamp has already been released.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchClamp_Release",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "LaunchClamp"), ],
            None
        )

    def _return_type_release(self) -> TypeBase:
        return None

    def _build_call_release(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchClamp_Release",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "LaunchClamp"), ],
            None
        )



class LaunchSite(ClassBase):
    """
    A place where craft can be launched from.
    More of these can be added with mods like Kerbal Konstructs.
    """
    @property
    def body(self) -> CelestialBody:
        """
        The celestial body the launch site is on.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchSite_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    def _return_type_body(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CelestialBody")

    def _build_call_body(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchSite_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @property
    def editor_facility(self) -> EditorFacility:
        """
        Which editor is normally used for this launch site.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchSite_get_EditorFacility",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.enumeration_type("SpaceCenter", "EditorFacility")
        )

    def _return_type_editor_facility(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "EditorFacility")

    def _build_call_editor_facility(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchSite_get_EditorFacility",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.enumeration_type("SpaceCenter", "EditorFacility")
        )

    @property
    def name(self) -> str:
        """
        The name of the launch site.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchSite_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchSite_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "LaunchSite")],
            self._client._types.string_type
        )



class Leg(ClassBase):
    """
    A landing leg. Obtained by calling SpaceCenter.Part#leg.
    """
    @property
    def deployable(self) -> bool:
        """
        Whether the leg is deployable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Leg_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    def _return_type_deployable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Leg_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the landing leg is deployed.

        Fixed landing legs are always deployed.
        Returns an error if you try to deploy fixed landing gear.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Leg_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Leg_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Leg"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Leg_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    @property
    def is_grounded(self) -> bool:
        """
        Returns whether the leg is touching the ground.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Leg_get_IsGrounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    def _return_type_is_grounded(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_grounded(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Leg_get_IsGrounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this landing leg.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Leg_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Leg_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> LegState:
        """
        The current state of the landing leg.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Leg_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.enumeration_type("SpaceCenter", "LegState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "LegState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Leg_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Leg")],
            self._client._types.enumeration_type("SpaceCenter", "LegState")
        )



class Light(ClassBase):
    """
    A light. Obtained by calling SpaceCenter.Part#light.
    """
    @property
    def active(self) -> bool:
        """
        Whether the light is switched on.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.bool_type
        )

    @active.setter
    def active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Light_set_Active",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Light"), self._client._types.bool_type],
            None
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.bool_type
        )

    @property
    def blink(self) -> bool:
        """
        Whether blinking is enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_Blink",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.bool_type
        )

    @blink.setter
    def blink(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Light_set_Blink",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Light"), self._client._types.bool_type],
            None
        )

    def _return_type_blink(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_blink(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_Blink",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.bool_type
        )

    @property
    def blink_rate(self) -> float:
        """
        The blink rate of the light.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_BlinkRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.float_type
        )

    @blink_rate.setter
    def blink_rate(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Light_set_BlinkRate",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Light"), self._client._types.float_type],
            None
        )

    def _return_type_blink_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_blink_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_BlinkRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.float_type
        )

    @property
    def color(self) -> Tuple[float,float,float]:
        """
        The color of the light, as an RGB triple.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_Color",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.tuple_type(self._client._types.float_type, self._client._types.float_type, self._client._types.float_type)
        )

    @color.setter
    def color(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Light_set_Color",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Light"), self._client._types.tuple_type(self._client._types.float_type, self._client._types.float_type, self._client._types.float_type)],
            None
        )

    def _return_type_color(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.float_type, self._client._types.float_type, self._client._types.float_type)

    def _build_call_color(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_Color",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.tuple_type(self._client._types.float_type, self._client._types.float_type, self._client._types.float_type)
        )

    @property
    def part(self) -> Part:
        """
        The part object for this light.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def power_usage(self) -> float:
        """
        The current power usage, in units of charge per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Light_get_PowerUsage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.float_type
        )

    def _return_type_power_usage(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_power_usage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Light_get_PowerUsage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Light")],
            self._client._types.float_type
        )



class Module(ClassBase):
    """
    This can be used to interact with a specific part module. This includes part modules in
    stock KSP, and those added by mods.

    In KSP, each part has zero or more
    `PartModules <https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation#MODULES>`_
    associated with it. Each one contains some of the functionality of the part.
    For example, an engine has a "ModuleEngines" part module that contains all the
    functionality of an engine.
    """
    @property
    def actions(self) -> List[str]:
        """
        A list of all the names of the modules actions. These are the parts actions that can
        be assigned to action groups in the in-game editor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_Actions",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_actions(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_actions(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_Actions",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def actions_by_id(self) -> List[str]:
        """
        A list of all the identifiers of the modules actions. These are the parts actions
        that can be assigned to action groups in the in-game editor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_ActionsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_actions_by_id(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_actions_by_id(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_ActionsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def events(self) -> List[str]:
        """
        A list of the names of all of the modules events. Events are the clickable buttons
        visible in the right-click menu of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_Events",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_events(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_events(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_Events",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def events_by_id(self) -> List[str]:
        """
        A list of the identifiers of all of the modules events. Events are the clickable buttons
        visible in the right-click menu of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_EventsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_events_by_id(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_events_by_id(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_EventsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def fields(self) -> Dict[str, str]:
        """
        The modules field names and their associated values, as a dictionary.
        These are the values visible in the right-click menu of the part.

        Throws an exception if there is more than one field with the same name.
        In that case, use SpaceCenter.Module#fieldsById to get the fields by identifier.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_Fields",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)
        )

    def _return_type_fields(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)

    def _build_call_fields(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_Fields",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)
        )

    @property
    def fields_by_id(self) -> Dict[str, str]:
        """
        The modules field identifiers and their associated values, as a dictionary.
        These are the values visible in the right-click menu of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_FieldsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)
        )

    def _return_type_fields_by_id(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)

    def _build_call_fields_by_id(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_FieldsById",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.string_type)
        )

    @property
    def name(self) -> str:
        """
        Name of the PartModule. For example, "ModuleEngines".
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.string_type
        )

    @property
    def part(self) -> Part:
        """
        The part that contains this module.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Module")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def get_field(self, name: str) -> str:
        """
        Returns the value of a field with the given name.

        :param name: Name of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_GetField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.string_type
        )

    def _return_type_get_field(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_get_field(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_GetField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.string_type
        )

    def get_field_by_id(self, id: str) -> str:
        """
        Returns the value of a field with the given identifier.

        :param id: Identifier of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_GetFieldById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.string_type
        )

    def _return_type_get_field_by_id(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_get_field_by_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_GetFieldById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.string_type
        )

    def has_action(self, name: str) -> bool:
        """
        ``True`` if the part has an action with the given name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasAction",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_action(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_action(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasAction",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_action_with_id(self, id: str) -> bool:
        """
        ``True`` if the part has an action with the given identifier.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasActionWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_action_with_id(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_action_with_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasActionWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_event(self, name: str) -> bool:
        """
        ``True`` if the module has an event with the given name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasEvent",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_event(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_event(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasEvent",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_event_with_id(self, id: str) -> bool:
        """
        ``True`` if the module has an event with the given identifier.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasEventWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_event_with_id(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_event_with_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasEventWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_field(self, name: str) -> bool:
        """
        Returns ``True`` if the module has a field with the given name.

        :param name: Name of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_field(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_field(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_field_with_id(self, id: str) -> bool:
        """
        Returns ``True`` if the module has a field with the given identifier.

        :param id: Identifier of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_HasFieldWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_field_with_id(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_field_with_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_HasFieldWithId",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def reset_field(self, name: str) -> None:
        """
        Set the value of a field to its original value.

        :param name: Name of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_ResetField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def _return_type_reset_field(self) -> TypeBase:
        return None

    def _build_call_reset_field(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_ResetField",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def reset_field_by_id(self, id: str) -> None:
        """
        Set the value of a field to its original value.

        :param id: Identifier of the field.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_ResetFieldById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def _return_type_reset_field_by_id(self) -> TypeBase:
        return None

    def _build_call_reset_field_by_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_ResetFieldById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def set_action(self, name: str, value: bool = True) -> None:
        """
        Set the value of an action with the given name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetAction",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def _return_type_set_action(self) -> TypeBase:
        return None

    def _build_call_set_action(self, name: str, value: bool = True) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetAction",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def set_action_by_id(self, id: str, value: bool = True) -> None:
        """
        Set the value of an action with the given identifier.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetActionById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def _return_type_set_action_by_id(self) -> TypeBase:
        return None

    def _build_call_set_action_by_id(self, id: str, value: bool = True) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetActionById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def set_field_bool(self, name: str, value: bool) -> None:
        """
        Set the value of a field to true or false.

        :param name: Name of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldBool",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def _return_type_set_field_bool(self) -> TypeBase:
        return None

    def _build_call_set_field_bool(self, name: str, value: bool) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldBool",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def set_field_bool_by_id(self, id: str, value: bool) -> None:
        """
        Set the value of a field to true or false.

        :param id: Identifier of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldBoolById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def _return_type_set_field_bool_by_id(self) -> TypeBase:
        return None

    def _build_call_set_field_bool_by_id(self, id: str, value: bool) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldBoolById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def set_field_float(self, name: str, value: float) -> None:
        """
        Set the value of a field to the given floating point number.

        :param name: Name of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldFloat",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type],
            None
        )

    def _return_type_set_field_float(self) -> TypeBase:
        return None

    def _build_call_set_field_float(self, name: str, value: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldFloat",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type],
            None
        )

    def set_field_float_by_id(self, id: str, value: float) -> None:
        """
        Set the value of a field to the given floating point number.

        :param id: Identifier of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldFloatById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type],
            None
        )

    def _return_type_set_field_float_by_id(self) -> TypeBase:
        return None

    def _build_call_set_field_float_by_id(self, id: str, value: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldFloatById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type],
            None
        )

    def set_field_int(self, name: str, value: int) -> None:
        """
        Set the value of a field to the given integer number.

        :param name: Name of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldInt",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.sint32_type],
            None
        )

    def _return_type_set_field_int(self) -> TypeBase:
        return None

    def _build_call_set_field_int(self, name: str, value: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldInt",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.sint32_type],
            None
        )

    def set_field_int_by_id(self, id: str, value: int) -> None:
        """
        Set the value of a field to the given integer number.

        :param id: Identifier of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldIntById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.sint32_type],
            None
        )

    def _return_type_set_field_int_by_id(self) -> TypeBase:
        return None

    def _build_call_set_field_int_by_id(self, id: str, value: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldIntById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.sint32_type],
            None
        )

    def set_field_string(self, name: str, value: str) -> None:
        """
        Set the value of a field to the given string.

        :param name: Name of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldString",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.string_type],
            None
        )

    def _return_type_set_field_string(self) -> TypeBase:
        return None

    def _build_call_set_field_string(self, name: str, value: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldString",
            [self, name, value],
            ["self", "name", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.string_type],
            None
        )

    def set_field_string_by_id(self, id: str, value: str) -> None:
        """
        Set the value of a field to the given string.

        :param id: Identifier of the field.

        :param value: Value to set.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_SetFieldStringById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.string_type],
            None
        )

    def _return_type_set_field_string_by_id(self) -> TypeBase:
        return None

    def _build_call_set_field_string_by_id(self, id: str, value: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_SetFieldStringById",
            [self, id, value],
            ["self", "id", "value"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.string_type],
            None
        )

    def trigger_event(self, name: str) -> None:
        """
        Trigger the named event. Equivalent to clicking the button in the right-click menu
        of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_TriggerEvent",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def _return_type_trigger_event(self) -> TypeBase:
        return None

    def _build_call_trigger_event(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_TriggerEvent",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def trigger_event_by_id(self, id: str) -> None:
        """
        Trigger the event with the given identifier.
        Equivalent to clicking the button in the right-click menu of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Module_TriggerEventById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )

    def _return_type_trigger_event_by_id(self) -> TypeBase:
        return None

    def _build_call_trigger_event_by_id(self, id: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Module_TriggerEventById",
            [self, id],
            ["self", "id"],
            [self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            None
        )



class Node(ClassBase):
    """
    Represents a maneuver node. Can be created using SpaceCenter.Control#addNode.
    """
    @property
    def delta_v(self) -> float:
        """
        The delta-v of the maneuver node, in meters per second.

        Does not change when executing the maneuver node. See SpaceCenter.Node#remainingDeltaV.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_DeltaV",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @delta_v.setter
    def delta_v(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Node_set_DeltaV",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type],
            None
        )

    def _return_type_delta_v(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_delta_v(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_DeltaV",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def normal(self) -> float:
        """
        The magnitude of the maneuver nodes delta-v in the normal direction,
        in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_Normal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @normal.setter
    def normal(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Node_set_Normal",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type],
            None
        )

    def _return_type_normal(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_normal(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_Normal",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def orbit(self) -> Orbit:
        """
        The orbit that results from executing the maneuver node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    def _return_type_orbit(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Orbit")

    def _build_call_orbit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    @property
    def orbital_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the maneuver node, and
        orientated with the orbital prograde/normal/radial directions of the
        original orbit at the maneuver node's position.
         * The origin is at the position of the maneuver node.
         * The x-axis points in the orbital anti-radial direction of the original
           orbit, at the position of the maneuver node.
         * The y-axis points in the orbital prograde direction of the original
           orbit, at the position of the maneuver node.
         * The z-axis points in the orbital normal direction of the original orbit,
           at the position of the maneuver node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_orbital_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_orbital_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def prograde(self) -> float:
        """
        The magnitude of the maneuver nodes delta-v in the prograde direction,
        in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_Prograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @prograde.setter
    def prograde(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Node_set_Prograde",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type],
            None
        )

    def _return_type_prograde(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_prograde(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_Prograde",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def radial(self) -> float:
        """
        The magnitude of the maneuver nodes delta-v in the radial direction,
        in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_Radial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @radial.setter
    def radial(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Node_set_Radial",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type],
            None
        )

    def _return_type_radial(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_radial(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_Radial",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the maneuver node's burn.
         * The origin is at the position of the maneuver node.
         * The y-axis points in the direction of the burn.
         * The x-axis and z-axis point in arbitrary but fixed directions.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def remaining_delta_v(self) -> float:
        """
        Gets the remaining delta-v of the maneuver node, in meters per second. Changes as the
        node is executed. This is equivalent to the delta-v reported in-game.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_RemainingDeltaV",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    def _return_type_remaining_delta_v(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_remaining_delta_v(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_RemainingDeltaV",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def time_to(self) -> float:
        """
        The time until the maneuver node will be encountered, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_TimeTo",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    def _return_type_time_to(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_to(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_TimeTo",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @property
    def ut(self) -> float:
        """
        The universal time at which the maneuver will occur, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_get_UT",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    @ut.setter
    def ut(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Node_set_UT",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.double_type],
            None
        )

    def _return_type_ut(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_ut(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_get_UT",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Node")],
            self._client._types.double_type
        )

    def burn_vector(self, reference_frame: Optional[ReferenceFrame] = None) -> Tuple[float,float,float]:
        """
        Returns the burn vector for the maneuver node.

        :param reference_frame: The reference frame that the returned vector is in.
                                Defaults to SpaceCenter.Vessel#orbitalReferenceFrame.

        :returns: A vector whose direction is the direction of the maneuver node burn, and
                  magnitude is the delta-v of the burn in meters per second.

        Does not change when executing the maneuver node. See SpaceCenter.Node#remainingBurnVector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_BurnVector",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_burn_vector(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_burn_vector(self, reference_frame: Optional[ReferenceFrame] = None) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_BurnVector",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction of the maneuver nodes burn.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position vector of the maneuver node in the given reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def remaining_burn_vector(self, reference_frame: Optional[ReferenceFrame] = None) -> Tuple[float,float,float]:
        """
        Returns the remaining burn vector for the maneuver node.

        :param reference_frame: The reference frame that the returned vector is in.
                                Defaults to SpaceCenter.Vessel#orbitalReferenceFrame.

        :returns: A vector whose direction is the direction of the maneuver node burn, and
                  magnitude is the delta-v of the burn in meters per second.

        Changes as the maneuver node is executed. See SpaceCenter.Node#burnVector.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_RemainingBurnVector",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_remaining_burn_vector(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_remaining_burn_vector(self, reference_frame: Optional[ReferenceFrame] = None) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_RemainingBurnVector",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Node"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def remove(self) -> None:
        """
        Removes the maneuver node.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Node_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Node"), ],
            None
        )

    def _return_type_remove(self) -> TypeBase:
        return None

    def _build_call_remove(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Node_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Node"), ],
            None
        )



class Orbit(ClassBase):
    """
    Describes an orbit. For example, the orbit of a vessel, obtained by calling
    SpaceCenter.Vessel#orbit, or a celestial body, obtained by calling
    SpaceCenter.CelestialBody#orbit.
    """
    @property
    def apoapsis(self) -> float:
        """
        Gets the apoapsis of the orbit, in meters, from the center of mass
        of the body being orbited.

        For the apoapsis altitude reported on the in-game map view,
        use SpaceCenter.Orbit#apoapsisAltitude.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Apoapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_apoapsis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_apoapsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Apoapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def apoapsis_altitude(self) -> float:
        """
        The apoapsis of the orbit, in meters, above the sea level of the body being orbited.

        This is equal to SpaceCenter.Orbit#apoapsis minus the equatorial radius of the body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_ApoapsisAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_apoapsis_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_apoapsis_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_ApoapsisAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def argument_of_periapsis(self) -> float:
        """
        The `argument of periapsis <https://en.wikipedia.org/wiki/Argument_of_periapsis>`_, in radians.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_ArgumentOfPeriapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_argument_of_periapsis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_argument_of_periapsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_ArgumentOfPeriapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def body(self) -> CelestialBody:
        """
        The celestial body (e.g. planet or moon) around which the object is orbiting.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    def _return_type_body(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CelestialBody")

    def _build_call_body(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @property
    def eccentric_anomaly(self) -> float:
        """
        The `eccentric anomaly <https://en.wikipedia.org/wiki/Eccentric_anomaly>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_EccentricAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_eccentric_anomaly(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_eccentric_anomaly(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_EccentricAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def eccentricity(self) -> float:
        """
        The `eccentricity <https://en.wikipedia.org/wiki/Orbital_eccentricity>`_
        of the orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Eccentricity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_eccentricity(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_eccentricity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Eccentricity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def epoch(self) -> float:
        """
        The time since the epoch (the point at which the
        `mean anomaly at epoch <https://en.wikipedia.org/wiki/Mean_anomaly>`_
        was measured, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Epoch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_epoch(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_epoch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Epoch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def inclination(self) -> float:
        """
        The `inclination <https://en.wikipedia.org/wiki/Orbital_inclination>`_
        of the orbit,
        in radians.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Inclination",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_inclination(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_inclination(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Inclination",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def longitude_of_ascending_node(self) -> float:
        """
        The `longitude of the ascending node <https://en.wikipedia.org/wiki/Longitude_of_the_ascending_node>`_, in radians.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_LongitudeOfAscendingNode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_longitude_of_ascending_node(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_longitude_of_ascending_node(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_LongitudeOfAscendingNode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def mean_anomaly(self) -> float:
        """
        The `mean anomaly <https://en.wikipedia.org/wiki/Mean_anomaly>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_MeanAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_mean_anomaly(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mean_anomaly(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_MeanAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def mean_anomaly_at_epoch(self) -> float:
        """
        The `mean anomaly at epoch <https://en.wikipedia.org/wiki/Mean_anomaly>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_MeanAnomalyAtEpoch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_mean_anomaly_at_epoch(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mean_anomaly_at_epoch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_MeanAnomalyAtEpoch",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def next_orbit(self) -> Optional[Orbit]:
        """
        If the object is going to change sphere of influence in the future, returns the new
        orbit after the change. Otherwise returns ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_NextOrbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    def _return_type_next_orbit(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Orbit")

    def _build_call_next_orbit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_NextOrbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    @property
    def orbital_speed(self) -> float:
        """
        The current orbital speed in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_OrbitalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_orbital_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_orbital_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_OrbitalSpeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def periapsis(self) -> float:
        """
        The periapsis of the orbit, in meters, from the center of mass
        of the body being orbited.

        For the periapsis altitude reported on the in-game map view,
        use SpaceCenter.Orbit#periapsisAltitude.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Periapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_periapsis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_periapsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Periapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def periapsis_altitude(self) -> float:
        """
        The periapsis of the orbit, in meters, above the sea level of the body being orbited.

        This is equal to SpaceCenter.Orbit#periapsis minus the equatorial radius of the body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_PeriapsisAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_periapsis_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_periapsis_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_PeriapsisAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def period(self) -> float:
        """
        The orbital period, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Period",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_period(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_period(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Period",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def radius(self) -> float:
        """
        The current radius of the orbit, in meters. This is the distance between the center
        of mass of the object in orbit, and the center of mass of the body around which it
        is orbiting.

        This value will change over time if the orbit is elliptical.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Radius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_radius(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_radius(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Radius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def semi_major_axis(self) -> float:
        """
        The semi-major axis of the orbit, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_SemiMajorAxis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_semi_major_axis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_semi_major_axis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_SemiMajorAxis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def semi_minor_axis(self) -> float:
        """
        The semi-minor axis of the orbit, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_SemiMinorAxis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_semi_minor_axis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_semi_minor_axis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_SemiMinorAxis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def speed(self) -> float:
        """
        The current orbital speed of the object in meters per second.

        This value will change over time if the orbit is elliptical.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_speed(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_speed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_Speed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def time_to_apoapsis(self) -> float:
        """
        The time until the object reaches apoapsis, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_TimeToApoapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_time_to_apoapsis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_to_apoapsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_TimeToApoapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def time_to_periapsis(self) -> float:
        """
        The time until the object reaches periapsis, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_TimeToPeriapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_time_to_periapsis(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_to_periapsis(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_TimeToPeriapsis",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def time_to_soi_change(self) -> float:
        """
        The time until the object changes sphere of influence, in seconds. Returns ``NaN``
        if the object is not going to change sphere of influence.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_TimeToSOIChange",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_time_to_soi_change(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_to_soi_change(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_TimeToSOIChange",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    @property
    def true_anomaly(self) -> float:
        """
        The `true anomaly <https://en.wikipedia.org/wiki/True_anomaly>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_get_TrueAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_true_anomaly(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_true_anomaly(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_get_TrueAnomaly",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def distance_at_closest_approach(self, target: Orbit) -> float:
        """
        Estimates and returns the distance at closest approach to a target orbit, in meters.

        :param target: Target orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_DistanceAtClosestApproach",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_distance_at_closest_approach(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_distance_at_closest_approach(self, target: Orbit) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_DistanceAtClosestApproach",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def eccentric_anomaly_at_ut(self, ut: float) -> float:
        """
        The eccentric anomaly at the given universal time.

        :param ut: The universal time, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_EccentricAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_eccentric_anomaly_at_ut(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_eccentric_anomaly_at_ut(self, ut: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_EccentricAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def list_closest_approaches(self, target: Orbit, orbits: int) -> List[List[float]]:
        """
        Returns the times at closest approach and corresponding distances, to a target orbit.

        :returns: A list of two lists.
                  The first is a list of times at closest approach, as universal times in seconds.
                  The second is a list of corresponding distances at closest approach, in meters.

        :param target: Target orbit.

        :param orbits: The number of future orbits to search.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_ListClosestApproaches",
            [self, target, orbits],
            ["self", "target", "orbits"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.list_type(self._client._types.double_type))
        )

    def _return_type_list_closest_approaches(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.list_type(self._client._types.double_type))

    def _build_call_list_closest_approaches(self, target: Orbit, orbits: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_ListClosestApproaches",
            [self, target, orbits],
            ["self", "target", "orbits"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.list_type(self._client._types.double_type))
        )

    def mean_anomaly_at_ut(self, ut: float) -> float:
        """
        The mean anomaly at the given time.

        :param ut: The universal time in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_MeanAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_mean_anomaly_at_ut(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mean_anomaly_at_ut(self, ut: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_MeanAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def orbital_speed_at(self, time: float) -> float:
        """
        The orbital speed at the given time, in meters per second.

        :param time: Time from now, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_OrbitalSpeedAt",
            [self, time],
            ["self", "time"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_orbital_speed_at(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_orbital_speed_at(self, time: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_OrbitalSpeedAt",
            [self, time],
            ["self", "time"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def position_at(self, ut: float, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position at a given time, in the specified reference frame.

        :returns: The position as a vector.

        :param ut: The universal time to measure the position at.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_PositionAt",
            [self, ut, reference_frame],
            ["self", "ut", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position_at(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position_at(self, ut: float, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_PositionAt",
            [self, ut, reference_frame],
            ["self", "ut", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type, self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def radius_at(self, ut: float) -> float:
        """
        The orbital radius at the given time, in meters.

        :param ut: The universal time to measure the radius at.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_RadiusAt",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_radius_at(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_radius_at(self, ut: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_RadiusAt",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def radius_at_true_anomaly(self, true_anomaly: float) -> float:
        """
        The orbital radius at the point in the orbit given by the true anomaly.

        :param true_anomaly: The true anomaly.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_RadiusAtTrueAnomaly",
            [self, true_anomaly],
            ["self", "true_anomaly"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_radius_at_true_anomaly(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_radius_at_true_anomaly(self, true_anomaly: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_RadiusAtTrueAnomaly",
            [self, true_anomaly],
            ["self", "true_anomaly"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def relative_inclination(self, target: Orbit) -> float:
        """
        Relative inclination of this orbit and the target orbit, in radians.

        :param target: Target orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_RelativeInclination",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_relative_inclination(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_relative_inclination(self, target: Orbit) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_RelativeInclination",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def time_of_closest_approach(self, target: Orbit) -> float:
        """
        Estimates and returns the time at closest approach to a target orbit.

        :returns: The universal time at closest approach, in seconds.

        :param target: Target orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_TimeOfClosestApproach",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_time_of_closest_approach(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_time_of_closest_approach(self, target: Orbit) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_TimeOfClosestApproach",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def true_anomaly_at_an(self, target: Orbit) -> float:
        """
        The true anomaly of the ascending node with the given target orbit.

        :param target: Target orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtAN",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_true_anomaly_at_an(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_true_anomaly_at_an(self, target: Orbit) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtAN",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def true_anomaly_at_dn(self, target: Orbit) -> float:
        """
        The true anomaly of the descending node with the given target orbit.

        :param target: Target orbit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtDN",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def _return_type_true_anomaly_at_dn(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_true_anomaly_at_dn(self, target: Orbit) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtDN",
            [self, target],
            ["self", "target"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.class_type("SpaceCenter", "Orbit")],
            self._client._types.double_type
        )

    def true_anomaly_at_radius(self, radius: float) -> float:
        """
        The true anomaly at the given orbital radius.

        :param radius: The orbital radius in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtRadius",
            [self, radius],
            ["self", "radius"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_true_anomaly_at_radius(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_true_anomaly_at_radius(self, radius: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtRadius",
            [self, radius],
            ["self", "radius"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def true_anomaly_at_ut(self, ut: float) -> float:
        """
        The true anomaly at the given time.

        :param ut: The universal time in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_true_anomaly_at_ut(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_true_anomaly_at_ut(self, ut: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_TrueAnomalyAtUT",
            [self, ut],
            ["self", "ut"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def ut_at_true_anomaly(self, true_anomaly: float) -> float:
        """
        The universal time, in seconds, corresponding to the given true anomaly.

        :param true_anomaly: True anomaly.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Orbit_UTAtTrueAnomaly",
            [self, true_anomaly],
            ["self", "true_anomaly"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    def _return_type_ut_at_true_anomaly(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_ut_at_true_anomaly(self, true_anomaly: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_UTAtTrueAnomaly",
            [self, true_anomaly],
            ["self", "true_anomaly"],
            [self._client._types.class_type("SpaceCenter", "Orbit"), self._client._types.double_type],
            self._client._types.double_type
        )

    @classmethod
    def reference_plane_direction(cls, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction from which the orbits longitude of ascending node is measured,
        in the given reference frame.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "Orbit_static_ReferencePlaneDirection",
            [reference_frame],
            ["reference_frame"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @classmethod
    def _return_type_reference_plane_direction(cls) -> TypeBase:
        self = cls
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    @classmethod
    def _build_call_reference_plane_direction(cls, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_static_ReferencePlaneDirection",
            [reference_frame],
            ["reference_frame"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @classmethod
    def reference_plane_normal(cls, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction that is normal to the orbits reference plane,
        in the given reference frame.
        The reference plane is the plane from which the orbits inclination is measured.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "Orbit_static_ReferencePlaneNormal",
            [reference_frame],
            ["reference_frame"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @classmethod
    def _return_type_reference_plane_normal(cls) -> TypeBase:
        self = cls
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    @classmethod
    def _build_call_reference_plane_normal(cls, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "Orbit_static_ReferencePlaneNormal",
            [reference_frame],
            ["reference_frame"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class Parachute(ClassBase):
    """
    A parachute. Obtained by calling SpaceCenter.Part#parachute.
    """
    @property
    def armed(self) -> bool:
        """
        Whether the parachute has been armed or deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_Armed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.bool_type
        )

    def _return_type_armed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_armed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_Armed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.bool_type
        )

    @property
    def deploy_altitude(self) -> float:
        """
        The altitude at which the parachute will full deploy, in meters.
        Only applicable to stock parachutes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_DeployAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.float_type
        )

    @deploy_altitude.setter
    def deploy_altitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_set_DeployAltitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Parachute"), self._client._types.float_type],
            None
        )

    def _return_type_deploy_altitude(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_deploy_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_DeployAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.float_type
        )

    @property
    def deploy_min_pressure(self) -> float:
        """
        The minimum pressure at which the parachute will semi-deploy, in atmospheres.
        Only applicable to stock parachutes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_DeployMinPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.float_type
        )

    @deploy_min_pressure.setter
    def deploy_min_pressure(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_set_DeployMinPressure",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Parachute"), self._client._types.float_type],
            None
        )

    def _return_type_deploy_min_pressure(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_deploy_min_pressure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_DeployMinPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.float_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the parachute has been deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.bool_type
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this parachute.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> ParachuteState:
        """
        The current state of the parachute.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.enumeration_type("SpaceCenter", "ParachuteState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ParachuteState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parachute")],
            self._client._types.enumeration_type("SpaceCenter", "ParachuteState")
        )

    def arm(self) -> None:
        """
        Deploys the parachute. This has no effect if the parachute has already
        been armed or deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_Arm",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )

    def _return_type_arm(self) -> TypeBase:
        return None

    def _build_call_arm(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_Arm",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )

    def cut(self) -> None:
        """
        Cuts the parachute.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_Cut",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )

    def _return_type_cut(self) -> TypeBase:
        return None

    def _build_call_cut(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_Cut",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )

    def deploy(self) -> None:
        """
        Deploys the parachute. This has no effect if the parachute has already
        been deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parachute_Deploy",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )

    def _return_type_deploy(self) -> TypeBase:
        return None

    def _build_call_deploy(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parachute_Deploy",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Parachute"), ],
            None
        )



class Part(ClassBase):
    """
    Represents an individual part. Vessels are made up of multiple parts.
    Instances of this class can be obtained by several methods in SpaceCenter.Parts.
    """
    @property
    def antenna(self) -> Optional[Antenna]:
        """
        An SpaceCenter.Antenna if the part is an antenna, otherwise ``None``.

        If RemoteTech is installed, this will always return ``None``.
        To interact with RemoteTech antennas, use the RemoteTech service APIs.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Antenna",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Antenna")
        )

    def _return_type_antenna(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Antenna")

    def _build_call_antenna(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Antenna",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Antenna")
        )

    @property
    def auto_strut_mode(self) -> AutoStrutMode:
        """
        Auto-strut mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_AutoStrutMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.enumeration_type("SpaceCenter", "AutoStrutMode")
        )

    def _return_type_auto_strut_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "AutoStrutMode")

    def _build_call_auto_strut_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_AutoStrutMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.enumeration_type("SpaceCenter", "AutoStrutMode")
        )

    @property
    def available_seats(self) -> int:
        """
        How many open seats the part has.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_AvailableSeats",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.uint32_type
        )

    def _return_type_available_seats(self) -> TypeBase:
        return self._client._types.uint32_type

    def _build_call_available_seats(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_AvailableSeats",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.uint32_type
        )

    @property
    def axially_attached(self) -> bool:
        """
        Whether the part is axially attached to its parent, i.e. on the top
        or bottom of its parent. If the part has no parent, returns ``False``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_AxiallyAttached",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_axially_attached(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_axially_attached(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_AxiallyAttached",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def cargo_bay(self) -> Optional[CargoBay]:
        """
        A SpaceCenter.CargoBay if the part is a cargo bay, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_CargoBay",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "CargoBay")
        )

    def _return_type_cargo_bay(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CargoBay")

    def _build_call_cargo_bay(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_CargoBay",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "CargoBay")
        )

    @property
    def center_of_mass_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to this part, and centered on its
        center of mass.
         * The origin is at the center of mass of the part, as returned by
           SpaceCenter.Part#centerOfMass.
         * The axes rotate with the part.
         * The x, y and z axis directions depend on the design of the part.

        For docking port parts, this reference frame is not necessarily equivalent to the
        reference frame for the docking port, returned by
        SpaceCenter.DockingPort#referenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_CenterOfMassReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_center_of_mass_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_center_of_mass_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_CenterOfMassReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def children(self) -> List[Part]:
        """
        The parts children. Returns an empty list if the part has no children.
        This, in combination with SpaceCenter.Part#parent, can be used to traverse the vessels
        parts tree.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Children",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_children(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_children(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Children",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    @property
    def control_surface(self) -> Optional[ControlSurface]:
        """
        A SpaceCenter.ControlSurface if the part is an aerodynamic control surface,
        otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ControlSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ControlSurface")
        )

    def _return_type_control_surface(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ControlSurface")

    def _build_call_control_surface(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ControlSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ControlSurface")
        )

    @property
    def cost(self) -> float:
        """
        The cost of the part, in units of funds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Cost",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_cost(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_cost(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Cost",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def crossfeed(self) -> bool:
        """
        Whether this part is crossfeed capable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Crossfeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_crossfeed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_crossfeed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Crossfeed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def decouple_stage(self) -> int:
        """
        The stage in which this part will be decoupled. Returns -1 if the part is never
        decoupled from the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_DecoupleStage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.sint32_type
        )

    def _return_type_decouple_stage(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_decouple_stage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_DecoupleStage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.sint32_type
        )

    @property
    def decoupler(self) -> Optional[Decoupler]:
        """
        A SpaceCenter.Decoupler if the part is a decoupler, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Decoupler",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Decoupler")
        )

    def _return_type_decoupler(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Decoupler")

    def _build_call_decoupler(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Decoupler",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Decoupler")
        )

    @property
    def docking_port(self) -> Optional[DockingPort]:
        """
        A SpaceCenter.DockingPort if the part is a docking port, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_DockingPort",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "DockingPort")
        )

    def _return_type_docking_port(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "DockingPort")

    def _build_call_docking_port(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_DockingPort",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "DockingPort")
        )

    @property
    def dry_mass(self) -> float:
        """
        The mass of the part, not including any resources it contains, in kilograms.
        Returns zero if the part is massless.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_DryMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_dry_mass(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_dry_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_DryMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def dynamic_pressure(self) -> float:
        """
        The dynamic pressure acting on the part, in Pascals.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_DynamicPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_dynamic_pressure(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_dynamic_pressure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_DynamicPressure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def engine(self) -> Optional[Engine]:
        """
        An SpaceCenter.Engine if the part is an engine, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Engine",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Engine")
        )

    def _return_type_engine(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Engine")

    def _build_call_engine(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Engine",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Engine")
        )

    @property
    def experiment(self) -> Optional[Experiment]:
        """
        An SpaceCenter.Experiment if the part contains a
        single science experiment, otherwise ``None``.

        Throws an exception if the part contains more than one experiment.
        In that case, use SpaceCenter.Part#experiments to get the list of experiments in the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Experiment",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Experiment")
        )

    def _return_type_experiment(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Experiment")

    def _build_call_experiment(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Experiment",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Experiment")
        )

    @property
    def experiments(self) -> Optional[List[Experiment]]:
        """
        A list of SpaceCenter.Experiment objects that the part contains.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Experiments",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))
        )

    def _return_type_experiments(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))

    def _build_call_experiments(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Experiments",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))
        )

    @property
    def fairing(self) -> Optional[Fairing]:
        """
        A SpaceCenter.Fairing if the part is a fairing, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Fairing",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Fairing")
        )

    def _return_type_fairing(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Fairing")

    def _build_call_fairing(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Fairing",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Fairing")
        )

    @property
    def flag_url(self) -> str:
        """
        The asset URL for the part's flag.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_FlagURL",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @flag_url.setter
    def flag_url(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Part_set_FlagURL",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.string_type],
            None
        )

    def _return_type_flag_url(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_flag_url(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_FlagURL",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @property
    def fuel_lines_from(self) -> List[Part]:
        """
        The parts that are connected to this part via fuel lines, where the direction of the
        fuel line is into this part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_FuelLinesFrom",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_fuel_lines_from(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_fuel_lines_from(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_FuelLinesFrom",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    @property
    def fuel_lines_to(self) -> List[Part]:
        """
        The parts that are connected to this part via fuel lines, where the direction of the
        fuel line is out of this part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_FuelLinesTo",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_fuel_lines_to(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_fuel_lines_to(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_FuelLinesTo",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    @property
    def glow(self) -> bool:
        raise NotImplementedError

    @glow.setter
    def glow(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Part_set_Glow",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.bool_type],
            None
        )

    @property
    def highlight_color(self) -> Tuple[float,float,float]:
        """
        The color used to highlight the part, as an RGB triple.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_HighlightColor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @highlight_color.setter
    def highlight_color(self, value: Tuple[float,float,float]) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Part_set_HighlightColor",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            None
        )

    def _return_type_highlight_color(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_highlight_color(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_HighlightColor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def highlighted(self) -> bool:
        """
        Whether the part is highlighted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Highlighted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @highlighted.setter
    def highlighted(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Part_set_Highlighted",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.bool_type],
            None
        )

    def _return_type_highlighted(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_highlighted(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Highlighted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def impact_tolerance(self) -> float:
        """
        The impact tolerance of the part, in meters per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ImpactTolerance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_impact_tolerance(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_impact_tolerance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ImpactTolerance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def inertia_tensor(self) -> List[float]:
        """
        The inertia tensor of the part in the parts reference frame
        (SpaceCenter.ReferenceFrame).
        Returns the 3x3 matrix as a list of elements, in row-major order.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_InertiaTensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.double_type)
        )

    def _return_type_inertia_tensor(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.double_type)

    def _build_call_inertia_tensor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_InertiaTensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.double_type)
        )

    @property
    def intake(self) -> Optional[Intake]:
        """
        An SpaceCenter.Intake if the part is an intake, otherwise ``None``.

        This includes any part that generates thrust. This covers many different types
        of engine, including liquid fuel rockets, solid rocket boosters and jet engines.
        For RCS thrusters see SpaceCenter.RCS.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Intake",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Intake")
        )

    def _return_type_intake(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Intake")

    def _build_call_intake(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Intake",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Intake")
        )

    @property
    def is_fuel_line(self) -> bool:
        """
        Whether this part is a fuel line.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_IsFuelLine",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_is_fuel_line(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_fuel_line(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_IsFuelLine",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def launch_clamp(self) -> Optional[LaunchClamp]:
        """
        A SpaceCenter.LaunchClamp if the part is a launch clamp, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_LaunchClamp",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "LaunchClamp")
        )

    def _return_type_launch_clamp(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "LaunchClamp")

    def _build_call_launch_clamp(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_LaunchClamp",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "LaunchClamp")
        )

    @property
    def leg(self) -> Optional[Leg]:
        """
        A SpaceCenter.Leg if the part is a landing leg, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Leg",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Leg")
        )

    def _return_type_leg(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Leg")

    def _build_call_leg(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Leg",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Leg")
        )

    @property
    def light(self) -> Optional[Light]:
        """
        A SpaceCenter.Light if the part is a light, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Light",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Light")
        )

    def _return_type_light(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Light")

    def _build_call_light(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Light",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Light")
        )

    @property
    def mass(self) -> float:
        """
        The current mass of the part, including resources it contains, in kilograms.
        Returns zero if the part is massless.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_mass(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def massless(self) -> bool:
        """
        Whether the part is
        `massless <https://wiki.kerbalspaceprogram.com/wiki/Massless_part>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Massless",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_massless(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_massless(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Massless",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def max_skin_temperature(self) -> float:
        """
        Maximum temperature that the skin of the part can survive, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_MaxSkinTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_max_skin_temperature(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_max_skin_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_MaxSkinTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def max_temperature(self) -> float:
        """
        Maximum temperature that the part can survive, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_MaxTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_max_temperature(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_max_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_MaxTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def modules(self) -> List[Module]:
        """
        The modules for this part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Modules",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))
        )

    def _return_type_modules(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))

    def _build_call_modules(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Modules",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))
        )

    @property
    def moment_of_inertia(self) -> Tuple[float,float,float]:
        """
        The moment of inertia of the part in ``kg.m^2`` around its center of mass
        in the parts reference frame (SpaceCenter.ReferenceFrame).
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_MomentOfInertia",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_moment_of_inertia(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_moment_of_inertia(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_MomentOfInertia",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def name(self) -> str:
        """
        Internal name of the part, as used in
        `part cfg files <https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation>`_.
        For example "Mark1-2Pod".
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @property
    def parachute(self) -> Optional[Parachute]:
        """
        A SpaceCenter.Parachute if the part is a parachute, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Parachute",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Parachute")
        )

    def _return_type_parachute(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Parachute")

    def _build_call_parachute(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Parachute",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Parachute")
        )

    @property
    def parent(self) -> Optional[Part]:
        """
        The parts parent. Returns ``None`` if the part does not have a parent.
        This, in combination with SpaceCenter.Part#children, can be used to traverse the vessels
        parts tree.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Parent",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_parent(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_parent(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Parent",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def radially_attached(self) -> bool:
        """
        Whether the part is radially attached to its parent, i.e. on the side of its parent.
        If the part has no parent, returns ``False``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RadiallyAttached",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_radially_attached(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_radially_attached(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RadiallyAttached",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def radiator(self) -> Optional[Radiator]:
        """
        A SpaceCenter.Radiator if the part is a radiator, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Radiator",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Radiator")
        )

    def _return_type_radiator(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Radiator")

    def _build_call_radiator(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Radiator",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Radiator")
        )

    @property
    def rcs(self) -> Optional[RCS]:
        """
        A SpaceCenter.RCS if the part is an RCS block/thruster, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RCS")
        )

    def _return_type_rcs(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RCS")

    def _build_call_rcs(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RCS")
        )

    @property
    def reaction_wheel(self) -> Optional[ReactionWheel]:
        """
        A SpaceCenter.ReactionWheel if the part is a reaction wheel, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ReactionWheel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReactionWheel")
        )

    def _return_type_reaction_wheel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReactionWheel")

    def _build_call_reaction_wheel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ReactionWheel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReactionWheel")
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to this part, and centered on a fixed
        position within the part, defined by the parts model.
         * The origin is at the position of the part, as returned by
           SpaceCenter.Part#position.
         * The axes rotate with the part.
         * The x, y and z axis directions depend on the design of the part.

        For docking port parts, this reference frame is not necessarily equivalent to the
        reference frame for the docking port, returned by
        SpaceCenter.DockingPort#referenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def resource_converter(self) -> Optional[ResourceConverter]:
        """
        A SpaceCenter.ResourceConverter if the part is a resource converter,
        otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ResourceConverter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceConverter")
        )

    def _return_type_resource_converter(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ResourceConverter")

    def _build_call_resource_converter(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ResourceConverter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceConverter")
        )

    @property
    def resource_drain(self) -> Optional[ResourceDrain]:
        """
        A SpaceCenter.ResourceDrain if the part is a resource drain, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ResourceDrain",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceDrain")
        )

    def _return_type_resource_drain(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ResourceDrain")

    def _build_call_resource_drain(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ResourceDrain",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceDrain")
        )

    @property
    def resource_harvester(self) -> Optional[ResourceHarvester]:
        """
        A SpaceCenter.ResourceHarvester if the part is a resource harvester,
        otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ResourceHarvester",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceHarvester")
        )

    def _return_type_resource_harvester(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ResourceHarvester")

    def _build_call_resource_harvester(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ResourceHarvester",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "ResourceHarvester")
        )

    @property
    def resources(self) -> Resources:
        """
        A SpaceCenter.Resources object for the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Resources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    def _return_type_resources(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Resources")

    def _build_call_resources(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Resources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    @property
    def robotic_controller(self) -> Optional[RoboticController]:
        """
        A SpaceCenter.RoboticController if the part is a robotic controller,
        otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RoboticController",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticController")
        )

    def _return_type_robotic_controller(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RoboticController")

    def _build_call_robotic_controller(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RoboticController",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticController")
        )

    @property
    def robotic_hinge(self) -> Optional[RoboticHinge]:
        """
        A SpaceCenter.RoboticHinge if the part is a robotic hinge, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RoboticHinge",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticHinge")
        )

    def _return_type_robotic_hinge(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RoboticHinge")

    def _build_call_robotic_hinge(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RoboticHinge",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticHinge")
        )

    @property
    def robotic_piston(self) -> Optional[RoboticPiston]:
        """
        A SpaceCenter.RoboticPiston if the part is a robotic piston, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RoboticPiston",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticPiston")
        )

    def _return_type_robotic_piston(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RoboticPiston")

    def _build_call_robotic_piston(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RoboticPiston",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticPiston")
        )

    @property
    def robotic_rotation(self) -> Optional[RoboticRotation]:
        """
        A SpaceCenter.RoboticRotation if the part is a robotic rotation servo, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RoboticRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticRotation")
        )

    def _return_type_robotic_rotation(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RoboticRotation")

    def _build_call_robotic_rotation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RoboticRotation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticRotation")
        )

    @property
    def robotic_rotor(self) -> Optional[RoboticRotor]:
        """
        A SpaceCenter.RoboticRotor if the part is a robotic rotor, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_RoboticRotor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticRotor")
        )

    def _return_type_robotic_rotor(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "RoboticRotor")

    def _build_call_robotic_rotor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_RoboticRotor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "RoboticRotor")
        )

    @property
    def sensor(self) -> Optional[Sensor]:
        """
        A SpaceCenter.Sensor if the part is a sensor, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Sensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Sensor")
        )

    def _return_type_sensor(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Sensor")

    def _build_call_sensor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Sensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Sensor")
        )

    @property
    def shielded(self) -> bool:
        """
        Whether the part is shielded from the exterior of the vessel, for example by a fairing.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Shielded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_shielded(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_shielded(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Shielded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    @property
    def skin_temperature(self) -> float:
        """
        Temperature of the skin of the part, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_SkinTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_skin_temperature(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_skin_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_SkinTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def solar_panel(self) -> Optional[SolarPanel]:
        """
        A SpaceCenter.SolarPanel if the part is a solar panel, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_SolarPanel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "SolarPanel")
        )

    def _return_type_solar_panel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "SolarPanel")

    def _build_call_solar_panel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_SolarPanel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "SolarPanel")
        )

    @property
    def stage(self) -> int:
        """
        The stage in which this part will be activated. Returns -1 if the part is not
        activated by staging.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Stage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.sint32_type
        )

    def _return_type_stage(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_stage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Stage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.sint32_type
        )

    @property
    def tag(self) -> str:
        """
        The name tag for the part. Can be set to a custom string using the
        in-game user interface.

        This string is shared with
        `kOS <https://forum.kerbalspaceprogram.com/index.php?/topic/61827-/>`_
        if it is installed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Tag",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @tag.setter
    def tag(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Part_set_Tag",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.string_type],
            None
        )

    def _return_type_tag(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_tag(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Tag",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @property
    def temperature(self) -> float:
        """
        Temperature of the part, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Temperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    def _return_type_temperature(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Temperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.double_type
        )

    @property
    def thermal_conduction_flux(self) -> float:
        """
        The rate at which heat energy is conducting into or out of the part via contact with
        other parts. Measured in energy per unit time, or power, in Watts.
        A positive value means the part is gaining heat energy, and negative means it is
        losing heat energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalConductionFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_conduction_flux(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_conduction_flux(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalConductionFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_convection_flux(self) -> float:
        """
        The rate at which heat energy is convecting into or out of the part from the
        surrounding atmosphere. Measured in energy per unit time, or power, in Watts.
        A positive value means the part is gaining heat energy, and negative means it is
        losing heat energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalConvectionFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_convection_flux(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_convection_flux(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalConvectionFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_internal_flux(self) -> float:
        """
        The rate at which heat energy is begin generated by the part.
        For example, some engines generate heat by combusting fuel.
        Measured in energy per unit time, or power, in Watts.
        A positive value means the part is gaining heat energy, and negative means it is losing
        heat energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalInternalFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_internal_flux(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_internal_flux(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalInternalFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_mass(self) -> float:
        """
        A measure of how much energy it takes to increase the internal temperature of the part,
        in Joules per Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_mass(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_radiation_flux(self) -> float:
        """
        The rate at which heat energy is radiating into or out of the part from the surrounding
        environment. Measured in energy per unit time, or power, in Watts.
        A positive value means the part is gaining heat energy, and negative means it is
        losing heat energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalRadiationFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_radiation_flux(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_radiation_flux(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalRadiationFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_resource_mass(self) -> float:
        """
        A measure of how much energy it takes to increase the temperature of the resources
        contained in the part, in Joules per Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalResourceMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_resource_mass(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_resource_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalResourceMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_skin_mass(self) -> float:
        """
        A measure of how much energy it takes to increase the skin temperature of the part,
        in Joules per Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalSkinMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_skin_mass(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_skin_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalSkinMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def thermal_skin_to_internal_flux(self) -> float:
        """
        The rate at which heat energy is transferring between the part's skin and its internals.
        Measured in energy per unit time, or power, in Watts.
        A positive value means the part's internals are gaining heat energy,
        and negative means its skin is gaining heat energy.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_ThermalSkinToInternalFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    def _return_type_thermal_skin_to_internal_flux(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_skin_to_internal_flux(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_ThermalSkinToInternalFlux",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.float_type
        )

    @property
    def title(self) -> str:
        """
        Title of the part, as shown when the part is right clicked in-game. For example "Mk1-2 Command Pod".
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.string_type
        )

    @property
    def vessel(self) -> Vessel:
        """
        The vessel that contains this part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    def _return_type_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Vessel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @property
    def wheel(self) -> Optional[Wheel]:
        """
        A SpaceCenter.Wheel if the part is a wheel, otherwise ``None``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_get_Wheel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Wheel")
        )

    def _return_type_wheel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Wheel")

    def _build_call_wheel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_get_Wheel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.class_type("SpaceCenter", "Wheel")
        )

    def add_force(self, force: Tuple[float,float,float], position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> Force:
        """
        Exert a constant force on the part, acting at the given position.

        :returns: An object that can be used to remove or modify the force.

        :param force: A vector pointing in the direction that the force acts,
                      with its magnitude equal to the strength of the force in Newtons.

        :param position: The position at which the force acts, as a vector.

        :param reference_frame: The reference frame that the
                                force and position are in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_AddForce",
            [self, force, position, reference_frame],
            ["self", "force", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "Force")
        )

    def _return_type_add_force(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Force")

    def _build_call_add_force(self, force: Tuple[float,float,float], position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_AddForce",
            [self, force, position, reference_frame],
            ["self", "force", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "Force")
        )

    def bounding_box(self, reference_frame: ReferenceFrame) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The axis-aligned bounding box of the part in the given reference frame.

        :returns: The positions of the minimum and maximum vertices of the box,
                  as position vectors.

        :param reference_frame: The reference frame that the returned
                                position vectors are in.

        This is computed from the collision mesh of the part.
        If the part is not collidable, the box has zero volume and is centered on
        the SpaceCenter.Part#position of the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_BoundingBox",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_bounding_box(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_bounding_box(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_BoundingBox",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def center_of_mass(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the parts center of mass in the given reference frame.
        If the part is physicsless, this is equivalent to SpaceCenter.Part#position.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_CenterOfMass",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_center_of_mass(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_center_of_mass(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_CenterOfMass",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction the part points in, in the given reference frame.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def instantaneous_force(self, force: Tuple[float,float,float], position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> None:
        """
        Exert an instantaneous force on the part, acting at the given position.

        :param force: A vector pointing in the direction that the force acts,
                      with its magnitude equal to the strength of the force in Newtons.

        :param position: The position at which the force acts, as a vector.

        :param reference_frame: The reference frame that the
                                force and position are in.

        The force is applied instantaneously in a single physics update.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_InstantaneousForce",
            [self, force, position, reference_frame],
            ["self", "force", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            None
        )

    def _return_type_instantaneous_force(self) -> TypeBase:
        return None

    def _build_call_instantaneous_force(self, force: Tuple[float,float,float], position: Tuple[float,float,float], reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_InstantaneousForce",
            [self, force, position, reference_frame],
            ["self", "force", "position", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            None
        )

    def position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the part in the given reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.

        This is a fixed position in the part, defined by the parts model.
        It s not necessarily the same as the parts center of mass.
        Use SpaceCenter.Part#centerOfMass to get the parts center of mass.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def rotation(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float,float]:
        """
        The rotation of the part, in the given reference frame.

        :returns: The rotation as a quaternion of the form ``(x, y, z, w)``.

        :param reference_frame: The reference frame that the returned
                                rotation is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_rotation(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_rotation(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def velocity(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The linear velocity of the part in the given reference frame.

        :returns: The velocity as a vector. The vector points in the direction of travel,
                  and its magnitude is the speed of the body in meters per second.

        :param reference_frame: The reference frame that the returned
                                velocity vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Part_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_velocity(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Part_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class Parts(ClassBase):
    """
    Instances of this class are used to interact with the parts of a vessel.
    An instance can be obtained by calling SpaceCenter.Vessel#parts.
    """
    @property
    def all(self) -> List[Part]:
        """
        A list of all of the vessels parts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_All",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_all(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_all(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_All",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    @property
    def antennas(self) -> List[Antenna]:
        """
        A list of all antennas in the vessel.

        If RemoteTech is installed, this will always return an empty list.
        To interact with RemoteTech antennas, use the RemoteTech service APIs.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Antennas",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Antenna"))
        )

    def _return_type_antennas(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Antenna"))

    def _build_call_antennas(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Antennas",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Antenna"))
        )

    @property
    def cargo_bays(self) -> List[CargoBay]:
        """
        A list of all cargo bays in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_CargoBays",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CargoBay"))
        )

    def _return_type_cargo_bays(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CargoBay"))

    def _build_call_cargo_bays(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_CargoBays",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CargoBay"))
        )

    @property
    def control_surfaces(self) -> List[ControlSurface]:
        """
        A list of all control surfaces in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_ControlSurfaces",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ControlSurface"))
        )

    def _return_type_control_surfaces(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ControlSurface"))

    def _build_call_control_surfaces(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_ControlSurfaces",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ControlSurface"))
        )

    @property
    def controlling(self) -> Part:
        """
        The part from which the vessel is controlled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Controlling",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @controlling.setter
    def controlling(self, value: Part) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Parts_set_Controlling",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.class_type("SpaceCenter", "Part")],
            None
        )

    def _return_type_controlling(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_controlling(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Controlling",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def decouplers(self) -> List[Decoupler]:
        """
        A list of all decouplers in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Decouplers",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Decoupler"))
        )

    def _return_type_decouplers(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Decoupler"))

    def _build_call_decouplers(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Decouplers",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Decoupler"))
        )

    @property
    def docking_ports(self) -> List[DockingPort]:
        """
        A list of all docking ports in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_DockingPorts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "DockingPort"))
        )

    def _return_type_docking_ports(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "DockingPort"))

    def _build_call_docking_ports(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_DockingPorts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "DockingPort"))
        )

    @property
    def engines(self) -> List[Engine]:
        """
        A list of all engines in the vessel.

        This includes any part that generates thrust. This covers many different types
        of engine, including liquid fuel rockets, solid rocket boosters, jet engines and
        RCS thrusters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Engines",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Engine"))
        )

    def _return_type_engines(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Engine"))

    def _build_call_engines(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Engines",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Engine"))
        )

    @property
    def experiments(self) -> List[Experiment]:
        """
        A list of all science experiments in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Experiments",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))
        )

    def _return_type_experiments(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))

    def _build_call_experiments(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Experiments",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Experiment"))
        )

    @property
    def fairings(self) -> List[Fairing]:
        """
        A list of all fairings in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Fairings",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Fairing"))
        )

    def _return_type_fairings(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Fairing"))

    def _build_call_fairings(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Fairings",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Fairing"))
        )

    @property
    def intakes(self) -> List[Intake]:
        """
        A list of all intakes in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Intakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Intake"))
        )

    def _return_type_intakes(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Intake"))

    def _build_call_intakes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Intakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Intake"))
        )

    @property
    def launch_clamps(self) -> List[LaunchClamp]:
        """
        A list of all launch clamps attached to the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_LaunchClamps",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchClamp"))
        )

    def _return_type_launch_clamps(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchClamp"))

    def _build_call_launch_clamps(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_LaunchClamps",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchClamp"))
        )

    @property
    def legs(self) -> List[Leg]:
        """
        A list of all landing legs attached to the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Legs",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Leg"))
        )

    def _return_type_legs(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Leg"))

    def _build_call_legs(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Legs",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Leg"))
        )

    @property
    def lights(self) -> List[Light]:
        """
        A list of all lights in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Lights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Light"))
        )

    def _return_type_lights(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Light"))

    def _build_call_lights(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Lights",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Light"))
        )

    @property
    def parachutes(self) -> List[Parachute]:
        """
        A list of all parachutes in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Parachutes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Parachute"))
        )

    def _return_type_parachutes(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Parachute"))

    def _build_call_parachutes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Parachutes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Parachute"))
        )

    @property
    def radiators(self) -> List[Radiator]:
        """
        A list of all radiators in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Radiators",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Radiator"))
        )

    def _return_type_radiators(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Radiator"))

    def _build_call_radiators(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Radiators",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Radiator"))
        )

    @property
    def rcs(self) -> List[RCS]:
        """
        A list of all RCS blocks/thrusters in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RCS"))
        )

    def _return_type_rcs(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RCS"))

    def _build_call_rcs(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_RCS",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RCS"))
        )

    @property
    def reaction_wheels(self) -> List[ReactionWheel]:
        """
        A list of all reaction wheels in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_ReactionWheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ReactionWheel"))
        )

    def _return_type_reaction_wheels(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ReactionWheel"))

    def _build_call_reaction_wheels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_ReactionWheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ReactionWheel"))
        )

    @property
    def resource_converters(self) -> List[ResourceConverter]:
        """
        A list of all resource converters in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_ResourceConverters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceConverter"))
        )

    def _return_type_resource_converters(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceConverter"))

    def _build_call_resource_converters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_ResourceConverters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceConverter"))
        )

    @property
    def resource_drains(self) -> List[ResourceDrain]:
        """
        A list of all resource drains in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_ResourceDrains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceDrain"))
        )

    def _return_type_resource_drains(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceDrain"))

    def _build_call_resource_drains(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_ResourceDrains",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceDrain"))
        )

    @property
    def resource_harvesters(self) -> List[ResourceHarvester]:
        """
        A list of all resource harvesters in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_ResourceHarvesters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceHarvester"))
        )

    def _return_type_resource_harvesters(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceHarvester"))

    def _build_call_resource_harvesters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_ResourceHarvesters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "ResourceHarvester"))
        )

    @property
    def robotic_hinges(self) -> List[RoboticHinge]:
        """
        A list of all robotic hinges in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_RoboticHinges",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticHinge"))
        )

    def _return_type_robotic_hinges(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticHinge"))

    def _build_call_robotic_hinges(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_RoboticHinges",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticHinge"))
        )

    @property
    def robotic_pistons(self) -> List[RoboticPiston]:
        """
        A list of all robotic pistons in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_RoboticPistons",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticPiston"))
        )

    def _return_type_robotic_pistons(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticPiston"))

    def _build_call_robotic_pistons(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_RoboticPistons",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticPiston"))
        )

    @property
    def robotic_rotations(self) -> List[RoboticRotation]:
        """
        A list of all robotic rotations in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_RoboticRotations",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotation"))
        )

    def _return_type_robotic_rotations(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotation"))

    def _build_call_robotic_rotations(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_RoboticRotations",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotation"))
        )

    @property
    def robotic_rotors(self) -> List[RoboticRotor]:
        """
        A list of all robotic rotors in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_RoboticRotors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotor"))
        )

    def _return_type_robotic_rotors(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotor"))

    def _build_call_robotic_rotors(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_RoboticRotors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "RoboticRotor"))
        )

    @property
    def root(self) -> Part:
        """
        The vessels root part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Root",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_root(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_root(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Root",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def sensors(self) -> List[Sensor]:
        """
        A list of all sensors in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Sensors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Sensor"))
        )

    def _return_type_sensors(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Sensor"))

    def _build_call_sensors(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Sensors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Sensor"))
        )

    @property
    def solar_panels(self) -> List[SolarPanel]:
        """
        A list of all solar panels in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_SolarPanels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "SolarPanel"))
        )

    def _return_type_solar_panels(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "SolarPanel"))

    def _build_call_solar_panels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_SolarPanels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "SolarPanel"))
        )

    @property
    def wheels(self) -> List[Wheel]:
        """
        A list of all wheels in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_get_Wheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Wheel"))
        )

    def _return_type_wheels(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Wheel"))

    def _build_call_wheels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_get_Wheels",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Parts")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Wheel"))
        )

    def in_decouple_stage(self, stage: int) -> List[Part]:
        """
        A list of all parts that are decoupled in the given stage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_InDecoupleStage",
            [self, stage],
            ["self", "stage"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_in_decouple_stage(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_in_decouple_stage(self, stage: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_InDecoupleStage",
            [self, stage],
            ["self", "stage"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def in_stage(self, stage: int) -> List[Part]:
        """
        A list of all parts that are activated in the given stage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_InStage",
            [self, stage],
            ["self", "stage"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_in_stage(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_in_stage(self, stage: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_InStage",
            [self, stage],
            ["self", "stage"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def modules_with_name(self, module_name: str) -> List[Module]:
        """
        A list of modules (combined across all parts in the vessel) whose
        SpaceCenter.Module#name is module_name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_ModulesWithName",
            [self, module_name],
            ["self", "module_name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))
        )

    def _return_type_modules_with_name(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))

    def _build_call_modules_with_name(self, module_name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_ModulesWithName",
            [self, module_name],
            ["self", "module_name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Module"))
        )

    def with_module(self, module_name: str) -> List[Part]:
        """
        A list of all parts that contain a SpaceCenter.Module whose
        SpaceCenter.Module#name is module_name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_WithModule",
            [self, module_name],
            ["self", "module_name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_with_module(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_with_module(self, module_name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_WithModule",
            [self, module_name],
            ["self", "module_name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def with_name(self, name: str) -> List[Part]:
        """
        A list of parts whose SpaceCenter.Part#name is name.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_WithName",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_with_name(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_with_name(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_WithName",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def with_tag(self, tag: str) -> List[Part]:
        """
        A list of all parts whose SpaceCenter.Part#tag is tag.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_WithTag",
            [self, tag],
            ["self", "tag"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_with_tag(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_with_tag(self, tag: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_WithTag",
            [self, tag],
            ["self", "tag"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def with_title(self, title: str) -> List[Part]:
        """
        A list of all parts whose SpaceCenter.Part#title is title.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Parts_WithTitle",
            [self, title],
            ["self", "title"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )

    def _return_type_with_title(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))

    def _build_call_with_title(self, title: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Parts_WithTitle",
            [self, title],
            ["self", "title"],
            [self._client._types.class_type("SpaceCenter", "Parts"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Part"))
        )



class Propellant(ClassBase):
    """
    A propellant for an engine. Obtains by calling SpaceCenter.Engine#propellants.
    """
    @property
    def current_amount(self) -> float:
        """
        The current amount of propellant.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_CurrentAmount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    def _return_type_current_amount(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_current_amount(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_CurrentAmount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    @property
    def current_requirement(self) -> float:
        """
        The required amount of propellant.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_CurrentRequirement",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    def _return_type_current_requirement(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_current_requirement(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_CurrentRequirement",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    @property
    def draw_stack_gauge(self) -> bool:
        """
        If this propellant has a stack gauge or not.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_DrawStackGauge",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    def _return_type_draw_stack_gauge(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_draw_stack_gauge(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_DrawStackGauge",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    @property
    def ignore_for_isp(self) -> bool:
        """
        If this propellant should be ignored when calculating required mass flow
        given specific impulse.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_IgnoreForIsp",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    def _return_type_ignore_for_isp(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_ignore_for_isp(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_IgnoreForIsp",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    @property
    def ignore_for_thrust_curve(self) -> bool:
        """
        If this propellant should be ignored for thrust curve calculations.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_IgnoreForThrustCurve",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    def _return_type_ignore_for_thrust_curve(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_ignore_for_thrust_curve(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_IgnoreForThrustCurve",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    @property
    def is_deprived(self) -> bool:
        """
        If this propellant is deprived.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_IsDeprived",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    def _return_type_is_deprived(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_deprived(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_IsDeprived",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.bool_type
        )

    @property
    def name(self) -> str:
        """
        The name of the propellant.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.string_type
        )

    @property
    def ratio(self) -> float:
        """
        The propellant ratio.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_Ratio",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.float_type
        )

    def _return_type_ratio(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_ratio(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_Ratio",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.float_type
        )

    @property
    def total_resource_available(self) -> float:
        """
        The total amount of the underlying resource currently reachable given
        resource flow rules.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_TotalResourceAvailable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    def _return_type_total_resource_available(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_total_resource_available(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_TotalResourceAvailable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    @property
    def total_resource_capacity(self) -> float:
        """
        The total vehicle capacity for the underlying propellant resource,
        restricted by resource flow rules.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Propellant_get_TotalResourceCapacity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )

    def _return_type_total_resource_capacity(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_total_resource_capacity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Propellant_get_TotalResourceCapacity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Propellant")],
            self._client._types.double_type
        )



class RCS(ClassBase):
    """
    An RCS block or thruster. Obtained by calling SpaceCenter.Part#rCS.
    """
    @property
    def active(self) -> bool:
        """
        Whether the RCS thrusters are active.
        An RCS thruster is inactive if the RCS action group is disabled
        (SpaceCenter.Control#rCS), the RCS thruster itself is not enabled
        (SpaceCenter.RCS#enabled) or it is covered by a fairing (SpaceCenter.Part#shielded).
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def available_force(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The available force, in Newtons, that can be produced by this RCS,
        in the positive and negative x, y and z axes of the vessel. These axes
        correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        Returns zero if RCS is disabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_AvailableForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_force(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_force(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_AvailableForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_thrust(self) -> float:
        """
        The amount of thrust, in Newtons, that would be produced by the thruster when activated.
        Returns zero if the thruster does not have any fuel.
        Takes the thrusters current SpaceCenter.RCS#thrustLimit and atmospheric conditions
        into account.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_available_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_available_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def available_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The available torque, in Newton meters, that can be produced by this RCS,
        in the positive and negative pitch, roll and yaw axes of the vessel. These axes
        correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        Returns zero if RCS is disable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def enabled(self) -> bool:
        """
        Whether the RCS thrusters are enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @enabled.setter
    def enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_Enabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def forward_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when pitch control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_ForwardEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @forward_enabled.setter
    def forward_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_ForwardEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_forward_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_forward_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_ForwardEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def has_fuel(self) -> bool:
        """
        Whether the RCS has fuel available.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_HasFuel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    def _return_type_has_fuel(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_fuel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_HasFuel",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def kerbin_sea_level_specific_impulse(self) -> float:
        """
        The specific impulse of the RCS at sea level on Kerbin, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_kerbin_sea_level_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_kerbin_sea_level_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def max_thrust(self) -> float:
        """
        The maximum amount of thrust that can be produced by the RCS thrusters when active,
        in Newtons.
        Takes the thrusters current SpaceCenter.RCS#thrustLimit and atmospheric conditions
        into account.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_max_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def max_vacuum_thrust(self) -> float:
        """
        The maximum amount of thrust that can be produced by the RCS thrusters when active
        in a vacuum, in Newtons.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_max_vacuum_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_vacuum_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this RCS.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def pitch_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when pitch control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_PitchEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @pitch_enabled.setter
    def pitch_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_PitchEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_pitch_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_pitch_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_PitchEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def propellant_ratios(self) -> Dict[str, float]:
        """
        The ratios of resources that the RCS consumes. A dictionary mapping resource names
        to the ratios at which they are consumed by the RCS.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_PropellantRatios",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)
        )

    def _return_type_propellant_ratios(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)

    def _build_call_propellant_ratios(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_PropellantRatios",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.float_type)
        )

    @property
    def propellants(self) -> List[str]:
        """
        The names of resources that the RCS consumes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_Propellants",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_propellants(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_propellants(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_Propellants",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def right_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when roll control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_RightEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @right_enabled.setter
    def right_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_RightEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_right_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_right_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_RightEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def roll_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when roll control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_RollEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @roll_enabled.setter
    def roll_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_RollEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_roll_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_roll_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_RollEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def specific_impulse(self) -> float:
        """
        The current specific impulse of the RCS, in seconds. Returns zero
        if the RCS is not active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def thrust_limit(self) -> float:
        """
        The thrust limiter of the thruster. A value between 0 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_ThrustLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @thrust_limit.setter
    def thrust_limit(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_ThrustLimit",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.float_type],
            None
        )

    def _return_type_thrust_limit(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thrust_limit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_ThrustLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def thrusters(self) -> List[Thruster]:
        """
        A list of thrusters, one of each nozzel in the RCS part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_Thrusters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))
        )

    def _return_type_thrusters(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))

    def _build_call_thrusters(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_Thrusters",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Thruster"))
        )

    @property
    def up_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when yaw control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_UpEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @up_enabled.setter
    def up_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_UpEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_up_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_up_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_UpEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @property
    def vacuum_specific_impulse(self) -> float:
        """
        The vacuum specific impulse of the RCS, in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    def _return_type_vacuum_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_vacuum_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.float_type
        )

    @property
    def yaw_enabled(self) -> bool:
        """
        Whether the RCS thruster will fire when yaw control input is given.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RCS_get_YawEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )

    @yaw_enabled.setter
    def yaw_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RCS_set_YawEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RCS"), self._client._types.bool_type],
            None
        )

    def _return_type_yaw_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_yaw_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RCS_get_YawEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RCS")],
            self._client._types.bool_type
        )



class Radiator(ClassBase):
    """
    A radiator. Obtained by calling SpaceCenter.Part#radiator.
    """
    @property
    def deployable(self) -> bool:
        """
        Whether the radiator is deployable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Radiator_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.bool_type
        )

    def _return_type_deployable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Radiator_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.bool_type
        )

    @property
    def deployed(self) -> bool:
        """
        For a deployable radiator, ``True`` if the radiator is extended.
        If the radiator is not deployable, this is always ``True``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Radiator_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Radiator_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Radiator"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Radiator_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this radiator.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Radiator_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Radiator_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> RadiatorState:
        """
        The current state of the radiator.

        A fixed radiator is always SpaceCenter.RadiatorState#extended.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Radiator_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.enumeration_type("SpaceCenter", "RadiatorState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "RadiatorState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Radiator_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Radiator")],
            self._client._types.enumeration_type("SpaceCenter", "RadiatorState")
        )



class ReactionWheel(ClassBase):
    """
    A reaction wheel. Obtained by calling SpaceCenter.Part#reactionWheel.
    """
    @property
    def active(self) -> bool:
        """
        Whether the reaction wheel is active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.bool_type
        )

    @active.setter
    def active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_set_Active",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel"), self._client._types.bool_type],
            None
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ReactionWheel_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.bool_type
        )

    @property
    def available_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The available torque, in Newton meters, that can be produced by this reaction wheel,
        in the positive and negative pitch, roll and yaw axes of the vessel. These axes
        correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        Returns zero if the reaction wheel is inactive or broken.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ReactionWheel_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def broken(self) -> bool:
        """
        Whether the reaction wheel is broken.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_get_Broken",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.bool_type
        )

    def _return_type_broken(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_broken(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ReactionWheel_get_Broken",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.bool_type
        )

    @property
    def max_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque, in Newton meters, that can be produced by this reaction wheel,
        when it is active, in the positive and negative pitch, roll and yaw axes of the vessel.
        These axes correspond to the coordinate axes of the SpaceCenter.Vessel#referenceFrame.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_get_MaxTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_max_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_max_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ReactionWheel_get_MaxTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def part(self) -> Part:
        """
        The part object for this reaction wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ReactionWheel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ReactionWheel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ReactionWheel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )



class ReferenceFrame(ClassBase):
    """
    Represents a reference frame for positions, rotations and
    velocities. Contains:
     * The position of the origin.
     * The directions of the x, y and z axes.
     * The linear velocity of the frame.
     * The angular velocity of the frame.

    This class does not contain any properties or methods. It is only
    used as a parameter to other functions.
    """
    @classmethod
    def create_hybrid(cls, position: ReferenceFrame, rotation: ReferenceFrame = None, velocity: ReferenceFrame = None, angular_velocity: ReferenceFrame = None) -> ReferenceFrame:
        """
        Create a hybrid reference frame. This is a custom reference frame
        whose components inherited from other reference frames.

        :param position: The reference frame providing the position of the origin.

        :param rotation: The reference frame providing the rotation of the frame.

        :param velocity: The reference frame providing the linear velocity of the frame.

        :param angular_velocity: The reference frame providing the angular velocity
                                 of the frame.

        The position reference frame is required but all other
        reference frames are optional. If omitted, they are set to the
        position reference frame.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "ReferenceFrame_static_CreateHybrid",
            [position, rotation, velocity, angular_velocity],
            ["position", "rotation", "velocity", "angular_velocity"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @classmethod
    def _return_type_create_hybrid(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    @classmethod
    def _build_call_create_hybrid(cls, position: ReferenceFrame, rotation: ReferenceFrame = None, velocity: ReferenceFrame = None, angular_velocity: ReferenceFrame = None) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "ReferenceFrame_static_CreateHybrid",
            [position, rotation, velocity, angular_velocity],
            ["position", "rotation", "velocity", "angular_velocity"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @classmethod
    def create_relative(cls, reference_frame: ReferenceFrame, position: Tuple[float,float,float] = (0.0, 0.0, 0.0), rotation: Tuple[float,float,float,float] = (0.0, 0.0, 0.0, 1.0), velocity: Tuple[float,float,float] = (0.0, 0.0, 0.0), angular_velocity: Tuple[float,float,float] = (0.0, 0.0, 0.0)) -> ReferenceFrame:
        """
        Create a relative reference frame. This is a custom reference frame
        whose components offset the components of a parent reference frame.

        :param reference_frame: The parent reference frame on which to
                                base this reference frame.

        :param position: The offset of the position of the origin,
                         as a position vector. Defaults to ``(0, 0, 0)``

        :param rotation: The rotation to apply to the parent frames rotation,
                         as a quaternion of the form ``(x, y, z, w)``.
                         Defaults to ``(0, 0, 0, 1)`` (i.e. no rotation)

        :param velocity: The linear velocity to offset the parent frame by,
                         as a vector pointing in the direction of travel, whose magnitude is the speed in
                         meters per second. Defaults to ``(0, 0, 0)``.

        :param angular_velocity: The angular velocity to offset the parent frame by,
                                 as a vector. This vector points in the direction of the axis of rotation,
                                 and its magnitude is the speed of the rotation in radians per second.
                                 Defaults to ``(0, 0, 0)``.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "ReferenceFrame_static_CreateRelative",
            [reference_frame, position, rotation, velocity, angular_velocity],
            ["reference_frame", "position", "rotation", "velocity", "angular_velocity"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @classmethod
    def _return_type_create_relative(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    @classmethod
    def _build_call_create_relative(cls, reference_frame: ReferenceFrame, position: Tuple[float,float,float] = (0.0, 0.0, 0.0), rotation: Tuple[float,float,float,float] = (0.0, 0.0, 0.0, 1.0), velocity: Tuple[float,float,float] = (0.0, 0.0, 0.0), angular_velocity: Tuple[float,float,float] = (0.0, 0.0, 0.0)) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "ReferenceFrame_static_CreateRelative",
            [reference_frame, position, rotation, velocity, angular_velocity],
            ["reference_frame", "position", "rotation", "velocity", "angular_velocity"],
            [self._client._types.class_type("SpaceCenter", "ReferenceFrame"), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )



class Resource(ClassBase):
    """
    An individual resource stored within a part.
    Created using methods in the SpaceCenter.Resources class.
    """
    @property
    def amount(self) -> float:
        """
        The amount of the resource that is currently stored in the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Amount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    def _return_type_amount(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_amount(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Amount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    @property
    def density(self) -> float:
        """
        The density of the resource, in ``kg/l``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Density",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    def _return_type_density(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_density(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Density",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    @property
    def enabled(self) -> bool:
        """
        Whether use of this resource is enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.bool_type
        )

    @enabled.setter
    def enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Resource_set_Enabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Resource"), self._client._types.bool_type],
            None
        )

    def _return_type_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.bool_type
        )

    @property
    def flow_mode(self) -> ResourceFlowMode:
        """
        The flow mode of the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_FlowMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")
        )

    def _return_type_flow_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")

    def _build_call_flow_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_FlowMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")
        )

    @property
    def max(self) -> float:
        """
        The total amount of the resource that can be stored in the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Max",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    def _return_type_max(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Max",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.float_type
        )

    @property
    def name(self) -> str:
        """
        The name of the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.string_type
        )

    @property
    def part(self) -> Part:
        """
        The part containing the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resource_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resource_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.class_type("SpaceCenter", "Part")
        )



class ResourceConverter(ClassBase):
    """
    A resource converter. Obtained by calling SpaceCenter.Part#resourceConverter.
    """
    @property
    def core_temperature(self) -> float:
        """
        The core temperature of the converter, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_get_CoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    def _return_type_core_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_core_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_get_CoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    @property
    def count(self) -> int:
        """
        The number of converters in the part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_get_Count",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.sint32_type
        )

    def _return_type_count(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_count(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_get_Count",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.sint32_type
        )

    @property
    def optimum_core_temperature(self) -> float:
        """
        The core temperature at which the converter will operate with peak efficiency, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_get_OptimumCoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    def _return_type_optimum_core_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_optimum_core_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_get_OptimumCoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def thermal_efficiency(self) -> float:
        """
        The thermal efficiency of the converter, as a percentage of its maximum.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_get_ThermalEfficiency",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    def _return_type_thermal_efficiency(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_efficiency(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_get_ThermalEfficiency",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter")],
            self._client._types.float_type
        )

    def active(self, index: int) -> bool:
        """
        True if the specified converter is active.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Active",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.bool_type
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Active",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.bool_type
        )

    def inputs(self, index: int) -> List[str]:
        """
        List of the names of resources consumed by the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Inputs",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_inputs(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_inputs(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Inputs",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.string_type)
        )

    def name(self, index: int) -> str:
        """
        The name of the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Name",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.string_type
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Name",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.string_type
        )

    def outputs(self, index: int) -> List[str]:
        """
        List of the names of resources produced by the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Outputs",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_outputs(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_outputs(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Outputs",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.list_type(self._client._types.string_type)
        )

    def start(self, index: int) -> None:
        """
        Start the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Start",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            None
        )

    def _return_type_start(self) -> TypeBase:
        return None

    def _build_call_start(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Start",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            None
        )

    def state(self, index: int) -> ResourceConverterState:
        """
        The state of the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_State",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.enumeration_type("SpaceCenter", "ResourceConverterState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ResourceConverterState")

    def _build_call_state(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_State",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.enumeration_type("SpaceCenter", "ResourceConverterState")
        )

    def status_info(self, index: int) -> str:
        """
        Status information for the specified converter.
        This is the full status message shown in the in-game UI.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_StatusInfo",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.string_type
        )

    def _return_type_status_info(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_status_info(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_StatusInfo",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            self._client._types.string_type
        )

    def stop(self, index: int) -> None:
        """
        Stop the specified converter.

        :param index: Index of the converter.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceConverter_Stop",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            None
        )

    def _return_type_stop(self) -> TypeBase:
        return None

    def _build_call_stop(self, index: int) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceConverter_Stop",
            [self, index],
            ["self", "index"],
            [self._client._types.class_type("SpaceCenter", "ResourceConverter"), self._client._types.sint32_type],
            None
        )



class ResourceDrain(ClassBase):
    """
    A resource drain. Obtained by calling SpaceCenter.Part#resourceDrain.
    """
    @property
    def available_resources(self) -> List[Resource]:
        """
        List of available resources.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_AvailableResources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    def _return_type_available_resources(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))

    def _build_call_available_resources(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_AvailableResources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    @property
    def drain_mode(self) -> DrainMode:
        """
        The drain mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_DrainMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.enumeration_type("SpaceCenter", "DrainMode")
        )

    @drain_mode.setter
    def drain_mode(self, value: DrainMode) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_set_DrainMode",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.enumeration_type("SpaceCenter", "DrainMode")],
            None
        )

    def _return_type_drain_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "DrainMode")

    def _build_call_drain_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_DrainMode",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.enumeration_type("SpaceCenter", "DrainMode")
        )

    @property
    def max_rate(self) -> float:
        """
        Maximum possible drain rate.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_MaxRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    def _return_type_max_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_MaxRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    @property
    def min_rate(self) -> float:
        """
        Minimum possible drain rate
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_MinRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    def _return_type_min_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_min_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_MinRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this resource drain.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def rate(self) -> float:
        """
        Current drain rate.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    @rate.setter
    def rate(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_set_Rate",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.float_type],
            None
        )

    def _return_type_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain")],
            self._client._types.float_type
        )

    def check_resource(self, resource: Resource) -> bool:
        """
        Whether the provided resource is enabled for draining.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_CheckResource",
            [self, resource],
            ["self", "resource"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.bool_type
        )

    def _return_type_check_resource(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_check_resource(self, resource: Resource) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_CheckResource",
            [self, resource],
            ["self", "resource"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.class_type("SpaceCenter", "Resource")],
            self._client._types.bool_type
        )

    def set_resource(self, resource: Resource, enabled: bool) -> None:
        """
        Whether the given resource should be drained.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_SetResource",
            [self, resource, enabled],
            ["self", "resource", "enabled"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.class_type("SpaceCenter", "Resource"), self._client._types.bool_type],
            None
        )

    def _return_type_set_resource(self) -> TypeBase:
        return None

    def _build_call_set_resource(self, resource: Resource, enabled: bool) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_SetResource",
            [self, resource, enabled],
            ["self", "resource", "enabled"],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), self._client._types.class_type("SpaceCenter", "Resource"), self._client._types.bool_type],
            None
        )

    def start(self) -> None:
        """
        Activates resource draining for all enabled parts.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_Start",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), ],
            None
        )

    def _return_type_start(self) -> TypeBase:
        return None

    def _build_call_start(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_Start",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), ],
            None
        )

    def stop(self) -> None:
        """
        Turns off resource draining.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceDrain_Stop",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), ],
            None
        )

    def _return_type_stop(self) -> TypeBase:
        return None

    def _build_call_stop(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceDrain_Stop",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "ResourceDrain"), ],
            None
        )



class ResourceHarvester(ClassBase):
    """
    A resource harvester (drill). Obtained by calling SpaceCenter.Part#resourceHarvester.
    """
    @property
    def active(self) -> bool:
        """
        Whether the harvester is actively drilling.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.bool_type
        )

    @active.setter
    def active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_set_Active",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester"), self._client._types.bool_type],
            None
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.bool_type
        )

    @property
    def core_temperature(self) -> float:
        """
        The core temperature of the drill, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_CoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    def _return_type_core_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_core_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_CoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the harvester is deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.bool_type
        )

    @property
    def extraction_rate(self) -> float:
        """
        The rate at which the drill is extracting ore, in units per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_ExtractionRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    def _return_type_extraction_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_extraction_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_ExtractionRate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    @property
    def optimum_core_temperature(self) -> float:
        """
        The core temperature at which the drill will operate with peak efficiency, in Kelvin.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_OptimumCoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    def _return_type_optimum_core_temperature(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_optimum_core_temperature(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_OptimumCoreTemperature",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this harvester.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> ResourceHarvesterState:
        """
        The state of the harvester.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.enumeration_type("SpaceCenter", "ResourceHarvesterState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "ResourceHarvesterState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.enumeration_type("SpaceCenter", "ResourceHarvesterState")
        )

    @property
    def thermal_efficiency(self) -> float:
        """
        The thermal efficiency of the drill, as a percentage of its maximum.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceHarvester_get_ThermalEfficiency",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )

    def _return_type_thermal_efficiency(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thermal_efficiency(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceHarvester_get_ThermalEfficiency",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceHarvester")],
            self._client._types.float_type
        )



class ResourceTransfer(ClassBase):
    """
    Transfer resources between parts.
    """
    @property
    def amount(self) -> float:
        """
        The amount of the resource that has been transferred.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceTransfer_get_Amount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceTransfer")],
            self._client._types.float_type
        )

    def _return_type_amount(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_amount(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceTransfer_get_Amount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceTransfer")],
            self._client._types.float_type
        )

    @property
    def complete(self) -> bool:
        """
        Whether the transfer has completed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ResourceTransfer_get_Complete",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceTransfer")],
            self._client._types.bool_type
        )

    def _return_type_complete(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_complete(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ResourceTransfer_get_Complete",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ResourceTransfer")],
            self._client._types.bool_type
        )

    @classmethod
    def start(cls, from_part: Part, to_part: Part, resource: str, max_amount: float) -> ResourceTransfer:
        """
        Start transferring a resource transfer between a pair of parts. The transfer will move
        at most max_amount units of the resource, depending on how much of
        the resource is available in the source part and how much storage is available in the
        destination part.
        Use SpaceCenter.ResourceTransfer#complete to check if the transfer is complete.
        Use SpaceCenter.ResourceTransfer#amount to see how much of the resource has been transferred.

        :param from_part: The part to transfer to.

        :param to_part: The part to transfer from.

        :param resource: The name of the resource to transfer.

        :param max_amount: The maximum amount of resource to transfer.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "ResourceTransfer_static_Start",
            [from_part, to_part, resource, max_amount],
            ["from_part", "to_part", "resource", "max_amount"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "Part"), self._client._types.string_type, self._client._types.float_type],
            self._client._types.class_type("SpaceCenter", "ResourceTransfer")
        )

    @classmethod
    def _return_type_start(cls) -> TypeBase:
        self = cls
        return self._client._types.class_type("SpaceCenter", "ResourceTransfer")

    @classmethod
    def _build_call_start(cls, from_part: Part, to_part: Part, resource: str, max_amount: float) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "ResourceTransfer_static_Start",
            [from_part, to_part, resource, max_amount],
            ["from_part", "to_part", "resource", "max_amount"],
            [self._client._types.class_type("SpaceCenter", "Part"), self._client._types.class_type("SpaceCenter", "Part"), self._client._types.string_type, self._client._types.float_type],
            self._client._types.class_type("SpaceCenter", "ResourceTransfer")
        )



class Resources(ClassBase):
    """
    Represents the collection of resources stored in a vessel, stage or part.
    Created by calling SpaceCenter.Vessel#resources,
    SpaceCenter.Vessel#resourcesInDecoupleStage or
    SpaceCenter.Part#resources.
    """
    @property
    def all(self) -> List[Resource]:
        """
        All the individual resources that can be stored.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_get_All",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    def _return_type_all(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))

    def _build_call_all(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_get_All",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    @property
    def enabled(self) -> bool:
        """
        Whether use of all the resources are enabled.

        This is ``True`` if all of the resources are enabled.
        If any of the resources are not enabled, this is ``False``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.bool_type
        )

    @enabled.setter
    def enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Resources_set_Enabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.bool_type],
            None
        )

    def _return_type_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_get_Enabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.bool_type
        )

    @property
    def names(self) -> List[str]:
        """
        A list of resource names that can be stored.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_get_Names",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_names(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_names(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_get_Names",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Resources")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def amount(self, name: str) -> float:
        """
        Returns the amount of a resource that is currently stored.

        :param name: The name of the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_Amount",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.float_type
        )

    def _return_type_amount(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_amount(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_Amount",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.float_type
        )

    def has_resource(self, name: str) -> bool:
        """
        Check whether the named resource can be stored.

        :param name: The name of the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_HasResource",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_has_resource(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_resource(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_HasResource",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def max(self, name: str) -> float:
        """
        Returns the amount of a resource that can be stored.

        :param name: The name of the resource.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_Max",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.float_type
        )

    def _return_type_max(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_Max",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.float_type
        )

    def with_resource(self, name: str) -> List[Resource]:
        """
        All the individual resources with the given name that can be stored.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Resources_WithResource",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    def _return_type_with_resource(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))

    def _build_call_with_resource(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Resources_WithResource",
            [self, name],
            ["self", "name"],
            [self._client._types.class_type("SpaceCenter", "Resources"), self._client._types.string_type],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Resource"))
        )

    @classmethod
    def density(cls, name: str) -> float:
        """
        Returns the density of a resource, in ``kg/l``.

        :param name: The name of the resource.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "Resources_static_Density",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.float_type
        )

    @classmethod
    def _return_type_density(cls) -> TypeBase:
        self = cls
        return self._client._types.float_type

    @classmethod
    def _build_call_density(cls, name: str) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "Resources_static_Density",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.float_type
        )

    @classmethod
    def flow_mode(cls, name: str) -> ResourceFlowMode:
        """
        Returns the flow mode of a resource.

        :param name: The name of the resource.
        """
        self = cls
        return cls._client._invoke(
            "SpaceCenter",
            "Resources_static_FlowMode",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")
        )

    @classmethod
    def _return_type_flow_mode(cls) -> TypeBase:
        self = cls
        return self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")

    @classmethod
    def _build_call_flow_mode(cls, name: str) -> KRPC_pb2.ProcedureCall:
        self = cls
        return self._client._build_call(
            "SpaceCenter",
            "Resources_static_FlowMode",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.enumeration_type("SpaceCenter", "ResourceFlowMode")
        )



class RoboticController(ClassBase):
    """
    A robotic controller. Obtained by calling SpaceCenter.Part#roboticController.
    """
    @property
    def part(self) -> Part:
        """
        The part object for this controller.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticController")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticController")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def add_axis(self, module: Module, field_name: str) -> bool:
        """
        Add an axis to the controller.

        :returns: Returns ``True`` if the axis is added successfully.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_AddAxis",
            [self, module, field_name],
            ["self", "module", "field_name"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_add_axis(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_add_axis(self, module: Module, field_name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_AddAxis",
            [self, module, field_name],
            ["self", "module", "field_name"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def add_key_frame(self, module: Module, field_name: str, time: float, value: float) -> bool:
        """
        Add key frame value for controller axis.

        :returns: Returns ``True`` if the key frame is added successfully.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_AddKeyFrame",
            [self, module, field_name, time, value],
            ["self", "module", "field_name", "time", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type, self._client._types.float_type],
            self._client._types.bool_type
        )

    def _return_type_add_key_frame(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_add_key_frame(self, module: Module, field_name: str, time: float, value: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_AddKeyFrame",
            [self, module, field_name, time, value],
            ["self", "module", "field_name", "time", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type, self._client._types.float_type, self._client._types.float_type],
            self._client._types.bool_type
        )

    def axes(self) -> List[List[str]]:
        """
        The axes for the controller.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_Axes",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), ],
            self._client._types.list_type(self._client._types.list_type(self._client._types.string_type))
        )

    def _return_type_axes(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.list_type(self._client._types.string_type))

    def _build_call_axes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_Axes",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), ],
            self._client._types.list_type(self._client._types.list_type(self._client._types.string_type))
        )

    def clear_axis(self, module: Module, field_name: str) -> bool:
        """
        Clear axis.

        :returns: Returns ``True`` if the axis is cleared successfully.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_ClearAxis",
            [self, module, field_name],
            ["self", "module", "field_name"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def _return_type_clear_axis(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_clear_axis(self, module: Module, field_name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_ClearAxis",
            [self, module, field_name],
            ["self", "module", "field_name"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Module"), self._client._types.string_type],
            self._client._types.bool_type
        )

    def has_part(self, part: Part) -> bool:
        """
        Whether the controller has a part.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticController_HasPart",
            [self, part],
            ["self", "part"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )

    def _return_type_has_part(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_part(self, part: Part) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticController_HasPart",
            [self, part],
            ["self", "part"],
            [self._client._types.class_type("SpaceCenter", "RoboticController"), self._client._types.class_type("SpaceCenter", "Part")],
            self._client._types.bool_type
        )



class RoboticHinge(ClassBase):
    """
    A robotic hinge. Obtained by calling SpaceCenter.Part#roboticHinge.
    """
    @property
    def current_angle(self) -> float:
        """
        Current angle.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_CurrentAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    def _return_type_current_angle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_current_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_CurrentAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @property
    def damping(self) -> float:
        """
        Damping percentage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @damping.setter
    def damping(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_set_Damping",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), self._client._types.float_type],
            None
        )

    def _return_type_damping(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_damping(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @property
    def locked(self) -> bool:
        """
        Lock movement.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.bool_type
        )

    @locked.setter
    def locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_set_Locked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), self._client._types.bool_type],
            None
        )

    def _return_type_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.bool_type
        )

    @property
    def motor_engaged(self) -> bool:
        """
        Whether the motor is engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.bool_type
        )

    @motor_engaged.setter
    def motor_engaged(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_set_MotorEngaged",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_engaged(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_engaged(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this robotic hinge.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def rate(self) -> float:
        """
        Target movement rate in degrees per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @rate.setter
    def rate(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_set_Rate",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), self._client._types.float_type],
            None
        )

    def _return_type_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @property
    def target_angle(self) -> float:
        """
        Target angle.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_get_TargetAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    @target_angle.setter
    def target_angle(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_set_TargetAngle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), self._client._types.float_type],
            None
        )

    def _return_type_target_angle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_get_TargetAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge")],
            self._client._types.float_type
        )

    def move_home(self) -> None:
        """
        Move hinge to it's built position.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticHinge_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), ],
            None
        )

    def _return_type_move_home(self) -> TypeBase:
        return None

    def _build_call_move_home(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticHinge_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticHinge"), ],
            None
        )



class RoboticPiston(ClassBase):
    """
    A robotic piston part. Obtained by calling SpaceCenter.Part#roboticPiston.
    """
    @property
    def current_extension(self) -> float:
        """
        Current extension of the piston.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_CurrentExtension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    def _return_type_current_extension(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_current_extension(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_CurrentExtension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @property
    def damping(self) -> float:
        """
        Damping percentage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @damping.setter
    def damping(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_set_Damping",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), self._client._types.float_type],
            None
        )

    def _return_type_damping(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_damping(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @property
    def locked(self) -> bool:
        """
        Lock movement.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.bool_type
        )

    @locked.setter
    def locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_set_Locked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), self._client._types.bool_type],
            None
        )

    def _return_type_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.bool_type
        )

    @property
    def motor_engaged(self) -> bool:
        """
        Whether the motor is engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.bool_type
        )

    @motor_engaged.setter
    def motor_engaged(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_set_MotorEngaged",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_engaged(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_engaged(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this robotic piston.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def rate(self) -> float:
        """
        Target movement rate in degrees per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @rate.setter
    def rate(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_set_Rate",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), self._client._types.float_type],
            None
        )

    def _return_type_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @property
    def target_extension(self) -> float:
        """
        Target extension of the piston.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_get_TargetExtension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    @target_extension.setter
    def target_extension(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_set_TargetExtension",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), self._client._types.float_type],
            None
        )

    def _return_type_target_extension(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_extension(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_get_TargetExtension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston")],
            self._client._types.float_type
        )

    def move_home(self) -> None:
        """
        Move piston to it's built position.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticPiston_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), ],
            None
        )

    def _return_type_move_home(self) -> TypeBase:
        return None

    def _build_call_move_home(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticPiston_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticPiston"), ],
            None
        )



class RoboticRotation(ClassBase):
    """
    A robotic rotation servo. Obtained by calling SpaceCenter.Part#roboticRotation.
    """
    @property
    def current_angle(self) -> float:
        """
        Current angle.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_CurrentAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    def _return_type_current_angle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_current_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_CurrentAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @property
    def damping(self) -> float:
        """
        Damping percentage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @damping.setter
    def damping(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_set_Damping",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), self._client._types.float_type],
            None
        )

    def _return_type_damping(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_damping(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_Damping",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @property
    def locked(self) -> bool:
        """
        Lock Movement
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.bool_type
        )

    @locked.setter
    def locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_set_Locked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), self._client._types.bool_type],
            None
        )

    def _return_type_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.bool_type
        )

    @property
    def motor_engaged(self) -> bool:
        """
        Whether the motor is engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.bool_type
        )

    @motor_engaged.setter
    def motor_engaged(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_set_MotorEngaged",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_engaged(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_engaged(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this robotic rotation servo.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def rate(self) -> float:
        """
        Target movement rate in degrees per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @rate.setter
    def rate(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_set_Rate",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), self._client._types.float_type],
            None
        )

    def _return_type_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_Rate",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @property
    def target_angle(self) -> float:
        """
        Target angle.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_get_TargetAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    @target_angle.setter
    def target_angle(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_set_TargetAngle",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), self._client._types.float_type],
            None
        )

    def _return_type_target_angle(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_get_TargetAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation")],
            self._client._types.float_type
        )

    def move_home(self) -> None:
        """
        Move rotation servo to it's built position.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotation_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), ],
            None
        )

    def _return_type_move_home(self) -> TypeBase:
        return None

    def _build_call_move_home(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotation_MoveHome",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "RoboticRotation"), ],
            None
        )



class RoboticRotor(ClassBase):
    """
    A robotic rotor. Obtained by calling SpaceCenter.Part#roboticRotor.
    """
    @property
    def current_rpm(self) -> float:
        """
        Current RPM.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_CurrentRPM",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )

    def _return_type_current_rpm(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_current_rpm(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_CurrentRPM",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )

    @property
    def inverted(self) -> bool:
        """
        Whether the rotor direction is inverted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_Inverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @inverted.setter
    def inverted(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_set_Inverted",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor"), self._client._types.bool_type],
            None
        )

    def _return_type_inverted(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_inverted(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_Inverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @property
    def locked(self) -> bool:
        """
        Lock movement.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @locked.setter
    def locked(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_set_Locked",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor"), self._client._types.bool_type],
            None
        )

    def _return_type_locked(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_locked(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_Locked",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @property
    def motor_engaged(self) -> bool:
        """
        Whether the motor is engaged.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @motor_engaged.setter
    def motor_engaged(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_set_MotorEngaged",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_engaged(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_engaged(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_MotorEngaged",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this robotic rotor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def target_rpm(self) -> float:
        """
        Target RPM.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_TargetRPM",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )

    @target_rpm.setter
    def target_rpm(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_set_TargetRPM",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor"), self._client._types.float_type],
            None
        )

    def _return_type_target_rpm(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_target_rpm(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_TargetRPM",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )

    @property
    def torque_limit(self) -> float:
        """
        Torque limit percentage.
        """
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_get_TorqueLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )

    @torque_limit.setter
    def torque_limit(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "RoboticRotor_set_TorqueLimit",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor"), self._client._types.float_type],
            None
        )

    def _return_type_torque_limit(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_torque_limit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "RoboticRotor_get_TorqueLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "RoboticRotor")],
            self._client._types.float_type
        )



class ScienceData(ClassBase):
    """
    Obtained by calling SpaceCenter.Experiment#data.
    """
    @property
    def data_amount(self) -> float:
        """
        Data amount.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceData_get_DataAmount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )

    def _return_type_data_amount(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_data_amount(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceData_get_DataAmount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )

    @property
    def science_value(self) -> float:
        """
        Science value.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceData_get_ScienceValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )

    def _return_type_science_value(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_science_value(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceData_get_ScienceValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )

    @property
    def transmit_value(self) -> float:
        """
        Transmit value.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceData_get_TransmitValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )

    def _return_type_transmit_value(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_transmit_value(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceData_get_TransmitValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceData")],
            self._client._types.float_type
        )



class ScienceSubject(ClassBase):
    """
    Obtained by calling SpaceCenter.Experiment#scienceSubject.
    """
    @property
    def data_scale(self) -> float:
        """
        Multiply science value by this to determine data amount in mits.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_DataScale",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    def _return_type_data_scale(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_data_scale(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_DataScale",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    @property
    def is_complete(self) -> bool:
        """
        Whether the experiment has been completed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_IsComplete",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.bool_type
        )

    def _return_type_is_complete(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_is_complete(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_IsComplete",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.bool_type
        )

    @property
    def science(self) -> float:
        """
        Amount of science already earned from this subject, not updated until after
        transmission/recovery.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_Science",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    def _return_type_science(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_science(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_Science",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    @property
    def science_cap(self) -> float:
        """
        Total science allowable for this subject.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_ScienceCap",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    def _return_type_science_cap(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_science_cap(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_ScienceCap",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    @property
    def scientific_value(self) -> float:
        """
        Diminishing value multiplier for decreasing the science value returned from repeated
        experiments.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_ScientificValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    def _return_type_scientific_value(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_scientific_value(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_ScientificValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    @property
    def subject_value(self) -> float:
        """
        Multiplier for specific Celestial Body/Experiment Situation combination.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_SubjectValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    def _return_type_subject_value(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_subject_value(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_SubjectValue",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.float_type
        )

    @property
    def title(self) -> str:
        """
        Title of science subject, displayed in science archives
        """
        return self._client._invoke(
            "SpaceCenter",
            "ScienceSubject_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.string_type
        )

    def _return_type_title(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_title(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ScienceSubject_get_Title",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "ScienceSubject")],
            self._client._types.string_type
        )



class Sensor(ClassBase):
    """
    A sensor, such as a thermometer. Obtained by calling SpaceCenter.Part#sensor.
    """
    @property
    def active(self) -> bool:
        """
        Whether the sensor is active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Sensor_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.bool_type
        )

    @active.setter
    def active(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Sensor_set_Active",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Sensor"), self._client._types.bool_type],
            None
        )

    def _return_type_active(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_active(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Sensor_get_Active",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this sensor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Sensor_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Sensor_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def value(self) -> str:
        """
        The current value of the sensor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Sensor_get_Value",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.string_type
        )

    def _return_type_value(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_value(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Sensor_get_Value",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Sensor")],
            self._client._types.string_type
        )



class SolarPanel(ClassBase):
    """
    A solar panel. Obtained by calling SpaceCenter.Part#solarPanel.
    """
    @property
    def deployable(self) -> bool:
        """
        Whether the solar panel is deployable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.bool_type
        )

    def _return_type_deployable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.bool_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the solar panel is extended.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.bool_type
        )

    @property
    def energy_flow(self) -> float:
        """
        The current amount of energy being generated by the solar panel, in
        units of charge per second.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_EnergyFlow",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.float_type
        )

    def _return_type_energy_flow(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_energy_flow(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_EnergyFlow",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.float_type
        )

    @property
    def part(self) -> Part:
        """
        The part object for this solar panel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def state(self) -> SolarPanelState:
        """
        The current state of the solar panel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.enumeration_type("SpaceCenter", "SolarPanelState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "SolarPanelState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.enumeration_type("SpaceCenter", "SolarPanelState")
        )

    @property
    def sun_exposure(self) -> float:
        """
        The current amount of sunlight that is incident on the solar panel,
        as a percentage. A value between 0 and 1.
        """
        return self._client._invoke(
            "SpaceCenter",
            "SolarPanel_get_SunExposure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.float_type
        )

    def _return_type_sun_exposure(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_sun_exposure(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "SolarPanel_get_SunExposure",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "SolarPanel")],
            self._client._types.float_type
        )



class Thruster(ClassBase):
    """
    The component of an SpaceCenter.Engine or SpaceCenter.RCS part that generates thrust.
    Can obtained by calling SpaceCenter.Engine#thrusters or SpaceCenter.RCS#thrusters.

    Engines can consist of multiple thrusters.
    For example, the S3 KS-25x4 "Mammoth" has four rocket nozzels, and so consists of
    four thrusters.
    """
    @property
    def gimbal_angle(self) -> Tuple[float,float,float]:
        """
        The current gimbal angle in the pitch, roll and yaw axes, in degrees.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_get_GimbalAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_gimbal_angle(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_gimbal_angle(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_get_GimbalAngle",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def gimballed(self) -> bool:
        """
        Whether the thruster is gimballed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_get_Gimballed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.bool_type
        )

    def _return_type_gimballed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_gimballed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_get_Gimballed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.bool_type
        )

    @property
    def part(self) -> Part:
        """
        The SpaceCenter.Part that contains this thruster.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def thrust_reference_frame(self) -> ReferenceFrame:
        """
        A reference frame that is fixed relative to the thruster and orientated with
        its thrust direction (SpaceCenter.Thruster#thrustDirection).
        For gimballed engines, this takes into account the current rotation of the gimbal.
         * The origin is at the position of thrust for this thruster
           (SpaceCenter.Thruster#thrustPosition).
         * The axes rotate with the thrust direction.
           This is the direction in which the thruster expels propellant, including any gimballing.
         * The y-axis points along the thrust direction.
         * The x-axis and z-axis are perpendicular to the thrust direction.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_get_ThrustReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_thrust_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_thrust_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_get_ThrustReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Thruster")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def gimbal_position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        Position around which the gimbal pivots.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_GimbalPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_gimbal_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_gimbal_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_GimbalPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def initial_thrust_direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction of the force generated by the thruster, when the engine is in its
        initial position (no gimballing), in the given reference frame.
        This is opposite to the direction in which the thruster expels propellant.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_InitialThrustDirection",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_initial_thrust_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_initial_thrust_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_InitialThrustDirection",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def initial_thrust_position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position at which the thruster generates thrust, when the engine is in its
        initial position (no gimballing), in the given reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.

        This position can move when the gimbal rotates. This is because the thrust position and
        gimbal position are not necessarily the same.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_InitialThrustPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_initial_thrust_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_initial_thrust_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_InitialThrustPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def thrust_direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction of the force generated by the thruster, in the given reference frame.
        This is opposite to the direction in which the thruster expels propellant.
        For gimballed engines, this takes into account the current rotation of the gimbal.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_ThrustDirection",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_thrust_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_thrust_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_ThrustDirection",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def thrust_position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position at which the thruster generates thrust, in the given reference frame.
        For gimballed engines, this takes into account the current rotation of the gimbal.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Thruster_ThrustPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_thrust_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_thrust_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Thruster_ThrustPosition",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Thruster"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class Vessel(ClassBase):
    """
    These objects are used to interact with vessels in KSP. This includes getting
    orbital and flight data, manipulating control inputs and managing resources.
    Created using SpaceCenter#activeVessel or SpaceCenter#vessels.
    """
    @property
    def auto_pilot(self) -> AutoPilot:
        """
        An SpaceCenter.AutoPilot object, that can be used to perform
        simple auto-piloting of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AutoPilot",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "AutoPilot")
        )

    def _return_type_auto_pilot(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "AutoPilot")

    def _build_call_auto_pilot(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AutoPilot",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "AutoPilot")
        )

    @property
    def available_control_surface_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that the aerodynamic control surfaces can generate.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableControlSurfaceTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_control_surface_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_control_surface_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableControlSurfaceTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_engine_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that the currently active and gimballed engines can generate.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableEngineTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_engine_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_engine_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableEngineTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_other_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that parts (excluding reaction wheels, gimballed engines,
        RCS and control surfaces) can generate.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableOtherTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_other_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_other_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableOtherTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_rcs_force(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum force that the currently active RCS thrusters can generate.
        Returns the forces in ``N`` along each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the right, forward and bottom directions of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableRCSForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_rcs_force(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_rcs_force(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableRCSForce",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_rcs_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that the currently active RCS thrusters can generate.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableRCSTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_rcs_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_rcs_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableRCSTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_reaction_wheel_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that the currently active and powered reaction wheels can generate.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableReactionWheelTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_reaction_wheel_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_reaction_wheel_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableReactionWheelTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def available_thrust(self) -> float:
        """
        Gets the total available thrust that can be produced by the vessel's
        active engines, in Newtons. This is computed by summing
        SpaceCenter.Engine#availableThrust for every active engine in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_available_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_available_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def available_torque(self) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The maximum torque that the vessel generates. Includes contributions from
        reaction wheels, RCS, gimballed engines and aerodynamic control surfaces.
        Returns the torques in ``N.m`` around each of the coordinate axes of the
        vessels reference frame (SpaceCenter.ReferenceFrame).
        These axes are equivalent to the pitch, roll and yaw axes of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_available_torque(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_available_torque(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_AvailableTorque",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    @property
    def biome(self) -> str:
        """
        The name of the biome the vessel is currently in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Biome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.string_type
        )

    def _return_type_biome(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_biome(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Biome",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.string_type
        )

    @property
    def comms(self) -> Comms:
        """
        Returns a SpaceCenter.Comms object that can be used to interact
        with CommNet for this vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Comms",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Comms")
        )

    def _return_type_comms(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Comms")

    def _build_call_comms(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Comms",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Comms")
        )

    @property
    def control(self) -> Control:
        """
        Returns a SpaceCenter.Control object that can be used to manipulate
        the vessel's control inputs. For example, its pitch/yaw/roll controls,
        RCS and thrust.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Control",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Control")
        )

    def _return_type_control(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Control")

    def _build_call_control(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Control",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Control")
        )

    @property
    def crew(self) -> List[CrewMember]:
        """
        The crew in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Crew",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CrewMember"))
        )

    def _return_type_crew(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CrewMember"))

    def _build_call_crew(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Crew",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "CrewMember"))
        )

    @property
    def crew_capacity(self) -> int:
        """
        The number of crew that can occupy the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_CrewCapacity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.sint32_type
        )

    def _return_type_crew_capacity(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_crew_capacity(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_CrewCapacity",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.sint32_type
        )

    @property
    def crew_count(self) -> int:
        """
        The number of crew that are occupying the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_CrewCount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.sint32_type
        )

    def _return_type_crew_count(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_crew_count(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_CrewCount",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.sint32_type
        )

    @property
    def dry_mass(self) -> float:
        """
        The total mass of the vessel, excluding resources, in kg.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_DryMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_dry_mass(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_dry_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_DryMass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def inertia_tensor(self) -> List[float]:
        """
        The inertia tensor of the vessel around its center of mass,
        in the vessels reference frame (SpaceCenter.ReferenceFrame).
        Returns the 3x3 matrix as a list of elements, in row-major order.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_InertiaTensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.list_type(self._client._types.double_type)
        )

    def _return_type_inertia_tensor(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.double_type)

    def _build_call_inertia_tensor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_InertiaTensor",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.list_type(self._client._types.double_type)
        )

    @property
    def kerbin_sea_level_specific_impulse(self) -> float:
        """
        The combined specific impulse of all active engines at sea level on Kerbin, in seconds.
        This is computed using the formula
        `described here <https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_kerbin_sea_level_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_kerbin_sea_level_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_KerbinSeaLevelSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def mass(self) -> float:
        """
        The total mass of the vessel, including resources, in kg.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_mass(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_mass(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Mass",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def max_thrust(self) -> float:
        """
        The total maximum thrust that can be produced by the vessel's active
        engines, in Newtons. This is computed by summing
        SpaceCenter.Engine#maxThrust for every active engine.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_max_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_MaxThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def max_vacuum_thrust(self) -> float:
        """
        The total maximum thrust that can be produced by the vessel's active
        engines when the vessel is in a vacuum, in Newtons. This is computed by
        summing SpaceCenter.Engine#maxVacuumThrust for every active engine.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_max_vacuum_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_vacuum_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_MaxVacuumThrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def met(self) -> float:
        """
        The mission elapsed time in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_MET",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.double_type
        )

    def _return_type_met(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_met(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_MET",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.double_type
        )

    @property
    def moment_of_inertia(self) -> Tuple[float,float,float]:
        """
        The moment of inertia of the vessel around its center of mass in ``kg.m^2``.
        The inertia values in the returned 3-tuple are around the
        pitch, roll and yaw directions respectively.
        This corresponds to the vessels reference frame (SpaceCenter.ReferenceFrame).
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_MomentOfInertia",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_moment_of_inertia(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_moment_of_inertia(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_MomentOfInertia",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    @property
    def name(self) -> str:
        """
        The name of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.string_type
        )

    @name.setter
    def name(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_set_Name",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.string_type],
            None
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.string_type
        )

    @property
    def orbit(self) -> Orbit:
        """
        The current orbit of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    def _return_type_orbit(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Orbit")

    def _build_call_orbit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Orbit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Orbit")
        )

    @property
    def orbital_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the vessel,
        and orientated with the vessels orbital prograde/normal/radial directions.
         * The origin is at the center of mass of the vessel.
         * The axes rotate with the orbital prograde/normal/radial directions.
         * The x-axis points in the orbital anti-radial direction.
         * The y-axis points in the orbital prograde direction.
         * The z-axis points in the orbital normal direction.

        Be careful not to confuse this with 'orbit' mode on the navball.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_orbital_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_orbital_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_OrbitalReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def parts(self) -> Parts:
        """
        A SpaceCenter.Parts object, that can used to interact with the parts that make up this vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Parts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Parts")
        )

    def _return_type_parts(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Parts")

    def _build_call_parts(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Parts",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Parts")
        )

    @property
    def recoverable(self) -> bool:
        """
        Whether the vessel is recoverable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Recoverable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.bool_type
        )

    def _return_type_recoverable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_recoverable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Recoverable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.bool_type
        )

    @property
    def reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the vessel,
        and orientated with the vessel.
         * The origin is at the center of mass of the vessel.
         * The axes rotate with the vessel.
         * The x-axis points out to the right of the vessel.
         * The y-axis points in the forward direction of the vessel.
         * The z-axis points out of the bottom off the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_ReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def resources(self) -> Resources:
        """
        A SpaceCenter.Resources object, that can used to get information
        about resources stored in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Resources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    def _return_type_resources(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Resources")

    def _build_call_resources(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Resources",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    @property
    def situation(self) -> VesselSituation:
        """
        The situation the vessel is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Situation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.enumeration_type("SpaceCenter", "VesselSituation")
        )

    def _return_type_situation(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "VesselSituation")

    def _build_call_situation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Situation",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.enumeration_type("SpaceCenter", "VesselSituation")
        )

    @property
    def specific_impulse(self) -> float:
        """
        The combined specific impulse of all active engines, in seconds. This is computed using the formula
        `described here <https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_SpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def surface_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the vessel,
        and orientated with the surface of the body being orbited.
         * The origin is at the center of mass of the vessel.
         * The axes rotate with the north and up directions on the surface of the body.
         * The x-axis points in the `zenith <https://en.wikipedia.org/wiki/Zenith>`_
           direction (upwards, normal to the body being orbited, from the center of the body towards the center of
           mass of the vessel).
         * The y-axis points northwards towards the
           `astronomical horizon <https://en.wikipedia.org/wiki/Horizon>`_ (north, and tangential to the
           surface of the body -- the direction in which a compass would point when on the surface).
         * The z-axis points eastwards towards the
           `astronomical horizon <https://en.wikipedia.org/wiki/Horizon>`_ (east, and tangential to the
           surface of the body -- east on a compass when on the surface).

        Be careful not to confuse this with 'surface' mode on the navball.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_SurfaceReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_surface_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_surface_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_SurfaceReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def surface_velocity_reference_frame(self) -> ReferenceFrame:
        """
        The reference frame that is fixed relative to the vessel,
        and orientated with the velocity vector of the vessel relative
        to the surface of the body being orbited.
         * The origin is at the center of mass of the vessel.
         * The axes rotate with the vessel's velocity vector.
         * The y-axis points in the direction of the vessel's velocity vector,
           relative to the surface of the body being orbited.
         * The z-axis is in the plane of the
           `astronomical horizon <https://en.wikipedia.org/wiki/Horizon>`_.
         * The x-axis is orthogonal to the other two axes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_SurfaceVelocityReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    def _return_type_surface_velocity_reference_frame(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ReferenceFrame")

    def _build_call_surface_velocity_reference_frame(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_SurfaceVelocityReferenceFrame",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.class_type("SpaceCenter", "ReferenceFrame")
        )

    @property
    def thrust(self) -> float:
        """
        The total thrust currently being produced by the vessel's engines, in
        Newtons. This is computed by summing SpaceCenter.Engine#thrust for
        every engine in the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Thrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_thrust(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_thrust(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Thrust",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    @property
    def type(self) -> VesselType:
        """
        The type of the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.enumeration_type("SpaceCenter", "VesselType")
        )

    @type.setter
    def type(self, value: VesselType) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_set_Type",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.enumeration_type("SpaceCenter", "VesselType")],
            None
        )

    def _return_type_type(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "VesselType")

    def _build_call_type(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_Type",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.enumeration_type("SpaceCenter", "VesselType")
        )

    @property
    def vacuum_specific_impulse(self) -> float:
        """
        The combined vacuum specific impulse of all active engines, in seconds. This is computed using the formula
        `described here <https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines>`_.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def _return_type_vacuum_specific_impulse(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_vacuum_specific_impulse(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_get_VacuumSpecificImpulse",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            self._client._types.float_type
        )

    def angular_velocity(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The angular velocity of the vessel, in the given reference frame.

        :returns: The angular velocity as a vector. The magnitude of the vector is the rotational
                  speed of the vessel, in radians per second. The direction of the vector indicates the
                  axis of rotation, using the right-hand rule.

        :param reference_frame: The reference frame the returned
                                angular velocity is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_AngularVelocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_angular_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_angular_velocity(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_AngularVelocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def available_thrust_at(self, pressure: float) -> float:
        """
        Gets the total available thrust that can be produced by the vessel's
        active engines, in Newtons. This is computed by summing
        SpaceCenter.Engine#availableThrustAt for every active engine in the vessel.
        Takes the given pressure into account.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_AvailableThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_available_thrust_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_available_thrust_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_AvailableThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def bounding_box(self, reference_frame: ReferenceFrame) -> Tuple[Tuple[float,float,float],Tuple[float,float,float]]:
        """
        The axis-aligned bounding box of the vessel in the given reference frame.

        :returns: The positions of the minimum and maximum vertices of the box,
                  as position vectors.

        :param reference_frame: The reference frame that the returned
                                position vectors are in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_BoundingBox",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def _return_type_bounding_box(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))

    def _build_call_bounding_box(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_BoundingBox",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type), self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type))
        )

    def direction(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The direction in which the vessel is pointing, in the given reference frame.

        :returns: The direction as a unit vector.

        :param reference_frame: The reference frame that the returned
                                direction is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_direction(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_direction(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Direction",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def flight(self, reference_frame: Optional[ReferenceFrame] = None) -> Flight:
        """
        Returns a SpaceCenter.Flight object that can be used to get flight
        telemetry for the vessel, in the specified reference frame.

        :param reference_frame: Reference frame. Defaults to the vessel's surface reference frame
                                (SpaceCenter.Vessel#surfaceReferenceFrame).
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Flight",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "Flight")
        )

    def _return_type_flight(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Flight")

    def _build_call_flight(self, reference_frame: Optional[ReferenceFrame] = None) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Flight",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.class_type("SpaceCenter", "Flight")
        )

    def max_thrust_at(self, pressure: float) -> float:
        """
        The total maximum thrust that can be produced by the vessel's active
        engines, in Newtons. This is computed by summing
        SpaceCenter.Engine#maxThrustAt for every active engine.
        Takes the given pressure into account.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_MaxThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_max_thrust_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_max_thrust_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_MaxThrustAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def position(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The position of the center of mass of the vessel, in the given reference frame.

        :returns: The position as a vector.

        :param reference_frame: The reference frame that the returned
                                position vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_position(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_position(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Position",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def recover(self) -> None:
        """
        Recover the vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Recover",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Vessel"), ],
            None
        )

    def _return_type_recover(self) -> TypeBase:
        return None

    def _build_call_recover(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Recover",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Vessel"), ],
            None
        )

    def resources_in_decouple_stage(self, stage: int, cumulative: bool = True) -> Resources:
        """
        Returns a SpaceCenter.Resources object, that can used to get
        information about resources stored in a given stage.

        :param stage: Get resources for parts that are decoupled in this stage.

        :param cumulative: When ``False``, returns the resources for parts
                           decoupled in just the given stage. When ``True`` returns the resources decoupled in
                           the given stage and all subsequent stages combined.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_ResourcesInDecoupleStage",
            [self, stage, cumulative],
            ["self", "stage", "cumulative"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.sint32_type, self._client._types.bool_type],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    def _return_type_resources_in_decouple_stage(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Resources")

    def _build_call_resources_in_decouple_stage(self, stage: int, cumulative: bool = True) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_ResourcesInDecoupleStage",
            [self, stage, cumulative],
            ["self", "stage", "cumulative"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.sint32_type, self._client._types.bool_type],
            self._client._types.class_type("SpaceCenter", "Resources")
        )

    def rotation(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float,float]:
        """
        The rotation of the vessel, in the given reference frame.

        :returns: The rotation as a quaternion of the form ``(x, y, z, w)``.

        :param reference_frame: The reference frame that the returned
                                rotation is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_rotation(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_rotation(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Rotation",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def specific_impulse_at(self, pressure: float) -> float:
        """
        The combined specific impulse of all active engines, in seconds. This is computed using the formula
        `described here <https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines>`_.
        Takes the given pressure into account.

        :param pressure: Atmospheric pressure in atmospheres
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_SpecificImpulseAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def _return_type_specific_impulse_at(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_specific_impulse_at(self, pressure: float) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_SpecificImpulseAt",
            [self, pressure],
            ["self", "pressure"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.double_type],
            self._client._types.float_type
        )

    def velocity(self, reference_frame: ReferenceFrame) -> Tuple[float,float,float]:
        """
        The velocity of the center of mass of the vessel, in the given reference frame.

        :returns: The velocity as a vector. The vector points in the direction of travel,
                  and its magnitude is the speed of the body in meters per second.

        :param reference_frame: The reference frame that the returned
                                velocity vector is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Vessel_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )

    def _return_type_velocity(self) -> TypeBase:
        return self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)

    def _build_call_velocity(self, reference_frame: ReferenceFrame) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Vessel_Velocity",
            [self, reference_frame],
            ["self", "reference_frame"],
            [self._client._types.class_type("SpaceCenter", "Vessel"), self._client._types.class_type("SpaceCenter", "ReferenceFrame")],
            self._client._types.tuple_type(self._client._types.double_type, self._client._types.double_type, self._client._types.double_type)
        )



class Waypoint(ClassBase):
    """
    Represents a waypoint. Can be created using SpaceCenter.WaypointManager#addWaypoint.
    """
    @property
    def bedrock_altitude(self) -> float:
        """
        The altitude of the waypoint above the surface of the body, in meters.
        When over water, this is the altitude above the sea floor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_BedrockAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @bedrock_altitude.setter
    def bedrock_altitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_BedrockAltitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.double_type],
            None
        )

    def _return_type_bedrock_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_bedrock_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_BedrockAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @property
    def body(self) -> CelestialBody:
        """
        The celestial body the waypoint is attached to.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @body.setter
    def body(self, value: CelestialBody) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Body",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.class_type("SpaceCenter", "CelestialBody")],
            None
        )

    def _return_type_body(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CelestialBody")

    def _build_call_body(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Body",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @property
    def clustered(self) -> bool:
        """
        ``True`` if this waypoint is part of a set of clustered waypoints with greek letter
        names appended (Alpha, Beta, Gamma, etc).
        If ``True``, there is a one-to-one correspondence with the greek letter name and
        the SpaceCenter.Waypoint#index.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Clustered",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    def _return_type_clustered(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_clustered(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Clustered",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    @property
    def color(self) -> int:
        """
        The seed of the icon color. See SpaceCenter.WaypointManager#colors for example colors.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Color",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.sint32_type
        )

    @color.setter
    def color(self, value: int) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Color",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.sint32_type],
            None
        )

    def _return_type_color(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_color(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Color",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.sint32_type
        )

    @property
    def contract(self) -> Contract:
        """
        The associated contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Contract",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.class_type("SpaceCenter", "Contract")
        )

    def _return_type_contract(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Contract")

    def _build_call_contract(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Contract",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.class_type("SpaceCenter", "Contract")
        )

    @property
    def grounded(self) -> bool:
        """
        ``True`` if the waypoint is attached to the ground.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Grounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    def _return_type_grounded(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_grounded(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Grounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    @property
    def has_contract(self) -> bool:
        """
        Whether the waypoint belongs to a contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_HasContract",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    def _return_type_has_contract(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_contract(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_HasContract",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    @property
    def icon(self) -> str:
        """
        The icon of the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Icon",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.string_type
        )

    @icon.setter
    def icon(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Icon",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.string_type],
            None
        )

    def _return_type_icon(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_icon(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Icon",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.string_type
        )

    @property
    def index(self) -> int:
        """
        The integer index of this waypoint within its cluster of sibling waypoints.
        In other words, when you have a cluster of waypoints called "Somewhere Alpha",
        "Somewhere Beta" and "Somewhere Gamma", the alpha site has index 0, the beta
        site has index 1 and the gamma site has index 2.
        When SpaceCenter.Waypoint#clustered is ``False``, this is zero.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Index",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.sint32_type
        )

    def _return_type_index(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_index(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Index",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.sint32_type
        )

    @property
    def latitude(self) -> float:
        """
        The latitude of the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Latitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @latitude.setter
    def latitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Latitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.double_type],
            None
        )

    def _return_type_latitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_latitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Latitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @property
    def longitude(self) -> float:
        """
        The longitude of the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Longitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @longitude.setter
    def longitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Longitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.double_type],
            None
        )

    def _return_type_longitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_longitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Longitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @property
    def mean_altitude(self) -> float:
        """
        The altitude of the waypoint above sea level, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_MeanAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @mean_altitude.setter
    def mean_altitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_MeanAltitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.double_type],
            None
        )

    def _return_type_mean_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_mean_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_MeanAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @property
    def name(self) -> str:
        """
        The name of the waypoint as it appears on the map and the contract.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.string_type
        )

    @name.setter
    def name(self, value: str) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_Name",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.string_type],
            None
        )

    def _return_type_name(self) -> TypeBase:
        return self._client._types.string_type

    def _build_call_name(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_Name",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.string_type
        )

    @property
    def near_surface(self) -> bool:
        """
        ``True`` if the waypoint is near to the surface of a body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_NearSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    def _return_type_near_surface(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_near_surface(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_NearSurface",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.bool_type
        )

    @property
    def surface_altitude(self) -> float:
        """
        The altitude of the waypoint above the surface of the body or sea level,
        whichever is closer, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_get_SurfaceAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    @surface_altitude.setter
    def surface_altitude(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_set_SurfaceAltitude",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), self._client._types.double_type],
            None
        )

    def _return_type_surface_altitude(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_surface_altitude(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_get_SurfaceAltitude",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Waypoint")],
            self._client._types.double_type
        )

    def remove(self) -> None:
        """
        Removes the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Waypoint_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), ],
            None
        )

    def _return_type_remove(self) -> TypeBase:
        return None

    def _build_call_remove(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Waypoint_Remove",
            [self, ],
            ["self", ],
            [self._client._types.class_type("SpaceCenter", "Waypoint"), ],
            None
        )



class WaypointManager(ClassBase):
    """
    Waypoints are the location markers you can see on the map view showing you where contracts are targeted for.
    With this structure, you can obtain coordinate data for the locations of these waypoints.
    Obtained by calling SpaceCenter#waypointManager.
    """
    @property
    def colors(self) -> Dict[str, int]:
        """
        An example map of known color - seed pairs.
        Any other integers may be used as seed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "WaypointManager_get_Colors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.sint32_type)
        )

    def _return_type_colors(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.sint32_type)

    def _build_call_colors(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "WaypointManager_get_Colors",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.sint32_type)
        )

    @property
    def icons(self) -> List[str]:
        """
        Returns all available icons (from "GameData/Squad/Contracts/Icons/").
        """
        return self._client._invoke(
            "SpaceCenter",
            "WaypointManager_get_Icons",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.list_type(self._client._types.string_type)
        )

    def _return_type_icons(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.string_type)

    def _build_call_icons(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "WaypointManager_get_Icons",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.list_type(self._client._types.string_type)
        )

    @property
    def waypoints(self) -> List[Waypoint]:
        """
        A list of all existing waypoints.
        """
        return self._client._invoke(
            "SpaceCenter",
            "WaypointManager_get_Waypoints",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Waypoint"))
        )

    def _return_type_waypoints(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Waypoint"))

    def _build_call_waypoints(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "WaypointManager_get_Waypoints",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager")],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Waypoint"))
        )

    def add_waypoint(self, latitude: float, longitude: float, body: CelestialBody, name: str) -> Waypoint:
        """
        Creates a waypoint at the given position at ground level, and returns a
        SpaceCenter.Waypoint object that can be used to modify it.

        :param latitude: Latitude of the waypoint.

        :param longitude: Longitude of the waypoint.

        :param body: Celestial body the waypoint is attached to.

        :param name: Name of the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "WaypointManager_AddWaypoint",
            [self, latitude, longitude, body, name],
            ["self", "latitude", "longitude", "body", "name"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Waypoint")
        )

    def _return_type_add_waypoint(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Waypoint")

    def _build_call_add_waypoint(self, latitude: float, longitude: float, body: CelestialBody, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "WaypointManager_AddWaypoint",
            [self, latitude, longitude, body, name],
            ["self", "latitude", "longitude", "body", "name"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager"), self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Waypoint")
        )

    def add_waypoint_at_altitude(self, latitude: float, longitude: float, altitude: float, body: CelestialBody, name: str) -> Waypoint:
        """
        Creates a waypoint at the given position and altitude, and returns a
        SpaceCenter.Waypoint object that can be used to modify it.

        :param latitude: Latitude of the waypoint.

        :param longitude: Longitude of the waypoint.

        :param altitude: Altitude (above sea level) of the waypoint.

        :param body: Celestial body the waypoint is attached to.

        :param name: Name of the waypoint.
        """
        return self._client._invoke(
            "SpaceCenter",
            "WaypointManager_AddWaypointAtAltitude",
            [self, latitude, longitude, altitude, body, name],
            ["self", "latitude", "longitude", "altitude", "body", "name"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager"), self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Waypoint")
        )

    def _return_type_add_waypoint_at_altitude(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Waypoint")

    def _build_call_add_waypoint_at_altitude(self, latitude: float, longitude: float, altitude: float, body: CelestialBody, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "WaypointManager_AddWaypointAtAltitude",
            [self, latitude, longitude, altitude, body, name],
            ["self", "latitude", "longitude", "altitude", "body", "name"],
            [self._client._types.class_type("SpaceCenter", "WaypointManager"), self._client._types.double_type, self._client._types.double_type, self._client._types.double_type, self._client._types.class_type("SpaceCenter", "CelestialBody"), self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "Waypoint")
        )



class Wheel(ClassBase):
    """
    A wheel. Includes landing gear and rover wheels.
    Obtained by calling SpaceCenter.Part#wheel.
    Can be used to control the motors, steering and deployment of wheels, among other things.
    """
    @property
    def auto_friction_control(self) -> bool:
        """
        Whether automatic friction control is enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_AutoFrictionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @auto_friction_control.setter
    def auto_friction_control(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_AutoFrictionControl",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_auto_friction_control(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_auto_friction_control(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_AutoFrictionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def brakes(self) -> float:
        """
        The braking force, as a percentage of maximum, when the brakes are applied.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Brakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @brakes.setter
    def brakes(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_Brakes",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_brakes(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_brakes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Brakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def broken(self) -> bool:
        """
        Whether the wheel is broken.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Broken",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_broken(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_broken(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Broken",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def deflection(self) -> float:
        """
        Current deflection of the wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Deflection",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_deflection(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_deflection(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Deflection",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def deployable(self) -> bool:
        """
        Whether the wheel is deployable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_deployable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Deployable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def deployed(self) -> bool:
        """
        Whether the wheel is deployed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @deployed.setter
    def deployed(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_Deployed",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_deployed(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_deployed(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Deployed",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def drive_limiter(self) -> float:
        """
        Manual setting for the motor limiter.
        Only takes effect if the wheel has automatic traction control disabled.
        A value between 0 and 100 inclusive.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_DriveLimiter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @drive_limiter.setter
    def drive_limiter(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_DriveLimiter",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_drive_limiter(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_drive_limiter(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_DriveLimiter",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def grounded(self) -> bool:
        """
        Whether the wheel is touching the ground.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Grounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_grounded(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_grounded(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Grounded",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def has_brakes(self) -> bool:
        """
        Whether the wheel has brakes.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_HasBrakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_has_brakes(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_brakes(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_HasBrakes",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def has_suspension(self) -> bool:
        """
        Whether the wheel has suspension.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_HasSuspension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_has_suspension(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_has_suspension(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_HasSuspension",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def manual_friction_control(self) -> float:
        """
        Manual friction control value. Only has an effect if automatic friction control is disabled.
        A value between 0 and 5 inclusive.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_ManualFrictionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @manual_friction_control.setter
    def manual_friction_control(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_ManualFrictionControl",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_manual_friction_control(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_manual_friction_control(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_ManualFrictionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def motor_enabled(self) -> bool:
        """
        Whether the motor is enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_MotorEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @motor_enabled.setter
    def motor_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_MotorEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_MotorEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def motor_inverted(self) -> bool:
        """
        Whether the direction of the motor is inverted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_MotorInverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @motor_inverted.setter
    def motor_inverted(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_MotorInverted",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_motor_inverted(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_motor_inverted(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_MotorInverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def motor_output(self) -> float:
        """
        The output of the motor. This is the torque currently being generated, in Newton meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_MotorOutput",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_motor_output(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_motor_output(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_MotorOutput",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def motor_state(self) -> MotorState:
        """
        Whether the direction of the motor is inverted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_MotorState",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.enumeration_type("SpaceCenter", "MotorState")
        )

    def _return_type_motor_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "MotorState")

    def _build_call_motor_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_MotorState",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.enumeration_type("SpaceCenter", "MotorState")
        )

    @property
    def part(self) -> Part:
        """
        The part object for this wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    def _return_type_part(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Part")

    def _build_call_part(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Part",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.class_type("SpaceCenter", "Part")
        )

    @property
    def powered(self) -> bool:
        """
        Whether the wheel is powered by a motor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Powered",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_powered(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_powered(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Powered",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def radius(self) -> float:
        """
        Radius of the wheel, in meters.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Radius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_radius(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_radius(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Radius",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def repairable(self) -> bool:
        """
        Whether the wheel is repairable.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Repairable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_repairable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_repairable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Repairable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def slip(self) -> float:
        """
        Current slip of the wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Slip",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_slip(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_slip(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Slip",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def state(self) -> WheelState:
        """
        The current state of the wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.enumeration_type("SpaceCenter", "WheelState")
        )

    def _return_type_state(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "WheelState")

    def _build_call_state(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_State",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.enumeration_type("SpaceCenter", "WheelState")
        )

    @property
    def steerable(self) -> bool:
        """
        Whether the wheel has steering.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Steerable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    def _return_type_steerable(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_steerable(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Steerable",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def steering_angle_limit(self) -> float:
        """
        The steering angle limit.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SteeringAngleLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @steering_angle_limit.setter
    def steering_angle_limit(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_SteeringAngleLimit",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_steering_angle_limit(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_steering_angle_limit(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SteeringAngleLimit",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def steering_enabled(self) -> bool:
        """
        Whether the wheel steering is enabled.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SteeringEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @steering_enabled.setter
    def steering_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_SteeringEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_steering_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_steering_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SteeringEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def steering_inverted(self) -> bool:
        """
        Whether the wheel steering is inverted.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SteeringInverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @steering_inverted.setter
    def steering_inverted(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_SteeringInverted",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_steering_inverted(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_steering_inverted(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SteeringInverted",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @property
    def steering_response_time(self) -> float:
        """
        Steering response time.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SteeringResponseTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @steering_response_time.setter
    def steering_response_time(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_SteeringResponseTime",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_steering_response_time(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_steering_response_time(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SteeringResponseTime",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def stress(self) -> float:
        """
        Current stress on the wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_Stress",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_stress(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_stress(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_Stress",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def stress_percentage(self) -> float:
        """
        Current stress on the wheel as a percentage of its stress tolerance.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_StressPercentage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_stress_percentage(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_stress_percentage(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_StressPercentage",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def stress_tolerance(self) -> float:
        """
        Stress tolerance of the wheel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_StressTolerance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_stress_tolerance(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_stress_tolerance(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_StressTolerance",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def suspension_damper_strength(self) -> float:
        """
        Suspension damper strength, as set in the editor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SuspensionDamperStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_suspension_damper_strength(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_suspension_damper_strength(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SuspensionDamperStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def suspension_spring_strength(self) -> float:
        """
        Suspension spring strength, as set in the editor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_SuspensionSpringStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    def _return_type_suspension_spring_strength(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_suspension_spring_strength(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_SuspensionSpringStrength",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def traction_control(self) -> float:
        """
        Setting for the traction control.
        Only takes effect if the wheel has automatic traction control enabled.
        A value between 0 and 5 inclusive.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_TractionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @traction_control.setter
    def traction_control(self, value: float) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_TractionControl",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.float_type],
            None
        )

    def _return_type_traction_control(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_traction_control(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_TractionControl",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.float_type
        )

    @property
    def traction_control_enabled(self) -> bool:
        """
        Whether automatic traction control is enabled.
        A wheel only has traction control if it is powered.
        """
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_get_TractionControlEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )

    @traction_control_enabled.setter
    def traction_control_enabled(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "Wheel_set_TractionControlEnabled",
            [self, value],
            ["self", "value"],
            [self._client._types.class_type("SpaceCenter", "Wheel"), self._client._types.bool_type],
            None
        )

    def _return_type_traction_control_enabled(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_traction_control_enabled(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "Wheel_get_TractionControlEnabled",
            [self],
            ["self"],
            [self._client._types.class_type("SpaceCenter", "Wheel")],
            self._client._types.bool_type
        )



class SpaceCenter:
    """
    Provides functionality to interact with Kerbal Space Program. This includes controlling
    the active vessel, managing its resources, planning maneuver nodes and auto-piloting.
    """

    def __init__(self, client: Client) -> None:
        self._client = client

    def __getattribute__(self, name):
        # Intercepts calls to obtain classes from the service,
        # to inject the client instance so that it can be used
        # for static method calls
        classes = object.__getattribute__(self, "_classes")
        if name in classes:
            client = object.__getattribute__(self, "_client")
            return WrappedClass(client, classes[name])

        # Intercept calls to obtain enumeration types
        enumerations = object.__getattribute__(self, "_enumerations")
        if name in enumerations:
           return enumerations[name]

        # Intercept calls to obtain exception types
        exceptions = object.__getattribute__(self, "_exceptions")
        if name in exceptions:
           return exceptions[name]

        # Fall back to default behaviour
        return object.__getattribute__(self, name)

    def __dir__(self):
        result = object.__dir__(self)
        result.extend(object.__getattribute__(self, "_classes").keys())
        result.extend(object.__getattribute__(self, "_enumerations").keys())
        result.extend(object.__getattribute__(self, "_exceptions").keys())
        return result

    _classes = {
        "Alarm": Alarm,
        "AlarmManager": AlarmManager,
        "Antenna": Antenna,
        "AutoPilot": AutoPilot,
        "Camera": Camera,
        "CargoBay": CargoBay,
        "CelestialBody": CelestialBody,
        "CommLink": CommLink,
        "CommNode": CommNode,
        "Comms": Comms,
        "Contract": Contract,
        "ContractManager": ContractManager,
        "ContractParameter": ContractParameter,
        "Control": Control,
        "ControlSurface": ControlSurface,
        "CrewMember": CrewMember,
        "Decoupler": Decoupler,
        "DockingPort": DockingPort,
        "Engine": Engine,
        "Experiment": Experiment,
        "Fairing": Fairing,
        "Flight": Flight,
        "Force": Force,
        "Intake": Intake,
        "LaunchClamp": LaunchClamp,
        "LaunchSite": LaunchSite,
        "Leg": Leg,
        "Light": Light,
        "Module": Module,
        "Node": Node,
        "Orbit": Orbit,
        "Parachute": Parachute,
        "Part": Part,
        "Parts": Parts,
        "Propellant": Propellant,
        "RCS": RCS,
        "Radiator": Radiator,
        "ReactionWheel": ReactionWheel,
        "ReferenceFrame": ReferenceFrame,
        "Resource": Resource,
        "ResourceConverter": ResourceConverter,
        "ResourceDrain": ResourceDrain,
        "ResourceHarvester": ResourceHarvester,
        "ResourceTransfer": ResourceTransfer,
        "Resources": Resources,
        "RoboticController": RoboticController,
        "RoboticHinge": RoboticHinge,
        "RoboticPiston": RoboticPiston,
        "RoboticRotation": RoboticRotation,
        "RoboticRotor": RoboticRotor,
        "ScienceData": ScienceData,
        "ScienceSubject": ScienceSubject,
        "Sensor": Sensor,
        "SolarPanel": SolarPanel,
        "Thruster": Thruster,
        "Vessel": Vessel,
        "Waypoint": Waypoint,
        "WaypointManager": WaypointManager,
        "Wheel": Wheel,
    }
    _enumerations = {
        "AntennaState": AntennaState,
        "AutoStrutMode": AutoStrutMode,
        "CameraMode": CameraMode,
        "CargoBayState": CargoBayState,
        "CommLinkType": CommLinkType,
        "ContractState": ContractState,
        "ControlInputMode": ControlInputMode,
        "ControlSource": ControlSource,
        "ControlState": ControlState,
        "CrewMemberGender": CrewMemberGender,
        "CrewMemberType": CrewMemberType,
        "DockingPortState": DockingPortState,
        "DrainMode": DrainMode,
        "EditorFacility": EditorFacility,
        "GameMode": GameMode,
        "LegState": LegState,
        "MapFilterType": MapFilterType,
        "MotorState": MotorState,
        "ParachuteState": ParachuteState,
        "RadiatorState": RadiatorState,
        "ResourceConverterState": ResourceConverterState,
        "ResourceFlowMode": ResourceFlowMode,
        "ResourceHarvesterState": ResourceHarvesterState,
        "RosterStatus": RosterStatus,
        "SASMode": SASMode,
        "SolarPanelState": SolarPanelState,
        "SpeedMode": SpeedMode,
        "SuitType": SuitType,
        "VesselSituation": VesselSituation,
        "VesselType": VesselType,
        "WarpMode": WarpMode,
        "WheelState": WheelState,
    }
    _exceptions = {
    }

    @property
    def active_vessel(self) -> Vessel:
        """
        The currently active vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_ActiveVessel",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @active_vessel.setter
    def active_vessel(self, value: Vessel) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_ActiveVessel",
            [value],
            ["value"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            None
        )

    def _return_type_active_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_active_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_ActiveVessel",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @property
    def alarm_manager(self) -> AlarmManager:
        """
        The alarm manager.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_AlarmManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "AlarmManager")
        )

    def _return_type_alarm_manager(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "AlarmManager")

    def _build_call_alarm_manager(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_AlarmManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "AlarmManager")
        )

    @property
    def bodies(self) -> Dict[str, CelestialBody]:
        """
        A dictionary of all celestial bodies (planets, moons, etc.) in the game,
        keyed by the name of the body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Bodies",
            [],
            [],
            [],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "CelestialBody"))
        )

    def _return_type_bodies(self) -> TypeBase:
        return self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "CelestialBody"))

    def _build_call_bodies(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Bodies",
            [],
            [],
            [],
            self._client._types.dictionary_type(self._client._types.string_type, self._client._types.class_type("SpaceCenter", "CelestialBody"))
        )

    @property
    def camera(self) -> Camera:
        """
        An object that can be used to control the camera.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Camera",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Camera")
        )

    def _return_type_camera(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Camera")

    def _build_call_camera(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Camera",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Camera")
        )

    @property
    def contract_manager(self) -> ContractManager:
        """
        The contract manager.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_ContractManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "ContractManager")
        )

    def _return_type_contract_manager(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "ContractManager")

    def _build_call_contract_manager(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_ContractManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "ContractManager")
        )

    @property
    def far_available(self) -> bool:
        """
        Whether `Ferram Aerospace Research <https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/>`_ is installed.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_FARAvailable",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    def _return_type_far_available(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_far_available(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_FARAvailable",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    @property
    def funds(self) -> float:
        """
        The current amount of funds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Funds",
            [],
            [],
            [],
            self._client._types.double_type
        )

    def _return_type_funds(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_funds(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Funds",
            [],
            [],
            [],
            self._client._types.double_type
        )

    @property
    def g(self) -> float:
        """
        The value of the ` gravitational constant <https://en.wikipedia.org/wiki/Gravitational_constant>`_ G in ``N(m/kg)^2``.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_G",
            [],
            [],
            [],
            self._client._types.double_type
        )

    def _return_type_g(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_g(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_G",
            [],
            [],
            [],
            self._client._types.double_type
        )

    @property
    def game_mode(self) -> GameMode:
        """
        The current mode the game is in.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_GameMode",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "GameMode")
        )

    def _return_type_game_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "GameMode")

    def _build_call_game_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_GameMode",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "GameMode")
        )

    @property
    def launch_sites(self) -> List[LaunchSite]:
        """
        A list of available launch sites.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_LaunchSites",
            [],
            [],
            [],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchSite"))
        )

    def _return_type_launch_sites(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchSite"))

    def _build_call_launch_sites(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_LaunchSites",
            [],
            [],
            [],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "LaunchSite"))
        )

    @property
    def map_filter(self) -> MapFilterType:
        """
        The visible objects in map mode.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_MapFilter",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "MapFilterType")
        )

    @map_filter.setter
    def map_filter(self, value: MapFilterType) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_MapFilter",
            [value],
            ["value"],
            [self._client._types.enumeration_type("SpaceCenter", "MapFilterType")],
            None
        )

    def _return_type_map_filter(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "MapFilterType")

    def _build_call_map_filter(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_MapFilter",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "MapFilterType")
        )

    @property
    def maximum_rails_warp_factor(self) -> int:
        """
        The current maximum regular "on-rails" warp factor that can be set.
        A value between 0 and 7 inclusive. See
        `the KSP wiki <https://wiki.kerbalspaceprogram.com/wiki/Time_warp>`_
        for details.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_MaximumRailsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    def _return_type_maximum_rails_warp_factor(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_maximum_rails_warp_factor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_MaximumRailsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    @property
    def navball(self) -> bool:
        """
        Whether the navball is visible.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Navball",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    @navball.setter
    def navball(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_Navball",
            [value],
            ["value"],
            [self._client._types.bool_type],
            None
        )

    def _return_type_navball(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_navball(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Navball",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    @property
    def physics_warp_factor(self) -> int:
        """
        The physical time warp rate. A value between 0 and 3 inclusive. 0 means
        no time warp. Returns 0 if regular "on-rails" time warp is active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_PhysicsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    @physics_warp_factor.setter
    def physics_warp_factor(self, value: int) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_PhysicsWarpFactor",
            [value],
            ["value"],
            [self._client._types.sint32_type],
            None
        )

    def _return_type_physics_warp_factor(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_physics_warp_factor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_PhysicsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    @property
    def rails_warp_factor(self) -> int:
        """
        The time warp rate, using regular "on-rails" time warp. A value between
        0 and 7 inclusive. 0 means no time warp. Returns 0 if physical time warp
        is active.

        If requested time warp factor cannot be set, it will be set to the next
        lowest possible value. For example, if the vessel is too close to a
        planet. See ` the KSP wiki <https://wiki.kerbalspaceprogram.com/wiki/Time_warp>`_ for details.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_RailsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    @rails_warp_factor.setter
    def rails_warp_factor(self, value: int) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_RailsWarpFactor",
            [value],
            ["value"],
            [self._client._types.sint32_type],
            None
        )

    def _return_type_rails_warp_factor(self) -> TypeBase:
        return self._client._types.sint32_type

    def _build_call_rails_warp_factor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_RailsWarpFactor",
            [],
            [],
            [],
            self._client._types.sint32_type
        )

    @property
    def reputation(self) -> float:
        """
        The current amount of reputation.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Reputation",
            [],
            [],
            [],
            self._client._types.float_type
        )

    def _return_type_reputation(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_reputation(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Reputation",
            [],
            [],
            [],
            self._client._types.float_type
        )

    @property
    def science(self) -> float:
        """
        The current amount of science.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Science",
            [],
            [],
            [],
            self._client._types.float_type
        )

    def _return_type_science(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_science(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Science",
            [],
            [],
            [],
            self._client._types.float_type
        )

    @property
    def target_body(self) -> Optional[CelestialBody]:
        """
        The currently targeted celestial body.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_TargetBody",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @target_body.setter
    def target_body(self, value: CelestialBody) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_TargetBody",
            [value],
            ["value"],
            [self._client._types.class_type("SpaceCenter", "CelestialBody")],
            None
        )

    def _return_type_target_body(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CelestialBody")

    def _build_call_target_body(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_TargetBody",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "CelestialBody")
        )

    @property
    def target_docking_port(self) -> Optional[DockingPort]:
        """
        The currently targeted docking port.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_TargetDockingPort",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "DockingPort")
        )

    @target_docking_port.setter
    def target_docking_port(self, value: DockingPort) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_TargetDockingPort",
            [value],
            ["value"],
            [self._client._types.class_type("SpaceCenter", "DockingPort")],
            None
        )

    def _return_type_target_docking_port(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "DockingPort")

    def _build_call_target_docking_port(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_TargetDockingPort",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "DockingPort")
        )

    @property
    def target_vessel(self) -> Optional[Vessel]:
        """
        The currently targeted vessel.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_TargetVessel",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @target_vessel.setter
    def target_vessel(self, value: Vessel) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_TargetVessel",
            [value],
            ["value"],
            [self._client._types.class_type("SpaceCenter", "Vessel")],
            None
        )

    def _return_type_target_vessel(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "Vessel")

    def _build_call_target_vessel(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_TargetVessel",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "Vessel")
        )

    @property
    def ui_visible(self) -> bool:
        """
        Whether the UI is visible.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_UIVisible",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    @ui_visible.setter
    def ui_visible(self, value: bool) -> None:
        return self._client._invoke(
            "SpaceCenter",
            "set_UIVisible",
            [value],
            ["value"],
            [self._client._types.bool_type],
            None
        )

    def _return_type_ui_visible(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_ui_visible(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_UIVisible",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    @property
    def ut(self) -> float:
        """
        The current universal time in seconds.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_UT",
            [],
            [],
            [],
            self._client._types.double_type
        )

    def _return_type_ut(self) -> TypeBase:
        return self._client._types.double_type

    def _build_call_ut(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_UT",
            [],
            [],
            [],
            self._client._types.double_type
        )

    @property
    def vessels(self) -> List[Vessel]:
        """
        A list of all the vessels in the game.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_Vessels",
            [],
            [],
            [],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))
        )

    def _return_type_vessels(self) -> TypeBase:
        return self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))

    def _build_call_vessels(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_Vessels",
            [],
            [],
            [],
            self._client._types.list_type(self._client._types.class_type("SpaceCenter", "Vessel"))
        )

    @property
    def warp_factor(self) -> float:
        """
        The current warp factor. This is the index of the rate at which time
        is passing for either regular "on-rails" or physical time warp. Returns 0
        if time warp is not active. When in on-rails time warp, this is equal to
        SpaceCenter#railsWarpFactor, and in physics time warp, this is equal to
        SpaceCenter#physicsWarpFactor.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_WarpFactor",
            [],
            [],
            [],
            self._client._types.float_type
        )

    def _return_type_warp_factor(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_warp_factor(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_WarpFactor",
            [],
            [],
            [],
            self._client._types.float_type
        )

    @property
    def warp_mode(self) -> WarpMode:
        """
        The current time warp mode. Returns SpaceCenter.WarpMode#none if time
        warp is not active, SpaceCenter.WarpMode#rails if regular "on-rails" time warp
        is active, or SpaceCenter.WarpMode#physics if physical time warp is active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_WarpMode",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "WarpMode")
        )

    def _return_type_warp_mode(self) -> TypeBase:
        return self._client._types.enumeration_type("SpaceCenter", "WarpMode")

    def _build_call_warp_mode(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_WarpMode",
            [],
            [],
            [],
            self._client._types.enumeration_type("SpaceCenter", "WarpMode")
        )

    @property
    def warp_rate(self) -> float:
        """
        The current warp rate. This is the rate at which time is passing for
        either on-rails or physical time warp. For example, a value of 10 means
        time is passing 10x faster than normal. Returns 1 if time warp is not
        active.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_WarpRate",
            [],
            [],
            [],
            self._client._types.float_type
        )

    def _return_type_warp_rate(self) -> TypeBase:
        return self._client._types.float_type

    def _build_call_warp_rate(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_WarpRate",
            [],
            [],
            [],
            self._client._types.float_type
        )

    @property
    def waypoint_manager(self) -> WaypointManager:
        """
        The waypoint manager.
        """
        return self._client._invoke(
            "SpaceCenter",
            "get_WaypointManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "WaypointManager")
        )

    def _return_type_waypoint_manager(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "WaypointManager")

    def _build_call_waypoint_manager(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "get_WaypointManager",
            [],
            [],
            [],
            self._client._types.class_type("SpaceCenter", "WaypointManager")
        )

    def can_rails_warp_at(self, factor: int = 1) -> bool:
        """
        Returns ``True`` if regular "on-rails" time warp can be used, at the specified warp
        factor. The maximum time warp rate is limited by various things,
        including how close the active vessel is to a planet. See
        `the KSP wiki <https://wiki.kerbalspaceprogram.com/wiki/Time_warp>`_
        for details.

        :param factor: The warp factor to check.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CanRailsWarpAt",
            [factor],
            ["factor"],
            [self._client._types.sint32_type],
            self._client._types.bool_type
        )

    def _return_type_can_rails_warp_at(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_rails_warp_at(self, factor: int = 1) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CanRailsWarpAt",
            [factor],
            ["factor"],
            [self._client._types.sint32_type],
            self._client._types.bool_type
        )

    def can_revert_to_launch(self) -> bool:
        """
        Whether the current flight can be reverted to launch.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CanRevertToLaunch",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    def _return_type_can_revert_to_launch(self) -> TypeBase:
        return self._client._types.bool_type

    def _build_call_can_revert_to_launch(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CanRevertToLaunch",
            [],
            [],
            [],
            self._client._types.bool_type
        )

    def clear_target(self) -> None:
        """
        Clears the current target.
        """
        return self._client._invoke(
            "SpaceCenter",
            "ClearTarget",
            [],
            [],
            [],
            None
        )

    def _return_type_clear_target(self) -> TypeBase:
        return None

    def _build_call_clear_target(self) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "ClearTarget",
            [],
            [],
            [],
            None
        )

    def create_kerbal(self, name: str, job: str, male: bool) -> None:
        """
        Creates a Kerbal.
        """
        return self._client._invoke(
            "SpaceCenter",
            "CreateKerbal",
            [name, job, male],
            ["name", "job", "male"],
            [self._client._types.string_type, self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def _return_type_create_kerbal(self) -> TypeBase:
        return None

    def _build_call_create_kerbal(self, name: str, job: str, male: bool) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "CreateKerbal",
            [name, job, male],
            ["name", "job", "male"],
            [self._client._types.string_type, self._client._types.string_type, self._client._types.bool_type],
            None
        )

    def get_kerbal(self, name: str) -> Optional[CrewMember]:
        """
        Find a Kerbal by name.

        :returns: ``None`` if no Kerbal with the given name exists.
        """
        return self._client._invoke(
            "SpaceCenter",
            "GetKerbal",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "CrewMember")
        )

    def _return_type_get_kerbal(self) -> TypeBase:
        return self._client._types.class_type("SpaceCenter", "CrewMember")

    def _build_call_get_kerbal(self, name: str) -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "GetKerbal",
            [name],
            ["name"],
            [self._client._types.string_type],
            self._client._types.class_type("SpaceCenter", "CrewMember")
        )

    def launch_vessel(self, craft_directory: str, name: str, launch_site: str, recover: bool = True, crew: Optional[List[str]] = None, flag_url: str = '') -> None:
        """
        Launch a vessel.

        :param craft_directory: Name of the directory in the current saves
                                "Ships" directory, that contains the craft file.
                                For example ``"VAB"`` or ``"SPH"``.

        :param name: Name of the vessel to launch. This is the name of the ".craft" file
                     in the save directory, without the ".craft" file extension.

        :param launch_site: Name of the launch site. For example ``"LaunchPad"`` or
                            ``"Runway"``.

        :param recover: If true and there is a vessel on the launch site,
                        recover it before launching.

        :param crew: If not ``None``, a list of names of Kerbals to place in the craft. Otherwise the crew will use default assignments.

        :param flag_url: If not ``None``, the asset URL of the mission flag to use for the launch.

        Throws an exception if any of the games pre-flight checks fail.
        """
        return self._client._invoke(
            "SpaceCenter",
            "LaunchVessel",
            [craft_directory, name, launch_site, recover, crew, flag_url],
            ["craft_directory", "name", "launch_site", "recover", "crew", "flag_url"],
            [self._client._types.string_type, self._client._types.string_type, self._client._types.string_type, self._client._types.bool_type, self._client._types.list_type(self._client._types.string_type), self._client._types.string_type],
            None
        )

    def _return_type_launch_vessel(self) -> TypeBase:
        return None

    def _build_call_launch_vessel(self, craft_directory: str, name: str, launch_site: str, recover: bool = True, crew: Optional[List[str]] = None, flag_url: str = '') -> KRPC_pb2.ProcedureCall:
        return self._client._build_call(
            "SpaceCenter",
            "LaunchVessel",
            [craft_directory, name, launch_site, recover, crew, flag_url],
            ["craft_directory", "name", "launch_site", "recover", "crew", "flag_url"],
            [self._client._types.string_type, self._client._types.string_type, self._client._types.string_type, self._client._types.bool_type, self._client._types.list_type(self._client._types.string_type), self._client._types.string_type],
            None
        )

    def launch_vessel_from_sph(self, name: str, recover: bool = True) -> None:
        """
        Launch a new vessel from the SPH onto the runway.

        :param name: Name of the vessel to launch.

        :param recover: If true and there is a vessel on the runway,
                