#    Copyright 2013-2015 ARM Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

#pylint: disable=E1101,W0201

import os
import re

from wlauto import Workload, Parameter
from wlauto.exceptions import ConfigError


this_dir = os.path.dirname(__file__)


class Dhrystone(Workload):

    name = 'dhrystone'
    description = """
    Runs the Dhrystone benchmark.

    Original source from::

        http://classes.soe.ucsc.edu/cmpe202/benchmarks/standard/dhrystone.c

    This version has been modified to configure duration and the number of
    threads used.

    """

    bm_regex = re.compile(r'This machine benchmarks at (?P<score>\d+)')
    dmips_regex = re.compile(r'(?P<score>\d+) DMIPS')
    time_regex = re.compile(r'Total dhrystone run time: (?P<time>[0-9.]+)')

    default_mloops = 100

    parameters = [
        Parameter('duration', kind=int, default=0,
                  description='The duration, in seconds, for which dhrystone will be executed. '
                              'Either this or ``mloops`` should be specified but not both.'),
        Parameter('mloops', kind=int, default=0,
                  description='Millions of loops to run. Either this or ``duration`` should be '
                              'specified, but not both. If neither is specified, this will default '
                              'to ``{}``'.format(default_mloops)),
        Parameter('threads', kind=int, default=4,
                  description='The number of separate dhrystone "threads" that will be forked.'),
        Parameter('delay', kind=int, default=0,
                  description=('The delay, in seconds, between kicking off of dhrystone '
                               'threads (if ``threads`` > 1).')),
    ]

    def setup(self, context):
        host_exe = os.path.join(this_dir, 'dhrystone')
        self.device_exe = self.device.install(host_exe)
        execution_mode = '-l {}'.format(self.mloops) if self.mloops else '-r {}'.format(self.duration)
        self.command = '{} {} -t {} -d {}'.format(self.device_exe,
                                                  execution_mode,
                                                  self.threads, self.delay)
        self.timeout = self.duration and self.duration + self.delay * self.threads + 10 or 300

    def run(self, context):
        self.output = self.device.execute(self.command, timeout=self.timeout, check_exit_code=False)

    def update_result(self, context):
        outfile = os.path.join(context.output_directory, 'dhrystone.output')
        with open(outfile, 'w') as wfh:
            wfh.write(self.output)
        score_count = 0
        dmips_count = 0
        for line in self.output.split('\n'):
            match = self.time_regex.search(line)
            if match:
                context.result.add_metric('time', float(match.group('time')), 'seconds', lower_is_better=True)
            else:
                match = self.bm_regex.search(line)
                if match:
                    metric = 'thread {} score'.format(score_count)
                    value = int(match.group('score'))
                    context.result.add_metric(metric, value)
                    score_count += 1
                else:
                    match = self.dmips_regex.search(line)
                    if match:
                        metric = 'thread {} DMIPS'.format(dmips_count)
                        value = int(match.group('score'))
                        context.result.add_metric(metric, value)
                        dmips_count += 1

    def teardown(self, context):
        self.device.uninstall_executable('dhrystone')

    def validate(self):
        if self.mloops and self.duration:  # pylint: disable=E0203
            raise ConfigError('mloops and duration cannot be both specified at the same time for dhrystone.')
        if not self.mloops and not self.duration:  # pylint: disable=E0203
            self.mloops = self.default_mloops

