
# -*- coding: utf-8 -*-

u'''A Python version of part of I{Karney}'s C++ module U{GeodesicExactC4
<https://GeographicLib.SourceForge.io/C++/doc/classGeographicLib_1_1GeodesicExactC4.html>}.

Copyright (C) U{Charles Karney<mailto:Charles@Karney.com>} (2008-2022)
and licensed under the MIT/X11 License.  For more information, see the
U{GeographicLib<https://GeographicLib.SourceForge.io>} documentation.
'''
# A copy of comments from Karney's C{GeodesicExactC4.cpp}:
#
# If the coefficient is greater or equal to 2^63, express it as a pair [a, b] which is
# combined with a*2^52 + b.  The largest coefficient is 831281402884796906843926125 =
# 0x2af9eaf25d149c52a73ee6d = 184581550685 * 2^52 + 0x149c52a73ee6d which is less than
# 2^90.  Both a and b are less that 2^52 and so are exactly representable by doubles;
# then the computation of the full double coefficient involves only a single rounding
# operation.  (Actually integers up to and including 2^53 can be represented exactly
# as doubles.  Limiting b to 52 bits allows it to be represented in 13 digits in hex.)
#
# If the coefficient is less than 2^63, cast it to  _f if it isn't exactly representable
# as a float.  Thus 121722048 = 1901907*2^6 and 1901907 < 2^24 so the cast is not needed;
# 21708121824 = 678378807*2^5 and 678378807 >= 2^24 so the cast is needed.
#
# Large coefficients are split so that lo contains the low 52 bits and hi the rest.
# This choice avoids double rounding with doubles and higher precision types.  Note,
# float coefficients will suffer double rounding; however the accuracy is already
# lousy for floats.

from pygeodesy.geodesicx.gxbases import _f, _f2

__all__ = ()
__version__ = '22.05.14'

_coeffs_30 = (  # GEOGRAPHICLIB_GEODESICEXACT_ORDER == 30
    # Generated by Maxima on 2017-05-27 10:17:57-04:00
    # C4[0], coeff of eps^29, polynomial in n of order 0
    3361, _f(109067695),
    # C4[0], coeff of eps^28, polynomial in n of order 1
    _f(121722048), _f(30168404), _f(0x269c465a0c9),
    # C4[0], coeff of eps^27, polynomial in n of order 2
    _f(21708121824), -_f(10786479696), _f(8048130587),
    _f(0xbfa33c13e963),
    # C4[0], coeff of eps^26, polynomial in n of order 3
    _f(0x738319564e0), -_f(0x4c2475635c0), _f(0x25d0be52da0),
    _f(643173496654), _f(0xa0f21774b90225),
    # C4[0], coeff of eps^25, polynomial in n of order 4
    _f(0x7a99ea0a52f40), -_f(0x5a5f53e2c3b50), _f(0x3b83d2c0c8da0),
    -_f(0x1d8a81cb5cc70), _f(0x1605bd50459c1),
    _f(0x6fb2ae4757107d03),
    # C4[0], coeff of eps^24, polynomial in n of order 5
    _f(0x2507d929b7f89580), -_f(0x1ce7bf02c3715a00),
    _f(0x15463c23456c8680), -_f(0xdfecff0050dfd00),
    _f(0x6f141ba97196780), _f(0x1b71ab9c78b8b48),
    _f2(1520879, 0x957266bcf90f9),
    # C4[0], coeff of eps^23, polynomial in n of order 6
    _f2(5214, 0xb54b8c26f5620), -_f2(4202, 0x4ae5f5bcbf950),
    _f2(3272, 0xab988a50dfac0), -_f2(2404, 0x84ae60c9e7b30),
    _f(0x62be65b26227b760), -_f(0x30f2645200be8b10),
    _f(0x2472ebc3f09ad327), _f2(9429453, 0x6b5ee3606e93b),
    # C4[0], coeff of eps^22, polynomial in n of order 7
    _f2(213221, 0x21fe88963f0e0), -_f2(174746, 0x12fe03af82e40),
    _f2(140344, 0xd3dfad978d4a0), -_f2(109009, 0x13ee03d15f180),
    _f2(79932, 0x9fff01479b460), -_f2(52447, 0x53ea945b584c0),
    _f2(25976, 0xa5a6ee990f820), _f2(6403, 0x87dc4a069efc6),
    _f2(273454149, 0x29bfc1ec86baf),
    # C4[0], coeff of eps^21, polynomial in n of order 8
    _f2(1513769, 0x9572babb99080), -_f2(1247902, 0x66609b16e1250),
    _f2(1017692, 0x228016ac84e60), -_f2(814136, 0x86ec313455df0),
    _f2(630421, 0xa88f591713840), -_f2(461205, 0x487f023b60f90),
    _f2(302134, 0x36942691aea20), -_f2(149503, 0x5a1d9af94cb30),
    _f2(111169, 0xb14ab93d4ba6d), _f2(1367270745, 0xd0bec99ea1a6b),
    # C4[0], coeff of eps^20, polynomial in n of order 9
    _f2(2196138, 0xe1b60fe1808c0), -_f2(1802572, 0x3b4b1c2a34200),
    _f2(1475191, 0x47b8ccbe8340), -_f2(1196055, 0x2e2a401c46980),
    _f2(952413, 0x117e9e1fb75c0), -_f2(734856, 0x2e19f1e7be100),
    _f2(536171, 0x8daa599335040), -_f2(350594, 0xa58d466a3880),
    _f2(173293, 0x7b19cdc9682c0), _f2(42591, 0xb005bdeb82d74),
    _f2(1367270745, 0xd0bec99ea1a6b),
    # C4[0], coeff of eps^19, polynomial in n of order 10
    _f2(9954363, 0x5ecc5371ca720), -_f2(8035921, 0x7cc90565e0670),
    _f2(6522783, 0x32e1ec30d1a80), -_f2(5291286, 0x4172ef2beb090),
    _f2(4260231, 0x65c388ed45de0), -_f2(3373847, 0x4da61e8c704b0),
    _f2(2592185, 0xcd194d02dbd40), -_f2(1885401, 0xa08c9a20ef6d0),
    _f2(1230164, 0x4c527bc6a84a0), -_f2(607279, 0x24d6e51bd7af0),
    _f2(450701, 0xae98337b7d081), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^18, polynomial in n of order 11
    _f2(16160603, 0x85a3ec5761ce0), -_f2(12587219, 0x97b7f7c505ac0),
    _f2(9979192, 0xa0e43863a93a0), -_f2(7988280, 0xcfaf566027f00),
    _f2(6410314, 0xbffc30c12660), -_f2(5117692, 0xfd9318db4c340),
    _f2(4026292, 0x94c482b815d20), -_f2(3077917, 0x9c480ad851f80),
    _f2(2230377, 0x99db799d8bfe0), -_f2(1451530, 0xb0005d9658bc0),
    _f2(715485, 0xdbe6a2ef6d6a0), _f2(175141, 0x3547b8669b9be),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^17, polynomial in n of order 12
    _f2(30091817, 0x8745c27487540), -_f2(21716256, 0x7a4bb1495e170),
    _f2(16366670, 0xd4e8bc19a0660), -_f2(12670374, 0x9eda0f5df2ed0),
    _f2(9963727, 0x5ae4f6d3c8380), -_f2(7887824, 0x191034733ae30),
    _f2(6231873, 0x96448488ef0a0), -_f2(4863678, 0x67c3c74b1b90),
    _f2(3695513, 0x2e7ae0f4851c0), -_f2(2665992, 0xe6864878c32f0),
    _f2(1729741, 0xf881cba41aae0), -_f2(851104, 0x888fd5b7ab050),
    _f2(629987, 0x9ea5a19626943), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^16, polynomial in n of order 13
    _f2(79181861, 0x46beef62ca900), -_f2(45969492, 0x85a19d8425400),
    _f2(30736937, 0x10d9a95bb4f00), -_f2(22084618, 0xaf3a6659fa600),
    _f2(16548053, 0x58583f22e9500), -_f2(12711232, 0x3d7f1b1be3800),
    _f2(9889259, 0xbbf5d84b2bb00), -_f2(7711253, 0x36b17889dca00),
    _f2(5958759, 0x73d1ebe040100), -_f2(4493987, 0xfa374abbe1c00),
    _f2(3224517, 0x29027e04ea700), -_f2(2084431, 0x8d77e42beee00),
    _f2(1023433, 0xbf113370eed00), _f2(249103, 0x93cdbdabe0fb0),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^15, polynomial in n of order 14
    _f2(100415733, 0x1c7e0d98777e0), -_f2(220472579, 0x196c2a7ff77f0),
    _f2(81497972, 0xcf48e14d7b2c0), -_f2(47157604, 0xb4c79beff0c90),
    _f2(31400333, 0x3ade51fc905a0), -_f2(22437640, 0x62c8445afeb30),
    _f2(16688020, 0xb49b2cc64ec80), -_f2(12687475, 0x35a524f08d7d0),
    _f2(9727302, 0xc96eb1166e360), -_f2(7422875, 0x3574dc9ff9670),
    _f2(5546536, 0x3897621326640), -_f2(3953280, 0x7a61d237aeb10),
    _f2(2544043, 0x942757fc8f120), -_f2(1245848, 0x5f59e2e2499b0),
    _f2(918672, 0xb7e149f3f515d), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^14, polynomial in n of order 15
    -_f2(410150575, 0x33edeefdadd60), _f2(389451478, 0x4a8eb37cf8e40),
    _f2(102537774, 0xdf54e754057e0), -_f2(228145792, 0x9928ef6984980),
    _f2(84014235, 0x8c476a1354120), -_f2(48417903, 0x9486b64af140),
    _f2(32072368, 0xac5157de0d660), -_f2(22757026, 0x6fd3c1d71f100),
    _f2(16760216, 0x75de552320fa0), -_f2(12564203, 0xce657c7ead0c0),
    _f2(9433140, 0xee7b325fde4e0), -_f2(6966096, 0xc0a9d97231880),
    _f2(4923714, 0x7fe1a8c934e20), -_f2(3150864, 0xcacdc5bf45040),
    _f2(1538058, 0xc6e75548f4360), _f2(371250, 0x9b28ca926da22),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^13, polynomial in n of order 16
    _f2(10071346, 0xbead2787bab00), _f2(77935892, 0xc8037e807a610),
    -_f2(424974584, 0x95c58aa2abc60), _f2(405632040, 0xf37804095de30),
    _f2(104709205, 0x2c34dddf07040), -_f2(236671973, 0xc06ad427a5bb0),
    _f2(86756233, 0x36f6256b264e0), -_f2(49748360, 0xa42ca4c379390),
    _f2(32735340, 0x1aa6eba145580), -_f2(23012513, 0x41e6e60af5570),
    _f2(16722020, 0xa0e65eb557620), -_f2(12285046, 0x712c138942d50),
    _f2(8933912, 0x44131ea6cfac0), -_f2(6247309, 0xac4879043a730),
    _f2(3969671, 0x8774cc7c1760), -_f2(1929932, 0x2a739696c4f10),
    _f2(1414943, 0x9f9bcb791811f), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^12, polynomial in n of order 17
    _f2(1301009, 0x7885767b34dc0), _f2(3139452, 0x6299dbe8eac00),
    _f2(10399899, 0xe9c2f692aa40), _f2(80694987, 0xafcfc919b1e80),
    -_f2(441529449, 0x34f14f083e140), _f2(423985433, 0x2e9be95704100),
    _f2(106892519, 0x9a909730adb40), -_f2(246219322, 0x3cc21ecefbc80),
    _f2(89751674, 0x8e9ea1f760fc0), -_f2(51139306, 0x4d1fa35b2aa00),
    _f2(33357165, 0x391836578ec40), -_f2(23152852, 0x670df382e5780),
    _f2(16502135, 0xfb453b1baa0c0), -_f2(11755175, 0x732a395d89500),
    _f2(8105218, 0xa64658fb65d40), -_f2(5103238, 0xc9c658d3f3280),
    _f2(2468214, 0x7d6aacb2351c0), _f2(588064, 0xecbdce72e5104),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^11, polynomial in n of order 18
    _f2(365173, 0x141eb92882aa0), _f2(660579, 0x721db1cc80890),
    _f2(1339643, 0x6f3cff39e7d00), _f2(3240370, 0xc29100e665970),
    _f2(10762711, 0xac38fa6376f60), _f2(83769430, 0x6edf90fa38050),
    -_f2(460180081, 0xa7a2c15d05240), _f2(445039582, 0xb96af8d66e930),
    _f2(109020126, 0x840edc5d1e420), -_f2(257005247, 0x2ec795996fff0),
    _f2(93028106, 0x54adfb574be80), -_f2(52565819, 0x1d828e2b6cf10),
    _f2(33879206, 0x109475f98e8e0), -_f2(23088279, 0x158dbde3c1830),
    _f2(15975944, 0x7a6ca24c70f40), -_f2(10806612, 0x3c0d699b76f50),
    _f2(6721635, 0xd5a36326ddda0), -_f2(3228909, 0xe44dc20d06870),
    _f2(2345355, 0x81bdf10588059), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^10, polynomial in n of order 19
    _f2(142358, 0x43f28ef2bce60), _f2(224104, 0xc49bf70fb8540),
    _f2(374789, 0x29edb81ed2220), _f2(679606, 0x56dce126b3a00),
    _f2(1381751, 0x3315a15e701e0), _f2(3351469, 0xe4cb186e3aec0),
    _f2(11166107, 0x295c18ed1d5a0), _f2(87224183, 0xbf27e3cc5cb80),
    -_f2(481408924, 0xf800e4fbbfaa0), _f2(469519077, 0x9e18ca33e7840),
    _f2(110970854, 0x606788cedf920), -_f2(269315695, 0x90dadb20d6300),
    _f2(96606791, 0x8c213171618e0), -_f2(53972000, 0xd509f5454de40),
    _f2(34191407, 0x9021dc5d4cca0), -_f2(22654105, 0x9f8b9187f1180),
    _f2(14912791, 0x946e9b2907c60), -_f2(9121084, 0x6067cd3f714c0),
    _f2(4341360, 0x73b562399020), _f2(1011849, 0x75de66a5bdb46),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^9, polynomial in n of order 20
    _f2(66631, 0x784cbdfb1b2c0), _f2(96606, 0x3419bb8e05f90),
    _f2(145459, 0xb79bffbfb42e0), _f2(229589, 0x824d22506cd30),
    _f2(385010, 0x35e34fd0f4f00), _f2(700134, 0x4df5413db48d0),
    _f2(1427794, 0x581b23c083b20), _f2(3474469, 0x224df4c0f7670),
    _f2(11618119, 0x6c8cba4306b40), _f2(91144571, 0x713d14f45fa10),
    -_f2(505869523, 0xd3d937aa3bca0), _f2(498449385, 0x686859af477b0),
    _f2(112524504, 0x2ca5b0e042780), -_f2(283533725, 0xba4eec11a6cb0),
    _f2(100487121, 0xc424152de7ba0), -_f2(55236514, 0x8c4dd4ee50f10),
    _f2(34077723, 0x322bbe9b9a3c0), -_f2(21528502, 0x2ca44d130cb70),
    _f2(12851809, 0x7f1d30d5603e0), -_f2(6038295, 0xecbfc0da7fdd0),
    _f2(4313665, 0xa0fbedf62e95b), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^8, polynomial in n of order 21
    _f2(34939, 0x4781a8598a880), _f2(47986, 0x870a153a0ba00),
    _f2(67643, 0xf93c5a3d5fb80), _f2(98366, 0xdef5527b5d100),
    _f2(148567, 0x565e4f7b51e80), _f2(235242, 0x766e64b79c800),
    _f2(395796, 0x5614c84bc3180), _f2(722239, 0xc9f1a6fcbf00),
    _f2(1478257, 0xd3352c2795480), _f2(3611438, 0xfdbc40cced600),
    _f2(12129091, 0x5ec9e3d72a780), _f2(95645231, 0xe79e249b02d00),
    -_f2(534473300, 0x6333290e9b580), _f2(533336700, 0xd7635e240e400),
    _f2(113268651, 0x31e09daaa5d80), -_f2(300181610, 0x6cd38634ee500),
    _f2(104606327, 0x6a6e0bd3d0080), -_f2(56090968, 0xcfc000b8f0e00),
    _f2(33084425, 0x428f85e945380), -_f2(19025074, 0x3fea5ea1f7700),
    _f2(8768855, 0x59c11511e7680), _f2(1959911, 0x57aea52b92dd8),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^7, polynomial in n of order 22
    _f2(19712, 0xac93bc6991f60), _f2(26064, 0x47e63bb6f7b10),
    _f2(35129, 0x85349dd791940), _f2(48412, 0xcf2b50a5e4170),
    _f2(68486, 0xf23457a2e7b20), _f2(99959, 0x1aee9379bdd0),
    _f2(151547, 0xc976e86422100), _f2(240911, 0x67a8290f88c30),
    _f2(407002, 0x79f859786e6e0), _f2(745880, 0xf6e3b80f24890),
    _f2(1533569, 0xcfffb4a9fa8c0), _f2(3764807, 0xab1a08cbd8ef0),
    _f2(12712489, 0x4098eb8542a0), _f2(100884327, 0x9a754746dfb50),
    -_f2(568536969, 0xbcc82f5b36f80), _f2(576497219, 0x10ca042b229b0),
    _f2(112392819, 0xaecaa4a6c6e60), -_f2(319979712, 0xfe05e4aae49f0),
    _f2(108728942, 0x9b1cd9ac3b840), -_f2(55904982, 0xfebe8a174c390),
    _f2(30158727, 0xd0df7149f4a20), -_f2(13482566, 0x2ca2af46da730),
    _f2(9304222, 0x6328f1d67a7f5), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^6, polynomial in n of order 23
    _f2(11639, 0x4298ebe4bc020), _f2(14966, 0xe9089607c0a40),
    _f2(19534, 0x1996a62965260), _f2(25928, 0xdcaffa7bfcb80),
    _f2(35089, 0x59fa64f7d88a0), _f2(48563, 0x32ed377221cc0),
    _f2(69004, 0xe5c9403173ae0), _f2(101181, 0xf483b00105600),
    _f2(154143, 0xf39432e434120), _f2(246274, 0xfc90899a3cf40),
    _f2(418255, 0xdad9486cf7360), _f2(770731, 0xbf0321b55e080),
    _f2(1593877, 0xd61fe95ba9a0), _f2(3937200, 0x3820413b3e1c0),
    _f2(13385919, 0xf48ca237dbbe0), _f2(107086956, 0x9d1b10f932b00),
    -_f2(610048075, 0x6c1b2715a7de0), _f2(631706048, 0xcac1d46451440),
    _f2(108187733, 0xaf9fd1440d460), -_f2(343908890, 0x37b3c0b50a80),
    _f2(112109635, 0x3a73d439f8aa0), -_f2(53028119, 0x15d1799f5d940),
    _f2(22454404, 0x49a70d2177ce0), _f2(4553016, 0x22f700960daaa),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^5, polynomial in n of order 24
    _f2(7030, 0x634f92bbfec80), _f2(8852, 0x183ea9c784b10),
    _f2(11280, 0x864427e0ea420), _f2(14569, 0x4ed71f4155e30),
    _f2(19103, 0x13b2c1ad2ffc0), _f2(25480, 0x35983eb20bf50),
    _f2(34659, 0x18ad59c5f9360), _f2(48227, 0x95f2c0574270),
    _f2(68917, 0x8c5b3ac32f300), _f2(101660, 0x272f49f96bb90),
    _f2(155850, 0xbc628b339b2a0), _f2(250657, 0x122490d07feb0),
    _f2(428675, 0x21f5a97506640), _f2(795748, 0x8d9dd2ee8dfd0),
    _f2(1658420, 0x22b44d2c5a1e0), _f2(4130702, 0x814b60cb632f0),
    _f2(14171990, 0xb8691b29bf980), _f2(114585240, 0x7599d8275cc10),
    -_f2(662180135, 0x55c1167b3fee0), _f2(705602404, 0xf6219ee07f30),
    _f2(96655880, 0xe42cfbbc64cc0), -_f2(373149978, 0xd8d5a94d3dfb0),
    _f2(112272021, 0x704341a757060), -_f2(42251989, 0xbf5a94cca7c90),
    _f2(26498553, 0xea37274059c77), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^4, polynomial in n of order 25
    _f2(4244, 0x3972351df5940), _f2(5257, 0xaa8f87b5d5600),
    _f2(6578, 0xed6cb3b3fa2c0), _f2(8324, 0xb4008d853180),
    _f2(10662, 0x703b07259b440), _f2(13846, 0x8f2f6ca125d00),
    _f2(18261, 0x3a455b4269dc0), _f2(24508, 0x5045fb81ae880),
    _f2(33557, 0x1b3e945f36f40), _f2(47022, 0x9499ec44e400),
    _f2(67699, 0x7a940285938c0), _f2(100662, 0x403646e1e5f80),
    _f2(155637, 0xf20897fb50a40), _f2(252593, 0x7106d86756b00),
    _f2(436178, 0xe720d891ff3c0), _f2(818051, 0x1d79595b01680),
    _f2(1723706, 0xc365c92e70540), _f2(4344105, 0xb055b91247200),
    _f2(15096896, 0xe96c54f834ec0), _f2(123888911, 0x435c586708d80),
    -_f2(730395130, 0x8d07d85ee1fc0), _f2(811137162, 0xd7ccf03d27900),
    _f2(66848989, 0xdd39a234bc9c0), -_f2(407950245, 0xd67367b7fbb80),
    _f2(99073631, 0x21cb91dfe1b40), _f2(14205410, 0x589c3f44ce7ac),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^3, polynomial in n of order 26
    _f2(2481, 0x8d2c27b46b620), _f2(3034, 0xe44720f3fdf90),
    _f2(3743, 0xf82fc54a92780), _f2(4662, 0xb922ac44f6b70),
    _f2(5867, 0xae02c805f08e0), _f2(7469, 0x40a687e9b4d50),
    _f2(9629, 0xbb2099bca6640), _f2(12592, 0xa0727e14e5130),
    _f2(16731, 0xdc4cfea134ba0), _f2(22636, 0xbf84f9dc44310),
    _f2(31263, 0xfe99294d5c500), _f2(44220, 0x78f2e666feef0),
    _f2(64313, 0xe77c1f84fde60), _f2(96684, 0x43c9282e120d0),
    _f2(151281, 0x84eb0984fa3c0), _f2(248729, 0xa2c4a502aa4b0),
    _f2(435615, 0xd80deb212120), _f2(829647, 0x194fc60e84690),
    _f2(1777619, 0x17dfea7bc6280), _f2(4562307, 0x417bb8824d270),
    _f2(16175470, 0xd3a7db47373e0), _f2(135804489, 0xbb999e2601450),
    -_f2(825156505, 0xa8162cc9f9ec0), _f2(977623624, 0xd8c5ee7f4d830),
    -_f2(20397512, 0x4ab8f862cc960), -_f2(435632583, 0xf2b7943e115f0),
    _f2(143237887, 0xa8277df5ccab1), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^2, polynomial in n of order 27
    _f(0x52cac993243497e0), _f(0x6437dfaee57b9d40),
    _f(0x7a3f9cad4d2f48a0), _f2(2405, 0xee01eec3f2b00),
    _f2(2986, 0x65a22988df560), _f2(3743, 0xe8ba104bd58c0),
    _f2(4745, 0x82561551e620), _f2(6086, 0xa7581d3ddee80),
    _f2(7912, 0x8561dfdd262e0), _f2(10440, 0x7aa2aab74b440),
    _f2(14008, 0x9b1a2c148b3a0), _f2(19155, 0xcd3b8407d7200),
    _f2(26767, 0x9792b4f9c2060), _f2(38350, 0xb50c17257efc0),
    _f2(56574, 0xaf828f4edf120), _f2(86399, 0xb1bc40483f580),
    _f2(137581, 0x7d29442656de0), _f2(230687, 0xc9059cc5d4b40),
    _f2(413025, 0xcba5d91bbdea0), _f2(806439, 0xbad85d457b900),
    _f2(1777226, 0xdb254a1088b60), _f2(4709200, 0x187f6563b06c0),
    _f2(17312174, 0x4c53d944cbc20), _f2(151524377, 0x682a2ddefc80),
    -_f2(970338799, 0x73aba5c04720), _f2(1287957204, 0xb756685e76240),
    -_f2(416692036, 0xd1e73fe253660), -_f2(78129756, 0xe75b5bfa6fa32),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[0], coeff of eps^1, polynomial in n of order 28
    _f(0xb4c355cd41c92c0), _f(0xd8fea3a41cc7830),
    _f(0x1064f0c6b9a6ad20), _f(0x13f7a88902ef1b10),
    _f(0x1884a414973fcb80), _f(0x1e5fa2ae5243d7f0),
    _f(0x25fe0bb384ddd9e0), _f(0x3006f6e3e0e25ad0),
    _f(0x3d6c2c13c34ec440), _f(0x4f91f34825bd4fb0),
    _f(0x688ffb74f98676a0), _f2(2233, 0xdec33bb086290),
    _f2(3036, 0xe53843c2cdd00), _f2(4213, 0xb13e1137e3f70),
    _f2(5984, 0xaa1cca8abe360), _f2(8732, 0xb9880d6c69250),
    _f2(13152, 0x1eadcfcfd75c0), _f2(20566, 0x4e1752c3c0730),
    _f2(33653, 0xf4262a5798020), _f2(58247, 0x3a420e3524a10),
    _f2(108257, 0x7934f39e3ee80), _f2(221025, 0xaccc1c0dc06f0),
    _f2(514222, 0xffbb852faace0), _f2(1456965, 0x29e8a4070e9d0),
    _f2(5827860, 0xa7a2901c3a740), _f2(56821641, 0x6270fd1339eb0),
    -_f2(416692036, 0xd1e73fe253660), _f2(625038055, 0x3adadfd37d190),
    -_f2(273454149, 0x29bfc1ec86baf), _f2(1367270745, 0xd0bec99ea1a6b),
    # C4[0], coeff of eps^0, polynomial in n of order 29
    _f2(42171, 0xbca3d5a569b4), _f2(46862, 0xd0a41cdef9cf0),
    _f2(52277, 0xa2d5316ac1b2c), _f2(58560, 0x6f94d669a7a28),
    _f2(65892, 0x788629d238da4), _f2(74502, 0x6b99bdf690d60),
    _f2(84681, 0x87b277eadbb1c), _f2(96804, 0x8c76c6701c898),
    _f2(111359, 0x1427f62cd3d94), _f2(128987, 0x59921e2221dd0),
    _f2(150546, 0xaa0136eb20f0c), _f2(177198, 0x7742592373f08),
    _f2(210542, 0x4360b9bd64984), _f2(252821, 0x8a8c09196de40),
    _f2(307248, 0x66986780ae6fc), _f2(378530, 0x79d0ac77ed78),
    _f2(473750, 0x5114d83948174), _f2(603901, 0x80acdb5cb5eb0),
    _f2(786661, 0x2afc1dbf812ec), _f2(1051686, 0xda8ab314e3e8),
    _f2(1451326, 0xc0ede2017b564), _f2(2083956, 0x5d3b51a63af20),
    _f2(3149615, 0xde5c8fc3f62dc), _f2(5099378, 0x12ae3e18b3258),
    _f2(9106032, 0x45ee012c1b554), _f2(18940547, 0x20d0545bbdf90),
    _f2(52086504, 0x9a3ce7fc4a6cc), _f2(312519027, 0x9d6d6fe9be8c8),
    -_f2(1093816596, 0xa6ff07b21aebc),
    _f2(2734541491, 0xa17d933d434d6),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[1], coeff of eps^29, polynomial in n of order 0
    917561, _f(273868982145),
    # C4[1], coeff of eps^28, polynomial in n of order 1
    -_f(125915776), _f(90505212), _f(0x73d4d30e25b),
    # C4[1], coeff of eps^27, polynomial in n of order 2
    -_f(0x2f7e4f2fca0), _f(0x161b06db8f0), _f(379339642199),
    _f(0x145a25f15d59339),
    # C4[1], coeff of eps^26, polynomial in n of order 3
    -_f(0x780f9f651c0), _f(0x49cd6538080), -_f(0x275396e6f40),
    _f(0x1c1406225ea), _f(0x1e2d6465e2b066f),
    # C4[1], coeff of eps^25, polynomial in n of order 4
    -_f(0x226e68a74f6c2c0), _f(0x178fbd94c6e4130),
    -_f(0x10bafa7048ffb60), _f(0x7b204e43552d10),
    _f(0x1ebd785c76c649), _f2(369943, 0xaebaf6655156d),
    # C4[1], coeff of eps^24, polynomial in n of order 5
    -_f(0x26adfa4c2bcf8500), _f(0x1be7e116f09bc400),
    -_f(0x1641521374362300), _f(0xd7dd4a2b1831200),
    -_f(0x7449d087ac65100), _f(0x525502d56a2a1d8),
    _f2(4562638, 0xc0573436eb2eb),
    # C4[1], coeff of eps^23, polynomial in n of order 6
    -_f2(27299, 0x1e7fae46f2ae0), _f2(20250, 0xb050f61211530),
    -_f2(17170, 0x1ccacfb407b40), _f2(11560, 0x5557506ac7a50),
    -_f2(8300, 0x1ee1dfec0f3a0), _f2(3760, 0xc5da39149a170),
    _f(0x3aaaad07e2dbe15f), _f2(141441801, 0x4a8f52a67aa75),
    # C4[1], coeff of eps^22, polynomial in n of order 7
    -_f2(223720, 0xada70de871dc0), _f2(168212, 0x95f7a36b8e780),
    -_f2(147708, 0x4639d71413140), _f2(104570, 0x398040c96dd00),
    -_f2(84304, 0x27ca2fe2f28c0), _f2(50205, 0xd862a9f308280),
    -_f2(27426, 0xbe7e08935dc40), _f2(19210, 0x9794de13dcf52),
    _f2(820362447, 0x7d3f45c59430d),
    # C4[1], coeff of eps^21, polynomial in n of order 8
    -_f2(1591044, 0x45108afb80980), _f2(1200725, 0xfaaefe8d2aff0),
    -_f2(1074110, 0x244b18cc1fd20), _f2(779463, 0x6e55e2794e4d0),
    -_f2(667443, 0x7f273db50d4c0), _f2(440073, 0xbd38cdf5ffbb0),
    -_f2(320490, 0xb0902bc064460), _f2(142410, 0x1eb038cc00090),
    _f2(35531, 0x5cce3f7afbb81), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[1], coeff of eps^20, polynomial in n of order 9
    -_f2(6932123, 0xff59c6bb56f80), _f2(5207764, 0x9d4c81592dc00),
    -_f2(4682178, 0xdef9cf054a880), _f2(3431350, 0xdcd7f0ab97d00),
    -_f2(3036244, 0xeb9781cfe3980), _f2(2097463, 0x35c6f48ae00),
    -_f2(1714507, 0xab45478b85280), _f2(997568, 0xe75b4df283f00),
    -_f2(555001, 0x356f72a492380), _f2(383325, 0x3033ad4799914),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^19, polynomial in n of order 10
    -_f2(10475274, 0x80e3f984eb560), _f2(7761418, 0x6cb2d37d31d50),
    -_f2(6912729, 0x2574b8548f80), _f2(5061056, 0xbff13b9f8e7b0),
    -_f2(4542234, 0x9c8561f8559a0), _f2(3202970, 0x45874de1c0010),
    -_f2(2776395, 0x2331e9957c0), _f2(1780809, 0x24244086de270),
    -_f2(1321308, 0xb7d4404aacde0), _f2(572110, 0xf0d923e3d0ad0),
    _f2(142666, 0x15ad08c690505), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^18, polynomial in n of order 11
    -_f2(16991539, 0x3bfa3a952a5c0), _f2(12232630, 0xc216625651e80),
    -_f2(10582386, 0xca84c044c7740), _f2(7659664, 0x22fef68736200),
    -_f2(6852368, 0xbf4b993050cc0), _f2(4854746, 0x78ae9dfa88580),
    -_f2(4332124, 0x5850c11d91e40), _f2(2896859, 0x8330e6242d100),
    -_f2(2410777, 0x3c4e4b27563c0), _f2(1359574, 0x6f5bc7e308c80),
    -_f2(775169, 0xf705a84369540), _f2(525423, 0x9fd72933d2d3a),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^17, polynomial in n of order 12
    -_f2(31605635, 0x9b2a6245129c0), _f2(21349095, 0xec111ef51efd0),
    -_f2(17343382, 0xc6b59d854f620), _f2(12224940, 0xad54b9902f0),
    -_f2(10665275, 0xcb2c9d1586680), _f2(7495419, 0x2bbe593f97c10),
    -_f2(6731026, 0x5bd11498926e0), _f2(4567553, 0xbb95797dfef30),
    -_f2(4019270, 0xe17fb3dce340), _f2(2483542, 0x18261977df050),
    -_f2(1889445, 0x252a3b83f47a0), _f2(789608, 0x3727b34041370),
    _f2(196748, 0x5030b26b63d7f), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^16, polynomial in n of order 13
    -_f2(83651327, 0x7df35b769ce00), _f2(46183264, 0x6a662d0fec800),
    -_f2(32523895, 0xbf44a3e60200), _f2(21575930, 0xbd1dba7599c00),
    -_f2(17706525, 0xdbcb8c6749600), _f2(12151631, 0x7c587583d3000),
    -_f2(10707728, 0xa79806e6f4a00), _f2(7245171, 0x8aa6d7e27c400),
    -_f2(6517082, 0x9ff2c462fde00), _f2(4168671, 0x7a21919979800),
    -_f2(3551918, 0x26047c5101200), _f2(1918361, 0x786d4fd8aec00),
    -_f2(1131511, 0x7e7a26769a600), _f2(747310, 0xbb693903a2f10),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^15, polynomial in n of order 14
    -_f2(63372442, 0x2cb5338504ea0), _f2(236021120, 0xed659df2db350),
    -_f2(86667901, 0x5273be9be40), _f2(47209611, 0xc1161d91d1e30),
    -_f2(33537857, 0x3d1f3cdba35e0), _f2(21739691, 0xd5c3b2c9df710),
    -_f2(18074666, 0x2123c601d8980), _f2(11984705, 0x3d2e52a8729f0),
    -_f2(10682808, 0x1cfcfab158d20), _f2(6875060, 0xeec2e9924a2d0),
    -_f2(6158904, 0xf3892aedc14c0), _f2(3612073, 0x775a08e9d4db0),
    -_f2(2844696, 0x4fdad4b74f460), _f2(1130419, 0xe52285ff91690),
    _f2(281319, 0xf8ed6ce679421), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^14, polynomial in n of order 15
    _f2(377918798, 0xab0ca9f0672c0), -_f2(418618018, 0x8099eba53f80),
    -_f2(60854873, 0x3eafa33f453c0), _f2(245263030, 0xf5560cf897d00),
    -_f2(90083330, 0xb4182a1e90640), _f2(48226005, 0xa87e22e4ae980),
    -_f2(34666917, 0x2b03feac26cc0), _f2(21804113, 0xa9bac4593e00),
    -_f2(18434597, 0x75e58711b4f40), _f2(11683388, 0x18da60c9eb280),
    -_f2(10544255, 0x717858fde75c0), _f2(6335167, 0xce8110cc57f00),
    -_f2(5568830, 0x1a6ca9ba6a840), _f2(2826076, 0xf4ab3cac7db80),
    -_f2(1750284, 0x2ff80145eaec0), _f2(1113751, 0xd17a5fb748e66),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^13, polynomial in n of order 16
    -_f2(7676111, 0x5b2a6c5f6c100), -_f2(64415807, 0x4cf1fd08a9430),
    _f2(389009273, 0x614b445047d20), -_f2(437396877, 0xd309fa5941090),
    -_f2(57368388, 0x6af986a1a0c0), _f2(255600151, 0x61702d3245910),
    -_f2(94005962, 0x2924b0b2256a0), _f2(49188288, 0xa4967a4d0acb0),
    -_f2(35935634, 0xccf0586b2e080), _f2(21713831, 0x3869a07cfee50),
    -_f2(18759173, 0xcf3c8197a7a60), _f2(11187408, 0x277eed08021f0),
    -_f2(10209411, 0xbc33094486040), _f2(5549613, 0x5f33e35304b90),
    -_f2(4590963, 0x90f6e6e49ce20), _f2(1692490, 0x5de933ef26f30),
    _f2(420297, 0x50d0b3d8c1d9b), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^12, polynomial in n of order 17
    -_f2(852919, 0x6a82cfa963080), -_f2(2188759, 0x20ca5d762f800),
    -_f2(7786929, 0x3421dcca91f80), -_f2(65787035, 0x1d560be049100),
    _f2(401061675, 0x8c48395cfc980), -_f2(458713135, 0x22175c326fa00),
    -_f2(52544362, 0x54a9b8a28c580), _f2(267237346, 0x9f71e62ba7d00),
    -_f2(98592445, 0x567d144d01c80), _f2(50019657, 0x7efcd81e48400),
    -_f2(37374118, 0xabf7952238b80), _f2(21383288, 0xfc61768bbcb00),
    -_f2(18992011, 0x5234632e06280), _f2(10406178, 0xe1fef86250200),
    -_f2(9523344, 0xe57e66503f180), _f2(4398013, 0x8a16c0de4d900),
    -_f2(2932033, 0xa738784cb8880), _f2(1764194, 0xc6396b58af30c),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^11, polynomial in n of order 18
    -_f2(210362, 0x76b369d3025e0), -_f2(399459, 0x1eaf9acef0ab0),
    -_f2(856141, 0xe229f972ba700), -_f2(2206922, 0xef935c87bb50),
    -_f2(7896496, 0x6b0bc697c0820), -_f2(67217074, 0x2cc6331df1df0),
    _f2(414202467, 0x2b5605d0252c0), -_f2(483149583, 0xa02db175d690),
    -_f2(45836711, 0xc18042256fa60), _f2(280420397, 0xa9af8baa076d0),
    -_f2(104078404, 0x7a91f5b525380), _f2(50585814, 0x9d940e3bb2630),
    -_f2(39015494, 0x6a69555b81ca0), _f2(20678727, 0x5f0f1f3a9390),
    -_f2(19012332, 0x416957968b9c0), _f2(9200947, 0xc21b589061af0),
    -_f2(8178296, 0xad1e8ab768ee0), _f2(2676456, 0xd6956da2a1850),
    _f2(661843, 0xede00571b821d), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^10, polynomial in n of order 19
    -_f2(73282, 0x88acf774cdcc0), -_f2(119856, 0xfafc4232d6980),
    -_f2(209310, 0xc95dad3d9d040), -_f2(398728, 0xc3246fdb30c00),
    -_f2(857927, 0x8ca89fdf097c0), -_f2(2222415, 0x7f22a8f79ee80),
    -_f2(8002412, 0xa401cae100b40), -_f2(68698832, 0xcf05dd2d1e900),
    _f2(428572510, 0x4af905b8fd40), -_f2(511480829, 0xaa7af93dad380),
    -_f2(36412636, 0xa51695c145640), _f2(295430858, 0x62539c3ab7a00),
    -_f2(110834541, 0xf7ac6a286ddc0), _f2(50648730, 0xf42d6a1912780),
    -_f2(40882711, 0xc825af61d7140), _f2(19389515, 0xc578a6be65d00),
    -_f2(18548541, 0x30b0433e6e8c0), _f2(7353872, 0xa4f0c77ab4280),
    -_f2(5517208, 0xc642445621c40), _f2(3035548, 0x619b33f1391d2),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^9, polynomial in n of order 20
    -_f2(31116, 0x5ced59f2a6a40), -_f2(46466, 0x39ef1648a3c30),
    -_f2(72339, 0x13bec712995a0), -_f2(118591, 0xe96704ee23c10),
    -_f2(207681, 0xf3272ddf69500), -_f2(396975, 0x5586a3fda15f0),
    -_f2(857776, 0x96a9e394d3460), -_f2(2234014, 0x9c760527155d0),
    -_f2(8101033, 0x1f3b77f93fc0), -_f2(70217181, 0xc7476a97287b0),
    _f2(444320933, 0x84d59896b7ce0), -_f2(544755366, 0x60ab42e093790),
    -_f2(22958170, 0x5fc77e584ca80), _f2(312550991, 0xea91e4bc80e90),
    -_f2(119474190, 0x655c7a979e1e0), _f2(49778595, 0x69cfb591beb0),
    -_f2(42938053, 0xad555dfab9540), _f2(17185991, 0x9567a8e814cd0),
    -_f2(16947236, 0xc941a0517b0a0), _f2(4507394, 0xb6bfddcb2cf0),
    _f2(1103154, 0xee71952935057), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^8, polynomial in n of order 21
    -_f2(15013, 0x669ca85dbff00), -_f2(21081, 0x7f4d799198400),
    -_f2(30470, 0xbdb587d74d900), -_f2(45587, 0xe4badb51b1a00),
    -_f2(71124, 0x646ea35b6300), -_f2(116891, 0x8adb62aa4d000),
    -_f2(205315, 0x1aa2ab2ec7d00), -_f2(393884, 0x4b8d8eda78600),
    -_f2(855000, 0x2faa553050700), -_f2(2239966, 0xb31164c141c00),
    -_f2(8186764, 0x97347e701e100), -_f2(71742883, 0x7f111739b7200),
    _f2(461586973, 0x9a516d5401500), -_f2(584418823, 0xe1245bd6e6800),
    -_f2(3315305, 0x14110f9c0500), _f2(331936814, 0x28269ca022200),
    -_f2(131069117, 0x7ee7ad0730f00), _f2(47184778, 0x227a729454c00),
    -_f2(44897669, 0x9cd1b2a1e900), _f2(13574545, 0xcd96a182a3600),
    -_f2(12485695, 0x45db16a057300), _f2(5879734, 0x70bef82b8988),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^7, polynomial in n of order 22
    -_f2(7900, 0x638c66d8a8320), -_f2(10613, 0xf2ac3092c9cb0),
    -_f2(14565, 0xe107ae27501c0), -_f2(20489, 0xead89ce414d0),
    -_f2(29670, 0x849ce08edf860), -_f2(44482, 0xeb1f022729ef0),
    -_f2(69562, 0xbdfcfee35b00), -_f2(114632, 0x975e8fa16f10),
    -_f2(201989, 0x9411d71111da0), -_f2(389021, 0x33d7ff034b930),
    -_f2(848628, 0xc0285ec233440), -_f2(2237713, 0xb97d9ca55b150),
    -_f2(8250880, 0x9132887d792e0), -_f2(73221392, 0xf1ffe05c8b70),
    _f2(480452831, 0x383b5471fd280), -_f2(632496874, 0xca3591eba7b90),
    _f2(26233104, 0x13df159bb07e0), _f2(353203487, 0x101c2c33c4a50),
    -_f2(147596513, 0x7a337ff05e6c0), _f2(41406718, 0x88562e0e69230),
    -_f2(45513246, 0x22b5bfcbced60), _f2(7934370, 0xa8c8e9d8c2810),
    _f2(1869414, 0xdc5c61854a479), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^6, polynomial in n of order 23
    -_f2(4406, 0xf939ae5c97c40), -_f2(5729, 0xf863eba5bf80),
    -_f2(7570, 0xa927e082c4c0), -_f2(10189, 0xdc3d2b5930900),
    -_f2(14011, 0xfd72406188940), -_f2(19751, 0x4ee9330f94280),
    -_f2(28665, 0xa6c18d00fb1c0), -_f2(43078, 0xe8ed052a45400),
    -_f2(67543, 0xd4150add2640), -_f2(111634, 0xb28e55bb02580),
    -_f2(197389, 0xccdd68505cec0), -_f2(381765, 0x22e00b9b89f00),
    -_f2(837258, 0xa000eefe9340), -_f2(2223425, 0xd3d15b309a880),
    -_f2(8279438, 0xc28db224c5bc0), -_f2(74551261, 0xb7816e54f2a00),
    _f2(500824278, 0x3891b999befc0), -_f2(691847154, 0x918a2dd450b80),
    _f2(72461747, 0xa045596356740), _f2(374046829, 0x41b777218cb00),
    -_f2(172833056, 0x62b9485f4dd40), _f2(29915148, 0x80284d25e7180),
    -_f2(39423763, 0x40d338467c5c0), _f2(13659048, 0x68e501c228ffe),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^5, polynomial in n of order 24
    -_f2(2545, 0x1363104362d80), -_f2(3226, 0xe67b1424a4830),
    -_f2(4144, 0x8c711302fa660), -_f2(5400, 0xc1bfe2853af90),
    -_f2(7153, 0xb2c26c1682b40), -_f2(9653, 0x9e8ef4e7cf0f0),
    -_f2(13308, 0xeb09aee491820), -_f2(18810, 0x561040fe22850),
    -_f2(27375, 0xc35e0fb3fc900), -_f2(41260, 0x7d7f41fc271b0),
    -_f2(64893, 0xc7a96414399e0), -_f2(107622, 0xe02e2157de910),
    -_f2(191035, 0x6ce8a0a1be6c0), -_f2(371181, 0x96988a373aa70),
    -_f2(818768, 0xa91a46aa60ba0), -_f2(2191167, 0x9fde37effd1d0),
    -_f2(8249435, 0xe27cdc35b6480), -_f2(75540143, 0x55cc77d97b30),
    _f2(522119910, 0xf5aa540a8b2a0), -_f2(766397212, 0x64559a510c290),
    _f2(148547296, 0x8152775e2ddc0), _f2(385247751, 0x81b301a133c10),
    -_f2(213402544, 0x90fce845e3f20), _f2(10198756, 0x255c7c31664b0),
    _f2(1365904, 0xd74a19c69db33), _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^4, polynomial in n of order 25
    -_f(0x5cd20bbc3c672180), -_f(0x73720b2d98187c00),
    -_f2(2321, 0xc4eb857568680), -_f2(2952, 0xb2617088c8f00),
    -_f2(3804, 0x417bd8fa2e380), -_f2(4973, 0x5ec86f601d200),
    -_f2(6609, 0x998272f30a880), -_f2(8950, 0x197c7ab46b500),
    -_f2(12382, 0xcc481e2a44580), -_f2(17565, 0x5f7861969a800),
    -_f2(25660, 0x4a6f330e22a80), -_f2(38825, 0xe447100991b00),
    -_f2(61313, 0x47573aa0ec780), -_f2(102123, 0xa55bb6037e00),
    -_f2(182121, 0xfb4d0590e8c80), -_f2(355742, 0x340be91b74100),
    -_f2(789743, 0xf318e4285e980), -_f2(2131260, 0x2c59b0f82d400),
    -_f2(8121193, 0x3f9cc7c594e80), -_f2(75808472, 0x814742dd4a700),
    _f2(542406027, 0xe15955752d480), -_f2(860719085, 0xb088c959b2a00),
    _f2(281794203, 0x6d691a09a0f80), _f2(349671639, 0x4a19c69db3300),
    -_f2(268081590, 0x1f35e51280d80), _f2(42616231, 0x9d4bdce6b704),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^3, polynomial in n of order 26
    -_f(0x34f88b61ee2c2e60), -_f(0x40e8b73250ad02b0),
    -_f(0x50402824a1190680), -_f(0x643133a56bf6de50),
    -_f(0x7e70b50d7e53aea0), -_f2(2583, 0x89ee9103c6bf0),
    -_f2(3343, 0x2d56b6f20aac0), -_f2(4390, 0x9150bee746f90),
    -_f2(5862, 0xecb9ee1767ee0), -_f2(7978, 0x9b4551158ad30),
    -_f2(11096, 0x13774a5e7af00), -_f2(15825, 0x3f23db737e8d0),
    -_f2(23248, 0xf45a340cbf20), -_f2(35380, 0xaf4478627e670),
    -_f2(56209, 0x8a81f32e3340), -_f2(94205, 0x2f98ae2576a10),
    -_f2(169093, 0xeae4ad4ee8f60), -_f2(332577, 0xf0ed8664037b0),
    -_f2(743995, 0x906300fb45780), -_f2(2026493, 0x9c6e844791350),
    -_f2(7821602, 0x7531c16940fa0), -_f2(74557824, 0x1ed43b2e7c0f0),
    _f2(555703654, 0x34418f385c440), -_f2(974709694, 0x84f4a67130490),
    _f2(527421389, 0x42f7f1faaa020), _f2(94702735, 0xa411a5cab5dd0),
    -_f2(117194635, 0x5b0909f7a774b),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^2, polynomial in n of order 27
    -_f(0x1bd57a8f504dd3c0), -_f(0x21b6ff10b9172180),
    -_f(0x292825cda3a88940), -_f(0x32aacbfadedfca00),
    -_f(0x3ef38a62fa0322c0), -_f(0x4f013a1cfd80d280),
    -_f(0x64414a4729c69840), -_f2(2060, 0x90ead26a03300),
    -_f2(2683, 0x237c6d92be1c0), -_f2(3547, 0x3d9a05c33e380),
    -_f2(4770, 0x6ec9da59bf740), -_f2(6541, 0x1657e411dc00),
    -_f2(9170, 0x1a8b4944fd0c0), -_f2(13190, 0xb069410801480),
    -_f2(19554, 0x9e393a3b06640), -_f2(30047, 0xba30505448500),
    -_f2(48224, 0x707d4f4f6afc0), -_f2(81689, 0xf05ca40b52580),
    -_f2(148265, 0xab90de58ba540), -_f2(294962, 0x64373b047ee00),
    -_f2(667587, 0xc0c688fa83ec0), -_f2(1840377, 0xc842d822d680),
    -_f2(7199121, 0xfc41489b57440), -_f2(69934327, 0xdb9ec152bd700),
    _f2(541991040, 0xe60e5a413c240), -_f2(1060670639, 0x2d9274118e780),
    _f2(833384073, 0xa3ce7fc4a6cc0), -_f2(234389270, 0xb61213ef4ee96),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[1], coeff of eps^1, polynomial in n of order 28
    -_f(0xb4c355cd41c92c0), -_f(0xd8fea3a41cc7830),
    -_f(0x1064f0c6b9a6ad20), -_f(0x13f7a88902ef1b10),
    -_f(0x1884a414973fcb80), -_f(0x1e5fa2ae5243d7f0),
    -_f(0x25fe0bb384ddd9e0), -_f(0x3006f6e3e0e25ad0),
    -_f(0x3d6c2c13c34ec440), -_f(0x4f91f34825bd4fb0),
    -_f(0x688ffb74f98676a0), -_f2(2233, 0xdec33bb086290),
    -_f2(3036, 0xe53843c2cdd00), -_f2(4213, 0xb13e1137e3f70),
    -_f2(5984, 0xaa1cca8abe360), -_f2(8732, 0xb9880d6c69250),
    -_f2(13152, 0x1eadcfcfd75c0), -_f2(20566, 0x4e1752c3c0730),
    -_f2(33653, 0xf4262a5798020), -_f2(58247, 0x3a420e3524a10),
    -_f2(108257, 0x7934f39e3ee80), -_f2(221025, 0xaccc1c0dc06f0),
    -_f2(514222, 0xffbb852faace0), -_f2(1456965, 0x29e8a4070e9d0),
    -_f2(5827860, 0xa7a2901c3a740), -_f2(56821641, 0x6270fd1339eb0),
    _f2(416692036, 0xd1e73fe253660), -_f2(625038055, 0x3adadfd37d190),
    _f2(273454149, 0x29bfc1ec86baf),
    _f2(12305436712, 0x56b51693aedc3),
    # C4[2], coeff of eps^29, polynomial in n of order 0
    185528, _f(30429886905),
    # C4[2], coeff of eps^28, polynomial in n of order 1
    _f(17366491968), _f(4404238552), _f(0x74e318fa9c07f),
    # C4[2], coeff of eps^27, polynomial in n of order 2
    _f(412763643136), -_f(248137794944), _f(164642704408),
    _f(0x4d882f0532d9e9),
    # C4[2], coeff of eps^26, polynomial in n of order 3
    _f(0x11462b92d913a0), -_f(0xdd4620ebadc40),
    _f(0x5974730e46be0), _f(0x16bcec57851cc),
    _f2(33547, 0x1cf91962af003),
    # C4[2], coeff of eps^25, polynomial in n of order 4
    _f(0xc83679b433c00), -_f(0xb29b6d58dfb00), _f(0x5f4e3bdd4de00),
    -_f(0x3affd9960e900), _f(0x2665fb625f490),
    _f2(15809, 0x8f200ee7e2a7d),
    # C4[2], coeff of eps^24, polynomial in n of order 5
    _f(0x67b92a8524a18e80), -_f(0x609d7d3ca356ae00),
    _f(0x39db180d1b52d580), -_f(0x2fa1e9183dec9700),
    _f(0x1294d8f2627edc80), _f(0x4bc94ddbc9bad70),
    _f2(22813193, 0xc1b4051297e97),
    # C4[2], coeff of eps^23, polynomial in n of order 6
    _f2(24830, 0x3d0fb879bb600), -_f2(23212, 0xa100635ccdb00),
    _f2(14957, 0x147cd156ba400), -_f2(13653, 0x51ea4b9c89d00),
    _f2(7024, 0x2535370909200), -_f2(4511, 0x3af63b60c9f00),
    _f2(2865, 0xf50f5adcce1f0), _f2(235736335, 0x7c44346acc6c3),
    # C4[2], coeff of eps^22, polynomial in n of order 7
    _f2(1046092, 0x25a6222f26060), -_f2(949436, 0x14a3a722f1840),
    _f2(652845, 0xb96689ab42720), -_f2(615919, 0x6f1345ab50580),
    _f2(356624, 0x982d38f2a9de0), -_f2(303839, 0x22c37d5c832c0),
    _f2(113262, 0x286189b57e4a0), _f2(28978, 0x12ae8b059bc84),
    _f2(6836353729, 0x13b9f01928417),
    # C4[2], coeff of eps^21, polynomial in n of order 8
    _f2(4643688, 0x71b79cbf7cc00), -_f2(3959056, 0x83e38a4f9d180),
    _f2(2926140, 0x6f81ce5fc3900), -_f2(2722736, 0xdd03df5282c80),
    _f2(1710940, 0xc70403130e600), -_f2(1602990, 0x9ebb76967a780),
    _f2(787738, 0x6bf60987b1300), -_f2(530212, 0xcde2a88ab0280),
    _f2(326645, 0xab9033855e368), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^20, polynomial in n of order 9
    _f2(2366152, 0x4fc26559c91c0), -_f2(1830925, 0x4d73259824200),
    _f2(1477489, 0x62c9a90a52a40), -_f2(1299560, 0xe7bf798235180),
    _f2(885946, 0x5cb0a99f5e2c0), -_f2(843740, 0x47153eb842100),
    _f2(469359, 0x79db9d7cfb40), -_f2(417111, 0x1a4c5e2477080),
    _f2(146559, 0x51b0aa3dcb3c0), _f2(37677, 0x6dd5ee66abd48),
    _f2(6836353729, 0x13b9f01928417),
    # C4[2], coeff of eps^19, polynomial in n of order 10
    _f2(11390177, 0xa8f910291300), -_f2(7729638, 0x6f23cf47c2480),
    _f2(6929266, 0x5fb765e065c00), -_f2(5514735, 0x5eb0876136380),
    _f2(4148166, 0x27d6c40aa500), -_f2(3788609, 0xfef33001c8280),
    _f2(2322601, 0x1de03c2bc2e00), -_f2(2237878, 0x77b7642b94180),
    _f2(1037457, 0x571c66f013700), -_f2(742165, 0x8c39e6d5b6080),
    _f2(439349, 0xf7cfa6e796fc8), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^18, polynomial in n of order 11
    _f2(19643005, 0x3eb0d373a0e0), -_f2(11359402, 0x98e8f09139c0),
    _f2(11381255, 0xacc1b03fd73a0), -_f2(7834592, 0x92741bdd3b00),
    _f2(6664656, 0xa317edb25b660), -_f2(5516050, 0x3ff87cc43bc40),
    _f2(3774293, 0xd5e83edc68920), -_f2(3594547, 0xbec9f61701d80),
    _f2(1908400, 0x61c5f793c0be0), -_f2(1786093, 0xfaf3f7a19bec0),
    _f2(579905, 0x9d50696085ea0), _f2(150042, 0xa9efa9004c604),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^17, polynomial in n of order 12
    _f2(38321815, 0x1e48683dc9800), -_f2(18616913, 0x727791f8dfa00),
    _f2(20113440, 0xb841223d75400), -_f2(11495937, 0x9838f29931e00),
    _f2(11261630, 0x21fd3747b1000), -_f2(7960716, 0x75135ee9c200),
    _f2(6275150, 0xa8a2fa972cc00), -_f2(5471565, 0x945df446e600),
    _f2(3293426, 0x6eab44c698800), -_f2(3257897, 0x559df659f8a00),
    _f2(1401057, 0x756ea738a4400), -_f2(1086629, 0xf49cb94a8ae00),
    _f2(610116, 0x479bdc6c290e0), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^16, polynomial in n of order 13
    _f2(102781113, 0x98fe5a9192500), -_f2(40336104, 0xccc089a851400),
    _f2(40165652, 0x6e617f3b73300), -_f2(18616625, 0x95536d5576600),
    _f2(20514709, 0xd39b96f5ec100), -_f2(11691503, 0x7c1154bb0b800),
    _f2(10980290, 0x40d1adbe6cf00), -_f2(8104717, 0x4a433bfb60a00),
    _f2(5726151, 0xc3b2b2965d00), -_f2(5331323, 0xa4559d80c5c00),
    _f2(2689333, 0x7cf2f82446b00), -_f2(2678624, 0x7904ff2b8ae00),
    _f2(779755, 0xfacbca777f900), _f2(203539, 0xb4670b88476e0),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^15, polynomial in n of order 14
    -_f2(23295494, 0x8be82e34e6400), -_f2(256522224, 0x1264f586eb600),
    _f2(109420782, 0x9692235ce1800), -_f2(40005401, 0x76f47ac799a00),
    _f2(42210732, 0x9175627089400), -_f2(18637789, 0x360d04338fe00),
    _f2(20777547, 0x32d7f69c1000), -_f2(11978808, 0x3c6fce691e200),
    _f2(10467739, 0x890cbd2438c00), -_f2(8246695, 0x5d95a89294600),
    _f2(4981450, 0x2e83f5dba0800), -_f2(4997884, 0x48d2490e42a00),
    _f2(1949724, 0xd6b9d613a8400), -_f2(1687002, 0x42840cd678e00),
    _f2(881316, 0x5154c853b06e0), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^14, polynomial in n of order 15
    -_f2(315852553, 0x127aa1fb9560), _f2(452067016, 0x32f06289dc340),
    -_f2(36389203, 0xc905d2dd0bc20), -_f2(265701999, 0x414c3c9652f80),
    _f2(117462481, 0xb44ff33f8ed20), -_f2(39375172, 0xb9e521c5c6240),
    _f2(44443567, 0x98c20ae94660), -_f2(18737379, 0x9088d09ce7500),
    _f2(20789662, 0x74772cb6e2fa0), -_f2(12399165, 0xc39cbc16e07c0),
    _f2(9634015, 0x48be8ec7788e0), -_f2(8326007, 0x8f1246dddba80),
    _f2(4012687, 0x8a9763f933220), -_f2(4283805, 0xe15bd5742d40),
    _f2(1064918, 0x3e0322e890b60), _f2(281445, 0x189dacfa2913c),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^13, polynomial in n of order 16
    _f2(4607575, 0xc9d7900c88800), _f2(44527228, 0x61b96ac1eb380),
    -_f2(320302478, 0xa276d3450e900), _f2(471382647, 0x4d0623cc86a80),
    -_f2(52535715, 0x404f1a5b09a00), -_f2(275262322, 0xf3348bb543e80),
    _f2(127364360, 0xbf0504ec13500), -_f2(38376532, 0x74833ebc78780),
    _f2(46801690, 0x6a3245e5c4400), -_f2(19021914, 0x3bda110f1b080),
    _f2(20372666, 0xf7fc04d85300), -_f2(12992077, 0x825700022f980),
    _f2(8374681, 0xba502a56d2200), -_f2(8187369, 0x8d48a8bba280),
    _f2(2818780, 0x7113503f27100), -_f2(2834494, 0xf2038f04beb80),
    _f2(1337917, 0xc906f381aecf8), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^12, polynomial in n of order 17
    _f2(388658, 0x19c7c6f8ea2c0), _f2(1117971, 0xaadcbdb38ac00),
    _f2(4519560, 0xaee28ee393540), _f2(44278119, 0xe09b9f50af680),
    -_f2(324493551, 0x5c00bae29840), _f2(492697628, 0x7d1cc3fd18100),
    -_f2(72657626, 0xb42806bf185c0), -_f2(284925253, 0x57cc84a557480),
    _f2(139770748, 0x33e950dc3acc0), -_f2(36961790, 0xef70c005baa00),
    _f2(49119876, 0xa052562f03f40), -_f2(19681131, 0xbaa50226adf80),
    _f2(19252422, 0xc3af9265b71c0), -_f2(13755373, 0x2f0960c0cd500),
    _f2(6600104, 0x6565773f88440), -_f2(7462805, 0xbfb982e534a80),
    _f2(1452711, 0x6b2cd84feb6c0), _f2(390635, 0x965de9321fbe8),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^11, polynomial in n of order 18
    _f2(73868, 0xf53613318fd00), _f2(155158, 0x6bea1fc037e80),
    _f2(370865, 0xe686995a3a800), _f2(1077531, 0xb6b00d00e5180),
    _f2(4409046, 0x1d5f244685300), _f2(43860006, 0xf94485a638480),
    -_f2(328226208, 0x254b380304200), _f2(516242826, 0x48cfde1d3d780),
    -_f2(98028430, 0xc7227901d5700), -_f2(294125055, 0xf41dd5cbff580),
    _f2(155591277, 0xc58331ae9d400), -_f2(35168366, 0x6c3820d072280),
    _f2(51023141, 0xfcae9f00dff00), -_f2(21033813, 0x6b0840ce0ef80),
    _f2(17035669, 0xa0ab037f7ea00), -_f2(14520825, 0x209891efc9c80),
    _f2(4321952, 0xda1143d705500), -_f2(5322397, 0x9ed9b44796980),
    _f2(2165443, 0xa5af00ad58358), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^10, polynomial in n of order 19
    _f2(19809, 0x63304b335a660), _f2(35566, 0xcb4164f348e40),
    _f2(68577, 0xe86c972757e20), _f2(145245, 0xbc9cc7446e200),
    _f2(350489, 0x7e29a3d4285e0), _f2(1029750, 0x45087f82835c0),
    _f2(4270842, 0x2203011585da0), _f2(43220702, 0xa65b618eca980),
    -_f2(331199124, 0xa89ccd5235aa0), _f2(542217711, 0x200e3727c5d40),
    -_f2(130429686, 0x3b8b1d50d02e0), -_f2(301749371, 0x2c4d836f88f00),
    _f2(176097282, 0x8ddfe73d104e0), -_f2(33280999, 0x8c12e2a85fb40),
    _f2(51717673, 0x23cc103525ca0), -_f2(23558374, 0x76fe0e70fc780),
    _f2(13250268, 0x69c1c450ca460), -_f2(14595460, 0xd8a80a3d5d3c0),
    _f2(1848614, 0x7d3564e37c20), _f2(506231, 0x2a6100a6a6db4),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^9, polynomial in n of order 20
    _f2(6397, 0xfcd62c9faa400), _f2(10440, 0x3fc8ff8e75700),
    _f2(17841, 0xb7bede1dba00), _f2(32272, 0x7935213063d00),
    _f2(62742, 0x8933a9bfd5000), _f2(134128, 0x223daf23d6300),
    _f2(327129, 0xfca43cca0e600), _f2(973230, 0x31dda9e44900),
    _f2(4098328, 0x3528b970ffc00), _f2(42289297, 0xe5d54d5326f00),
    -_f2(332951092, 0xecfda756dee00), _f2(570709002, 0x2878cf4ff5500),
    -_f2(172380399, 0x5788b53115800), -_f2(305626020, 0x9c65fcc7d8500),
    _f2(202987914, 0xbd0aab0ad3e00), -_f2(32233434, 0x3f0406dec9f00),
    _f2(49604551, 0xc747777555400), -_f2(27757216, 0x323bffb167900),
    _f2(7652705, 0x1c15203ae6a00), -_f2(11782806, 0x2b7827f239300),
    _f2(3811565, 0x362856b8e6d30), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^8, polynomial in n of order 21
    _f2(2297, 0xe5959dcaf9680), _f2(3515, 0xaf44e93439a00),
    _f2(5557, 0xf844363205d80), _f2(9134, 0x3148872cf3100),
    _f2(15730, 0x1f27208afe480), _f2(28695, 0xbe2e993314800),
    _f2(56314, 0x2c7b05479ab80), _f2(121661, 0x287926e675f00),
    _f2(300328, 0xfc8a376113280), _f2(906274, 0xf1fb199eef600),
    _f2(3883000, 0x5f528c391f980), _f2(40968060, 0xe6e08c5558d00),
    -_f2(332763533, 0x8282a4a507f80), _f2(601507851, 0xf6ba284c8a400),
    -_f2(227453313, 0x642fd223ab880), -_f2(301473974, 0xbe5976c5a4500),
    _f2(238209921, 0x57c5b91e6ce80), -_f2(34582562, 0x41ecac4f5ae00),
    _f2(41696071, 0xee870caef9580), -_f2(33183269, 0xa456f79c1700),
    _f2(1407347, 0x27b05f0931c80), _f2(329283, 0x26010fabff570),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^7, polynomial in n of order 22
    _f(0x367dbe5da7953e00), _f(0x4f9a921ac6fb1900),
    _f(0x773454548df74400), _f2(2938, 0xbc18faed4af00),
    _f2(4681, 0x407a350a64a00), _f2(7756, 0xa0ed83ee90500),
    _f2(13477, 0x2fbfd87edd000), _f2(24826, 0x9ea174e739b00),
    _f2(49249, 0xd3391f1d95600), _f2(107696, 0xcac2013cff100),
    _f2(269571, 0xe064d3a745c00), _f2(826840, 0x70825da398700),
    _f2(3613882, 0x7ef0aa40a6200), _f2(39120270, 0xc5673698bdd00),
    -_f2(329492011, 0x53f65ac991800), _f2(633695353, 0xfeb5c44027300),
    -_f2(300630213, 0xecf09fbea9200), -_f2(280700646, 0xcee0a2073700),
    _f2(282664342, 0x7b726e8a17400), -_f2(46720160, 0x11dfe8c55a100),
    _f2(23527957, 0x90f427ad67a00), -_f2(33848503, 0x5eac35f0d4b00),
    _f2(7456233, 0x7c1f0b332cab0), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^6, polynomial in n of order 23
    _f(0x14f52a063dc5fc20), _f(0x1d93a1e9ceb48740),
    _f(0x2a911c303b723a60), _f(0x3ea26bba66a54980),
    _f(0x5e84fad71b3608a0), _f2(2349, 0x85d3117e94bc0),
    _f2(3776, 0x1c9d51cf2c6e0), _f2(6317, 0x5193932d16e00),
    _f2(11091, 0xc7716ff97d520), _f2(20667, 0xe33c2c4a29040),
    _f2(41523, 0x1a30a42ae9360), _f2(92100, 0xbd0a1f1419280),
    _f2(234309, 0x70b77706661a0), _f2(732507, 0x72fafb4df54c0),
    _f2(3276808, 0xe462aef209fe0), _f2(36551902, 0x4c4d10a4b700),
    -_f2(321265885, 0x720bf168351e0), _f2(664675522, 0x65892c55e9940),
    -_f2(398339257, 0x2b82ef41c13a0), -_f2(225754486, 0xf240500d62480),
    _f2(330356701, 0xbb7252695baa0), -_f2(82401980, 0x37f104ae0a240),
    -_f2(4970822, 0x52bf5cccc8720), -_f2(3278171, 0x9e4b710fe0e14),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^5, polynomial in n of order 24
    _f(0x7d5242068d47400), _f(0xac3832c9e621080),
    _f(0xf0840d5e59cf500), _f(0x155fabefd3362980),
    _f(0x1f01ffac4c30b600), _f(0x2e0489bbd6aca280),
    _f(0x461560bdbc05f700), _f(0x6df6210d29c3bb80),
    _f2(2857, 0xf2e1b87d2f800), _f2(4836, 0xd8d8f4249b480),
    _f2(8600, 0x17271d36df900), _f2(16248, 0x163bc1ffccd80),
    _f2(33146, 0xc23750bad3a00), _f2(74792, 0x260310eab4680),
    _f2(194024, 0xef2cdae46fb00), _f2(620545, 0xfcf47db535f80),
    _f2(2853712, 0x7228ad7b17c00), _f2(32984640, 0x1c4ce82435880),
    -_f2(304937768, 0x83ef272fd0300), _f2(687819348, 0xf9e0f9c397180),
    -_f2(526420007, 0xa1ce2482e4200), -_f2(101220737, 0xb065c6f7c1580),
    _f2(344186593, 0xf79ee4a13ff00), -_f2(151524377, 0x682a2ddefc80),
    _f2(15298134, 0x380aba4a19708), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^4, polynomial in n of order 25
    _f(0x2b077c634ede840), _f(0x39e80232e455600),
    _f(0x4f004399e9803c0), _f(0x6d6a8dd96e7d980),
    _f(0x9a16639c690ff40), _f(0xdd0eb6a29ee1d00),
    _f(0x143ca2e567649ac0), _f(0x1e583a687f6ce080),
    _f(0x2ebb5ae27bca9640), _f(0x4a366ef6d0a8e400),
    _f(0x7a244f6987aeb1c0), _f2(3355, 0xff6a995ee780),
    _f2(6059, 0x95d9afc38ad40), _f2(11647, 0x91c4ac30bab00),
    _f2(24220, 0xbe377a4d448c0), _f2(55835, 0xd9394a033ee80),
    _f2(148417, 0x27a782b394440), _f2(488256, 0xe5126fdac7200),
    _f2(2322515, 0xb040a0735fc0), _f2(28019858, 0x3d9464fe1f580),
    -_f2(275064197, 0x290d46715a4c0), _f2(686424553, 0x6984a82213900),
    -_f2(677745912, 0x9f6fb36960940), _f2(151524377, 0x682a2ddefc80),
    _f2(169007958, 0xfd6a53329f240), -_f2(85232462, 0x13a97b9cd6e08),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^3, polynomial in n of order 26
    _f(0xc4c78b5f73e700), _f(0x1046756e5efb980),
    _f(0x15cbc98d9fba400), _f(0x1d9279681ffce80),
    _f(0x28b2f34344c6100), _f(0x38e6214caec8380),
    _f(0x50f0f0d0c655e00), _f(0x7563dc0de2d1880),
    _f(0xadfad5eb325db00), _f(0x1083ab8775a8cd80),
    _f(0x19c9d8efc1ad1800), _f(0x29945e7f0056e280),
    _f(0x4594bf2102ba5500), _f(0x79a9d12705de9780),
    _f2(3587, 0xb2b264e0cd200), _f2(7053, 0x1d58043372c80),
    _f2(15040, 0x44c8073c3cf00), _f2(35667, 0x702872e47e180),
    _f2(97902, 0x6929355be8c00), _f2(334186, 0x1d1de4e87f680),
    _f2(1659947, 0xed2beccfc4900), _f2(21110207, 0x53559189eab80),
    -_f2(222144335, 0x8c70c0703ba00), _f2(617753229, 0x694fabb034080),
    -_f2(769277606, 0x6fd24e8e23d00), _f2(454573131, 0x1387e899cf580),
    -_f2(104173009, 0x3479cff894d98),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[2], coeff of eps^2, polynomial in n of order 27
    _f(0x24546bc28a93e0), _f(0x2f6c4d745b8e40),
    _f(0x3e90f252c210a0), _f(0x5380c389acd700),
    _f(0x70da9adde57d60), _f(0x9aa08aca5a9fc0),
    _f(0xd7127fe199fa20), _f(0x130248120008880),
    _f(0x1b6103e1c56a6e0), _f(0x283fa247b6e3140),
    _f(0x3c89da46fe8a3a0), _f(0x5d71643158b3a00),
    _f(0x948b363af771060), _f(0xf445a32263b42c0),
    _f(0x1a1d56e9fe070d20), _f(0x2ecb290f0241eb80),
    _f(0x58a5da95527fb9e0), _f2(2876, 0x680343126d440),
    _f2(6354, 0x3e35c062e36a0), _f2(15689, 0x7d2910c199d00),
    _f2(45107, 0x47d6102c9a360), _f2(162386, 0x35cf6d6d5e5c0),
    _f2(857038, 0x54e3334f72020), _f2(11655721, 0x4f45203874e80),
    -_f2(131126864, 0xbbc9aa7b23320), _f2(378810942, 0x9046972ad7740),
    -_f2(416692036, 0xd1e73fe253660), _f2(156259513, 0xceb6b7f4df464),
    _f2(20509061187, 0x3b2dd04b78c45),
    # C4[3], coeff of eps^29, polynomial in n of order 0
    594728, _f(456448303575),
    # C4[3], coeff of eps^28, polynomial in n of order 1
    -_f(3245452288), _f(1965206256), _f(0x17609e98859b3),
    # C4[3], coeff of eps^27, polynomial in n of order 2
    -_f(0x15f49b7dd3600), _f(0x7876e24c6900), _f(0x1f5dd75c0b28),
    _f2(4837, 0x68f14547adeb),
    # C4[3], coeff of eps^26, polynomial in n of order 3
    -_f(0x33418e8004000), _f(0x17b00d59dc000),
    -_f(0x11669ade1c000), _f(0xa37322475bc0),
    _f2(6709, 0x6c31d1e089667),
    # C4[3], coeff of eps^25, polynomial in n of order 4
    -_f(0xc3e38d2fc36800), _f(0x6a604d6faf7a00),
    -_f(0x650b3de948f400), _f(0x20a6596010be00),
    _f(0x88f534a1fae70), _f2(275086, 0x53fa9cf60167f),
    # C4[3], coeff of eps^24, polynomial in n of order 5
    -_f(0xdd5f9d233a5800), _f(0x8b724926c9e000),
    -_f(0x8af41510346800), _f(0x3d05686ce77000),
    -_f(0x2f9901c72df800), _f(0x1ae74f29ea4ce0),
    _f2(223345, 0xf3eec944ed143),
    # C4[3], coeff of eps^23, polynomial in n of order 6
    -_f2(81630, 0xcf55ff9c68c00), _f2(60811, 0x59dd5ef6a6e00),
    -_f2(57592, 0x6457f059a8800), _f2(30387, 0x2572e53b9c200),
    -_f2(30167, 0xe11b4690d8400), _f2(9044, 0xd72699d03d600),
    _f2(2392, 0x21f43a8f7f830), _f2(990092609, 0x9eb428d5a933),
    # C4[3], coeff of eps^22, polynomial in n of order 7
    -_f2(3070961, 0xf14af9164000), _f2(2767073, 0x4d2d51bbc4000),
    -_f2(2322170, 0xf623e90f3c000), _f2(1476552, 0x4ed8bf53f8000),
    -_f2(1490469, 0x7e13eaba44000), _f2(616004, 0x8b84c9ea6c000),
    -_f2(517487, 0xf3178ed39c000), _f2(279040, 0x23dc4dd774ec0),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^21, polynomial in n of order 8
    -_f2(3998482, 0x374a7520d6800), _f2(4351696, 0x89a9dbf785900),
    -_f2(3077852, 0x4b8dc9fbd6e00), _f2(2436308, 0x9b47462d3fb00),
    -_f2(2230379, 0xda399323b400), _f2(1147885, 0x7a5199072bd00),
    -_f2(1196012, 0x91bb473d37a00), _f2(325643, 0x5e75ef9e35f00),
    _f2(87110, 0x728c765d95698), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^20, polynomial in n of order 9
    -_f2(5536106, 0x41a6dc97e5400), _f2(6819318, 0x7020ae33aa000),
    -_f2(3996497, 0x7d04a5d65ec00), _f2(4026336, 0x4a526eb153800),
    -_f2(3081046, 0x922df73cac400), _f2(2027203, 0x8c3cc70035000),
    -_f2(2046086, 0x4cc9bc51b5c00), _f2(787253, 0x8fa9057e6800),
    -_f2(725367, 0x21dd9ffc63400), _f2(368582, 0x69a43eb914890),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^19, polynomial in n of order 10
    -_f2(8942538, 0x3b8622ae62a00), _f2(10481872, 0x1e7c948175300),
    -_f2(5381394, 0x830498d800800), _f2(6645195, 0x535f47efddd00),
    -_f2(4043713, 0x9ba9cf138e600), _f2(3563786, 0x6253b3df24700),
    -_f2(3045580, 0xe2f1f7a110400), _f2(1548984, 0x4828fbf665100),
    -_f2(1694435, 0x63dcfc138a200), _f2(406057, 0xe76a74dc3bb00),
    _f2(110280, 0xa64ca1bbeb438), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^18, polynomial in n of order 11
    -_f2(18204995, 0x3f490d6ed8000), _f2(15367333, 0xa666c37198000),
    -_f2(8424707, 0xb9613a5da8000), _f2(10765521, 3190860555 << 17),
    -_f2(5300295, 0xd300940f58000), _f2(6273886, 0xba1b2aa228000),
    -_f2(4137511, 0x6a32b5bc28000), _f2(2951915, 0x3ffeb65fb0000),
    -_f2(2898950, 0x38c8743c58000), _f2(1027617, 0x2c3889c5b8000),
    -_f2(1062542, 0x7c8a4a4828000), _f2(500325, 0x147f19cd83980),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^17, polynomial in n of order 12
    -_f2(46659673, 0x7940546261000), _f2(20576887, 0xb72d09f420c00),
    -_f2(17371112, 0xc460beb873800), _f2(16552256, 0x8d133b2d84400),
    -_f2(7883306, 0x3c181b1016000), _f2(10867815, 0x95ba8c80bfc00),
    -_f2(5343012, 0x31a34980f8800), _f2(5640245, 0x12558783a3400),
    -_f2(4241979, 0x47a64b12cb000), _f2(2204426, 0xf7d60f21fec00),
    -_f2(2506924, 0x6e46ed413d800), _f2(503732, 0xa322eb69a2400),
    _f2(139663, 0x777cb98300b20), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^16, polynomial in n of order 13
    -_f2(156865464, 0x9b4a437ced000), _f2(26751997, 0x84cabd1d8c000),
    -_f2(47510066, 0xf418e3e50b000), _f2(22667291, 0xeea5410a3a000),
    -_f2(16175537, 0xc4ceea20b9000), _f2(17818506, 0xfb6c54d608000),
    -_f2(7402653, 0x2459922697000), _f2(10650742, 0xeb52d29456000),
    -_f2(5558253, 0xfdda6aad45000), _f2(4690304, 0xc3737ed884000),
    -_f2(4248624, 0xb4bb4dab63000), _f2(1382140, 0xc755b095f2000),
    -_f2(1646389, 0x4c787b5791000), _f2(701746, 0xdc0286e009640),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^15, polynomial in n of order 14
    _f2(158569992, 0x763cf17d39800), _f2(242045827, 0xf358b9d531400),
    -_f2(171801710, 0xfbdaa54751000), _f2(26564510, 0xe59a1e6b54c00),
    -_f2(47715397, 0x8fdbdb93bb800), _f2(25503418, 0x124aa89300400),
    -_f2(14593564, 0x65519680b6000), _f2(19028249, 0x27fd86c303c00),
    -_f2(7127523, 0x40a42052f0800), _f2(9926805, 0x1876eddc2f400),
    -_f2(5956098, 0xfb7e2f3f1b000), _f2(3422018, 0xde3cf0f552c00),
    -_f2(3909386, 0x4ce6da2de5800), _f2(606166, 0xec68c0e73e400),
    _f2(172919, 0x9ad62b665b520), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^14, polynomial in n of order 15
    _f2(234628808, 0x48818da828000), -_f2(452308383, 0x26baa88038000),
    _f2(184630907, 0xde7b734758000), _f2(240946965, 0x4db221ae90000),
    -_f2(189474421, 0xed4c1e36d8000), _f2(27214973, 0x55324802d8000),
    -_f2(46882338, 0xe5fcdfdca8000), _f2(29262846, 2319362995 << 17),
    -_f2(12682237, 0x3cee53d458000), _f2(19904432, 0x70537f02e8000),
    -_f2(7274198, 0xbf917ba828000), _f2(8480909, 0x438c3da230000),
    -_f2(6415713, 0xc95c9b8258000), _f2(1960896, 0x685dc04df8000),
    -_f2(2745254, 0xf883406d28000), _f2(1023946, 0x4eef421f04580),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^13, polynomial in n of order 16
    -_f2(2272755, 0x57fd708a77000), -_f2(26091168, 0x1366cec7d9d00),
    _f2(231976719, 0xafe6927fcde00), -_f2(464894868, 0x24c5c39795700),
    _f2(215184123, 0xaf8273d716c00), _f2(236438336, 0xab29f0bfd4f00),
    -_f2(210344218, 0x367ffa8b78600), _f2(29454299, 0x2f129bee9500),
    -_f2(44460297, 0xf9cfdfb8bb800), _f2(34058265, 0xda8305b9abb00),
    -_f2(10677799, 0x93543d448ea00), _f2(19950418, 0xbb16c712a0100),
    -_f2(8097327, 0xc3857f1ecdc00), _f2(6164437, 0x8a1d8a85ca700),
    -_f2(6487914, 0xa92c56ec54e00), _f2(653539, 0x4a58f163aed00),
    _f2(193289, 0xc4fa7fb371708), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^12, polynomial in n of order 17
    -_f2(136365, 0x73a1fcfe6ac00), -_f2(450638, 0xd074750f34000),
    -_f2(2128024, 0x54e7feac4d400), -_f2(24952088, 0x92a9c1fc91800),
    _f2(228113259, 0x85d44607e4400), -_f2(477191195, 0x7e69e50f07000),
    _f2(251096618, 0x1896eb4cd1c00), _f2(226763725, 0xac7cda7d93800),
    -_f2(234776156, 0x14cc4b0edcc00), _f2(34557325, 0x4230b4bd66000),
    -_f2(39741101, 0x3a85821c7f400), _f2(39764072, 0x42dd69fc98800),
    -_f2(9161206, 0x9c1a792d6dc00), _f2(18380268, 0xf302f56753000),
    -_f2(9708385, 0x581708d300400), _f2(3148914, 0x8380fab1bd800),
    -_f2(5050904, 0x8a565e3e8ec00), _f2(1566765, 0x6fd98617e9df0),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^11, polynomial in n of order 18
    -_f2(18810, 0x4977f6cdda600), -_f2(44617, 0xf507aa2256700),
    -_f2(121680, 0x26c8d0378b000), -_f2(408670, 0xadcc6d8f87900),
    -_f2(1967116, 0xd731d207dba00), -_f2(23614778, 0x5c1a1fadbeb00),
    _f2(222693980, 0x695506ba87c00), -_f2(488598159, 0xe2ab67bc47d00),
    _f2(293333811, 0x10f016a3f3200), _f2(209273530, 0x4db1c2b811100),
    -_f2(262769616, 0x9b49f60945800), _f2(44647130, 0x3acb33bfff00),
    -_f2(31983858, 0x227f1389ce200), _f2(45626356, 0x9e16c6ccb8d00),
    -_f2(9276161, 0xf8fb16a652c00), _f2(14205372, 0x289c377eefb00),
    -_f2(11490116, 0xc948e407f600), _f2(414830, 0x163387d5d8900),
    _f2(117690, 0xc756ec17c4aa8), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^10, polynomial in n of order 19
    -_f2(3667, 0x8ba48fb7ec000), -_f2(7355, 0xde5d961edc000),
    -_f2(15963, 0x138d280434000), -_f2(38393, 53315683 << 17),
    -_f2(106358, 0x1cca460dcc000), -_f2(363723, 0x77fed5aee4000),
    -_f2(1788619, 0xb46088e414000), -_f2(22045766, 0x7d53064fc8000),
    _f2(215267089, 0x7c4e47994000), -_f2(498143540, 0xc077eb386c000),
    _f2(342855614, 0x4b25e0bbcc000), _f2(179961617, 0x7ca6ea4dd0000),
    -_f2(293329289, 0xb4e43f9ccc000), _f2(63137066, 0xbcee02f98c000),
    -_f2(20920174, 0xdceb909f94000), _f2(49479848, 0x7088e98168000),
    -_f2(12768344, 0x1ee1d8cbec000), _f2(6948560, 0xd8f6969c04000),
    -_f2(10643749, 0x466c677134000), _f2(2529930, 0x161dcdf222440),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^9, polynomial in n of order 20
    -_f(0x354d49acec3dd800), -_f(0x606a7d34c50a0200),
    -_f2(2939, 0xdc47a7c209c00), -_f2(5971, 0x671f2d9dad600),
    -_f2(13140, 0xcdf9f327fe000), -_f2(32101, 0x6baea5bb9ea00),
    -_f2(90511, 0x408ba9a232400), -_f2(315893, 0xc97e5e852be00),
    -_f2(1591343, 0xfce30d8d1e800), -_f2(20207205, 0x8b4272e60d200),
    _f2(205238828, 0x21c1cf60c5400), -_f2(504251582, 0xb2b181bcfa600),
    _f2(400330413, 0xa384192d01000), _f2(132810886, 0x4094526254600),
    -_f2(323039224, 0xd5680dd0e3400), _f2(95085342, 0xbfbbc74d27200),
    -_f2(8279837, 0x6ce790195f800), _f2(46514941, 0x8e0e73ffc5e00),
    -_f2(20732718, 0x38ef4b2eebc00), -_f2(922541, 0xf2a1d94487600),
    -_f2(491669, 0x5bd07d195db30), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^8, polynomial in n of order 21
    -_f(0xd828cefda55a800), -_f(0x16c6eac98e7b6000),
    -_f(0x27e1e798049c9800), -_f(0x490330552dbbf000),
    -_f2(2255, 0x88ea2b8740800), -_f2(4647, 0x88c66c31f8000),
    -_f2(10390, 0xd13f35560f800), -_f2(25836, 0xfcd55e2db1000),
    -_f2(74324, 0xc0bfff0e86800), -_f2(265480, 0xf5ce67923a000),
    -_f2(1374647, 0xa0b10ca8f5800), -_f2(18058373, 0x723761b2e3000),
    _f2(191831943, 0xc85920c253800), -_f2(504361484, 0x6e935002fc000),
    _f2(465423127, 0xbaa71ebb04800), _f2(59036306, 0xf120275a2b000),
    -_f2(342905949, 0x5a93131732800), _f2(146354899, 0x9f9c2b8142000),
    -_f2(1641748, 0x1e8ba62ca1800), _f2(28969072, 0x51c8dabef9000),
    -_f2(27136540, 0x3d9359d98800), _f2(4249105, 0xd55e5a0325120),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^7, polynomial in n of order 22
    -_f(0x38123cee860f400), -_f(0x59d375c04e8be00),
    -_f(0x942bf86bd4c1800), -_f(0xfcbda8858afb200),
    -_f(0x1c02af2dc3443c00), -_f(0x33fc822f8d2b6600),
    -_f(0x65e35fc07de4e000), -_f2(3414, 0xc7eb297eb5a00),
    -_f2(7775, 0x1c0e884298400), -_f2(19731, 0x6a31912ef0e00),
    -_f2(58089, 0x9471e600da800), -_f2(213111, 0x15a6331c60200),
    -_f2(1139019, 0x77ee6ce2ccc00), -_f2(15560104, 0x33d66a0afb600),
    _f2(174045800, 0x2f0a20e9d9000), -_f2(494300177, 0xd9e4761bbaa00),
    _f2(535087920, 0xe9f8f195ec00), -_f2(53102016, 0x93f6bbbe95e00),
    -_f2(331738553, 0x77bff637f3800), _f2(216985631, 0x987f3afb7ae00),
    -_f2(21074121, 0x8043eaffd5c00), -_f2(4185955, 0xa3ff769180600),
    -_f2(4713710, 0xd2e19a34f30b0), _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^6, polynomial in n of order 23
    -_f(0xe0ca252d14c000), -_f(0x15a70af15f24000),
    -_f(0x222b3f817554000), -_f(0x375f97b48cd8000),
    -_f(0x5c7b9631f8ac000), -_f(0x9fe2527c7fcc000),
    -_f(0x11face3d5ef34000), -_f(0x21e77d8dabde0000),
    -_f(0x439dcbf7fdccc000), -_f2(2310, 0x1731d0ccf4000),
    -_f2(5373, 0x35ee2c1554000), -_f2(13965, 0xf39edc32e8000),
    -_f2(42247, 0xa0aa0b1cac000), -_f2(159930, 0xa2319a759c000),
    -_f2(887131, 0xc123fa86b4000), -_f2(12685735, 0x6243721af0000),
    _f2(150650948, 0x968da6a8b4000), -_f2(467294064, 0x1610ada8c4000),
    _f2(599544322, 0x5feb9b1dac000), -_f2(214883240, 0x150075a4f8000),
    -_f2(244806233, 0x53bd4b2bac000), _f2(272520146, 0x88b0e96a94000),
    -_f2(87760725, 0x27ae1fc734000), _f2(5827860, 0xa7a2901c3a740),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^5, polynomial in n of order 24
    -_f(0x32b69e04189800), -_f(0x4bd39320660300),
    -_f(0x73a508e7ef1600), -_f(0xb44a7ec206b900),
    -_f(0x1200d9d52c6d400), -_f(0x1d916a5ad4bcf00),
    -_f(0x321a3f994641200), -_f(0x57fce6d660f8500),
    -_f(0xa10c564a22b1000), -_f(0x1356fa3ebba41b00),
    -_f(0x275fd13435900e00), -_f(0x5604e2d76283d100),
    -_f2(3283, 0xdf8f52c874c00), -_f2(8783, 0x8ddc09700e700),
    -_f2(27451, 0x143e179f50a00), -_f2(107903, 0xe48c7d6f59d00),
    -_f2(625732, 0xe2abef41d8800), -_f2(9446536, 0xacc19c0743300),
    _f2(120325828, 0x5507fb0eafa00), -_f2(412649247, 0xc3fe82376e900),
    _f2(633089704, 0xd19d26ed03c00), -_f2(418090362, 0x84d33548fff00),
    -_f2(13712613, 0x4e3334f720200), _f2(163180098, 0x55c7c31664b00),
    -_f2(61921019, 0x751f3b2bed108),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[3], coeff of eps^4, polynomial in n of order 25
    -_f(0x30fab48eb2c00), -_f(0x4779db0cde000),
    -_f(0x6a1a5308c1400), -_f(0xa07c7893bf800),
    -_f(0xf7d15b087bc00), -_f(0x1878e181999000),
    -_f(0x27ab652bf7a400), -_f(0x422ed0b6682800),
    -_f(0x721448fff54c00), -_f(0xcc1e5699294000),
    -_f(0x17d5829db9a3400), -_f(0x2ed74923dde5800),
    -_f(0x61c84aba5ffdc00), -_f(0xdbaa1b53c88f000),
    -_f(0x21cc8beefe3fc400), -_f(0x5da8efb832aa8800),
    -_f2(4876, 0x5d83861736c00), -_f2(20082, 0x8bb9af0c4a000),
    -_f2(123005, 0x97d1502b45400), -_f2(1983151, 0x65e045fd8b800),
    _f2(27425226, 0x9c6669ee40400), -_f2(105081920, 0xe8c662ae85000),
    _f2(191976586, 0x46cce583c1c00), -_f2(186491540, 0xf45203874e800),
    _f2(93245770, 0x7a2901c3a7400), -_f2(18940547, 0x20d0545bbdf90),
    _f2(9570895220, 0xb53783566b8ed),
    # C4[3], coeff of eps^3, polynomial in n of order 26
    -_f(0x10330cb256200), -_f(0x172cb16211100),
    -_f(0x21a8187537800), -_f(0x31b06260f1f00),
    -_f(0x4ab014ab28e00), -_f(0x7280309c9cd00),
    -_f(0xb366eef7be400), -_f(0x11ff8a58b05b00),
    -_f(0x1dae666558ba00), -_f(0x327547ac4a0900),
    -_f(0x58c9207d125000), -_f(0xa2826b77361700),
    -_f(0x137557a5841e600), -_f(0x275355b4b1bc500),
    -_f(0x54b37d85300bc00), -_f(0xc517d06239a5300),
    -_f(0x1f8f2f623d981200), -_f(0x5b85a3034c390100),
    -_f2(5020, 0xa2ee6bc312800), -_f2(21965, 0x48d3177570f00),
    -_f2(144343, 0x4c469a2853e00), -_f2(2526007, 0xb6d389c1bbd00),
    _f2(38395317, 0x415c2de726c00), -_f2(163180098, 0x55c7c31664b00),
    _f2(326360196, 0xab8f862cc9600), -_f2(303048754, 0xd0545bbdf900),
    _f2(104173009, 0x3479cff894d98),
    _f2(28712685662, 0x1fa68a0342ac7),
    # C4[4], coeff of eps^29, polynomial in n of order 0
    4519424, _f(0x13ed3512585),
    # C4[4], coeff of eps^28, polynomial in n of order 1
    _f(322327509504), _f(86419033792), _f(0x12e7203d54087bd),
    # C4[4], coeff of eps^27, polynomial in n of order 2
    _f(0xdf868e997000), -_f(0xc54488fde800), _f(0x67996a8dfb80),
    _f2(6219, 0x86ed0fee71e5),
    # C4[4], coeff of eps^26, polynomial in n of order 3
    _f(0x1e30d5f17398800), -_f(0x20335f44c005000),
    _f(0x8656a9da59d800), _f(0x246f3281df3200),
    _f2(1871928, 0xea4bbbb5bea41),
    # C4[4], coeff of eps^25, polynomial in n of order 4
    _f(0x640278dc982000), -_f(0x64de2b5e388800),
    _f(0x266cf1cb211000), -_f(0x24af02897bd800),
    _f(0x125236c4932c80), _f2(225070, 0xa1cd0c0f186c5),
    # C4[4], coeff of eps^24, polynomial in n of order 5
    _f(0x183393315f62f400), -_f(0x147c8a635ba4f000),
    _f(0xaadb07a361e2c00), -_f(0xbd0a07cdca37800),
    _f(0x2c490db64a86400), _f(0xc3000bbe3e2580),
    _f2(8327613, 0x62a2be2e87a79),
    # C4[4], coeff of eps^23, polynomial in n of order 6
    _f2(7399, 0xe4703b1ceb000), -_f2(4925, 0x718bf750ef800),
    _f2(3656, 0xc01290e152000), -_f2(3594, 0x9ae0aefbbc800),
    _f(0x5080258211e79000), -_f(0x5458466826cf9800),
    _f(0x27a09e95cf36b080), _f2(97921247, 0xc3bd6c206251),
    # C4[4], coeff of eps^22, polynomial in n of order 7
    _f2(4319137, 0xe5044c1364800), -_f2(2259378, 0xc043aee633000),
    _f2(2431286, 0xcceb783bf5800), -_f2(1865690, 0x884902c9a2000),
    _f2(996566, 0x94ae3b7946800), -_f2(1135368, 0x2cb1c30811000),
    _f2(231629, 0x92b25177d7800), _f2(64961, 0x89605803fda00),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^21, polynomial in n of order 8
    _f2(6174501, 0x53f34a829c000), -_f2(2885765, 0xddf01a0f35800),
    _f2(4089976, 0x588848e445000), -_f2(2309244, 0x73683320c8800),
    _f2(1950621, 0xac1b944ace000), -_f2(1810054, 0xa24c07eb4b800),
    _f2(609590, 0x74daa18497000), -_f2(712107, 0x16cff78e5e800),
    _f2(310317, 0x16957f6a36b80), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^20, polynomial in n of order 9
    _f2(7763095, 0xd98a0c3214600), -_f2(4551997, 0xf65d38a54d000),
    _f2(6348004, 0x7dcc619ba1a00), -_f2(2777846, 0x11091dc381c00),
    _f2(3645151, 0x5af876afd6e00), -_f2(2403756, 0x12692c3266800),
    _f2(1377366, 0xde24866584200), -_f2(1585712, 0xf2192bea6b400),
    _f2(268682, 0xb0f056b079600), _f2(77255, 0xca5a822ebf740),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^19, polynomial in n of order 10
    _f2(8073134, 0x8bff962f2e000), -_f2(9331256, 0xe8e10405e1000),
    _f2(8608510, 0x42ad0321d8000), -_f2(3959617, 0x4c778c1e2f000),
    _f2(6283090, 0x55033b3d82000), -_f2(2832307, 0xbbdb17809d000),
    _f2(2955095, 0x929c8347ec000), -_f2(2459067, 0xd43d49c36b000),
    _f2(787004, 0x9cc4866d6000), -_f2(1039103, 0x6b1983acd9000),
    _f2(412222, 0xf695367aa1b00), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^18, polynomial in n of order 11
    _f2(8586281, 0xffd2991fd000), -_f2(20926106, 0xdd733d721a000),
    _f2(9282973, 0x193483c94f000), -_f2(8121077, 0x9b55004148000),
    _f2(9430655, 0x90c0e29221000), -_f2(3512067, 0x80c2ac76000),
    _f2(5840995, 0x1886eb4173000), -_f2(3061324, 0xab1a78b4a4000),
    _f2(2049544, 0x4067911445000), -_f2(2292525, 0x617c054ad2000),
    _f2(297833, 0x966e637f97000), _f2(88539, 0x9a2e50b8c6400),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^17, polynomial in n of order 12
    _f2(32196457, 0xd679f8ae1c000), -_f2(40594018, 0x37167c5ef5000),
    _f2(8052650, 0x2eda271162000), -_f2(20325613, 0xcd34eeff17000),
    _f2(11030346, 0x5827875768000), -_f2(6662972, 0x9685f0fc59000),
    _f2(10015916, 0xfa65faac6e000), -_f2(3377057, 0x1ef6021e7b000),
    _f2(4892320, 0x94cb79bcb4000), -_f2(3369439, 0x93437f1d3d000),
    _f2(1068721, 0xdee482d47a000), -_f2(1596884, 0xcb3e26805f000),
    _f2(562334, 0xcf5270735f500), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^16, polynomial in n of order 13
    _f2(239019678, 0x7928c61a8b800), -_f2(41200119, 0x147c0b11e000),
    _f2(27063572, 0xac3757be98800), -_f2(45155983, 0xc412cf1f79000),
    _f2(8354845, 0xf8b6ea7445800), -_f2(18750027, 0x4e7377c014000),
    _f2(13292220, 0xfed958edd2800), -_f2(5165101, 0x26aa3105af000),
    _f2(10025000, 0x43fec217f800), -_f2(3715677, 0xed5a4430a000),
    _f2(3405288, 0xc16fe1018c800), -_f2(3440521, 0x6cb0e4f2e5000),
    _f2(291108, 0x30be23439800), _f2(90314, 0xe93f4121c6900),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^15, polynomial in n of order 14
    -_f2(301344600, 0x1f7a69f35a000), -_f2(137666269, 0x81776c9d9b000),
    _f2(257500426, 0xa27a71193c000), -_f2(52745704, 0xa8e59f44d000),
    _f2(20527629, 0x3707e00852000), -_f2(49389175, 0x1679a6a55f000),
    _f2(10057417, 0xa546ce8428000), -_f2(15960633, 0x79a78f6a91000),
    _f2(15828795, 0x3b7a7e96fe000), -_f2(4041479, 0x5385608da3000),
    _f2(9015452, 0x8a056dcb14000), -_f2(4531739, 0xb18fd7c855000),
    _f2(1608583, 0x5c81da4aaa000), -_f2(2620079, 0xb9c03a2467000),
    _f2(790676, 0xf12036cb88d00), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^14, polynomial in n of order 15
    -_f2(152316078, 0x9ee9710b1f000), _f2(396132268, 0xf6300698d2000),
    -_f2(331944543, 0x2a26efc8bd000), -_f2(111967823, 0x409ccb544c000),
    _f2(276102802, 0x8592b62d25000), -_f2(69409637, 0x2e4659b6a000),
    _f2(12806364, 0xaa4a38387000), -_f2(52382533, 0xaa3aad6588000),
    _f2(13858261, 0x7d9fda6f69000), -_f2(11925525, 0x17f68feba6000),
    _f2(17994828, 0x2633a57dcb000), -_f2(3926621, 0x9c334da6c4000),
    _f2(6610729, 0xa84ec063ad000), -_f2(5341800, 0xcfe0c57fe2000),
    _f2(171304, 0xc92dc0ce0f000), _f2(53498, 0x8a12fdd94c400),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^13, polynomial in n of order 16
    _f2(945329, 0x3e694a5630000), _f2(13046260, 0xd11553dc81000),
    -_f2(145063327, 0x6c5bbd04f6000), _f2(395288944, 0x9758cc3483000),
    -_f2(364989750, 0x4da45c465c000), -_f2(77659847, 0x7f601a5fdb000),
    _f2(293261136, 0xdb46a6c9be000), -_f2(92956699, 0x68d702f4d9000),
    _f2(4748491, 0xd717292318000), -_f2(52641236, 0xde7217eeb7000),
    _f2(20401071, 0xa831b35d72000), -_f2(7165143, 0xe2daef21b5000),
    _f2(18530179, 0x70f1fa908c000), -_f2(5449998, 0x995f61f213000),
    _f2(2985284, 0xf423c13426000), -_f2(4674955, 0x4c99b17411000),
    _f2(1148405, 0xaa811667d8300), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^12, polynomial in n of order 17
    _f2(39064, 0xc457745427a00), _f2(149707, 0xe179ab818a000),
    _f2(834482, 0xb3de3faf4c600), _f2(11844090, 0x43801d34c0c00),
    -_f2(136492367, 0x606ac4f4b6e00), _f2(391413380, 0x8b1b355567800),
    -_f2(399991879, 0xf56c51d232200), -_f2(32313943, 0x670cb1cd91c00),
    _f2(306137820, 0x47c0d4df8aa00), -_f2(125355715, 0x12c37db13b000),
    -_f2(1549012, 0x61de67b1d0a00), -_f2(48002827, 0x1ef791fca4400),
    _f2(29707099, 0x80264b6e6c200), -_f2(3304868, 0xd90dacdedd800),
    _f2(15595740, 0x1c41b85df0e00), -_f2(8339676, 0x731c5b6cf6c00),
    -_f2(264319, 0x3253133a92600), -_f2(128183, 0x1fd72f4c70540),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^11, polynomial in n of order 18
    _f2(3796, 0xb8b80a685d000), _f2(10243, 0xe5415b1644800),
    _f2(32134, 0x75fe9c2f28000), _f2(125896, 0x13cc0b67cb800),
    _f2(720062, 0x2eb5ef2cf3000), _f2(10542664, 0x8e7784ebe2800),
    -_f2(126401502, 0xa942d02d22000), _f2(383396973, 0xa914c081a9800),
    -_f2(435856143, 0x9e18e4ddf7000), _f2(26921352, 0xa17bcee040800),
    _f2(309790567, 0x432113bb94000), -_f2(168177156, 0xf5a6b5d938800),
    -_f2(1732899, 0x7848d10f61000), -_f2(36033193, 0x6ff05a93a1800),
    _f2(39850986, 0x4a7ce5d24a000), -_f2(3520516, 0x12d4d9afda800),
    _f2(7904559, 0x47211641b5000), -_f2(9293198, 0x11e52b76c3800),
    _f2(1712350, 0xd1c47193d5a80), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^10, polynomial in n of order 19
    _f(0x20b0c3dbe662b800), _f(0x49a4ee6b654d5000),
    _f2(2895, 0xbb9a481b3e800), _f2(7963, 0xd6290c9168000),
    _f2(25525, 0x742091bd91800), _f2(102493, 0xec03f49fb000),
    _f2(603292, 0x6fe940faa4800), _f2(9144553, 0x3f081030e000),
    -_f2(114581171, 0x9502f66408800), _f2(369767644, 0x159b783921000),
    -_f2(470438620, 0x42537ac0f5800), _f2(102998223, 0x33db2118b4000),
    _f2(295924658, 0xfd504b0d5d800), -_f2(220875824, 0xd68590c9b9000),
    _f2(12088406, 0x3b87c77470800), -_f2(15966308, 0xf7cc70b9a6000),
    _f2(44660638, 0xbb68d3ddc3800), -_f2(11155854, 0x316b572a93000),
    -_f2(1400757, 0x91d7719929800), -_f2(909990, 0x5b4dcbdcd9200),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^9, polynomial in n of order 20
    _f(0x55091490e3fe000), _f(0xab3101736f26800),
    _f(0x16d77945c4e3b000), _f(0x345d2a91137d7800),
    _f2(2099, 0xc55d2c398000), _f2(5898, 0x424192198800),
    _f2(19366, 0xa6f5f449f5000), _f2(79943, 0x847cdfac49800),
    _f2(486014, 0x6a1dc16732000), _f2(7659629, 0x94cc8fca800),
    -_f2(100839015, 0x651046eed1000), _f2(348607247, 0x22ddc22bfb800),
    -_f2(499815073, 0x4df2756234000), _f2(197958555, 0x77a0b2f8bc800),
    _f2(251323198, 0x2663cfb2e9000), -_f2(276534810, 0xe292670a12800),
    _f2(51555588, 0x6a67a23666000), _f2(5587968, 0x5e92831b6e800),
    _f2(32523682, 0xed2ae23e23000), -_f2(21111776, 0x46401336e0800),
    _f2(2489921, 0xe3c1e337a6d80), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^8, polynomial in n of order 21
    _f(0xeb8379f6b27c00), _f(0x1b6c4de1f1d7000),
    _f(0x355a1dadc956400), _f(0x6d308de46411800),
    _f(0xed54313f63d4c00), _f(0x22ae87428a2ac000),
    _f(0x58ce5dd980bc3400), _f2(4090, 0xd3c824bc46800),
    _f2(13806, 0x44b4a8a441c00), _f2(58809, 0x7ab991df81000),
    _f2(370898, 0xe410033e70400), _f2(6109620, 0x6402b9f6fb800),
    -_f2(85053139, 0x4bf446ca91400), _f2(317515928, 0x1b63894556000),
    -_f2(517123103, 0xa7a388b5a2c00), _f2(310296682, 0xe98bc80130800),
    _f2(156996715, 0xaa3cf3c05bc00), -_f2(312601560, 0xdd28200ed5000),
    _f2(125126811, 0xf01e02788a400), _f2(4091818, 0xb5091207e5800),
    -_f2(866059, 0xc9a79cf1f7400), -_f2(4943757, 0xf4721fe538b80),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^7, polynomial in n of order 22
    _f(0x2814d49c0c5000), _f(0x468b0d3a3db800),
    _f(0x80724d98876000), _f(0xf31dbc49b20800),
    _f(0x1e12cb4a6a67000), _f(0x3eb5a58b5455800),
    _f(0x8b1eef20fbf8000), _f(0x14cb29a266eda800),
    _f(0x36974c82ca289000), _f2(2585, 0xefae20720f800),
    _f2(9007, 0x1d6baf437a000), _f2(39779, 0x24ec74fd54800),
    _f2(261696, 0x442f64f42b000), _f2(4534975, 0xa5b17f809800),
    -_f2(67279179, 0x4d9bf05604000), _f2(273758534, 0xd27122c18e800),
    -_f2(510920394, 0x40d515b3000), _f2(428723861, 0x53ee2b6143800),
    -_f2(7330129, 0x37be948582000), -_f2(275708250, 0xae16364977800),
    _f2(204390109, 0xe684af0fef000), -_f2(52540960, 0x7463315742800),
    _f2(2056891, 0xfeee14beab380), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^6, polynomial in n of order 23
    _f(0x628e4f4bb7800), _f(0xa60e374943000), _f(0x11fae77940e800),
    _f(0x2022ddc061a000), _f(0x3b7f2e2d7a5800),
    _f(0x72aa26ca9f1000), _f(0xe77392a11fc800),
    _f(0x1ed1e51d0348000), _f(0x460248a5fa93800),
    _f(0xabd9e84dc89f000), _f(0x1d078c2cd5cea800),
    _f(0x58c9fda5cf076000), _f2(5134, 0xa77137081800),
    _f2(23653, 0x63d76094d000), _f2(163469, 0x772f4630d8800),
    _f2(3004667, 0x8d384291a4000), -_f2(47956830, 0xd53f134a90800),
    _f2(214953528, 0xfe0a5a4ffb000), -_f2(463620631, 0xbff95a7639800),
    _f2(519033396, 0x411553aad2000), -_f2(237300381, 0xd565fafaa2800),
    -_f2(84296486, 0x10fabff57000), _f2(142611178, 0x607af3a3b4800),
    -_f2(46622885, 0x3d1480e1d3a00), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^5, polynomial in n of order 24
    _f(0xc0b5b2cac000), _f(0x139ac5d2ed800), _f(0x20abe97223000),
    _f(0x37e2f8cba0800), _f(0x6269b1d1ba000), _f(0xb3074a8a43800),
    _f(0x151de1e3911000), _f(0x298e5ccaa76800),
    _f(0x55d208375c8000), _f(0xbb7ea958fd9800),
    _f(0x1b5e1854857f000), _f(0x4547c4b8360c800),
    _f(0xc1cdc899e5d6000), _f(0x2682d6f5e00af800),
    _f2(2326, 0xf44888e46d000), _f2(11275, 0x7d4afe8b62800),
    _f2(82638, 0x859516eee4000), _f2(1628359, 0xc1653179c5800),
    -_f2(28286265, 0xc31f9b1d25000), _f2(141205400, 0x2bb5164778800),
    -_f2(353352393, 0x632221a20e000), _f2(504046796, 0x730ece181b800),
    -_f2(416863444, 0x7c7b16f237000), _f2(186491540, 0xf45203874e800),
    -_f2(34967163, 0xedcf60a95eb80), _f2(36916310137, 0x41f43bb0c949),
    # C4[4], coeff of eps^4, polynomial in n of order 25
    _f(0xe07098dae00), _f(0x16338b625000), _f(0x23dda179f200),
    _f(0x3b41a69cf400), _f(0x645a89a6b600), _f(0xaeabe0e09800),
    _f(0x1397028dcfa00), _f(0x246014e923c00), _f(0x4633de275be00),
    _f(0x8d95c8a56e000), _f(0x12c670f9ba0200),
    _f(0x2a433484738400), _f(0x6608a70542c600),
    _f(0x10c10ac322d2800), _f(0x30ddb4b92590a00),
    _f(0xa2e30513d28cc00), _f(0x289386109855ce00),
    _f2(3347, 0x17499d2cb7000), _f2(26358, 0x5763b5c021200),
    _f2(564821, 0x99c65b39a1400), -_f2(10825747, 0x58af29d092a00),
    _f2(60624185, 0x23d4ea299b800), -_f2(172778927, 0xa61ece902e600),
    _f2(279737311, 0x6e7b054af5c00), -_f2(233114426, 0x3166846922200),
    _f2(75762188, 0x8341516ef7e40), _f2(36916310137, 0x41f43bb0c949),
    # C4[5], coeff of eps^29, polynomial in n of order 0
    3108352, _f(0x4338129a0b3),
    # C4[5], coeff of eps^28, polynomial in n of order 1
    -_f(4961047 << 17), _f(304969986048), _f(0x171a7cbcbc0a5e7),
    # C4[5], coeff of eps^27, polynomial in n of order 2
    -_f(0xb7a8cf8589000), _f(0x25cdf8a9f5800), _f(0xaa8ee05df480),
    _f2(53207, 0x4825dfa147919),
    # C4[5], coeff of eps^26, polynomial in n of order 3
    -_f(0x4519d2e6066000), _f(0x17b1d503134000),
    -_f(0x1b53dc2d3c2000), _f(0xc104a529c3b00),
    _f2(207992, 0x1a086a30a3679),
    # C4[5], coeff of eps^25, polynomial in n of order 4
    -_f(0xe48436400f9e000), _f(0x825cbe3b5113800),
    -_f(0x9657faac8f9f000), _f(0x1ac735d19d16800),
    _f(0x7b639e59c13780), _f2(8527676, 0x2b5901ca2b961),
    # C4[5], coeff of eps^24, polynomial in n of order 5
    -_f(0x13b86e0d5c5dc000), _f(0x135f9b0385fb0000),
    -_f(0x10df1064c3304000), _f(0x58b0ae17a818000),
    -_f(0x70d05036b8ec000), _f(0x2e5299a0b610e00),
    _f2(10178194, 0x2338af8e3405b),
    # C4[5], coeff of eps^23, polynomial in n of order 6
    -_f2(126383, 0x5f6b81564f000), _f2(192332, 0x2215a4d90d800),
    -_f2(113392, 0x893928fcaa000), _f2(71665, 0x3fb557978e800),
    -_f2(81791, 0xa6f9503f45000), _f2(12036, 0x1a6fad5adf800),
    _f2(3561, 0x9aef6f2cefa80), _f2(3470764200, 0xea81d86b4b937),
    # C4[5], coeff of eps^22, polynomial in n of order 7
    -_f2(191647, 0x188f775ada000), _f2(308186, 0x45ee8f2434000),
    -_f2(124928, 0xd21a49314e000), _f2(153616, 0xaed0e35eb8000),
    -_f2(118466, 0xc4b6a2a9a2000), _f2(38029, 0x77ad4b77bc000),
    -_f2(53612, 0x41f60b8316000), _f2(20169, 0xecfa5f7fa8900),
    _f2(3470764200, 0xea81d86b4b937),
    # C4[5], coeff of eps^21, polynomial in n of order 8
    -_f2(5169843, 0xc81db86efc000), _f2(5341939, 0xe957aa505800),
    -_f2(2049228, 0x2e9753666d000), _f2(3734678, 0xdcd2e44998800),
    -_f2(1762099, 0xebebc251fe000), _f2(1337844, 0xa441c7cbb800),
    -_f2(1455577, 0x7e18adc04f000), _f2(163809, 0xd9aab3cbce800),
    _f2(50215, 0x8f7a6f7ead780), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^20, polynomial in n of order 9
    -_f2(11201228, 0x9af12fea90000), _f2(5330620, 7096189457 << 19),
    -_f2(4084126, 0xa473ecba70000), _f2(5776338, 0xc1238f4360000),
    -_f2(1850318, 0x7e36514750000), _f2(3091001, 2788978033 << 18),
    -_f2(1978996, 0x9854b5b30000), _f2(651396, 0xde4e2e0920000),
    -_f2(1009381, 0x5e1878c010000), _f2(341219, 0x67868049b6800),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^19, polynomial in n of order 10
    -_f2(19364139, 0xf3aad6c27e000), _f2(3661269, 0x231a8ee911000),
    -_f2(10171658, 0x9bc1444518000), _f2(6650152, 0x1449aa44ff000),
    -_f2(2982446, 0xb2f133d6b2000), _f2(5796709, 0x225c7b8fcd000),
    -_f2(2004712, 0xb33d0f538c000), _f2(2087887, 0x2718a4e53b000),
    -_f2(2041244, 0xb9c4a8d7e6000), _f2(150337, 0x64e8ec0109000),
    _f2(48205, 0x4eea8f2f13300), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^18, polynomial in n of order 11
    -_f2(17821498, 0x43ce2fe394000), _f2(8113989, 0x34042cf6f8000),
    -_f2(21055211, 0x1d823792dc000), _f2(4458324, 0xaba1762760000),
    -_f2(8384573, 0x54084121e4000), _f2(8079221, 0xcbb99849c8000),
    -_f2(2172398, 0x503335ed2c000), _f2(5129813, 0x3b8a4c21b0000),
    -_f2(2481567, 0xadec795134000), _f2(934125, 9279934035 << 15),
    -_f2(1531704, 0x9cc504aa7c000), _f2(453383, 0xd34e451346a00),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^17, polynomial in n of order 12
    _f2(4095301, 0x789aeb9e64000), _f2(49542396, 0x46ab457e8d000),
    -_f2(24303219, 0x1ccf0dd62000), _f2(4679495, 0x21a30e03df000),
    -_f2(21666597, 0xecbbb1868000), _f2(6429258, 0x6611bb6911000),
    -_f2(5963806, 0x7f45fe6c6e000), _f2(9141324, 0xab5773fc63000),
    -_f2(2043796, 0x5ca6f33334000), _f2(3626747, 0xd85dd12c15000),
    -_f2(2919955, 0xba0fdf867a000), _f2(85758, 0x333e03c667000),
    _f2(28339, 0x9119c9ad54d00), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^16, polynomial in n of order 13
    -_f2(273240474, 0x43c43c74c8000), _f2(133674826, 0x952bfc30e0000),
    _f2(7048142, 0x68e4684408000), _f2(44883009, 0xdb6a70b90000),
    -_f2(32370151, 0x153b9e91a8000), _f2(2006331, 0xa0ac245340000),
    -_f2(20459012, 0x9d1a27ed8000), _f2(9634139, 0x6e1e5ebef0000),
    -_f2(3415127, 0x8d101d0c88000), _f2(9090639, 8214448173 << 17),
    -_f2(2849328, 0xea461fc3b8000), _f2(1554483, 7516134885 << 16),
    -_f2(2460922, 0x6540542d68000), _f2(615586, 0x6f27f96118400),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^15, polynomial in n of order 14
    _f2(385255297, 0xc522d651da000), -_f2(58599463, 0x810289e63d000),
    -_f2(271784816, 0x96bdc01bbc000), _f2(164665597, 0xfc4f4e3665000),
    _f2(6169937, 0xa7ea1cfd2e000), _f2(36278794, 0xf1d4bf77a7000),
    -_f2(41327996, 0x5935502f28000), _f2(1406713, 0xae66a659c9000),
    -_f2(16753028, 0x6b0d0fac7e000), _f2(13550589, 0x7d5a3390b000),
    -_f2(1765295, 0x851b6e8694000), _f2(7142364, 0xca525091ad000),
    -_f2(4183412, 0x818c59892a000), -_f2(96164, 0xa4307ac011000),
    -_f2(44020, 0x281c2d0515b00), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^14, polynomial in n of order 15
    _f2(85300002, 0xc7e70a9f1c000), -_f2(294351273, 0xafb8edef98000),
    _f2(403760509, 0xda2cbc2e94000), -_f2(107444454, 0x9ae8f34870000),
    -_f2(261509454, 0x4bda846b4000), _f2(200593259, 0xcaf344c1b8000),
    -_f2(1492598, 0x1c0b3e713c000), _f2(23203659, 0x98196f9e60000),
    -_f2(49434335, 0xf8209c0184000), _f2(4620325, 0x4eb0e8bd08000),
    -_f2(10475101, 0x343acca80c000), _f2(16597245, 8542632147 << 16),
    -_f2(2356576, 0x3bbee61554000), _f2(3249396, 0x1edbdd7e58000),
    -_f2(4240477, 0x930e83f9dc000), _f2(851256, 0x2b979a0197a00),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^13, polynomial in n of order 16
    -_f2(334885, 0xc6bdc7fcb0000), -_f2(5563880, 0xa3a405a9f1000),
    _f2(77196254, 0x955c2ca786000), -_f2(280592470, 0x60fd2cd013000),
    _f2(419465490, 0x135ebd637c000), -_f2(164134806, 0xd03e535795000),
    -_f2(238238642, 0xf95f61c30e000), _f2(239782224, 0x6d53e5d49000),
    -_f2(20068072, 0x4afa414658000), _f2(6399560, 0x53e56b4c47000),
    -_f2(53380994, 0xb54d3160a2000), _f2(13179100, 0x7f23319325000),
    -_f2(3190623, 0x71f1454c2c000), _f2(15946535, 0x7112262fa3000),
    -_f2(5597132, 0xd891768336000), -_f2(517466, 0x3872db407f000),
    -_f2(280398, 0x37b65ce5ca500), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^12, polynomial in n of order 17
    -_f2(9362, 0x69735ac9d0000), -_f2(41698, 3327447843 << 20),
    -_f2(274851, 0x56e2bdf830000), -_f2(4724425, 0xa83b5c01a0000),
    _f2(68370240, 0x5baadc4870000), -_f2(262946254, 0xff686b9240000),
    _f2(430395020, 0x66a0aab610000), -_f2(228360148, 0x64a23696e0000),
    -_f2(196492193, 0xc6f6cbf150000), _f2(277855749, 243039325 << 19),
    -_f2(54565881, 0x3f0390efb0000), -_f2(10430670, 3478671393 << 17),
    -_f2(48232829, 0x9769bd8710000), _f2(26504611, 0xd8be140f40000),
    _f2(733724, 0x9fb250690000), _f2(8992810, 0x9e09f3a6a0000),
    -_f2(7946224, 0xca1f6288d0000), _f2(1176502, 0x79934ee544800),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^11, polynomial in n of order 18
    -_f(0x274a66713f785000), -_f(0x78cbe0a9df914800),
    -_f2(6986, 0x5cd0ed6f68000), -_f2(31980, 0xbaca6835fb800),
    -_f2(217574, 0x7dc41d384b000), -_f2(3882916, 0x2edd7dacd2800),
    _f2(58859398, 0xdc7c0f67f2000), -_f2(240755855, 0x78dc5ddf79800),
    _f2(433769587, 0x318800cb6f000), -_f2(298315443, 0xab75c9fd0800),
    -_f2(129660149, 0x66ef2473b4000), _f2(305615878, 0x94b6a51048800),
    -_f2(109156237, 0x593300db57000), -_f2(18007247, 0x43b21e10e800),
    -_f2(29424146, 0x61ad17715a000), _f2(38156138, 0xf0096c8a4a800),
    -_f2(4683041, 0xee399b1b9d000), -_f2(1149725, 0xbf46657f8c800),
    -_f2(1106736, 0x8c2ceac93e180), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^10, polynomial in n of order 19
    -_f(0x3bd4906e474e000), -_f(0x97941b80ce3c000),
    -_f(0x1a66716bc5afa000), -_f(0x532298a0bc3e0000),
    -_f2(4939, 0xda9250746000), -_f2(23308, 0x7863f72384000),
    -_f2(164254, 0x558c90eef2000), -_f2(3056120, 0xcef6e5fe8000),
    _f2(48766418, 0xafc6204b42000), -_f2(213414260, 0xdc9b1ebcc000),
    _f2(425806905, 0x15318e0496000), -_f2(369415923, 0x757d6c39f0000),
    -_f2(31178847, 0x2c748765b6000), _f2(306118804, 0x213b4942ec000),
    -_f2(181898310, 0x263b289662000), _f2(568685, 0x4686791808000),
    -_f2(309548, 0x34bb55302e000), _f2(32975540, 0x34fcc4d2a4000),
    -_f2(16246779, 0x8dca2dd5da000), _f2(1477949, 0xdae92a7065f00),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^9, polynomial in n of order 20
    -_f(0x69d018a3b9e000), -_f(0xed437c3919a800),
    -_f(0x237e48279feb000), -_f(0x5bea2151a0b3800),
    -_f(0x10666acb6ec18000), -_f(0x350c7e1643d3c800),
    -_f2(3247, 0xe2be74bf45000), -_f2(15860, 0x268da19a55800),
    -_f2(116263, 0x5e4790b892000), -_f2(2266502, 0x8314b6fb1e800),
    _f2(38294967, 0xecf46ee8e1000), -_f2(180538484, 0x555f9ed2b7800),
    _f2(401643505, 0x9c33fda5f4000), -_f2(432258273, 0xf8da98e440800),
    _f2(101814780, 0x5dd5e11f87000), _f2(252370005, 0x80f91f9d26800),
    -_f2(252307179, 0x99e21a8986000), _f2(63455824, 0x191a53ee5d800),
    _f2(12621880, 0x95e41abad000), _f2(2033357, 0xc3307b9c44800),
    -_f2(4727243, 0x20838a8bae80), _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^8, polynomial in n of order 21
    -_f(0xc09a6adbf4000), -_f(0x18cab6e3030000),
    -_f(0x359d0ace62c000), -_f(0x7ab7d9cc438000),
    -_f(0x12c67ab580a4000), -_f(856171152199 << 18),
    -_f(0x9233f1c13ddc000), -_f(0x1e779de654b48000),
    -_f(0x789f22a00b054000), -_f2(9796, 7021023797 << 16),
    -_f2(75089, 0xae07706a8c000), -_f2(1543001, 0x638fcd4c58000),
    _f2(27798321, 0x1e96e700fc000), -_f2(142306959, 0xd3ad6eb8e0000),
    _f2(355697955, 0xce7f78ffc4000), -_f2(469861249, 0x5989105b68000),
    _f2(259457720, 0x1370b4ff4c000), _f2(112194489, 0x36d40ed990000),
    -_f2(260872269, 0xf8005192ec000), _f2(151422395, 0x58f7b5f388000),
    -_f2(32332898, 0xbdc6e34964000), _f2(433029, 0xe4d3ce78fba00),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^7, polynomial in n of order 22
    -_f(0x1441fa2f35000), -_f(0x272c726527800),
    -_f(0x4ebdd7b856000), -_f(0xa564301b74800),
    -_f(0x16d6333bd37000), -_f(0x3580dec1951800),
    -_f(0x865ae53c178000), -_f(0x16ec61d7f65e800),
    -_f(0x455fa2e228b9000), -_f(0xef77f4cbfa3b800),
    -_f(0x3d9c6e708569a000), -_f2(5230, 0x8a511fbc88800),
    -_f2(42196, 0xcfdba8cebb000), -_f2(920786, 0xf57a80c4e5800),
    _f2(17837247, 0x2fc56aab44000), -_f2(100064916, 0x5e72032af2800),
    _f2(283253574, 0xc37962f3c3000), -_f2(455567530, 0xe21e28364f800),
    _f2(400948026, 0xf028b16722000), -_f2(118913774, 0x549816fe9c800),
    -_f2(112010399, 0x36034a3e3f000), _f2(121825743, 0x78c43cf486800),
    -_f2(36338425, 0x426e19287b880),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^6, polynomial in n of order 23
    -_f(0x1b5badebe000), -_f(0x326332ca4000), -_f(0x5fd1bd93a000),
    -_f(0xbcd8e5378000), -_f(0x1837bef256000),
    -_f(0x3404424ccc000), -_f(0x75bf8cd1d2000),
    -_f(38025986691 << 17), -_f(0x2dc96f11f6e000),
    -_f(0x811a6e895f4000), -_f(0x195036bc82ea000),
    -_f(0x5af70d135548000), -_f(0x187d57cdaa406000),
    -_f2(2189, 0x32d399c61c000), -_f2(18742, 0x385cb42a82000),
    -_f2(438375, 0xd6a8872030000), _f2(9224813, 0x89f7eb41e2000),
    -_f2(57288808, 0xfdc8999b44000), _f2(184899999, 0x331692f966000),
    -_f2(357870966, 0x3154fb6f18000), _f2(431875147, 0x7929b7544a000),
    -_f2(318710001, 0xe0f19bd36c000), _f2(131641087, 0xbb852faace000),
    -_f2(23311442, 0x9e8a4070e9d00),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[5], coeff of eps^5, polynomial in n of order 24
    -_f(92116035 << 14), -_f(0x26e7bc2d800), -_f(0x46d3779b000),
    -_f(0x84e1d0c0800), -_f(0x101cbc30a000), -_f(0x2073376e3800),
    -_f(0x442adb8b9000), -_f(0x963884ff6800), -_f(0x15dbd71e08000),
    -_f(0x363ebc6d59800), -_f(0x9122bbd857000),
    -_f(0x1a90a4ab06c800), -_f(0x56f0a68cd06000),
    -_f(0x147a29992a8f800), -_f(0x5d1402e6c175000),
    -_f(0x228e263277d22800), -_f2(5078, 0x584c613b04000),
    -_f2(128863, 0x92233985800), _f2(2982258, 0xd360aa0ed000),
    -_f2(20710125, 0x5bbe664118800), _f2(76213261, 0x519df32cfe000),
    -_f2(171479837, 0xf7a363253b800), _f2(241341994, 0x2d1ed763cf000),
    -_f2(186491540, 0xf45203874e800), _f2(58278606, 0x8c59a11a48880),
    _f2(45119934611, 0xe897fd72d67cb),
    # C4[6], coeff of eps^29, polynomial in n of order 0
    139264, _f(63626127165),
    # C4[6], coeff of eps^28, polynomial in n of order 1
    _f(247833 << 16), _f(4782743552), _f(0x219ae3fb400f15),
    # C4[6], coeff of eps^27, polynomial in n of order 2
    _f(420150473 << 18), -_f(0x876551ce0000), _f(0x350bfa156000),
    _f2(4837, 0x68f14547adeb),
    # C4[6], coeff of eps^26, polynomial in n of order 3
    _f(0x297e6b0e9e1000), -_f(0x2e90de909aa000),
    _f(0x6148b0a84b000), _f(0x1d77336bca600),
    _f2(207992, 0x1a086a30a3679),
    # C4[6], coeff of eps^25, polynomial in n of order 4
    _f(0x10bc6a9e4ee30000), -_f(0xc179e3d40c9c000),
    _f(0x3edf483df118000), -_f(0x5c91fff78634000),
    _f(0x216fdab58654400), _f2(10078162, 0xbedd8dc0620e7),
    # C4[6], coeff of eps^24, polynomial in n of order 5
    _f2(17715, 0xdb1cfba26000), -_f2(7689, 0x9976d7f948000),
    _f2(6474, 0xb1047d5d4a000), -_f2(6855, 0xa6eeabbaa4000),
    _f(0x2ac3e335ea26e000), _f(0xd6d2e7c22e28400),
    _f2(372892021, 0x96057cce2c163),
    # C4[6], coeff of eps^23, polynomial in n of order 6
    _f2(279883, 0xa92c150938000), -_f2(86797, 0xd10c69f53c000),
    _f2(160072, 0xfd9d58a4d0000), -_f2(96731, 0xc2b3d16724000),
    _f2(32938, 0x46d62be868000), -_f2(52162, 0xc27e2d9b0c000),
    _f2(17103, 0x67a9fde667c00), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[6], coeff of eps^22, polynomial in n of order 7
    _f2(293467, 0x7db7c77729000), -_f2(146628, 0x46fd92fe6000),
    _f2(282074, 0xcdca0f3f8b000), -_f2(92435, 0x174eb2c344000),
    _f2(105774, 0xf5edeb18ed000), -_f2(100726, 0x78839052a2000),
    _f2(6619, 0xde4489894f000), _f2(2174, 0xdeb0a21cf2e00),
    _f2(4101812237, 0x723c5cdbe4f41),
    # C4[6], coeff of eps^21, polynomial in n of order 8
    _f2(183603, 8337878185 << 19), -_f2(387951, 0x8934978f10000),
    _f2(363243, 0x9b8677d760000), -_f2(100927, 0x6adc79e30000),
    _f2(246790, 7131746729 << 18), -_f2(115867, 0xce56197550000),
    _f2(45470, 0x976a005d20000), -_f2(74789, 0x6bec0ac470000),
    _f2(21823, 0x7d1eb3d72b000), _f2(4101812237, 0x723c5cdbe4f41),
    # C4[6], coeff of eps^20, polynomial in n of order 9
    _f2(2390210, 0x71ea4526d8000), -_f2(11473167, 6397281565 << 18),
    _f2(3566140, 0xe9fdb6daa8000), -_f2(3459649, 0xbdbfad5d70000),
    _f2(5328875, 0xe507b89678000), -_f2(1202839, 0xbeff1963a0000),
    _f2(2208040, 0x527339ea48000), -_f2(1770989, 0xb71cae09d0000),
    _f2(48626, 0x557ebf6618000), _f2(16670, 0x4a1716aa8d000),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^19, polynomial in n of order 10
    _f2(16170911, 0xf66942f9a0000), -_f2(15946100, 0x87937e1ff0000),
    _f2(1191966, 5683381737 << 19), -_f2(10381645, 0x67a9610710000),
    _f2(5401104, 0xec5f94af60000), -_f2(1916345, 0x9f2b7d6630000),
    _f2(5166787, 7293640425 << 18), -_f2(1681428, 0xa094a5ad50000),
    _f2(912008, 0xad6a83a520000), -_f2(1452992, 0x3f13404c70000),
    _f2(367621, 0xca46f4fdbb000), _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^18, polynomial in n of order 11
    _f2(51879505, 0x1c6021da42000), -_f2(3388727, 0x452f2e2244000),
    _f2(10993546, 0x58785d1036000), -_f2(19450323, 0x2862de39d0000),
    _f2(1456775, 0xebc764482a000), -_f2(7922511, 0x8d8f4f815c000),
    _f2(7390372, 0xfe1ce59e1e000), -_f2(1065019, 0x2a2a06ce8000),
    _f2(3871757, 0x7ef447ee12000), -_f2(2395461, 0x8df44bf074000),
    -_f2(40351, 0xb597a7abfa000), -_f2(17707, 0xeba2dcf1c1400),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^17, polynomial in n of order 12
    _f2(18941665, 0xd940803e20000), -_f2(2462456, 0xc647b5b638000),
    _f2(55543449, 0x9a9f25d270000), -_f2(10182797, 0xdffcb19ee8000),
    _f2(4836527, 0xb44e233ec0000), -_f2(21402374, 0x58dcab98000),
    _f2(3817083, 0xbef1c88b10000), -_f2(4459099, 0x992120d448000),
    _f2(8502561, 0xac3fb5bf60000), -_f2(1525489, 0x80b8b610f8000),
    _f2(1649611, 0x4cebe6e3b0000), -_f2(2280763, 0x4f507e59a8000),
    _f2(482782, 0x1ffc428c24800), _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^16, polynomial in n of order 13
    _f2(169672066, 0xfc4e53058c000), -_f2(255936417, 0xcd4166f930000),
    _f2(43044311, 0x58bada2414000), _f2(10984552, 0x79ecf34458000),
    _f2(54615551, 0xb3c2ab069c000), -_f2(20672829, 0x547b9ae620000),
    -_f2(762958, 0xc96d76adc000), -_f2(20252510, 0xad74c43098000),
    _f2(8266131, 0x9541dc37ac000), -_f2(1263055, 0x9458475310000),
    _f2(7416125, 0xebded0d634000), -_f2(3121438, 0x16f54c0588000),
    -_f2(225538, 0xf843322744000), -_f2(111163, 0x41ef8785bb800),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^15, polynomial in n of order 14
    -_f2(371272727, 0xe93844d330000), _f2(258600199, 0x3ab9b44ef8000),
    _f2(127447726, 0xd7dad2fc20000), -_f2(278220404, 0x7730102b8000),
    _f2(77869881, 0xad9b189b70000), _f2(21813766, 0xb09d2ff98000),
    _f2(46644312, 9197745227 << 18), -_f2(33841430, 0x25b28aa218000),
    -_f2(3096455, 0x6fa54a95f0000), -_f2(14807144, 0xa86ee6dfc8000),
    _f2(13281582, 0xf66e06a960000), -_f2(452377, 0x35cd9cb178000),
    _f2(3621811, 0x85d91d8b0000), -_f2(3791781, 0x3a80710f28000),
    _f2(636887, 0x5f8cc1d1bc800), _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^14, polynomial in n of order 15
    -_f2(40751652, 0x879256f716000), _f2(182461023, 0x62c00442f4000),
    -_f2(366891419, 0xe235688602000), _f2(303920923, 0x2a6218fe88000),
    _f2(70640959, 0xa70aa30512000), -_f2(290919308, 0xf0cc1f4de4000),
    _f2(124435738, 0x116d522626000), _f2(24575054, 0x49539549b0000),
    _f2(29829722, 0x6d4c4f193a000), -_f2(46205497, 0xcd680acebc000),
    _f2(1253661, 0x8798d15a4e000), -_f2(5829398, 0x329c172b28000),
    _f2(15178042, 0x87d0f72562000), -_f2(3413258, 0x604057df94000),
    -_f2(544537, 0x1343d1098a000), -_f2(371792, 0x5ec0380ab3400),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^13, polynomial in n of order 16
    _f2(100946, 21976965 << 20), _f2(2010862, 0x3c46708bb0000),
    -_f2(34502092, 0x6e09dbf3a0000), _f2(163298206, 0x527fb2e110000),
    -_f2(355839921, 948516465 << 18), _f2(347383598, 0x3243b82e70000),
    -_f2(2611762, 0xae3f6124e0000), -_f2(286060486, 421499843 << 16),
    _f2(181022396, 2339564421 << 19), _f2(11053843, 0x8ea9e8f130000),
    _f2(5354229, 0xc704cb69e0000), -_f2(50862137, 0xf12aeaf970000),
    _f2(14064844, 5665935493 << 18), _f2(1748678, 0x2e869553f0000),
    _f2(9719088, 0x671cfc38a0000), -_f2(6714197, 0x76aa8fd6b0000),
    _f2(816805, 0x9ce5b98e4f000), _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^12, polynomial in n of order 17
    _f(0x75cff722d22b8000), _f2(9742, 5260669319 << 19),
    _f2(75734, 0x79163f0448000), _f2(1568684, 0xd935dd4310000),
    -_f2(28213944, 0x88db35f228000), _f2(141802366, 0xe4716652a0000),
    -_f2(336424367, 0x7aaa4f7098000), _f2(384795625, 0xe2aff0230000),
    -_f2(92516926, 0xbd45322708000), -_f2(252728877, 4730701433 << 18),
    _f2(239978666, 0xfd893c3a88000), -_f2(28528394, 5445461995 << 16),
    -_f2(18370370, 0x5cd8a4fbe8000), -_f2(38961300, 0x78b7628f20000),
    _f2(30014507, 0xb37b1485a8000), -_f2(654615, 0xa96a2bf90000),
    -_f2(667571, 0x85c41bf0c8000), -_f2(1181523, 0x1c81baa857000),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^11, polynomial in n of order 18
    _f(0x55d873de6520000), _f(0x12c7cfeef6810000),
    _f(0x4e200e3f1e1 << 20), _f2(6671, 0xd2467fb9f0000),
    _f2(53806, 3275978471 << 17), _f2(1163348, 0xd1cfb7f3d0000),
    -_f2(22032298, 0xf3cc53d740000), _f2(118198962, 4397370971 << 16),
    -_f2(306929389, 0x72efa76b60000), _f2(409945031, 0xba4df5f90000),
    -_f2(195574008, 5584443935 << 19), -_f2(178055138, 0x4cd4f3ce90000),
    _f2(282861404, 0xd715020c60000), -_f2(99637722, 0xf11193d4b0000),
    -_f2(20986520, 0xfb661347c0000), -_f2(8771627, 7018708525 << 16),
    _f2(31360164, 0xdb2c51c420000), -_f2(12477955, 8590832271 << 16),
    _f2(873590, 0xbe0d3e9693000), _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^10, polynomial in n of order 19
    _f(0x5808512b12b000), _f(0xfaa729276e2000),
    _f(0x3175560e4519000), _f(0xb21b680b3a90000),
    _f(0x2fcbc5fe71407000), _f2(4229, 0xf0de326e3e000),
    _f2(35532, 0x38e22907f5000), _f2(805604, 0x42db4fa3ec000),
    -_f2(16150031, 0xfe4d67d51d000), _f2(93034137, 0xf6628ead9a000),
    -_f2(265995225, 0x398943192f000), _f2(414315266, 0x970145dd48000),
    -_f2(301204836, 0xc549c7ba41000), -_f2(51738066, 0x4e1063bb0a000),
    _f2(275650719, 0x10481031ad000), -_f2(187610845, 0x85f00095c000),
    _f2(25230256, 0x4ada23b49b000), _f2(13917204, 0x3da6dc4452000),
    _f2(4066715, 0x8660f73889000), -_f2(4361677, 0xea98323d07e00),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^9, polynomial in n of order 20
    _f(0x65fa8c6bf0000), _f(0xfe88642ae4000), _f(0x2aa82304e58000),
    _f(0x7ca8bddcccc000), _f(434853972467 << 18),
    _f(0x5e16320d44b4000), _f(0x1a2859bf40b28000),
    _f2(2409, 0x1b825da69c000), _f2(21179, 0xabe6860d90000),
    _f2(506292, 0x5b6e5f0684000), -_f2(10810252, 0xeee1886808000),
    _f2(67327238, 0xa18a80786c000), -_f2(213364581, 0xe79aac41a0000),
    _f2(387619687, 0x51e3ba1054000), -_f2(387180015, 0xd550406b38000),
    _f2(121695298, 0x2400c6e23c000), _f2(172879787, 0x9e57682f30000),
    -_f2(230507460, 0xb74e70fddc000), _f2(112381926, 0x4eee70a198000),
    -_f2(20283371, 0x42949e7bf4000), -_f2(288686, 0x988d3450a7c00),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^8, polynomial in n of order 21
    _f(0x72e86a7de000), _f(8772831327 << 15), _f(0x273ffc1812000),
    _f(0x64635c5cac000), _f(0x11473cdd246000),
    _f(0x33fd816c260000), _f(0xae6e2137a7a000),
    _f(0x29ff10928814000), _f(0xc26a115cf4ae000),
    _f(0x492994f20c1c8000), _f2(10833, 0x80f3c9e4e2000),
    _f2(274842, 0xd406a2037c000), -_f2(6296293, 0xca802ed0ea000),
    _f2(42731189, 0xb6f3d1e130000), -_f2(151191524, 0x41a7e788b6000),
    _f2(320575109, 0xae49526ee4000), -_f2(416345568, 0xb8c8445e82000),
    _f2(298319523, 0xb52957c098000), -_f2(42956565, 0x78799bae4e000),
    -_f2(119892798, 0x70342c95b4000), _f2(103927174, 0x8691916be6000),
    -_f2(29157346, 0x2fb5a3d22ec00),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[6], coeff of eps^7, polynomial in n of order 22
    _f(74709635 << 15), _f(0x4ba47734000), _f(0xa7b994d0000),
    _f(0x1869c5c6c000), _f(0x3c23e3d88000), _f(0x9e1c8b7a4000),
    _f(1882100649 << 18), _f(0x573ad5a4dc000), _f(0x12f915ab6f8000),
    _f(0x4c1f4084014000), _f(0x170ced7cbfb0000),
    _f(0x921b89aca54c000), _f(0x599b4a7922068000),
    _f2(38914, 0x1efa73f084000), -_f2(964915, 0x51a6da0ae0000),
    _f2(7200274, 0x92a23dbc000), -_f2(28652022, 0x356dea628000),
    _f2(70837833, 0x39cdeca8f4000), -_f2(114872161, 0xfcdf3a9570000),
    _f2(122704354, 0xd9bfe74e2c000), -_f2(83141739, 0x9edadabcb8000),
    _f2(32332898, 0xbdc6e34964000), -_f2(5485045, 0x527ae1fc73400),
    _f2(17774519695, 0x99b03d0e3576f),
    # C4[6], coeff of eps^6, polynomial in n of order 23
    _f(257316433920), _f(517719121920), _f(0xfb6e649000),
    _f(0x221f7064000), _f(0x4d84a37f000), _f(0xb958155a000),
    _f(0x1d5dd0db5000), _f(0x4faa5a050000), _f(0xea04686eb000),
    _f(0x2f40e3db46000), _f(0xab8623d121000), _f(0x2d147c4903c000),
    _f(0xe63ae874e57000), _f(0x60cd21bcc932000),
    _f(0x3f869e23e408d000), _f2(29814, 0xcc97221028000),
    -_f2(808726, 0x6d837bf63d000), _f2(6700876, 0x1daf27af1e000),
    -_f2(30153942, 0x8594329407000), _f2(86154121, 0x7da76bf014000),
    -_f2(165128732, 0xdb80e436d1000), _f2(210163841, 0xd18cc55d0a000),
    -_f2(153581269, 0x570b79c9b000), _f2(46622885, 0x3d1480e1d3a00),
    _f2(53323559086, 0xcd10b72aa064d),
    # C4[7], coeff of eps^29, polynomial in n of order 0
    _f(13087612928), _f(0x90e6983c364f3d),
    # C4[7], coeff of eps^28, polynomial in n of order 1
    -_f(161707 << 21), _f(7239297 << 14), _f(0xcf8f801ee602cd),
    # C4[7], coeff of eps^27, polynomial in n of order 2
    -_f(3500022825 << 20), _f(630513507 << 19), _f(0x6038c37fa000),
    _f2(72555, 0x626230f3330c5),
    # C4[7], coeff of eps^26, polynomial in n of order 3
    -_f(92252949633 << 21), _f(16187170389 << 22),
    -_f(51975912235 << 21), _f(0x7c00d0f2b78000),
    _f2(3119881, 0x867e38d993117),
    # C4[7], coeff of eps^25, polynomial in n of order 4
    -_f(0x64d0a86bae7c0000), _f(0x7c07ce24c65f0000),
    -_f(0x739ece76489e0000), _f(0x6e7bce15f550000),
    _f(0x24fc420030b8400), _f2(127915142, 0x8a371ad88dcaf),
    # C4[7], coeff of eps^24, polynomial in n of order 5
    -_f2(5990, 0xbd2326cc40000), _f2(14992, 4018200301 << 20),
    -_f2(6873, 8929851351 << 18), _f2(2782, 8051012645 << 19),
    -_f2(4583, 0xc89924b340000), _f(0x52aed30dcf988800),
    _f2(430260024, 0xe82db7640b7c1),
    # C4[7], coeff of eps^23, polynomial in n of order 6
    -_f2(169326, 4206873009 << 17), _f2(261065, 0x25b4e353d0000),
    -_f2(59142, 0xf0c50992c0000), _f2(111182, 4597550539 << 16),
    -_f2(88869, 504433083 << 17), _f2(2313, 0xe34bfe3f90000),
    _f(0x32dc48b9e1d23400), _f2(4732860273, 0xf9f6e14c7e54b),
    # C4[7], coeff of eps^22, polynomial in n of order 7
    -_f2(467157, 1100000847 << 20), _f2(258178, 755278933 << 21),
    -_f2(91474, 559664221 << 20), _f2(248285, 171426119 << 22),
    -_f2(82821, 231309675 << 20), _f2(44668, 65972935 << 21),
    -_f2(71456, 2669582201 << 20), _f2(18220, 0x9846e079d4000),
    _f2(4732860273, 0xf9f6e14c7e54b),
    # C4[7], coeff of eps^21, polynomial in n of order 8
    -_f2(10858183, 1145150433 << 21), _f2(1155453, 0xa514064740000),
    -_f2(4408275, 1110140307 << 19), _f2(4494002, 0xa8330ec1c0000),
    -_f2(693747, 3759921697 << 20), _f2(2336198, 8880970129 << 18),
    -_f2(1499288, 4981657777 << 19), -_f2(18466, 6402610053 << 18),
    -_f2(7818, 0x6ee4879b83000), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^20, polynomial in n of order 9
    -_f2(7907170, 4058896835 << 20), _f2(1601483, 335338375 << 23),
    -_f2(11238504, 3427529005 << 20), _f2(2745284, 1787777405 << 21),
    -_f2(2325455, 2252860775 << 20), _f2(4939939, 712213223 << 22),
    -_f2(1021126, 555773201 << 20), _f2(952760, 1631005375 << 21),
    -_f2(1365312, 965324491 << 20), _f2(299618, 0x2f589c3f22000),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^19, polynomial in n of order 10
    _f2(5811147, 7891888051 << 19), _f2(19155879, 6260648859 << 18),
    -_f2(13234724, 832589145 << 21), -_f2(473729, 0xaccee67ac0000),
    -_f2(9690431, 2460044795 << 19), _f2(5218195, 5282091375 << 18),
    -_f2(699193, 3313511321 << 20), _f2(4032431, 0xb01d955a40000),
    -_f2(1901524, 5999844905 << 19), -_f2(111197, 715304509 << 18),
    -_f2(51622, 0xdda253af9f000), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^18, polynomial in n of order 11
    -_f2(34477536, 1085877825 << 20), _f2(44845230, 817114545 << 21),
    _f2(4606432, 1572161669 << 20), _f2(12496576, 210421693 << 23),
    -_f2(18271471, 1543698101 << 20), -_f2(128700, 742574025 << 21),
    -_f2(6139017, 689151983 << 20), _f2(7385046, 100502461 << 22),
    -_f2(590509, 2783893289 << 20), _f2(1800602, 699157181 << 21),
    -_f2(2092277, 3566080099 << 20), _f2(381025, 0x99466ecd7c000),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^17, polynomial in n of order 12
    -_f2(152644671, 981125379 << 19), -_f2(24136152, 0xd3514f38e0000),
    -_f2(16909786, 8097141141 << 18), _f2(53988238, 0xc115854860000),
    -_f2(2192558, 3293732289 << 20), _f2(3853073, 2819007469 << 17),
    -_f2(20689919, 5309095411 << 18), _f2(3514368, 0xf1b4463ee0000),
    -_f2(1814216, 3975618817 << 19), _f2(7354899, 0xbd88356420000),
    -_f2(2207882, 191252177 << 18), -_f2(269543, 3717910997 << 17),
    -_f2(156646, 0x7bcb3b3a6a800), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^16, polynomial in n of order 13
    _f2(52565396, 753292423 << 19), _f2(252855342, 568744119 << 21),
    -_f2(197183211, 7281644191 << 19), -_f2(6678358, 3552459447 << 20),
    _f2(4519131, 7283469291 << 19), _f2(56648760, 112164189 << 22),
    -_f2(15289276, 2020707835 << 19), -_f2(3713103, 1403767329 << 20),
    -_f2(17880720, 7304289905 << 19), _f2(9494998, 1497636157 << 21),
    _f2(492167, 2907561065 << 19), _f2(3952538, 4294903605 << 20),
    -_f2(3358139, 5130468237 << 19), _f2(480004, 0x1e727719e9000),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^15, polynomial in n of order 14
    _f2(279617399, 0xd9972cba40000), -_f2(353187715, 0x687b832220000),
    _f2(118965967, 4456434973 << 19), _f2(220096359, 3595022681 << 17),
    -_f2(240814657, 8170991797 << 18), _f2(28075084, 0xec7a345460000),
    _f2(24758769, 1818605983 << 20), _f2(48013974, 0xb5345431a0000),
    -_f2(32373431, 0xc7bac8f4c0000), -_f2(5075135, 8642954025 << 17),
    -_f2(9094832, 6469786017 << 19), _f2(13639028, 3685620545 << 17),
    -_f2(1773068, 1431802737 << 18), -_f2(460476, 0x51ab5a8ea0000),
    -_f2(423738, 0x5d98934922800), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^14, polynomial in n of order 15
    _f2(16417106, 2408387839 << 20), -_f2(93245803, 1562234793 << 21),
    _f2(256985456, 250552029 << 20), -_f2(365861944, 857240429 << 22),
    _f2(190902238, 1499270843 << 20), _f2(163412998, 1423242741 << 21),
    -_f2(274443985, 2668181351 << 20), _f2(82958237, 163620913 << 23),
    _f2(33859016, 1729347703 << 20), _f2(25275487, 1495319443 << 21),
    -_f2(45844273, 3794232747 << 20), _f2(4490176, 231613489 << 22),
    _f2(1010900, 690735667 << 20), _f2(10013483, 1036831025 << 21),
    -_f2(5637707, 2068106223 << 20), _f2(570308, 0x8f0afe45ec000),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^13, polynomial in n of order 16
    -_f2(25657, 393048869 << 22), -_f2(608651, 0xacbc40d5c0000),
    _f2(12764052, 2144856077 << 19), -_f2(76823449, 3121867141 << 18),
    _f2(228672619, 4131243473 << 20), -_f2(367062288, 0xf756dca4c0000),
    _f2(263470997, 8569317111 << 19), _f2(78573490, 0xffb10f8fc0000),
    -_f2(283548774, 1794660389 << 21), _f2(154702622, 9227087281 << 18),
    _f2(16937276, 1939608161 << 19), -_f2(6432822, 7897704317 << 18),
    -_f2(43016670, 946798949 << 20), _f2(22087851, 0xaa7600dd40000),
    _f2(1665577, 8523064651 << 19), -_f2(163221, 0xe0acad2e40000),
    -_f2(1189371, 0x766c2260a3000), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^12, polynomial in n of order 17
    -_f(0x13bc5107d5f << 20), -_f(506650109317 << 24),
    -_f2(17217, 2185571073 << 20), -_f2(426469, 557216187 << 21),
    _f2(9411503, 1140836685 << 20), -_f2(60299258, 66945391 << 22),
    _f2(194753933, 1835852139 << 20), -_f2(352928157, 2046106529 << 21),
    _f2(328231147, 2405007161 << 20), -_f2(34561926, 360605189 << 23),
    -_f2(248371006, 3915897705 << 20), _f2(226668375, 1401273273 << 21),
    -_f2(40114392, 2920598683 << 20), -_f2(25898188, 1028871717 << 22),
    -_f2(16043876, 2538787453 << 20), _f2(28698456, 1825641427 << 21),
    -_f2(9602688, 2437057327 << 20), _f2(502063, 0xa52218333a000),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^11, polynomial in n of order 18
    -_f(81880241733 << 19), -_f(651169421489 << 18),
    -_f(194261131981 << 22), -_f(0x4616f301f1bc0000),
    -_f2(10659, 7786635659 << 19), -_f2(276843, 0xf150eaf340000),
    _f2(6459374, 425055961 << 20), -_f2(44283297, 2370521611 << 18),
    _f2(156003403, 8328479919 << 19), -_f2(319848045, 0xab86b09a40000),
    _f2(372382116, 1407449139 << 21), -_f2(166870261, 0xbaeb2e09c0000),
    -_f2(148815577, 7753476247 << 19), _f2(260443738, 1330203003 << 18),
    -_f2(131653575, 428167437 << 20), _f2(2775725, 691412797 << 18),
    _f2(12306214, 6299226531 << 19), _f2(5355345, 9401097695 << 18),
    -_f2(3966302, 0xcbc08bfb17000), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^10, polynomial in n of order 19
    -_f(1704454843 << 20), -_f(2722537665 << 21), -_f(19434970697 << 20),
    -_f(4989045369 << 24), -_f(394962411735 << 20),
    -_f(0x128b33efecf << 21), -_f2(5903, 789230693 << 20),
    -_f2(161527, 569013611 << 22), _f2(4006338, 1271698701 << 20),
    -_f2(29564239, 1312346333 << 21), _f2(114267945, 2347153791 << 20),
    -_f2(265827046, 63320697 << 23), _f2(379233361, 4202669809 << 20),
    -_f2(292689947, 1148927723 << 21), _f2(19915451, 3747715939 << 20),
    _f2(197711494, 385979271 << 22), -_f2(197401730, 911113003 << 20),
    _f2(83971818, 387288839 << 21), -_f2(12852829, 1345691321 << 20),
    -_f2(602476, 0x5fc28370ac000), _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^9, polynomial in n of order 20
    -_f(304621785 << 18), -_f(0xc9814e4b0000), -_f(5069418237 << 17),
    -_f(0x7c4fe70d90000), -_f(7691534469 << 20),
    -_f(0x7a02179d470000), -_f(0x274586580a60000),
    -_f(0x10907db87bd50000), -_f2(2773, 9732262223 << 18),
    -_f2(80424, 78339267 << 16), _f2(2134032, 0xd8c3d9bae0000),
    -_f2(17066460, 0x8709888510000), _f2(72842964, 6932239995 << 19),
    -_f2(192914141, 0x448548ebf0000), _f2(332328916, 0xa61d5e5020000),
    -_f2(364348462, 0x260e7984d0000), _f2(215166704, 0xfd6630ec0000),
    _f2(5301792, 6304582341 << 16), -_f2(118567350, 0x6a550b6aa0000),
    _f2(89166503, 0x5c73fd2370000), -_f2(23960987, 0x75c7f62663400),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^8, polynomial in n of order 21
    -_f(11869221 << 18), -_f(7450235 << 20), -_f(79397539 << 18),
    -_f(113271327 << 19), -_f(700448177 << 18), -_f(148973407 << 22),
    -_f(9118660335 << 18), -_f(20216702289 << 19),
    -_f(0xcadd965ff40000), -_f(386512744317 << 20),
    -_f(0xf93c68aca7b << 18), -_f2(30847, 5279995331 << 19),
    _f2(882325, 8584251383 << 18), -_f2(7706931, 2116826591 << 21),
    _f2(36580048, 2730390969 << 18), -_f2(110604386, 3847062005 << 19),
    _f2(227103584, 0x9e98f54ac0000), -_f2(323034443, 1752619391 << 20),
    _f2(314251676, 0xb5ebcf2b40000), -_f2(199218854, 3061725287 << 19),
    _f2(73903768, 9476063903 << 18), -_f2(12124837, 0x72a953b85800),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[7], coeff of eps^7, polynomial in n of order 22
    -_f(575575 << 17), -_f(2681133 << 16), -_f(1637545 << 18),
    -_f(16890107 << 16), -_f(23159565 << 17), -_f(0x8210e690000),
    -_f(27276821 << 20), -_f(0x5bebf1b70000), -_f(3075032387 << 17),
    -_f(0x6a5f183250000), -_f(40477467135 << 18),
    -_f(0x11b5c31caf30000), -_f(0xd14cd352ff20000),
    -_f2(6969, 0xb17d189610000), _f2(216834, 7757873387 << 19),
    -_f2(2087035, 0xf153506af0000), _f2(11091105, 0x4b9d7f7a20000),
    -_f2(38290720, 0xa99fbe31d0000), _f2(91897729, 0x9718fbaac0000),
    -_f2(156643857, 0x418d7e6eb0000), _f2(184759421, 0x6102c9a360000),
    -_f2(129331594, 0xf71b8d2590000), _f2(38395317, 0x415c2de726c00),
    _f2(61527183561, 0xb18970e26a4cf),
    # C4[8], coeff of eps^29, polynomial in n of order 0
    _f(7241 << 16), _f(0x112c657acf71b),
    # C4[8], coeff of eps^28, polynomial in n of order 1
    _f(1165359 << 20), _f(3168035 << 17), _f(0x21ffb4a731cf423f),
    # C4[8], coeff of eps^27, polynomial in n of order 2
    _f(41827383 << 21), -_f(137865429 << 20), _f(631109843 << 16),
    _f2(4837, 0x68f14547adeb),
    # C4[8], coeff of eps^26, polynomial in n of order 3
    _f(54350489115 << 22), -_f(21656377197 << 23), _f(1080358617 << 22),
    _f(0x5c4a2579a0000), _f2(3535865, 0xba8f0d3ad9e09),
    # C4[8], coeff of eps^25, polynomial in n of order 4
    _f2(4480, 63845967 << 22), -_f(0x5f0bc8cec07 << 20),
    _f(0x198015cca1f << 21), -_f(0x51d1e6f78cd << 20),
    _f(0x14fb331d33f30000), _f2(144970494, 0xe0e91e6ce4f71),
    # C4[8], coeff of eps^24, polynomial in n of order 5
    _f2(226427, 7535956641 << 17), -_f2(36730, 6647829291 << 19),
    _f2(116830, 5936429895 << 17), -_f2(76966, 613785099 << 18),
    -_f(0x2a948e8d73a60000), -_f(0x116572b5168a4000),
    _f2(5363908310, 0x81b165bd17b55),
    # C4[8], coeff of eps^23, polynomial in n of order 6
    _f2(151394, 3866446399 << 20), -_f2(105723, 1435687723 << 19),
    _f2(240417, 2090106533 << 21), -_f2(54672, 3991575693 << 19),
    _f2(46185, 3230210197 << 20), -_f2(67790, 4028416911 << 19),
    _f2(15270, 0xa469197488000), _f2(5363908310, 0x81b165bd17b55),
    # C4[8], coeff of eps^22, polynomial in n of order 7
    -_f2(105618, 1394014919 << 21), -_f2(5351753, 377020849 << 22),
    _f2(3446650, 1690522763 << 21), -_f2(453181, 286167171 << 23),
    _f2(2431204, 1637447437 << 21), -_f2(1239333, 63204475 << 22),
    -_f2(60030, 1665832481 << 21), -_f2(26716, 0x6a2a5d69d0000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^21, polynomial in n of order 8
    _f2(4362900, 465328075 << 22), -_f2(10560212, 802403079 << 19),
    _f2(656010, 2976408017 << 20), -_f2(3068612, 8162681445 << 19),
    _f2(4482659, 1990068235 << 21), -_f2(516359, 5969201251 << 19),
    _f2(1022585, 502576667 << 20), -_f2(1273161, 3447687361 << 19),
    _f2(245310, 0x45a78ad538000), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^20, polynomial in n of order 9
    _f2(23624906, 1629010283 << 19), -_f2(5851601, 324958949 << 22),
    _f2(255419, 6850290885 << 19), -_f2(10559838, 1365338319 << 20),
    _f2(3058631, 5351542623 << 19), -_f2(782822, 266312293 << 21),
    _f2(4071490, 3032871865 << 19), -_f2(1457911, 2387656005 << 20),
    -_f2(144540, 929274797 << 19), -_f2(76349, 0x2c1c25d590000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^19, polynomial in n of order 10
    _f2(23056909, 2395766741 << 20), _f2(8427619, 3212023717 << 19),
    _f2(19522568, 367619617 << 22), -_f2(12637641, 5752438869 << 19),
    -_f2(1859539, 2126155853 << 20), -_f2(7720368, 2358643951 << 19),
    _f2(5969641, 447801057 << 21), -_f2(4464, 2860310889 << 19),
    _f2(1954609, 2968726289 << 20), -_f2(1904959, 7710818563 << 19),
    _f2(302310, 0x7de136fc28000), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^18, polynomial in n of order 11
    -_f2(34760584, 1377594673 << 21), -_f2(45089279, 700199389 << 22),
    _f2(38964787, 642296389 << 21), _f2(8377867, 242649263 << 24),
    _f2(10805340, 270655563 << 21), -_f2(18348308, 77894251 << 22),
    -_f2(8504, 712824639 << 21), -_f2(2874058, 104939633 << 23),
    _f2(7002991, 271121287 << 21), -_f2(1456031, 497148985 << 22),
    -_f2(262921, 1640850307 << 21), -_f2(186713, 0x66184da2b0000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^17, polynomial in n of order 12
    _f2(266733950, 1060079417 << 21), -_f2(92315127, 311764467 << 19),
    -_f2(39784767, 2743383633 << 20), -_f2(24124714, 5368290721 << 19),
    _f2(52035773, 16490707 << 22), -_f2(214469, 3779317103 << 19),
    -_f2(32744, 3406796695 << 20), -_f2(19012957, 4710528797 << 19),
    _f2(5897141, 767669011 << 21), _f2(758445, 547828629 << 19),
    _f2(4185368, 186448291 << 20), -_f2(2955613, 5547446041 << 19),
    _f2(363691, 0xed908404b8000), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^16, polynomial in n of order 13
    -_f2(254507630, 0xe2b8b6bb40000), -_f2(58148124, 1471923579 << 20),
    _f2(270522720, 3187458133 << 18), -_f2(149985652, 7726894061 << 19),
    -_f2(27328603, 0x99e6e9ea40000), _f2(4011861, 407374679 << 21),
    _f2(54943982, 0xaf3dd22640000), -_f2(17478454, 3922351195 << 19),
    -_f2(6848089, 0x9bbe86d940000), -_f2(11885922, 3297252137 << 20),
    _f2(11706960, 678889437 << 18), -_f2(595378, 2432546249 << 19),
    -_f2(329167, 0xe066968840000), -_f2(450081, 0x595d162958000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^15, polynomial in n of order 14
    -_f2(166710239, 3480741959 << 20), _f2(313421255, 2329933911 << 19),
    -_f2(299209385, 1287661491 << 21), _f2(24383199, 5307535921 << 19),
    _f2(248029318, 3243559867 << 20), -_f2(210032461, 8189928917 << 19),
    _f2(10907073, 960500783 << 22), _f2(29948106, 2038678405 << 19),
    _f2(40306034, 2725271357 << 20), -_f2(36745958, 6196825729 << 19),
    -_f2(1934460, 123839633 << 21), -_f2(492518, 4761069735 << 19),
    _f2(9949315, 1255380799 << 20), -_f2(4720329, 388065197 << 19),
    _f2(398374, 0x9081f25c18000), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^14, polynomial in n of order 15
    -_f2(5499415, 942753073 << 21), _f2(38811064, 349279653 << 22),
    -_f2(140017234, 1709558915 << 21), _f2(290760665, 163783697 << 23),
    -_f2(332595868, 714890693 << 21), _f2(118902683, 730607999 << 22),
    _f2(189429058, 237701737 << 21), -_f2(256456610, 243945477 << 24),
    _f2(78365400, 1246868327 << 21), _f2(35514427, 78282137 << 22),
    _f2(8045132, 96899221 << 21), -_f2(42695880, 422981029 << 23),
    _f2(15212575, 506177875 << 21), _f2(2863173, 903918451 << 22),
    _f2(284029, 1922203905 << 21), -_f2(1161079, 0x6c18f2ad70000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^13, polynomial in n of order 16
    _f2(5407, 439728533 << 23), _f2(151556, 693836399 << 19),
    -_f2(3836797, 3870271773 << 20), _f2(28742693, 8016450573 << 19),
    -_f2(111747677, 1508361473 << 21), _f2(256964119, 236840267 << 19),
    -_f2(347691811, 711701031 << 20), _f2(216072654, 2622689769 << 19),
    _f2(88295276, 790755477 << 22), -_f2(263658780, 5516898905 << 19),
    _f2(163753678, 37603151 << 20), -_f2(1803857, 6339257275 << 19),
    -_f2(22560155, 108468139 << 21), -_f2(21197875, 3801517693 << 19),
    _f2(25658955, 3111371333 << 20), -_f2(7416706, 6937931487 << 19),
    _f2(268690, 0x9ce0757848000), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^12, polynomial in n of order 17
    _f(369814360487 << 19), _f(159053188703 << 23),
    _f2(3152, 1136779065 << 19), _f2(92558, 2295771257 << 20),
    -_f2(2473330, 1734833909 << 19), _f2(19757571, 360351901 << 21),
    -_f2(83162616, 6619531363 << 19), _f2(212413714, 2066066043 << 20),
    -_f2(337117487, 5524436113 << 19), _f2(299293348, 697772127 << 22),
    -_f2(51076102, 4535292671 << 19), -_f2(200831521, 1217539203 << 20),
    _f2(227963885, 2651839699 << 19), -_f2(87452614, 163103009 << 21),
    -_f2(9664164, 7186873499 << 19), _f2(9739937, 3920029055 << 20),
    _f2(6101400, 4999938359 << 19), -_f2(3588081, 0x84422b3e50000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^11, polynomial in n of order 18
    _f(3576016431 << 20), _f(32208729499 << 19), _f(10983028711 << 23),
    _f(0x9286be006280000), _f(0x65e9f47db41 << 20),
    _f2(50386, 4528870031 << 19), -_f2(1428014, 1685009291 << 21),
    _f2(12227031, 6176103481 << 19), -_f2(56007028, 2392678701 << 20),
    _f2(159476659, 5817614083 << 19), -_f2(295263705, 809939737 << 22),
    _f2(344605761, 6427356205 << 19), -_f2(202719833, 951923227 << 20),
    -_f2(50658828, 5629491977 << 19), _f2(201884255, 1512264231 << 21),
    -_f2(166564947, 5368120671 << 19), _f2(63259557, 2614639991 << 20),
    -_f2(8140125, 1990873493 << 19), -_f2(722971, 0xa61c9dba68000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^10, polynomial in n of order 19
    _f(45596577 << 21), _f(81531441 << 22), _f(656187675 << 21),
    _f(191463201 << 25), _f(17391213765 << 21), _f(65094511967 << 22),
    _f(0x16272ee843f << 21), _f2(23168, 382193603 << 23),
    -_f2(700305, 441535191 << 21), _f2(6465118, 134564813 << 22),
    -_f2(32414063, 913166045 << 21), _f2(103314135, 145041825 << 24),
    -_f2(222332965, 1267927603 << 21), _f2(326070132, 55789563 << 22),
    -_f2(309964302, 1355885369 << 21), _f2(149975409, 308317249 << 23),
    _f2(35633361, 576916401 << 21), -_f2(113104897, 1027785623 << 22),
    _f2(77116975, 1889590315 << 21), -_f2(20082545, 0xcd53c80110000),
    _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^9, polynomial in n of order 20
    _f(1123785 << 21), _f(13838643 << 19), _f(23159565 << 20),
    _f(171251217 << 19), _f(44667189 << 23), _f(3472549135 << 19),
    _f(10302054723 << 20), _f(162001999341 << 19),
    _f(500351698399 << 21), _f2(8102, 6578411627 << 19),
    -_f2(262912, 1458939143 << 20), _f2(2634746, 8016047177 << 19),
    -_f2(14551212, 731365323 << 22), _f2(52133305, 8561410375 << 19),
    -_f2(129964645, 2819080401 << 20), _f2(232230181, 2215647013 << 19),
    -_f2(298362920, 1016411147 << 21), _f2(269480987, 8039357859 << 19),
    -_f2(161945649, 1493828379 << 20), _f2(57837731, 7816254593 << 19),
    -_f2(9237971, 9476063903 << 15), _f2(69730808036, 0x96022a9a34351),
    # C4[8], coeff of eps^8, polynomial in n of order 21
    _f(292383 << 17), _f(202215 << 19), _f(2386137 << 17),
    _f(3789747 << 18), _f(26247507 << 17), _f(6294651 << 21),
    _f(437764365 << 17), _f(1112245757 << 18), _f(0x67551030e0000),
    _f(28804895217 << 19), _f(0x2c0f1d988820000),
    _f(0x66a336663d1c0000), -_f2(57641, 8501165381 << 17),
    _f2(631918, 3696102011 << 20), -_f2(3870503, 720372107 << 17),
    _f2(15639991, 0xcc8b836440000), -_f2(44892569, 0xd7d7de220000),
    _f2(94682509, 2360318459 << 19), -_f2(147486216, 0x5ecdb08ae0000),
    _f2(163873573, 0xbeaba7b6c0000), -_f2(110855652, 0xd3ce78fba0000),
    _f2(32332898, 0xbdc6e34964000), _f2(69730808036, 0x96022a9a34351),
    # C4[9], coeff of eps^29, polynomial in n of order 0
    _f(16847 << 16), _f(0x3d2e2985830503),
    # C4[9], coeff of eps^28, polynomial in n of order 1
    -_f(207753 << 23), _f(1712087 << 18), _f(0x438da32e1600335),
    # C4[9], coeff of eps^27, polynomial in n of order 2
    -_f(3127493161 << 21), -_f(38277317 << 20), -_f(0xe4960490000),
    _f2(161925, 0x30e683ffe0741),
    # C4[9], coeff of eps^26, polynomial in n of order 3
    -_f(9299582409 << 22), _f(3656674463 << 23), -_f(10918261107 << 22),
    _f(80278491423 << 17), _f2(1317283, 0x4f8aa089603a9),
    # C4[9], coeff of eps^25, polynomial in n of order 4
    -_f(711479186953 << 22), _f2(3279, 1361598081 << 20),
    -_f(0x3749d192179 << 21), -_f(309897952117 << 20),
    -_f(0x1f18264b9990000), _f2(162025847, 0x379b22013c233),
    # C4[9], coeff of eps^24, polynomial in n of order 5
    -_f2(133856, 15001023 << 25), _f2(223946, 23087107 << 27),
    -_f2(32028, 12079289 << 25), _f2(48931, 2142027 << 26),
    -_f2(63933, 112742755 << 25), _f2(12842, 2153614949 << 20),
    _f2(5994956347, 0x96bea2db115f),
    # C4[9], coeff of eps^23, polynomial in n of order 6
    -_f2(5988742, 4056322469 << 20), _f2(2349145, 7648181561 << 19),
    -_f2(426462, 344885543 << 21), _f2(2475174, 940948911 << 19),
    -_f2(999559, 3441325239 << 20), -_f2(83146, 4971496059 << 19),
    -_f2(41198, 0x4f02423cb8000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^22, polynomial in n of order 7
    -_f2(8631189, 1052889985 << 21), -_f2(629492, 634245703 << 22),
    -_f2(3874477, 505696163 << 21), _f2(3866974, 513650043 << 23),
    -_f2(159710, 1408881461 << 21), _f2(1100061, 714281683 << 22),
    -_f2(1180171, 586380503 << 21), _f2(201643, 0x9fcf910730000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^21, polynomial in n of order 8
    _f2(341632, 721850923 << 22), _f2(2597220, 4632100393 << 19),
    -_f2(10372056, 1528523471 << 20), _f2(1205419, 4719051179 << 19),
    -_f2(1145316, 921601685 << 21), _f2(3990959, 6117017869 << 19),
    -_f2(1073059, 3486842565 << 20), -_f2(153111, 7776387185 << 19),
    -_f2(94130, 0x280827bb28000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^20, polynomial in n of order 9
    _f2(3783713, 134627971 << 22), _f2(23115315, 66415493 << 25),
    -_f2(6392067, 518305043 << 22), -_f2(1733013, 275013225 << 23),
    -_f2(8816564, 833972409 << 22), _f2(4468878, 247379557 << 24),
    _f2(300534, 553592433 << 22), _f2(2085027, 317816093 << 23),
    -_f2(1725044, 456624309 << 22), _f2(240877, 0x8d28f00d60000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^19, polynomial in n of order 10
    -_f2(58776429, 1067354331 << 20), _f2(18829393, 7579267909 << 19),
    _f2(10681211, 592856305 << 22), _f2(16946593, 1116323851 << 19),
    -_f2(14277877, 2775669533 << 20), -_f2(2149268, 8027942223 << 19),
    -_f2(4056423, 828321103 << 21), _f2(6443828, 4480734455 << 19),
    -_f2(857619, 751312863 << 20), -_f2(227988, 4599783267 << 19),
    -_f2(205805, 0x3340b739f8000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^18, polynomial in n of order 11
    -_f2(8326980, 196156635 << 21), -_f2(34821146, 552451591 << 22),
    -_f2(46791029, 557069513 << 21), _f2(38334852, 177025053 << 24),
    _f2(8937287, 838991609 << 21), _f2(5317827, 1033371567 << 22),
    -_f2(18378967, 400717301 << 21), _f2(2844411, 12754877 << 23),
    _f2(593018, 1659418637 << 21), _f2(4310821, 76308389 << 22),
    -_f2(2591282, 1217948513 << 21), _f2(276451, 0x9f1a0fb950000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^17, polynomial in n of order 12
    -_f2(182057178, 279202431 << 21), _f2(246730983, 7282837989 << 19),
    -_f2(61609386, 3656132889 << 20), -_f2(45340440, 795982425 << 19),
    -_f2(20534253, 134894613 << 22), _f2(51951312, 243959241 << 19),
    -_f2(4823611, 581671439 << 20), -_f2(5624597, 8387045493 << 19),
    -_f2(13789372, 989768405 << 21), _f2(9667615, 6509295661 << 19),
    _f2(215496, 1395596667 << 20), -_f2(184969, 6596889361 << 19),
    -_f2(459826, 0xaf07be5d28000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^16, polynomial in n of order 13
    _f2(316814308, 524172905 << 23), -_f2(186563878, 63588247 << 25),
    -_f2(110274143, 526845649 << 23), _f2(263023219, 83035351 << 24),
    -_f2(130904637, 509865531 << 23), -_f2(30397924, 20206077 << 26),
    _f2(13683269, 123318603 << 23), _f2(48158450, 141529345 << 24),
    -_f2(26437768, 420249375 << 23), -_f2(5662772, 129791197 << 25),
    -_f2(2185901, 350246489 << 23), _f2(9629298, 177188459 << 24),
    -_f2(3949112, 493038403 << 23), _f2(276643, 3634960421 << 18),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^15, polynomial in n of order 14
    _f2(78761274, 365004673 << 20), -_f2(201515576, 8484307809 << 19),
    _f2(313194006, 1602645493 << 21), -_f2(252751914, 5568714583 << 19),
    -_f2(13191170, 2167441005 << 20), _f2(241719441, 7606152787 << 19),
    -_f2(201822768, 404840345 << 22), _f2(21302083, 5136432093 << 19),
    _f2(37050656, 1255073061 << 20), _f2(20598069, 641077127 << 19),
    -_f2(39565379, 1270355289 << 21), _f2(9630032, 7047419793 << 19),
    _f2(3352897, 1867330359 << 20), _f2(651457, 7128437307 << 19),
    -_f2(1114108, 0x7475455348000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^14, polynomial in n of order 15
    _f2(1503684, 1762694997 << 21), -_f2(12945810, 457623793 << 22),
    _f2(59109631, 1997027375 << 21), -_f2(165337541, 436423661 << 23),
    _f2(292248408, 1310925625 << 21), -_f2(302358268, 80333091 << 22),
    _f2(101329883, 1625451155 << 21), _f2(168206436, 256940945 << 24),
    -_f2(245462494, 1698275235 << 21), _f2(106772813, 575322475 << 22),
    _f2(20184277, 1515722423 << 21), -_f2(15841583, 115367503 << 23),
    -_f2(24343366, 297803839 << 21), _f2(22619454, 642864953 << 22),
    -_f2(5751128, 1751200357 << 21), _f2(119914, 0x778fad9290000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^13, polynomial in n of order 16
    -_f(494538685723 << 23), -_f2(30244, 7532247025 << 19),
    _f2(913230, 2357276371 << 20), -_f2(8356271, 5886749331 << 19),
    _f2(41054740, 50726383 << 21), -_f2(125953300, 8377060373 << 19),
    _f2(252781900, 3961145001 << 20), -_f2(323011393, 7392450487 << 19),
    _f2(214671336, 735258085 << 22), _f2(38642307, 2838366023 << 19),
    -_f2(221064383, 2701482241 << 20), _f2(190191489, 7753766309 << 19),
    -_f2(54203341, 2021364059 << 21), -_f2(15936650, 4639479773 << 19),
    _f2(7069294, 1728459477 << 20), _f2(6475976, 7904740225 << 19),
    -_f2(3243677, 0x65d7af1058000), _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^12, polynomial in n of order 17
    -_f(4941153649 << 22), -_f(2434362319 << 26),
    -_f(480183190319 << 22), -_f2(15428, 422153761 << 23),
    _f2(492912, 506448323 << 22), -_f2(4815395, 177795021 << 24),
    _f2(25573504, 64498885 << 22), -_f2(86374812, 63633203 << 23),
    _f2(196725482, 856584503 << 22), -_f2(303474922, 105041487 << 25),
    _f2(296000607, 1045914233 << 22), -_f2(124541003, 470657541 << 23),
    -_f2(96946363, 592229397 << 22), _f2(194787320, 217061649 << 24),
    -_f2(139624521, 484247315 << 22), _f2(48044895, 435975913 << 23),
    -_f2(5082038, 276187937 << 22), -_f2(749748, 0x6069872020000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^11, polynomial in n of order 18
    -_f(231323121 << 20), -_f(2351460757 << 19), -_f(912558841 << 23),
    -_f(0xdfda7610580000), -_f(777314384543 << 20),
    -_f2(6590, 3852961377 << 19), _f2(223861, 17176789 << 21),
    -_f2(2346980, 3623268951 << 19), _f2(13542533, 3871010099 << 20),
    -_f2(50565862, 7643343277 << 19), _f2(130735502, 766383623 << 22),
    -_f2(239800507, 7719641123 << 19), _f2(308448660, 3395101317 << 20),
    -_f2(258446712, 3844536697 << 19), _f2(99709743, 227483207 << 21),
    _f2(54337873, 5199140497 << 19), -_f2(105984135, 215955881 << 20),
    _f2(67263140, 627338299 << 19), -_f2(17110329, 0x5fa94e648000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[9], coeff of eps^10, polynomial in n of order 19
    -_f(538707 << 21), -_f(1075491 << 22), -_f(9728097 << 21),
    -_f(3213907 << 25), -_f(333357375 << 21), -_f(1438804621 << 22),
    -_f(39246385997 << 21), -_f(379094211993 << 23),
    _f2(25645, 1674653973 << 21), -_f2(290249, 472854199 << 22),
    _f2(1830100, 1274307463 << 21), -_f2(7588281, 99130323 << 24),
    _f2(22282256, 82312105 << 21), -_f2(48025833, 432719649 << 22),
    _f2(76964476, 1304326427 << 21), -_f2(91125940, 162742323 << 23),
    _f2(77471536, 1478654845 << 21), -_f2(44556474, 1023100235 << 22),
    _f2(15423395, 377918063 << 21), -_f2(2409905, 0x7f0a0dc2b0000),
    _f2(25978144170, 0x7e28f6c5ff5f1),
    # C4[9], coeff of eps^9, polynomial in n of order 20
    -_f(16575 << 21), -_f(226005 << 19), -_f(421083 << 20),
    -_f(3487431 << 19), -_f(1025715 << 23), -_f(90604825 << 19),
    -_f(308056405 << 20), -_f(5606626571 << 19), -_f(20270111449 << 21),
    -_f(0x30ab7cf8ddd << 19), _f2(15220, 1707177905 << 20),
    -_f2(187210, 7636838095 << 19), _f2(1297995, 534056013 << 22),
    -_f2(6003229, 1506461473 << 19), _f2(20010763, 3942424887 << 20),
    -_f2(50026909, 6827222547 << 19), _f2(95435950, 2132760845 << 21),
    -_f2(138382128, 8075045605 << 19), _f2(146522254, 737992253 << 20),
    -_f2(96396219, 7300467927 << 19), _f2(27713913, 0x34f39e3ee8000),
    _f2(77934432511, 0x7a7ae451fe1d3),
    # C4[10], coeff of eps^29, polynomial in n of order 0
    _f(14059 << 19), _f(0x168a4531304537),
    # C4[10], coeff of eps^28, polynomial in n of order 1
    -_f(1004279 << 22), -_f(3373361 << 19), _f2(3807, 0xdf0925caacfb9),
    # C4[10], coeff of eps^27, polynomial in n of order 2
    _f(78580619 << 24), -_f(212705597 << 23), _f(705875469 << 19),
    _f2(59656, 0xa639fabc960fd),
    # C4[10], coeff of eps^26, polynomial in n of order 3
    _f(927832218729 << 21), -_f(204500125453 << 22),
    -_f(29157611613 << 21), -_f(0x66c4e2e4040000),
    _f2(23087123, 0x49a60b16d9e77),
    # C4[10], coeff of eps^25, polynomial in n of order 4
    _f(26024288967 << 27), -_f(7678900515 << 25), _f(13514191015 << 26),
    -_f(31097026337 << 25), _f(89826688809 << 21),
    _f2(25583028, 0x820b055e82c23),
    # C4[10], coeff of eps^24, polynomial in n of order 5
    _f2(1328855, 126349401 << 24), -_f2(550962, 13774891 << 26),
    _f2(2464835, 125518543 << 24), -_f2(784466, 25625323 << 25),
    -_f2(93184, 68528187 << 24), -_f2(52198, 1190112709 << 21),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^23, polynomial in n of order 6
    -_f2(1114607, 27405733 << 26), -_f2(4563722, 53821803 << 25),
    _f2(3169393, 348585 << 27), _f2(68182, 92955763 << 25),
    _f2(1172595, 45755337 << 26), -_f2(1088988, 13585007 << 25),
    _f2(166307, 46143431 << 21), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^22, polynomial in n of order 7
    _f2(5480278, 504127481 << 20), -_f2(9293162, 155326547 << 21),
    -_f2(197247, 3072475525 << 20), -_f2(1644302, 932629169 << 22),
    _f2(3811061, 3287215741 << 20), -_f2(747954, 323686257 << 21),
    -_f2(145848, 3935467265 << 20), -_f2(106662, 0xb8d6e5aaa0000),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^21, polynomial in n of order 8
    _f2(22796753, 23076841 << 25), -_f2(927290, 180149865 << 22),
    -_f2(369606, 74581989 << 23), -_f2(9303996, 552920075 << 22),
    _f2(3036817, 71115369 << 24), _f2(396000, 898162707 << 22),
    _f2(2181010, 484753161 << 23), -_f2(1556188, 389640079 << 22),
    _f2(192540, 3401040927 << 18), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^20, polynomial in n of order 9
    _f2(862955, 1266777839 << 21), _f2(7027949, 96012647 << 24),
    _f2(20762590, 1932890753 << 21), -_f2(9559408, 1057130891 << 22),
    -_f2(3064719, 1040728173 << 21), -_f2(5129237, 23711641 << 23),
    _f2(5760893, 1279205669 << 21), -_f2(394463, 240514009 << 22),
    -_f2(178786, 1942994377 << 21), -_f2(217080, 8651652815 << 18),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^19, polynomial in n of order 10
    -_f2(10913096, 468931943 << 23), -_f2(57356320, 139563275 << 22),
    _f2(21632622, 17971173 << 25), _f2(12352870, 1067519707 << 22),
    _f2(10546968, 197307279 << 23), -_f2(16476123, 321986815 << 22),
    _f2(466396, 220388453 << 24), _f2(183297, 876676071 << 22),
    _f2(4340035, 31265157 << 23), -_f2(2266775, 500956659 << 22),
    _f2(210337, 0xe1ea7a84c0000), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^18, polynomial in n of order 11
    _f2(183220667, 2590575043 << 20), _f2(3573393, 1902991101 << 21),
    -_f2(38433982, 657724943 << 20), -_f2(39892891, 403988263 << 23),
    _f2(42677900, 967722655 << 20), _f2(4292702, 1711217099 << 21),
    -_f2(2792039, 799969587 << 20), -_f2(14767346, 746757159 << 22),
    _f2(7703673, 1133060475 << 20), _f2(747527, 637790873 << 21),
    -_f2(45502, 3704918615 << 20), -_f2(458872, 0xc21d355260000),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^17, polynomial in n of order 12
    -_f2(61780842, 49135749 << 25), -_f2(196091506, 391376453 << 23),
    _f2(232013693, 187926637 << 24), -_f2(59475550, 301219495 << 23),
    -_f2(46331600, 62864215 << 26), -_f2(5439903, 151048009 << 23),
    _f2(49627120, 102515675 << 24), -_f2(16690048, 509576107 << 23),
    -_f2(7354945, 21733079 << 25), -_f2(3769052, 366616397 << 23),
    _f2(9145462, 214930505 << 24), -_f2(3305318, 371590575 << 23),
    _f2(189374, 6271289399 << 19), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^16, polynomial in n of order 13
    -_f2(246951312, 552772347 << 22), _f2(295555190, 29721595 << 24),
    -_f2(151972143, 664869293 << 22), -_f2(114414430, 423169395 << 23),
    _f2(248915402, 492058657 << 22), -_f2(140322148, 99500631 << 25),
    -_f2(15757705, 299151505 << 22), _f2(29401843, 368167099 << 23),
    _f2(29725213, 39057725 << 22), -_f2(34936824, 2445655 << 24),
    _f2(5290998, 483472011 << 22), _f2(3412892, 270211305 << 23),
    _f2(939828, 308185177 << 22), -_f2(1058440, 2262901433 << 19),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^15, polynomial in n of order 14
    -_f2(29237793, 21929809 << 24), _f2(96597143, 85827693 << 23),
    -_f2(210653294, 75090197 << 25), _f2(297719766, 264531499 << 23),
    -_f2(232859751, 332419 << 24), _f2(779198, 466262825 << 23),
    _f2(210565738, 49075321 << 26), -_f2(210231291, 35636249 << 23),
    _f2(60435795, 147172683 << 24), _f2(30790678, 95503589 << 23),
    -_f2(8341137, 27440903 << 25), -_f2(25891285, 147600733 << 23),
    _f2(19770912, 119140889 << 24), -_f2(4475853, 348372575 << 23),
    _f2(24304, 4909664935 << 19), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^14, polynomial in n of order 15
    -_f2(326980, 1465789373 << 20), _f2(3379554, 1566468779 << 21),
    -_f2(19029758, 226591575 << 20), _f2(68313947, 940737655 << 22),
    -_f2(165836985, 3033756273 << 20), _f2(273579872, 472941105 << 21),
    -_f2(286670501, 2169744907 << 20), _f2(131674848, 489609853 << 23),
    _f2(102478771, 1504412891 << 20), -_f2(220713363, 225877065 << 21),
    _f2(153302553, 1201451329 << 20), -_f2(29968476, 1046042243 << 22),
    -_f2(18498599, 2914872793 << 20), _f2(4637884, 270502717 << 21),
    _f2(6604171, 2668065421 << 20), -_f2(2936921, 0x8973648be0000),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^13, polynomial in n of order 16
    _f(8181919521 << 26), _f2(4651, 463423847 << 22),
    -_f2(165627, 528682553 << 23), _f2(1821092, 755660373 << 22),
    -_f2(11021646, 91392285 << 24), _f2(43145010, 992272131 << 22),
    -_f2(116748177, 310953403 << 23), _f2(223068773, 47271409 << 22),
    -_f2(294932999, 99296991 << 25), _f2(242263024, 286305695 << 22),
    -_f2(60276785, 30271037 << 23), -_f2(125363778, 717272947 << 22),
    _f2(182026841, 37372833 << 24), -_f2(116787755, 417593029 << 22),
    _f2(36759949, 346012225 << 23), -_f2(3061422, 962217431 << 22),
    -_f2(731281, 0xaaf6b13240000), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^12, polynomial in n of order 17
    _f(777809483 << 21), _f(436668683 << 25), _f(99139014933 << 21),
    _f(0x1cfc4bfd58d << 22), -_f2(70023, 1623299233 << 21),
    _f2(822756, 446933025 << 23), -_f2(5376526, 2111656919 << 21),
    _f2(23042396, 297910775 << 22), -_f2(69630161, 297782669 << 21),
    _f2(153266731, 112309515 << 24), -_f2(247130955, 1577554627 << 21),
    _f2(284460705, 580739169 << 22), -_f2(212091093, 1801700473 << 21),
    _f2(61297082, 319619083 << 23), _f2(65462218, 2096893009 << 21),
    -_f2(98426842, 1047683893 << 22), _f2(59152561, 1235484315 << 21),
    -_f2(14780753, 0xb5d74354c0000),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^11, polynomial in n of order 18
    _f(1233981 << 23), _f(14104237 << 22), _f(6201077 << 26),
    _f(933195507 << 22), _f(6966040851 << 23), _f(592370721657 << 22),
    -_f2(22184, 189431713 << 24), _f2(279989, 474035391 << 22),
    -_f2(1985627, 52832535 << 23), _f2(9357698, 635528005 << 22),
    -_f2(31645438, 92987147 << 25), _f2(79909927, 996806539 << 22),
    -_f2(153562851, 494252097 << 23), _f2(225351987, 543734289 << 22),
    -_f2(249473559, 252214923 << 24), _f2(201676475, 478217047 << 22),
    -_f2(111804841, 353712747 << 23), _f2(37701632, 700509149 << 22),
    -_f2(5783773, 3281237837 << 18), _f2(86138056986, 0x5ef39e09c8055),
    # C4[10], coeff of eps^10, polynomial in n of order 19
    _f(57057 << 20), _f(126819 << 21), _f(1284843 << 20),
    _f(478667 << 24), _f(56414325 << 20), _f(279062861 << 21),
    _f(8810413183 << 20), _f(99625441377 << 22),
    -_f2(3997, 1800115191 << 20), _f2(54510, 559965495 << 21),
    -_f2(421909, 1796318189 << 20), _f2(2196607, 410595787 << 23),
    -_f2(8328804, 1896277603 << 20), _f2(24012916, 1506461473 << 21),
    -_f2(53875133, 2685050457 << 20), _f2(94820235, 193579723 << 22),
    -_f2(129680615, 3269639887 << 20), _f2(131955714, 608596747 << 21),
    -_f2(84828673, 2009615045 << 20), _f2(24099054, 0xf664899ae0000),
    _f2(86138056986, 0x5ef39e09c8055),
    # C4[11], coeff of eps^29, polynomial in n of order 0
    -_f(255169 << 19), _f(0xbdc79d6e266b55f),
    # C4[11], coeff of eps^28, polynomial in n of order 1
    -_f(535829 << 26), _f(6461547 << 20), _f(0x56e2cdab4666fea1),
    # C4[11], coeff of eps^27, polynomial in n of order 2
    -_f(54075943 << 25), -_f(11012147 << 24), -_f(184884229 << 19),
    _f2(65338, 0x3c271ece8bf8f),
    # C4[11], coeff of eps^26, polynomial in n of order 3
    -_f(29189823 << 30), _f(157366885 << 32), -_f(637753597 << 30),
    _f(13332470307 << 23), _f2(19666808, 0xb9ff38da93b23),
    # C4[11], coeff of eps^25, polynomial in n of order 4
    -_f2(768828, 16543417 << 28), _f2(2405043, 41201001 << 26),
    -_f2(595679, 17511625 << 27), -_f2(94147, 42169149 << 26),
    -_f2(60455, 661597895 << 21), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^24, polynomial in n of order 5
    -_f2(5042070, 2793567 << 28), _f2(2454743, 3154771 << 30),
    _f2(191058, 8757223 << 28), _f2(1233521, 5992667 << 29),
    -_f2(1001395, 9125891 << 28), _f2(137539, 51052897 << 22),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^23, polynomial in n of order 6
    -_f2(7620321, 6390387 << 27), -_f2(1118882, 49853273 << 26),
    -_f2(2175696, 13938625 << 28), _f2(3557797, 45648113 << 26),
    -_f2(479218, 13589073 << 27), -_f2(128928, 23280229 << 26),
    -_f2(115227, 1709406351 << 21), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^22, polynomial in n of order 7
    _f2(3030693, 3978133 << 30), _f2(1618004, 874507 << 32),
    -_f2(9217736, 134985 << 30), _f2(1767041, 97063 << 33),
    _f2(345531, 3069873 << 30), _f2(2240563, 263433 << 32),
    -_f2(1400217, 895853 << 30), _f2(154222, 296573467 << 23),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^21, polynomial in n of order 8
    _f2(304504, 49998275 << 26), _f2(21950325, 110791689 << 23),
    -_f2(5005332, 65785063 << 24), -_f2(3038456, 102319349 << 23),
    -_f2(5977063, 34913149 << 25), _f2(5022754, 485487661 << 23),
    -_f2(45293, 7681997 << 24), -_f2(123970, 179449809 << 23),
    -_f2(222792, 7672407751 << 18), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^20, polynomial in n of order 9
    -_f2(56432361, 120691497 << 25), _f2(6246807, 9955417 << 28),
    _f2(11410351, 83254233 << 25), _f2(14692579, 49664915 << 26),
    -_f2(13833928, 124401461 << 25), -_f2(1245123, 9835207 << 27),
    -_f2(339985, 114545011 << 25), _f2(4291293, 22713457 << 26),
    -_f2(1980685, 98627585 << 25), _f2(159775, 7030690975 << 19),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^19, polynomial in n of order 10
    _f2(40826627, 234278683 << 24), -_f2(19124677, 161584861 << 23),
    -_f2(51024100, 50981393 << 26), _f2(30646271, 384869645 << 23),
    _f2(9815197, 6859997 << 24), _f2(639236, 244693975 << 23),
    -_f2(14951689, 12090449 << 25), _f2(5916250, 151054657 << 23),
    _f2(1073676, 226322463 << 24), _f2(80953, 380993803 << 23),
    -_f2(451111, 0xff79096c0000), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^18, polynomial in n of order 11
    -_f2(233788807, 3535193 << 28), _f2(177892093, 3762413 << 30),
    -_f2(5486729, 8981851 << 28), -_f2(45008759, 222901 << 32),
    -_f2(22295157, 5977845 << 28), _f2(46499690, 3347131 << 30),
    -_f2(8381947, 991351 << 28), -_f2(7636513, 1723229 << 31),
    -_f2(5108235, 6476849 << 28), _f2(8568922, 940153 << 30),
    -_f2(2769555, 11314291 << 28), _f2(126172, 1159668425 << 21),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^17, polynomial in n of order 12
    _f2(246666787, 23370677 << 26), -_f2(50685638, 204720607 << 24),
    -_f2(179803952, 51059789 << 25), _f2(226753224, 100010811 << 24),
    -_f2(83690537, 703961 << 27), -_f2(36110826, 15584139 << 24),
    _f2(17880019, 26951173 << 25), _f2(35367319, 73259919 << 24),
    -_f2(29730133, 51577369 << 26), _f2(2029349, 105583177 << 24),
    _f2(3224077, 120316375 << 25), _f2(1158582, 83501667 << 24),
    -_f2(999784, 6146420159 << 19), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^16, polynomial in n of order 13
    _f2(135472555, 2919565 << 26), -_f2(240891001, 9823619 << 28),
    _f2(277187915, 48314475 << 26), -_f2(153860890, 22130685 << 27),
    -_f2(78071452, 50966567 << 26), _f2(224257271, 6520287 << 29),
    -_f2(168898235, 23643785 << 26), _f2(25365615, 30758325 << 27),
    _f2(33991594, 22223845 << 26), -_f2(1325267, 13358465 << 28),
    -_f2(26274549, 1352253 << 26), _f2(17195323, 12709799 << 27),
    -_f2(3493469, 55138895 << 26), -_f2(37171, 2996514251 << 20),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^15, polynomial in n of order 14
    _f2(8369149, 65829073 << 25), -_f2(34468304, 77612041 << 24),
    _f2(98238486, 50466661 << 26), -_f2(197855127, 257258223 << 24),
    _f2(275634083, 126808451 << 25), -_f2(237329411, 109823349 << 24),
    _f2(57709083, 378039 << 27), _f2(144028485, 10992165 << 24),
    -_f2(208082193, 86131531 << 25), _f2(120116321, 182851999 << 24),
    -_f2(12733337, 27687817 << 26), -_f2(18886416, 178008391 << 24),
    _f2(2557866, 23705959 << 25), _f2(6572718, 173475635 << 24),
    -_f2(2666354, 4022017967 << 19), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^14, polynomial in n of order 15
    _f2(54399, 8224347 << 28), -_f2(665621, 1410497 << 30),
    _f2(4527846, 10561865 << 28), -_f2(20181039, 1593975 << 31),
    _f2(63299017, 4532479 << 28), -_f2(144047710, 3737571 << 30),
    _f2(238046961, 3527085 << 28), -_f2(274611219, 485845 << 32),
    _f2(189061064, 3080067 << 28), -_f2(9459768, 127989 << 30),
    -_f2(141083601, 3627343 << 28), _f2(166868825, 1278147 << 31),
    -_f2(97711641, 16702617 << 28), _f2(28303864, 4120681 << 30),
    -_f2(1707991, 14300715 << 28), -_f2(691965, 964491519 << 21),
    _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^13, polynomial in n of order 16
    -_f(394848061 << 27), -_f(277855615551 << 23),
    _f2(21507, 221049445 << 24), -_f2(280152, 15918397 << 23),
    _f2(2046623, 76965257 << 25), -_f2(9908745, 30179611 << 23),
    _f2(34287090, 9035647 << 24), -_f2(88042794, 304571673 << 23),
    _f2(170266683, 41403331 << 26), -_f2(246546803, 514564215 << 23),
    _f2(257558635, 22204697 << 24), -_f2(171596509, 74164853 << 23),
    _f2(32098211, 100286083 << 25), _f2(71621283, 185724333 << 23),
    -_f2(91026815, 208301517 << 24), _f2(52421624, 501338287 << 23),
    -_f2(12919309, 7987587551 << 18), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^12, polynomial in n of order 17
    -_f(2506701 << 25), -_f(1595211 << 29), -_f(414133331 << 25),
    -_f(9611154693 << 26), _f2(6318, 129560535 << 25),
    -_f2(88018, 7994433 << 27), _f2(693686, 99032081 << 25),
    -_f2(3663195, 37640223 << 26), _f2(14022738, 83451835 << 25),
    -_f2(40598902, 6803915 << 28), _f2(90961965, 119128629 << 25),
    -_f2(159220781, 788729 << 26), _f2(217217490, 112380639 << 25),
    -_f2(227284915, 26366059 << 27), _f2(176075660, 9208089 << 25),
    -_f2(94610548, 45670931 << 26), _f2(31201351, 21556291 << 25),
    -_f2(4712704, 700509149 << 19), _f2(94341681461, 0x436c57c191ed7),
    # C4[11], coeff of eps^11, polynomial in n of order 18
    -_f(13041 << 24), -_f(166957 << 23), -_f(82777 << 27),
    -_f(14154867 << 23), -_f(121102751 << 24), -_f(11919970777 << 23),
    _f(140288886837 << 25), -_f2(15649, 252654239 << 23),
    _f2(133731, 225409843 << 24), -_f2(773734, 115698949 << 23),
    _f2(3285982, 23309223 << 26), -_f2(10716783, 181102411 << 23),
    _f2(27557442, 232845957 << 24), -_f2(56645854, 420384689 << 23),
    _f2(93299054, 125728615 << 25), -_f2(121534295, 132369527 << 23),
    _f2(119605179, 120525655 << 24), -_f2(75403265, 163638237 << 23),
    _f2(21207168, 6304582341 << 18), _f2(94341681461, 0x436c57c191ed7),
    # C4[12], coeff of eps^29, polynomial in n of order 0
    _f(2113 << 23), _f(0x495846bc80a035),
    # C4[12], coeff of eps^28, polynomial in n of order 1
    -_f(5059597 << 25), -_f(23775299 << 22),
    _f2(61953, 0x75e619a89ce07),
    # C4[12], coeff of eps^27, polynomial in n of order 2
    _f(30823201 << 29), -_f(55301563 << 28), _f(131431881 << 24),
    _f2(497138, 0xbe8dd4238d2e7),
    # C4[12], coeff of eps^26, polynomial in n of order 3
    _f(8059635627 << 28), -_f(757042391 << 29), -_f(311216327 << 28),
    -_f(7273579 << 33), _f2(21376966, 0x1d2a1f8b6ccd),
    # C4[12], coeff of eps^25, polynomial in n of order 4
    _f2(590308, 751003 << 30), _f2(77521, 16047653 << 28),
    _f2(426657, 125003 << 29), -_f2(306166, 5244457 << 28),
    _f2(37995, 207060411 << 24), _f2(34181768645, 0x62a1b07dc9473),
    # C4[12], coeff of eps^24, polynomial in n of order 5
    -_f2(1599658, 2394579 << 27), -_f2(2671318, 5123391 << 29),
    _f2(3256460, 16377243 << 27), -_f2(261261, 3982303 << 28),
    -_f2(106562, 1204279 << 27), -_f2(120793, 1029973 << 24),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^23, polynomial in n of order 6
    _f2(1248773, 4542469 << 29), -_f2(2889741, 9813249 << 28),
    _f2(234885, 3135591 << 30), _f2(66922, 9908313 << 28),
    _f2(755418, 635863 << 29), -_f2(419245, 13200621 << 28),
    _f2(41213, 192739239 << 24), _f2(34181768645, 0x62a1b07dc9473),
    # C4[12], coeff of eps^22, polynomial in n of order 7
    _f2(20885911, 4938503 << 28), -_f2(1107830, 1234733 << 29),
    -_f2(2370377, 3771643 << 28), -_f2(6561451, 624527 << 30),
    _f2(4279851, 10797315 << 28), _f2(210660, 3761905 << 29),
    -_f2(68724, 2033407 << 28), -_f2(224555, 1152577 << 29),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^21, polynomial in n of order 8
    -_f2(5562062, 38325 << 31), _f2(7741765, 4470897 << 28),
    _f2(17399328, 6149297 << 29), -_f2(10890962, 10744893 << 28),
    -_f2(2368414, 446197 << 30), -_f2(891562, 9146315 << 28),
    _f2(4183586, 393403 << 29), -_f2(1730085, 6072185 << 28),
    _f2(120797, 18742483 << 24), _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^20, polynomial in n of order 9
    _f2(3332722, 179104103 << 24), -_f2(54245156, 21887465 << 27),
    _f2(18428910, 34326153 << 24), _f2(12293411, 106053413 << 25),
    _f2(4024929, 78680811 << 24), -_f2(14528270, 1262953 << 26),
    _f2(4350569, 36952333 << 24), _f2(1251271, 92345015 << 25),
    _f2(191236, 5049199 << 24), -_f2(439124, 1983321823 << 21),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^19, polynomial in n of order 10
    _f2(117817828, 9756529 << 27), _f2(29304818, 21859033 << 26),
    -_f2(33857646, 3348243 << 29), -_f2(34756825, 30241097 << 26),
    _f2(40576649, 11012471 << 27), -_f2(1809964, 37385419 << 26),
    -_f2(7014724, 9945043 << 28), -_f2(6163099, 62399597 << 26),
    _f2(7950550, 2557949 << 27), -_f2(2323999, 3159279 << 26),
    _f2(80031, 1030811061 << 22), _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^18, polynomial in n of order 11
    _f2(37677439, 43610729 << 26), -_f2(212417438, 31724009 << 27),
    _f2(188774384, 60946099 << 26), -_f2(37276694, 6182933 << 29),
    -_f2(44097735, 31570179 << 26), _f2(5696664, 10497345 << 27),
    _f2(38054298, 7154503 << 26), -_f2(24525159, 12952085 << 28),
    -_f2(350073, 57822319 << 26), _f2(2901654, 18012267 << 27),
    _f2(1319354, 63457243 << 26), -_f2(941373, 59576227 << 26),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^17, polynomial in n of order 12
    -_f2(253394431, 1462439 << 28), _f2(237669216, 409221 << 26),
    -_f2(76296320, 28335185 << 27), -_f2(133872864, 17217849 << 26),
    _f2(218174046, 2622387 << 29), -_f2(127998192, 57914967 << 26),
    _f2(363472, 30718057 << 27), _f2(32681168, 4189163 << 26),
    _f2(4677048, 16088179 << 28), -_f2(25857930, 7142835 << 26),
    _f2(14914891, 9312419 << 27), -_f2(2731797, 52301425 << 26),
    -_f2(76352, 726189181 << 22), _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^16, polynomial in n of order 13
    -_f2(53643409, 97684423 << 25), _f2(127408278, 3174879 << 27),
    -_f2(219370358, 126672641 << 25), _f2(262176902, 49024297 << 26),
    -_f2(182579118, 132254331 << 25), -_f2(3956056, 15289739 << 28),
    _f2(167880537, 57011019 << 25), -_f2(188895775, 46555265 << 26),
    _f2(91621970, 25449425 << 25), -_f2(762367, 26232331 << 27),
    -_f2(18049661, 12932969 << 25), _f2(839158, 10679509 << 26),
    _f2(6440415, 29973277 << 25), -_f2(2428550, 982597961 << 22),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^15, polynomial in n of order 14
    -_f2(1786573, 13634499 << 27), _f2(8939902, 21088027 << 26),
    -_f2(31907799, 6174271 << 28), _f2(84216248, 4944333 << 26),
    -_f2(166330228, 11364729 << 27), _f2(242706491, 8863071 << 26),
    -_f2(246937724, 7698133 << 29), _f2(139651898, 50060433 << 26),
    _f2(29493809, 19297617 << 27), -_f2(148014628, 18656733 << 26),
    _f2(151165884, 622571 << 28), -_f2(81883041, 55488299 << 26),
    _f2(21903841, 15379355 << 27), -_f2(792996, 14574745 << 26),
    -_f2(644309, 457907141 << 22), _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^14, polynomial in n of order 15
    -_f2(6504, 28793619 << 26), _f2(93075, 33271365 << 27),
    -_f2(752118, 6462873 << 26), _f2(4061558, 11832697 << 28),
    -_f2(15840997, 35193887 << 26), _f2(46482714, 19239327 << 27),
    -_f2(104709924, 13039269 << 26), _f2(181860520, 7828435 << 29),
    -_f2(240159499, 36173099 << 26), _f2(229992094, 10037497 << 27),
    -_f2(136854274, 43911089 << 26), _f2(9990763, 2550227 << 28),
    _f2(74520212, 5689801 << 26), -_f2(84057599, 709549 << 27),
    _f2(46786776, 54603587 << 26), -_f2(11406945, 39298831 << 26),
    _f2(102545305936, 0x27e511795bd59),
    # C4[12], coeff of eps^13, polynomial in n of order 16
    _f(1030055 << 30), _f(829418525 << 26), -_f(19924010015 << 27),
    _f2(9050, 10804695 << 26), -_f2(78488, 9283787 << 28),
    _f2(459151, 22444081 << 26), -_f2(1962716, 17985613 << 27),
    _f2(6408338, 7820523 << 26), -_f2(16395176, 1626457 << 29),
    _f2(33311385, 35536325 << 26), -_f2(53946341, 7054843 << 27),
    _f2(69201696, 61410431 << 26), -_f2(69012103, 3773337 << 28),
    _f2(51544754, 7834329 << 26), -_f2(26961871, 12889641 << 27),
    _f2(8722958, 26104467 << 26), -_f2(1300056, 320360129 << 22),
    _f2(34181768645, 0x62a1b07dc9473),
    # C4[12], coeff of eps^12, polynomial in n of order 17
    _f(127075 << 24), _f(91195 << 28), _f(26902525 << 24),
    _f(715607165 << 25), -_f(73094160425 << 24),
    _f2(4440, 35913265 << 26), -_f2(41519, 978831 << 24),
    _f2(264211, 112928967 << 25), -_f2(1241795, 175695285 << 24),
    _f2(4515620, 18853435 << 27), -_f2(13069247, 261014043 << 24),
    _f2(30619380, 129358865 << 25), -_f2(58537051, 226171969 << 24),
    _f2(91194564, 65482939 << 26), -_f2(113993206, 58979239 << 24),
    _f2(109036979, 115740763 << 25), -_f2(67602927, 138149837 << 24),
    _f2(18850816, 700509149 << 21), _f2(102545305936, 0x27e511795bd59),
    # C4[13], coeff of eps^29, polynomial in n of order 0
    -_f(634219 << 23), _f2(3193, 0x402148867236b),
    # C4[13], coeff of eps^28, polynomial in n of order 1
    -_f(400561 << 32), _f(1739049 << 27), _f2(66909, 0xbcc54ee94d445),
    # C4[13], coeff of eps^27, polynomial in n of order 2
    -_f(6387996953 << 29), -_f(3461245957 << 28), -_f(49206438547 << 24),
    _f2(286172946, 0xcc6f5fc7e64c9),
    # C4[13], coeff of eps^26, polynomial in n of order 3
    _f(7296571113 << 30), _f2(10661, 1488313 << 31),
    -_f2(6836, 2507629 << 30), _f(103233906747 << 25),
    _f2(900397808, 0x384bb07b32421),
    # C4[13], coeff of eps^25, polynomial in n of order 4
    -_f2(1030602, 1434287 << 30), _f2(976249, 6303335 << 28),
    -_f2(29214, 7243007 << 29), -_f2(27193, 4360723 << 28),
    -_f2(41363, 170006437 << 24), _f2(36916310137, 0x41f43bb0c949),
    # C4[13], coeff of eps^24, polynomial in n of order 5
    -_f2(2597630, 109963 << 31), -_f2(46366, 88065 << 33),
    _f(835763379 << 31), _f2(754229, 667751 << 32),
    -_f2(376195, 1000319 << 31), _f2(33027, 62908623 << 26),
    _f2(36916310137, 0x41f43bb0c949),
    # C4[13], coeff of eps^23, polynomial in n of order 6
    _f2(1907767, 7512493 << 29), -_f2(1322539, 10099505 << 28),
    -_f2(6889396, 2784065 << 30), _f2(3566231, 12064393 << 28),
    _f2(392016, 1668111 << 29), -_f2(16024, 9677725 << 28),
    -_f2(223530, 46890859 << 24), _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^22, polynomial in n of order 7
    _f2(2768819, 1533979 << 30), _f2(18682895, 1051157 << 31),
    -_f2(7962826, 2061455 << 30), -_f2(3008388, 501585 << 32),
    -_f2(1419492, 411945 << 30), _f2(4033867, 1208903 << 31),
    -_f2(1511425, 98131 << 30), _f2(90538, 115806565 << 25),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^21, polynomial in n of order 8
    -_f2(51332124, 214119 << 31), _f2(7579765, 3153587 << 28),
    _f2(12475441, 1655707 << 29), _f2(7005177, 5256105 << 28),
    -_f2(13679833, 119207 << 30), _f2(3016909, 4530623 << 28),
    _f2(1323928, 2024201 << 29), _f2(284896, 6925237 << 28),
    -_f2(424636, 138679005 << 24), _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^20, polynomial in n of order 9
    _f2(47250711, 14679 << 32), -_f2(18472981, 62575 << 35),
    -_f2(42459575, 893863 << 32), _f2(33307537, 106651 << 33),
    _f2(3060800, 842635 << 32), -_f2(5867540, 102671 << 34),
    -_f2(6941741, 849331 << 32), _f2(7324844, 423881 << 33),
    -_f2(1953157, 771073 << 32), _f2(46148, 28524089 << 27),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^19, polynomial in n of order 10
    -_f2(218954922, 27801141 << 27), _f2(144947317, 36456347 << 26),
    -_f2(2491117, 129025 << 29), -_f2(43746541, 53566187 << 26),
    -_f2(5414513, 33128531 << 27), _f2(38475112, 39418671 << 26),
    -_f2(19653214, 3660993 << 28), -_f2(2031714, 8235735 << 26),
    _f2(2517568, 31068687 << 27), _f2(1433299, 8158787 << 26),
    -_f2(884985, 911850811 << 22), _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^18, polynomial in n of order 11
    _f2(186784429, 10521821 << 28), -_f2(7066121, 6171767 << 29),
    -_f2(168709085, 159265 << 28), _f2(199772613, 1997709 << 31),
    -_f2(91000398, 5796399 << 28), -_f2(16385586, 3500385 << 29),
    _f2(28845005, 1198547 << 28), _f2(9523912, 3994797 << 30),
    -_f2(24921512, 7172347 << 28), _f2(12920997, 2065141 << 29),
    -_f2(2137442, 11262329 << 28), -_f2(100773, 90157665 << 23),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^17, polynomial in n of order 12
    _f2(153014747, 10291963 << 28), -_f2(229746959, 27043849 << 26),
    _f2(237324258, 14238909 << 27), -_f2(127910449, 9268979 << 26),
    -_f2(52661288, 2811671 << 29), _f2(178449477, 48064707 << 26),
    -_f2(166899831, 11458293 << 27), _f2(67870692, 24951769 << 26),
    _f2(7326612, 206249 << 28), -_f2(16569622, 39442673 << 26),
    -_f2(550002, 21806887 << 27), _f2(6246699, 62490405 << 26),
    -_f2(2219585, 747027389 << 22), _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^16, polynomial in n of order 13
    _f2(15145062, 3114639 << 29), -_f2(45473383, 71569 << 31),
    _f2(104280194, 4043033 << 29), -_f2(182691434, 3191599 << 30),
    _f2(238813543, 4302931 << 29), -_f2(215430056, 686779 << 32),
    _f2(95643898, 4170781 << 29), _f2(58502156, 871831 << 30),
    -_f2(149008930, 6169513 << 29), _f2(135928257, 1117477 << 31),
    -_f2(68778720, 227103 << 29), _f2(17013972, 3743517 << 30),
    -_f2(171458, 5381733 << 29), -_f2(594747, 43724235 << 24),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^15, polynomial in n of order 14
    _f2(268265, 12727175 << 27), -_f2(1599130, 17232215 << 26),
    _f2(6942204, 4883571 << 28), -_f2(22914299, 20494129 << 26),
    _f2(58880733, 8011269 << 27), -_f2(118985431, 7979051 << 26),
    _f2(188592645, 4054545 << 29), -_f2(229762161, 35295621 << 26),
    _f2(203148724, 31300867 << 27), -_f2(107385077, 53637247 << 26),
    -_f2(6680614, 2406191 << 28), _f2(75281884, 8527271 << 26),
    -_f2(77627899, 32310399 << 27), _f2(42028799, 34263981 << 26),
    -_f2(10160264, 35032709 << 22), _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^14, polynomial in n of order 15
    _f(8350913025 << 28), -_f2(8241, 2495877 << 29),
    _f2(78008, 13111987 << 28), -_f2(500800, 4045265 << 30),
    _f2(2364509, 9424021 << 28), -_f2(8593646, 4773407 << 29),
    _f2(24709323, 11418567 << 28), -_f2(57114100, 2066875 << 31),
    _f2(106928260, 4889705 << 28), -_f2(162113251, 5033977 << 29),
    _f2(197269922, 8596123 << 28), -_f2(188736396, 4070811 << 30),
    _f2(136566807, 16720509 << 28), -_f2(69783667, 938643 << 29),
    _f2(22203894, 1359919 << 28), -_f2(3271109, 212531129 << 23),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[13], coeff of eps^13, polynomial in n of order 16
    -_f(94185 << 30), -_f(86179275 << 26), _f(2372802705 << 27),
    -_f(83726038305 << 26), _f2(12668, 1555717 << 28),
    -_f2(87922, 39994007 << 26), _f2(452934, 19637187 << 27),
    -_f2(1815855, 62281965 << 26), _f2(5835571, 1574167 << 29),
    -_f2(15318374, 24668899 << 26), _f2(33211265, 14617205 << 27),
    -_f2(59722012, 27257657 << 26), _f2(88729847, 57943 << 28),
    -_f2(107054489, 21433519 << 26), _f2(99917523, 12239271 << 27),
    -_f2(61060708, 48513541 << 26), _f2(16900731, 943456205 << 22),
    _f2(110748930411, 0xc5dcb3125bdb),
    # C4[14], coeff of eps^29, polynomial in n of order 0
    _f(41 << 28), _f(0x3fbc634a12a6b1),
    # C4[14], coeff of eps^28, polynomial in n of order 1
    -_f(6907093 << 31), -_f(59887787 << 28),
    _f2(5739014, 0x909af11944e4b),
    # C4[14], coeff of eps^27, polynomial in n of order 2
    _f2(3432, 499601 << 33), -_f(2083199471 << 32), _f(3406572267 << 28),
    _f2(307370942, 0xdb94118adae9f),
    # C4[14], coeff of eps^26, polynomial in n of order 3
    _f2(287986, 4314073 << 29), _f2(5344, 3636147 << 30),
    -_f2(6205, 2906637 << 29), -_f2(13964, 12467885 << 26),
    _f2(13216950542, 0xe1def252c54b5),
    # C4[14], coeff of eps^25, polynomial in n of order 4
    -_f2(258061, 515595 << 33), -_f2(74790, 1657665 << 31),
    _f2(745027, 493173 << 32), -_f2(337382, 84843 << 31),
    _f2(26418, 5099583 << 27), _f2(39650851628, 0xa59cd6f84fe1f),
    # C4[14], coeff of eps^24, polynomial in n of order 5
    -_f2(100052, 3082133 << 30), -_f2(6991386, 428305 << 32),
    _f2(2902871, 3549453 << 30), _f2(514674, 1320943 << 31),
    _f2(32543, 4070319 << 30), -_f2(220557, 2292103 << 27),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^23, polynomial in n of order 6
    _f2(6249633, 975799 << 32), -_f2(1750517, 1286063 << 31),
    -_f2(1090661, 209219 << 33), -_f2(631632, 1802089 << 31),
    _f2(1285387, 761149 << 32), -_f2(440347, 2020899 << 31),
    _f2(22303, 14762615 << 27), _f2(39650851628, 0xa59cd6f84fe1f),
    # C4[14], coeff of eps^22, polynomial in n of order 7
    -_f2(1155507, 7367607 << 29), _f2(11090657, 3295693 << 30),
    _f2(9416360, 3921899 << 29), -_f2(12562252, 1614097 << 31),
    _f2(1905484, 7990669 << 29), _f2(1324142, 2135535 << 30),
    _f2(362851, 6226223 << 29), -_f2(408795, 45924241 << 26),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^21, polynomial in n of order 8
    -_f2(2556392, 83451 << 35), -_f2(45924405, 628445 << 32),
    _f2(25722566, 76303 << 33), _f2(6427311, 738073 << 32),
    -_f2(4460754, 79355 << 34), -_f2(7474566, 842481 << 32),
    _f2(6714086, 421381 << 33), -_f2(1643964, 835835 << 32),
    _f2(21175, 2825543 << 28), _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^20, polynomial in n of order 9
    _f2(101794762, 1213867 << 31), _f2(21384252, 53331 << 34),
    -_f2(38294895, 814203 << 31), -_f2(14666563, 397319 << 32),
    _f2(37283642, 1395551 << 31), -_f2(15279397, 125869 << 33),
    -_f2(3174330, 433863 << 31), _f2(2115734, 458067 << 32),
    _f2(1510128, 1624339 << 31), -_f2(831539, 10478291 << 28),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^19, polynomial in n of order 10
    _f2(50295468, 469581 << 33), -_f2(186185623, 531407 << 32),
    _f2(174713425, 41641 << 35), -_f2(59412258, 3489 << 32),
    -_f2(26728127, 294149 << 33), _f2(23787374, 31373 << 32),
    _f2(13262792, 54953 << 34), -_f2(23669724, 520005 << 32),
    _f2(11190603, 172969 << 33), -_f2(1670792, 243479 << 32),
    -_f2(115324, 7271069 << 28), _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^18, polynomial in n of order 11
    -_f2(229751836, 26450113 << 27), _f2(205122059, 12799441 << 28),
    -_f2(76881886, 7573243 << 27), -_f2(89213757, 3943587 << 30),
    _f2(179505441, 31406283 << 27), -_f2(144430080, 11541225 << 28),
    _f2(48475411, 26026641 << 27), _f2(12591449, 3614557 << 29),
    -_f2(14798010, 26226089 << 27), -_f2(1654995, 15989667 << 28),
    _f2(6017860, 18394141 << 27), -_f2(2035659, 55899187 << 24),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^17, polynomial in n of order 12
    -_f2(60003627, 419631 << 31), _f2(122260158, 890121 << 29),
    -_f2(193015194, 2586489 << 30), _f2(228332901, 6912147 << 29),
    -_f2(182676146, 109669 << 32), _f2(57596630, 2823965 << 29),
    _f2(79437172, 3055697 << 30), -_f2(146103578, 5035353 << 29),
    _f2(121669517, 1691035 << 31), -_f2(57926620, 1249999 << 29),
    _f2(13245099, 2677787 << 30), _f2(250593, 3348667 << 29),
    -_f2(546524, 56364575 << 25), _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^16, polynomial in n of order 13
    -_f2(2910026, 15317989 << 28), _f2(10671028, 169493 << 30),
    -_f2(30788418, 3245427 << 28), _f2(70862414, 261923 << 29),
    -_f2(130581700, 1010945 << 28), _f2(191189814, 642567 << 31),
    -_f2(216782974, 13106831 << 28), _f2(177827671, 7710997 << 29),
    -_f2(82572754, 6587933 << 28), -_f2(19188450, 2518521 << 30),
    _f2(74652545, 11169877 << 28), -_f2(71762036, 443641 << 29),
    _f2(37978089, 1743047 << 28), -_f2(9119456, 66783679 << 25),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^15, polynomial in n of order 14
    -_f2(25313, 471763 << 30), _f2(176943, 4508751 << 29),
    -_f2(914488, 1483519 << 31), _f2(3661023, 8037561 << 29),
    -_f2(11683077, 3602217 << 30), _f2(30253421, 7276067 << 29),
    -_f2(64215578, 725077 << 32), _f2(112133608, 2030221 << 29),
    -_f2(160624032, 1744767 << 30), _f2(186713478, 2165751 << 29),
    -_f2(172286017, 2016853 << 31), _f2(121247637, 6964321 << 29),
    -_f2(60696359, 459733 << 30), _f2(19031909, 1781195 << 29),
    -_f2(2775486, 102023215 << 25), _f2(118952554885, 0xf0d684e8efa5d),
    # C4[14], coeff of eps^14, polynomial in n of order 15
    -_f(614557125 << 27), _f(5831464275 << 28),
    -_f2(3808, 17097199 << 27), _f2(28626, 5026047 << 29),
    -_f2(160361, 32462873 << 27), _f2(702411, 15495849 << 28),
    -_f2(2480054, 13665347 << 27), _f2(7201343, 703573 << 30),
    -_f2(17420896, 11395693 << 27), _f2(35365729, 6899711 << 28),
    -_f2(60346284, 10497687 << 27), _f2(86059048, 7827541 << 29),
    -_f2(100689087, 8447169 << 27), _f2(91987561, 3237205 << 28),
    -_f2(55509735, 6799595 << 27), _f2(15265177, 48513541 << 24),
    _f2(118952554885, 0xf0d684e8efa5d),
    # C4[15], coeff of eps^29, polynomial in n of order 0
    -_f(204761 << 28), _f2(20426, 0xaa7b82b97d24f),
    # C4[15], coeff of eps^28, polynomial in n of order 1
    -_f(34699 << 42), _f(26415501 << 29), _f2(6134808, 0xac3bb24726559),
    # C4[15], coeff of eps^27, polynomial in n of order 2
    _f2(16894, 439 << 40), -_f2(3396, 5539 << 38),
    -_f2(13997, 7293149 << 28), _f2(14128464373, 0x6d08ce11dbba7),
    # C4[15], coeff of eps^26, polynomial in n of order 3
    -_f2(50643, 63489 << 36), _f2(243167, 8553 << 37),
    -_f2(100839, 3467 << 36), _f2(7018, 548741 << 30),
    _f2(14128464373, 0x6d08ce11dbba7),
    # C4[15], coeff of eps^25, polynomial in n of order 4
    -_f2(6907413, 21379 << 36), _f2(2301071, 198931 << 34),
    _f2(591806, 32973 << 35), _f2(76262, 38289 << 34),
    -_f2(216244, 23833777 << 27), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^24, polynomial in n of order 5
    -_f2(2869395, 52521 << 36), -_f2(3255913, 8819 << 38),
    -_f2(2304160, 33823 << 36), _f2(3661540, 28261 << 37),
    -_f2(1155441, 18213 << 36), _f2(48366, 13607837 << 28),
    _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^23, polynomial in n of order 6
    _f2(8754539, 110435 << 35), _f2(11218727, 249609 << 34),
    -_f2(11298141, 31087 << 36), _f2(996220, 194783 << 34),
    _f2(1275763, 41633 << 35), _f2(426630, 95573 << 34),
    -_f2(392368, 19533817 << 27), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^22, polynomial in n of order 7
    -_f2(46020147, 1607 << 36), _f2(18483914, 31121 << 37),
    _f2(8546239, 40923 << 36), -_f2(2972379, 4277 << 38),
    -_f2(7799822, 49315 << 36), _f2(6131851, 9051 << 37),
    -_f2(1385578, 60289 << 36), _f2(2743, 3362879 << 30),
    _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^21, polynomial in n of order 8
    _f2(36025526, 1303 << 40), -_f2(30131090, 26093 << 37),
    -_f2(21835156, 15459 << 38), _f2(35026415, 31673 << 37),
    -_f2(11464406, 5705 << 39), -_f2(3907909, 12145 << 37),
    _f2(1722090, 1439 << 38), _f2(1557916, 18869 << 37),
    -_f2(781446, 6381283 << 28), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^20, polynomial in n of order 9
    -_f2(190262221, 2387 << 40), _f2(146996287, 1227 << 41),
    -_f2(33573688, 3541 << 40), -_f2(32294922, 2067 << 39),
    _f2(18331180, 2115 << 40), _f2(16022794, 2331 << 40),
    -_f2(22246846, 3383 << 40), _f2(9695242, 4143 << 39),
    -_f2(1302304, 2671 << 40), -_f2(123235, 5577019 << 29),
    _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^19, polynomial in n of order 10
    _f2(169057636, 9637 << 37), -_f2(31515275, 17095 << 36),
    -_f2(115123722, 7359 << 39), _f2(174060780, 58071 << 36),
    -_f2(122862790, 20125 << 37), _f2(32880337, 12981 << 36),
    _f2(15824026, 705 << 38), -_f2(12943852, 57005 << 36),
    -_f2(2522257, 22495 << 37), _f2(5771316, 18225 << 36),
    -_f2(1873338, 7714415 << 28), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^18, polynomial in n of order 11
    _f2(137352006, 26079 << 36), -_f2(197705648, 26891 << 37),
    _f2(213128803, 51077 << 36), -_f2(150461460, 3135 << 39),
    _f2(25445949, 34251 << 36), _f2(93962136, 11667 << 37),
    -_f2(140732252, 24207 << 36), _f2(108614245, 6273 << 38),
    -_f2(48923563, 62665 << 36), _f2(10316934, 1969 << 37),
    _f2(535285, 33757 << 36), -_f2(501186, 3348667 << 30),
    _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^17, polynomial in n of order 12
    _f2(15155809, 205049 << 34), -_f2(39104030, 957115 << 32),
    _f2(81967212, 139599 << 33), -_f2(139468172, 993913 << 32),
    _f2(190415844, 61587 << 35), -_f2(202311488, 967447 << 32),
    _f2(154439958, 403401 << 33), -_f2(61783996, 889045 << 32),
    -_f2(28504911, 66989 << 34), _f2(73132006, 1030157 << 32),
    -_f2(66442314, 293949 << 33), _f2(34502754, 346959 << 32),
    -_f2(8240730, 128798053 << 25), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[15], coeff of eps^16, polynomial in n of order 13
    _f2(114172, 142577 << 34), -_f2(499141, 33119 << 36),
    _f2(1750098, 174183 << 34), -_f2(5016097, 114721 << 35),
    _f2(11893006, 66221 << 34), -_f2(23470909, 19093 << 37),
    _f2(38591591, 188131 << 34), -_f2(52613301, 65223 << 35),
    _f2(58753809, 139305 << 34), -_f2(52512562, 23925 << 36),
    _f2(36059714, 158111 << 34), -_f2(17726185, 93229 << 35),
    _f2(5486676, 138853 << 34), -_f2(792996, 14574745 << 26),
    _f2(42385393120, 0x471a6a35932f5),
    # C4[15], coeff of eps^15, polynomial in n of order 14
    _f(592706205 << 33), -_f2(9147, 839013 << 32),
    _f2(55355, 240865 << 34), -_f2(262940, 644275 << 32),
    _f2(1011310, 29095 << 33), -_f2(3215965, 1023905 << 32),
    _f2(8575909, 35467 << 35), -_f2(19352216, 329839 << 32),
    _f2(37124659, 455473 << 33), -_f2(60529336, 778589 << 32),
    _f2(83288367, 93771 << 34), -_f2(94856196, 165035 << 32),
    _f2(85043486, 110139 << 33), -_f2(50751757, 943257 << 32),
    _f2(13877433, 107462891 << 25), _f2(127156179360, 0xd54f3ea0b98df),
    # C4[16], coeff of eps^29, polynomial in n of order 0
    _f(553 << 31), _f(0x292ecb9a960d27d1),
    # C4[16], coeff of eps^28, polynomial in n of order 1
    -_f(61453 << 36), -_f(4754645 << 34),
    _f2(19591808, 0x57955a5f17535),
    # C4[16], coeff of eps^27, polynomial in n of order 2
    _f2(33770, 14237 << 36), -_f2(12917, 115767 << 35),
    _f(1665987897 << 31), _f2(2148568314, 0xda506166fe05f),
    # C4[16], coeff of eps^26, polynomial in n of order 3
    _f2(1765351, 9719 << 36), _f2(634098, 16193 << 37),
    _f2(114937, 5021 << 36), -_f2(211035, 902511 << 32),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^25, polynomial in n of order 4
    -_f2(3041817, 11535 << 37), -_f2(2643315, 63657 << 35),
    _f2(3458443, 225 << 36), -_f2(1011407, 29251 << 35),
    _f2(33755, 354965 << 31), _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^24, polynomial in n of order 5
    _f2(12443946, 111847 << 35), -_f2(9978547, 23661 << 37),
    _f2(264818, 24689 << 35), _f2(1196082, 9587 << 36),
    _f2(477961, 17339 << 35), -_f2(375862, 243659 << 32),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^23, polynomial in n of order 6
    _f2(11971225, 59849 << 36), _f2(9677599, 56595 << 35),
    -_f2(1515717, 2317 << 37), -_f2(7956047, 112667 << 35),
    _f2(5585704, 62147 << 36), -_f2(1169086, 64041 << 35),
    -_f2(10840, 1435009 << 31), _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^22, polynomial in n of order 7
    -_f2(20905609, 47207 << 36), -_f2(27003899, 18815 << 37),
    _f2(32128586, 62715 << 36), -_f2(8207156, 6437 << 38),
    -_f2(4335741, 20931 << 36), _f2(1351161, 14763 << 37),
    _f2(1583200, 44703 << 36), -_f2(734819, 355141 << 32),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^21, polynomial in n of order 8
    _f2(119271221, 13241 << 38), -_f2(13185134, 117885 << 35),
    -_f2(34424757, 12741 << 36), _f2(12971898, 62105 << 35),
    _f2(17958221, 23929 << 37), -_f2(20751983, 45233 << 35),
    _f2(8405753, 5609 << 36), -_f2(1009805, 84123 << 35),
    -_f2(126669, 998091 << 31), _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^20, polynomial in n of order 9
    _f2(7281953, 46177 << 36), -_f2(132136826, 3687 << 39),
    _f2(164419146, 28463 << 36), -_f2(102943145, 13301 << 37),
    _f2(20499773, 15997 << 36), _f2(17609391, 14489 << 38),
    -_f2(11127728, 9397 << 36), -_f2(3194109, 31911 << 37),
    _f2(5518515, 63129 << 36), -_f2(1729623, 95963 << 34),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^19, polynomial in n of order 10
    -_f2(197461925, 11965 << 36), _f2(194820269, 1667 << 35),
    -_f2(119937281, 937 << 38), -_f2(1213288, 24915 << 35),
    _f2(103481944, 52469 << 36), -_f2(133891976, 7945 << 35),
    _f2(96822293, 18071 << 37), -_f2(41434588, 121951 << 35),
    _f2(8025452, 27431 << 36), _f2(724406, 126187 << 35),
    -_f2(459367, 1295029 << 31), _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^18, polynomial in n of order 11
    -_f2(47525285, 62545 << 36), _f2(91887649, 22453 << 37),
    -_f2(145781941, 19979 << 36), _f2(186991182, 8001 << 39),
    -_f2(187151585, 35749 << 36), _f2(133148350, 20179 << 37),
    -_f2(44425461, 13151 << 36), -_f2(35371425, 9983 << 38),
    _f2(71056997, 38023 << 36), -_f2(61631567, 21647 << 37),
    _f2(31499493, 50637 << 36), -_f2(7491423, 758351 << 32),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^17, polynomial in n of order 12
    -_f2(2259631, 24697 << 37), _f2(7098981, 78723 << 35),
    -_f2(18582556, 8879 << 36), _f2(40852668, 12369 << 35),
    -_f2(75692831, 12691 << 38), _f2(118080654, 84927 << 35),
    -_f2(154130872, 52073 << 36), _f2(166118829, 74381 << 35),
    -_f2(144327913, 2259 << 37), _f2(96964720, 98683 << 35),
    -_f2(46899246, 18595 << 36), _f2(14349769, 50505 << 35),
    -_f2(2057503, 1465135 << 31), _f2(135359803835, 0xb9c7f85883761),
    # C4[16], coeff of eps^16, polynomial in n of order 13
    -_f2(18695, 264305 << 33), _f2(95700, 8265 << 35),
    -_f2(398136, 419847 << 33), _f2(1375381, 177071 << 34),
    -_f2(4004787, 429789 << 33), _f2(9930000, 13635 << 36),
    -_f2(21101250, 231795 << 33), _f2(38532718, 52073 << 34),
    -_f2(60367925, 93257 << 33), _f2(80490566, 118467 << 35),
    -_f2(89511061, 246751 << 33), _f2(78923731, 162339 << 34),
    -_f2(46636750, 263349 << 33), _f2(12687939, 1991833 << 30),
    _f2(135359803835, 0xb9c7f85883761),
    # C4[17], coeff of eps^29, polynomial in n of order 0
    -_f(280331 << 31), _f2(154847, 0x4e6e7be138cdb),
    # C4[17], coeff of eps^28, polynomial in n of order 1
    -_f(82431 << 38), _f(142069 << 33), _f2(989485, 0x4511e2f2b39a3),
    # C4[17], coeff of eps^27, polynomial in n of order 2
    _f2(30957, 2723 << 36), _f2(7080, 38071 << 35),
    -_f2(9773, 1986585 << 31), _f2(6836353729, 0x13b9f01928417),
    # C4[17], coeff of eps^26, polynomial in n of order 3
    -_f2(138771, 28785 << 37), _f2(154910, 14439 << 38),
    -_f2(42193, 29611 << 37), _f(1108797915 << 32),
    _f2(6836353729, 0x13b9f01928417),
    # C4[17], coeff of eps^25, polynomial in n of order 4
    -_f2(1238256, 21701 << 37), -_f2(44811, 81027 << 35),
    _f2(156785, 14859 << 36), _f2(74079, 77407 << 35),
    -_f2(51372, 1082481 << 31), _f2(20509061187, 0x3b2dd04b78c45),
    # C4[17], coeff of eps^24, polynomial in n of order 5
    _f2(10057115, 7495 << 39), -_f2(158283, 1579 << 41),
    -_f2(7978477, 2703 << 39), _f2(5079175, 3021 << 40),
    -_f2(987192, 2101 << 39), -_f2(20806, 242401 << 34),
    _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^23, polynomial in n of order 6
    -_f2(30405369, 16203 << 36), _f2(28904813, 10839 << 35),
    -_f2(5472666, 28841 << 37), -_f2(4538327, 21695 << 35),
    _f2(1010309, 2151 << 36), _f2(1591197, 61387 << 35),
    -_f2(691600, 842821 << 31), _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^22, polynomial in n of order 7
    _f2(2360974, 20517 << 37), -_f2(34168343, 7885 << 38),
    _f2(7988557, 399 << 37), _f2(19220645, 2761 << 39),
    -_f2(19251394, 21847 << 37), _f2(7294617, 7633 << 38),
    -_f2(776533, 25709 << 37), -_f2(127091, 628387 << 32),
    _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^21, polynomial in n of order 8
    -_f2(141970389, 5875 << 38), _f2(152285106, 31 << 35),
    -_f2(85013706, 54665 << 36), _f2(10784519, 74157 << 35),
    _f2(18376223, 22989 << 37), -_f2(9415616, 123045 << 35),
    -_f2(3707065, 39939 << 36), _f2(5266887, 19945 << 35),
    -_f2(1601936, 974407 << 31), _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^20, polynomial in n of order 9
    _f2(174732199, 7199 << 38), -_f2(91781661, 1783 << 41),
    -_f2(22947906, 1007 << 38), _f2(109147441, 451 << 39),
    -_f2(126268040, 11085 << 38), _f2(86262862, 2409 << 40),
    -_f2(35185382, 1435 << 38), _f2(6220582, 7041 << 39),
    _f2(846498, 391 << 38), -_f2(421214, 84365 << 33),
    _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^19, polynomial in n of order 10
    _f2(100447726, 5039 << 36), -_f2(149758021, 121873 << 35),
    _f2(181554380, 1939 << 38), -_f2(171878757, 123903 << 35),
    _f2(113962110, 29289 << 36), -_f2(29967290, 80205 << 35),
    -_f2(40353928, 13613 << 37), _f2(68655180, 86853 << 35),
    -_f2(57285125, 57949 << 36), _f2(28886745, 8439 << 35),
    -_f2(6846764, 2025561 << 31), _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^18, polynomial in n of order 11
    _f2(9163438, 32371 << 37), -_f2(22188557, 9937 << 38),
    _f2(45681407, 15569 << 37), -_f2(80085759, 21 << 40),
    _f2(119267529, 32127 << 37), -_f2(149784698, 12951 << 38),
    _f2(156408668, 30941 << 37), -_f2(132494258, 5045 << 39),
    _f2(87286969, 10059 << 37), -_f2(41608633, 10397 << 38),
    _f2(12599797, 16681 << 37), -_f2(1793721, 181577 << 32),
    _f2(143563428310, 0x9e40b2104d5e3),
    # C4[17], coeff of eps^17, polynomial in n of order 12
    _f2(152058, 3531 << 37), -_f2(566838, 109449 << 35),
    _f2(1788421, 65069 << 36), -_f2(4828739, 49907 << 35),
    _f2(11241509, 10969 << 38), -_f2(22666304, 107837 << 35),
    _f2(39633653, 283 << 36), -_f2(59939783, 113319 << 35),
    _f2(77715030, 3737 << 37), -_f2(84609105, 47985 << 35),
    _f2(73498818, 52617 << 36), -_f2(43049308, 20443 << 35),
    _f2(11659187, 1311925 << 31), _f2(143563428310, 0x9e40b2104d5e3),
    # C4[18], coeff of eps^29, polynomial in n of order 0
    _f(35 << 34), _f(0x29845c2bcb5c10d7),
    # C4[18], coeff of eps^28, polynomial in n of order 1
    _f2(3628, 18373 << 37), -_f2(4063, 232509 << 34),
    _f2(3097286791, 0x8a812bfedbe75),
    # C4[18], coeff of eps^27, polynomial in n of order 2
    _f2(435730, 613 << 39), -_f2(110987, 3811 << 38), _f(489021323 << 34),
    _f2(21681007540, 0xc98833f803533),
    # C4[18], coeff of eps^26, polynomial in n of order 3
    -_f2(762945, 31179 << 36), _f2(988791, 87 << 37),
    _f2(550009, 38375 << 36), -_f2(343815, 323189 << 33),
    _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^25, polynomial in n of order 4
    _f2(1063744, 27 << 41), -_f2(7897635, 7767 << 39),
    _f2(4613149, 699 << 40), -_f2(833936, 93 << 39),
    -_f2(28054, 94387 << 35), _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^24, polynomial in n of order 5
    _f2(25578507, 4379 << 38), -_f2(3209600, 1553 << 40),
    -_f2(4577572, 5923 << 38), _f2(702466, 1583 << 39),
    _f2(1586031, 287 << 38), -_f2(651636, 122639 << 35),
    _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^23, polynomial in n of order 6
    -_f2(32324739, 1815 << 40), _f2(3520775, 1207 << 39),
    _f2(19946468, 259 << 41), -_f2(17787966, 4575 << 39),
    _f2(6336978, 3235 << 40), -_f2(589727, 5685 << 39),
    -_f2(125505, 20667 << 35), _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^22, polynomial in n of order 7
    _f2(138884729, 22203 << 36), -_f2(69168625, 14473 << 37),
    _f2(3249237, 4577 << 36), _f2(18436830, 10301 << 38),
    -_f2(7840055, 31609 << 36), -_f2(4091705, 30595 << 37),
    _f2(5021146, 27565 << 36), -_f2(1488082, 115687 << 33),
    _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^21, polynomial in n of order 8
    -_f2(66334778, 1299 << 41), -_f2(40377625, 16285 << 38),
    _f2(111882749, 4839 << 39), -_f2(118325119, 4711 << 38),
    _f2(76858390, 3693 << 40), -_f2(29952902, 3569 << 38),
    _f2(4790818, 4941 << 39), _f2(921311, 4421 << 38),
    -_f2(386621, 181821 << 34), _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^20, polynomial in n of order 9
    -_f2(151679112, 16629 << 37), _f2(174648786, 1667 << 40),
    -_f2(156892091, 15835 << 37), _f2(96799837, 4169 << 38),
    -_f2(17949188, 6721 << 37), -_f2(43885384, 7293 << 39),
    _f2(66080580, 25305 << 37), -_f2(53357084, 1853 << 38),
    _f2(26599572, 17011 << 37), -_f2(6287689, 169979 << 34),
    _f2(151767052785, 0x82b96bc817465),
    # C4[18], coeff of eps^19, polynomial in n of order 10
    -_f2(8594193, 5169 << 39), _f2(16702080, 5475 << 38),
    -_f2(27882498, 1245 << 41), _f2(39843622, 14413 << 38),
    -_f2(48340851, 951 << 39), _f2(49066184, 11639 << 38),
    -_f2(40627946, 3165 << 40), _f2(26296855, 15713 << 38),
    -_f2(12371894, 1597 << 39), _f2(3711568, 4235 << 38),
    -_f2(524991, 147555 << 34), _f2(50589017595, 0x2b9323ed5d177),
    # C4[18], coeff of eps^18, polynomial in n of order 11
    -_f2(768539, 29011 << 36), _f2(2243105, 18035 << 37),
    -_f2(5671852, 39713 << 36), _f2(12494515, 7255 << 39),
    -_f2(24051943, 5231 << 36), _f2(40468348, 22085 << 37),
    -_f2(59307062, 46653 << 36), _f2(74994737, 5975 << 38),
    -_f2(80108014, 59787 << 36), _f2(68664012, 25623 << 37),
    -_f2(39899358, 51033 << 36), _f2(10762327, 20443 << 33),
    _f2(151767052785, 0x82b96bc817465),
    # C4[19], coeff of eps^29, polynomial in n of order 0
    -_f(69697 << 34), _f2(220556, 0x6c98ea537e51f),
    # C4[19], coeff of eps^28, polynomial in n of order 1
    -_f(1238839 << 41), _f(675087 << 35),
    _f2(141943813, 0x222cc7846d81),
    # C4[19], coeff of eps^27, polynomial in n of order 2
    _f2(876102, 3999 << 40), _f2(573743, 1451 << 39),
    -_f2(328615, 14973 << 34), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^26, polynomial in n of order 3
    -_f2(7739083, 17 << 46), _f2(4186838, 53 << 45), -_f2(704448, 1 << 46),
    -_f2(33249, 11241 << 37), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^25, polynomial in n of order 4
    -_f2(1360864, 133 << 42), -_f2(4500609, 2667 << 40),
    _f2(427896, 299 << 41), _f2(1570943, 1191 << 40),
    -_f2(614728, 45789 << 35), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^24, polynomial in n of order 5
    -_f2(379105, 631 << 42), _f2(20252634, 139 << 44),
    -_f2(16388211, 705 << 42), _f2(5510947, 339 << 43),
    -_f2(439601, 699 << 42), -_f2(122601, 56745 << 36),
    _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^23, polynomial in n of order 6
    -_f2(55355388, 567 << 41), -_f2(2520461, 2117 << 40),
    _f2(18017708, 147 << 42), -_f2(6413373, 771 << 40),
    -_f2(4373212, 61 << 41), _f2(4784182, 2079 << 40),
    -_f2(1386197, 54485 << 35), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^22, polynomial in n of order 7
    -_f2(54112477, 29 << 46), _f2(112419812, 35 << 45),
    -_f2(110372726, 9 << 46), _f2(68510282, 53 << 46),
    -_f2(25556330, 19 << 46), _f2(3652507, 1 << 45), _f2(962676, 17 << 46),
    -_f2(355362, 30093 << 37), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^21, polynomial in n of order 8
    _f2(166723371, 209 << 42), -_f2(142457721, 7469 << 39),
    _f2(81530379, 2787 << 40), -_f2(7977897, 3383 << 39),
    -_f2(46298043, 1775 << 41), _f2(63437092, 799 << 39),
    -_f2(49803454, 3807 << 40), _f2(24585849, 2581 << 39),
    -_f2(5799325, 105875 << 34), _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^20, polynomial in n of order 9
    _f2(54095236, 1729 << 41), -_f2(86448328, 33 << 44),
    _f2(119042325, 527 << 41), -_f2(140012701, 875 << 42),
    _f2(138519104, 1133 << 41), -_f2(112357061, 257 << 43),
    _f2(71568963, 1275 << 41), -_f2(33272498, 441 << 42),
    _f2(9897515, 729 << 41), -_f2(1391838, 12705 << 35),
    _f2(159970677260, 0x6732257fe12e7),
    # C4[19], coeff of eps^19, polynomial in n of order 10
    _f2(2731650, 3225 << 40), -_f2(6520331, 5423 << 39),
    _f2(13678206, 885 << 42), -_f2(25266687, 5569 << 39),
    _f2(41073925, 3215 << 40), -_f2(58519302, 7091 << 39),
    _f2(72351138, 181 << 41), -_f2(75968694, 8133 << 39),
    _f2(64333849, 3333 << 40), -_f2(37115682, 4791 << 39),
    _f2(9974839, 182105 << 34), _f2(159970677260, 0x6732257fe12e7),
    # C4[20], coeff of eps^29, polynomial in n of order 0
    _f(1 << 39), _f2(386445, 0x44b61aebc827),
    # C4[20], coeff of eps^28, polynomial in n of order 1
    _f2(3670, 3431 << 40), -_f(63923791 << 37),
    _f2(1044560880, 0x57ec63f8653c9),
    # C4[20], coeff of eps^27, polynomial in n of order 2
    _f2(165149, 453 << 43), -_f2(25858, 471 << 42), -_f(26276299 << 38),
    _f2(7311926162, 0x6776bbcac4a7f),
    # C4[20], coeff of eps^26, polynomial in n of order 3
    -_f2(4343033, 595 << 42), _f2(185313, 303 << 43),
    _f2(1548473, 271 << 42), -_f2(580654, 777 << 40),
    _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^25, polynomial in n of order 4
    _f2(20236427, 149 << 44), -_f2(15067334, 133 << 42),
    _f2(4797544, 165 << 43), -_f2(318599, 375 << 42),
    -_f2(118861, 3875 << 38), _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^24, polynomial in n of order 5
    -_f2(6870833, 1979 << 41), _f2(17282399, 281 << 43),
    -_f2(5135975, 189 << 41), -_f2(4572111, 263 << 42),
    _f2(4557653, 1537 << 41), -_f2(1294702, 4061 << 38),
    _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^23, polynomial in n of order 6
    _f2(111332564, 131 << 43), -_f2(102611836, 439 << 42),
    _f2(61113705, 49 << 44), -_f2(21849131, 865 << 42),
    _f2(2742318, 257 << 43), _f2(980372, 533 << 42),
    -_f2(327159, 8391 << 38), _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^22, polynomial in n of order 7
    -_f2(128743521, 979 << 42), _f2(67998970, 481 << 43),
    _f2(279122, 855 << 42), -_f2(47847734, 245 << 44),
    _f2(60794248, 257 << 42), -_f2(46583621, 181 << 43),
    _f2(22803394, 43 << 42), -_f2(5369928, 2229 << 40),
    _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^21, polynomial in n of order 8
    -_f2(88564699, 121 << 45), _f2(117949702, 533 << 42),
    -_f2(134881895, 27 << 43), _f2(130376590, 239 << 42),
    -_f2(103788735, 57 << 44), _f2(65154071, 233 << 42),
    -_f2(29963298, 393 << 43), _f2(8844588, 195 << 42),
    -_f2(1237189, 6873 << 38), _f2(168174301735, 0x4baadf37ab169),
    # C4[20], coeff of eps^20, polynomial in n of order 9
    -_f2(7362630, 999 << 40), _f2(14785858, 137 << 43),
    -_f2(26321377, 9 << 40), _f2(41483460, 1083 << 41),
    -_f2(57615917, 1643 << 40), _f2(69797568, 521 << 42),
    -_f2(72155594, 1933 << 40), _f2(60438019, 617 << 41),
    -_f2(34641303, 3055 << 40), _f2(9278920, 21175 << 37),
    _f2(168174301735, 0x4baadf37ab169),
    # C4[21], coeff of eps^29, polynomial in n of order 0
    -_f(2017699 << 39), _f2(144690669, 0x92d5d14b2b5b9),
    # C4[21], coeff of eps^28, polynomial in n of order 1
    -_f2(21806, 31 << 47), -_f(1751493 << 42),
    _f2(7668605487, 0x6644548ff9f4d),
    # C4[21], coeff of eps^27, polynomial in n of order 2
    -_f(610053 << 43), _f2(66113, 223 << 42), -_f2(23877, 14131 << 38),
    _f2(7668605487, 0x6644548ff9f4d),
    # C4[21], coeff of eps^26, polynomial in n of order 3
    -_f2(601427, 223 << 44), _f2(181759, 65 << 45), -_f2(9602, 5 << 44),
    -_f2(4983, 2721 << 39), _f2(7668605487, 0x6644548ff9f4d),
    # C4[21], coeff of eps^25, polynomial in n of order 4
    _f2(16348405, 227 << 44), -_f2(4001511, 795 << 42),
    -_f2(4705038, 397 << 43), _f2(4342393, 855 << 42),
    -_f2(1212256, 1051 << 38), _f2(176377926210, 0x302398ef74feb),
    # C4[21], coeff of eps^24, polynomial in n of order 5
    -_f2(95167920, 19 << 45), _f2(54565817, 7 << 47),
    -_f2(18712410, 5 << 45), _f2(2011897, 15 << 46), _f2(981374, 25 << 45),
    -_f2(301721, 597 << 40), _f2(176377926210, 0x302398ef74feb),
    # C4[21], coeff of eps^23, polynomial in n of order 6
    _f2(56043535, 133 << 43), _f2(7101303, 759 << 42),
    -_f2(48732132, 249 << 44), _f2(58197907, 161 << 42),
    -_f2(43660867, 425 << 43), _f2(21217809, 619 << 42),
    -_f2(4990122, 11039 << 38), _f2(176377926210, 0x302398ef74feb),
    # C4[21], coeff of eps^22, polynomial in n of order 7
    _f2(38792824, 189 << 44), -_f2(43241527, 125 << 45),
    _f2(40920531, 151 << 44), -_f2(32022608, 39 << 46),
    _f2(19836099, 97 << 44), -_f2(9032168, 63 << 45),
    _f2(2647359, 187 << 44), -_f2(368524, 4161 << 39),
    _f2(58792642070, 0x100bdda526ff9),
    # C4[21], coeff of eps^21, polynomial in n of order 8
    _f2(15813930, 121 << 45), -_f2(27228018, 205 << 42),
    _f2(41726053, 443 << 43), -_f2(56628215, 983 << 42),
    _f2(67341662, 57 << 44), -_f2(68636694, 193 << 42),
    _f2(56918234, 105 << 43), -_f2(32430156, 715 << 42),
    _f2(8660325, 15343 << 38), _f2(176377926210, 0x302398ef74feb),
    # C4[22], coeff of eps^29, polynomial in n of order 0
    -_f(229 << 43), _f2(2018939, 0x935060fc493cd),
    # C4[22], coeff of eps^28, polynomial in n of order 1
    _f2(64733, 61 << 46), -_f2(22613, 493 << 43),
    _f2(8025284812, 0x6511ed552f41b),
    # C4[22], coeff of eps^27, polynomial in n of order 2
    _f2(158513, 3 << 48), -_f2(6162, 29 << 47), -_f2(4786, 487 << 43),
    _f2(8025284812, 0x6511ed552f41b),
    # C4[22], coeff of eps^26, polynomial in n of order 3
    -_f2(130438, 301 << 43), -_f2(208062, 47 << 44), _f2(179942, 497 << 43),
    -_f2(49466, 167 << 40), _f2(8025284812, 0x6511ed552f41b),
    # C4[22], coeff of eps^25, polynomial in n of order 4
    _f2(2120438, 3 << 47), -_f2(697803, 39 << 45), _f2(61914, 3 << 46),
    _f2(42203, 115 << 45), -_f2(12120, 543 << 41),
    _f2(8025284812, 0x6511ed552f41b),
    # C4[22], coeff of eps^24, polynomial in n of order 5
    _f2(12722577, 33 << 44), -_f2(49104495, 51 << 46),
    _f2(55677556, 71 << 44), -_f2(41002422, 115 << 45),
    _f2(19800840, 109 << 44), -_f2(4652345, 837 << 41),
    _f2(184581550685, 0x149c52a73ee6d),
    # C4[22], coeff of eps^23, polynomial in n of order 6
    -_f2(124610244, 57 << 46), _f2(115654934, 113 << 45),
    -_f2(89096506, 19 << 47), _f2(54518354, 119 << 45),
    -_f2(24598996, 19 << 46), _f2(7163443, 125 << 45),
    -_f2(992759, 1841 << 41), _f2(184581550685, 0x149c52a73ee6d),
    # C4[22], coeff of eps^22, polynomial in n of order 7
    -_f2(27999005, 155 << 43), _f2(41827085, 121 << 44),
    -_f2(55581037, 1 << 43), _f2(64987058, 83 << 45),
    -_f2(65383321, 103 << 43), _f2(53725829, 211 << 44),
    -_f2(30444636, 461 << 43), _f2(8107539, 715 << 40),
    _f2(184581550685, 0x149c52a73ee6d),
    # C4[23], coeff of eps^29, polynomial in n of order 0
    -_f2(4289, 21 << 43), _f2(1676392827, 0x7a5fe79ee0e95),
    # C4[23], coeff of eps^28, polynomial in n of order 1
    -_f(1351 << 51), -_f(234789 << 44), _f2(1676392827, 0x7a5fe79ee0e95),
    # C4[23], coeff of eps^27, polynomial in n of order 2
    -_f2(209744, 1 << 50), _f2(171585, 3 << 49), -_f2(46526, 469 << 43),
    _f2(8381964137, 0x63df861a648e9),
    # C4[23], coeff of eps^26, polynomial in n of order 3
    -_f2(599194, 1 << 51), _f2(41297, 0), _f2(41388, 1 << 51),
    -_f2(11218, 97 << 45), _f2(8381964137, 0x63df861a648e9),
    # C4[23], coeff of eps^25, polynomial in n of order 4
    -_f2(2134087, 7 << 49), _f2(2315275, 31 << 47), -_f2(1677358, 15 << 48),
    _f2(805613, 21 << 47), -_f2(189149, 1213 << 41),
    _f2(8381964137, 0x63df861a648e9),
    # C4[23], coeff of eps^24, polynomial in n of order 5
    _f2(4740508, 1 << 49), -_f2(3599518, 1 << 51), _f2(2177844, 7 << 49),
    -_f2(974429, 1 << 50), _f2(282071, 5 << 49), -_f2(38931, 779 << 42),
    _f2(8381964137, 0x63df861a648e9),
    # C4[23], coeff of eps^23, polynomial in n of order 6
    _f2(1817763, 3 << 48), -_f2(2369306, 23 << 47), _f2(2727592, 1 << 49),
    -_f2(2711734, 1 << 47), _f2(2209561, 1 << 48), -_f2(1245816, 11 << 47),
    _f2(330919, 1979 << 41), _f2(8381964137, 0x63df861a648e9),
    # C4[24], coeff of eps^29, polynomial in n of order 0
    -_f(1439 << 46), _f2(44813556, 0x37a4fd885dffd),
    # C4[24], coeff of eps^28, polynomial in n of order 1
    _f2(32742, 3 << 50), -_f2(8770, 21 << 47),
    _f2(1747728692, 0x7a229fc651f8b),
    # C4[24], coeff of eps^27, polynomial in n of order 2
    _f2(4928, 1 << 51), _f2(8067, 1 << 50), -_f2(2080, 43 << 46),
    _f2(1747728692, 0x7a229fc651f8b),
    # C4[24], coeff of eps^26, polynomial in n of order 3
    _f2(2214330, 0), -_f2(1581120, 0), _f2(755790, 0),
    -_f2(177363, 7 << 47), _f2(8738643462, 0x62ad1edf99db7),
    # C4[24], coeff of eps^25, polynomial in n of order 4
    -_f2(1116955, 0), _f2(668788, 3 << 50), -_f2(296917, 1 << 51),
    _f2(85476, 1 << 50), -_f2(11752, 63 << 46),
    _f2(2912881154, 0x20e45f9fddf3d),
    # C4[24], coeff of eps^24, polynomial in n of order 5
    -_f2(2320992, 3 << 48), _f2(2634056, 1 << 50), -_f2(2590155, 5 << 48),
    _f2(2094168, 1 << 49), -_f2(1175298, 7 << 48), _f2(311454, 11 << 45),
    _f2(8738643462, 0x62ad1edf99db7),
    # C4[25], coeff of eps^29, polynomial in n of order 0
    -_f(3707 << 46), _f2(12720731, 0x2bd144a4925ef),
    # C4[25], coeff of eps^28, polynomial in n of order 1
    _f(301 << 53), -_f(2379 << 48), _f2(139928042, 0xe1fdf3124a145),
    # C4[25], coeff of eps^27, polynomial in n of order 2
    -_f2(298603, 1 << 51), _f2(142145, 1 << 50), -_f2(33346, 63 << 46),
    _f2(1819064557, 0x79e557edc3081),
    # C4[25], coeff of eps^26, polynomial in n of order 3
    _f2(370617, 0), -_f2(163358, 0), _f2(46787, 0), -_f2(6410, 23 << 47),
    _f2(1819064557, 0x79e557edc3081),
    # C4[25], coeff of eps^25, polynomial in n of order 4
    _f2(508963, 0), -_f2(495426, 3 << 50), _f2(397689, 1 << 51),
    -_f2(222238, 1 << 50), _f2(58764, 59 << 46),
    _f2(1819064557, 0x79e557edc3081),
    # C4[26], coeff of eps^29, polynomial in n of order 0
    -_f(1 << 49), _f2(131359, 0xe834f81ee20c1),
    # C4[26], coeff of eps^28, polynomial in n of order 1
    _f2(10305, 0), -_f2(2417, 1 << 49), _f2(145415417, 0x1d0ced8b7a293),
    # C4[26], coeff of eps^27, polynomial in n of order 2
    -_f2(11556, 0), _f2(3294, 0), -_f(3599 << 49),
    _f2(145415417, 0x1d0ced8b7a293),
    # C4[26], coeff of eps^26, polynomial in n of order 3
    -_f2(36490, 1 << 51), _f2(29097, 0), -_f2(16195, 1 << 51),
    _f2(4273, 13 << 48), _f2(145415417, 0x1d0ced8b7a293),
    # C4[27], coeff of eps^29, polynomial in n of order 0
    -_f(2029 << 49), _f2(16766976, 0xd0e6a80084b19),
    # C4[27], coeff of eps^28, polynomial in n of order 1
    _f(7 << 56), -_f(61 << 50), _f2(5588992, 0x45a238002c3b3),
    # C4[27], coeff of eps^27, polynomial in n of order 2
    _f2(3080, 0), -_f(427 << 54), _f(3599 << 49),
    _f2(16766976, 0xd0e6a80084b19),
    # C4[28], coeff of eps^29, polynomial in n of order 0
    -_f(1 << 53), _f2(827461, 0x318a62b8e0a5b),
    # C4[28], coeff of eps^28, polynomial in n of order 1
    -_f(29 << 55), _f(61 << 52), _f2(2482383, 0x949f282aa1f11),
    # C4[29], coeff of eps^29, polynomial in n of order 0
    _f(1 << 53), _f2(88602, 0xec373d36a45df),  # PYCHOK exported
)  # 5425


# **) MIT License
#
# Copyright (C) 2016-2023 -- mrJean1 at Gmail -- All Rights Reserved.
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.
